##########################################
#TD_main.py
'''
SOLVE AND EXAMINE
A SIMPLE 1D 
DIFFUSION PROBLEM:
dC/dt = K d^2C/dx^2

With a time-varying 
diffusion coefficient:
K = K(t), 

K can have spatial structure
if it is prescribed

USE IMPLICIT TIME-STEPPING
AND CENTERED SPACE-DIFFERENCING 


SYNTAX OF THE CODE IS RELATIVE TO
T^3W PROBLEM, SO IT IS WRITTEN
WITH A VERTICAL GRID (z) AND
THE QUANTITY BEING DIFFUSED IS THOUGHT OF
AS A VELOCITY (v)

dv/dt = K d^2v/dz^2
'''
# DANIEL DAUHAJRE UCLA APRIL 2016
###########################################
################################
import os
import sys
import numpy as np
import scipy as sp
from pylab import *
import matplotlib.pyplot as plt
from scipy.sparse import *
from scipy.sparse.linalg import spsolve
from netCDF4 import Dataset
################################


    	#############################################################
			# PROBLEM SETUP	
'''
			-Create grid (spatial + temporal)
			-Create/setup temporal structure of diffusion coefficient
                        -Create initial condition for v(z,t=0)
'''
	################################################################

# SET PARAMETERS BY CALLING TD_params.py
#execfile('./src_code/TD_params.py') # for Python 2.7
exec(open('./src_code/TD_params.py').read()) # changed by Baird, for Python 3.5

print('		################################')
print('		     RUN_ID: ' + run_ID)
print('		################################')

print(' ')
print(' ')
print('################################')
print('H = ' +str(H) + 'm')
print('dz = ' + str(dz) + 'm')
print('tend_days: ' + str(tend_days))
print('Kv_time_choice: ' + Kv_time_choice)
print('v_IC_choice: ' + v_IC_choice)
print('################################')

##########################
# SPATIAL GRID
'''
H --> total depth
dz --> vertical spacing
z_r --> grid (rho-levels)
z_w --> grid(w-levels)
N --> number of vertical (rho) levels
'''
############################
z_r = np.arange(-H+(dz/2.),dz/2.,dz)
z_w = np.arange(-H,0+dz,dz) 
N = len(z_r)
################################
# TEMPORAL GRID
'''
t1 ---> length of simulation (units = seconds)
nt --> number of time steps
dt --> time step (seconds)
'''
################################
t1 = 86400 * tend_days
tsteps = np.arange(0,t1,dt)
tsteps_days = tsteps / 86400.
nt = len(tsteps)

num_days = (int(nt/dt)) + 1
len_day = nt / num_days
################################


##################################
# DIFFUSION COEFFICIENT
'''
Kv    ---> vertical mixing coefficient (m^2/s)
Kv0   ---> base amplitude of vertical mixing


DIFFERENT CHOICES FOR TEMPORAL SHAPE
OF MIXING (Kv_time_choice)
1) 'const' ---> constant in time
2) 'sine'--> sine wave

ADD more cases if you like
'''
##################################
Kv_nt = np.zeros([nt,N+1])
if Kv_time_choice == 'const':
   Kv_nt[:,:] = Kv0
if Kv_time_choice == 'sine':  
   for k in range(N+1):
       Kv_nt[:,k] = -(Kdiff/2)*np.sin( (omega*tsteps) + phase_shift) + (Kv0 - (Kdiff/2))



##########################################
# INITIAL CONDITION FOR v(z,t=0)
'''
BASIC FILAMENT PROFILE OF TRANSVERSE
VELOCITY IS THAT OF SYMMETRIC ABOUT A
ZERO-CROSSING IN THE VERTICAL
'''
########################################
v_nt = np.zeros([nt,N])
if v_IC_choice == 'SYMM_ZERO':
   Hscale = (H/2.+z_r)/H
   v_nt[0,:] = (Hscale * v0)        
   # SET B.C.S
   v_nt[0,-1] = v_nt[0,-2]
   v_nt[0,0] = v_nt[0,1]

##########################################
# FORCING F(t,z)
'''
RHS FORCING TERM IF WANTED
'''
########################################
F_nt = np.zeros([nt,N])
F_nt[:,:] = F0



#####################################################################################

    	#############################################################
			# TIME-STEPPING	
'''
			- SOLVE FOR NEW v AT EACH TIME STEP
                          WITH IMPLICIT TIME-STEP FOR THE 
                          TRIDIAGONAL SYSTEM USING
                          SPARSE MATRIX LIBRARY
'''
	################################################################

def solve_TD_implicit(v_n_1,Kv_n,z_r,z_w,dt,F_n):
    """
    Function that solves 
    diffusion equation for 
    each time step using 
    tridiagonal matrix inversion

    v_n_1 --> v(t=n-1)
    Kv_n --> Kv(t=n)
    z_w   --> vertical grid (w-levels)
    z_r   ---> vertical grid(rho-levels)
    dt ---> time step (seconds)
    F_n --> forcing term (F(t=n))
    """
    dz = z_r[1:] - z_r[:-1]
    Hz = z_w[1:] - z_w[:-1]

    ###################################
    # SET UP COEFFICIENT MATRIX
    ###################################
    A = lil_matrix((N,N)) 
    R = np.zeros(N)

    # BOTTOM
    idx = 0
    
    D6 = Hz[0] + (dt * (Kv_n[1]/dz[0]))
    D7 = (dt * (Kv_n[1] / dz[0]))
   
    A[idx,idx] = D6
    A[idx,idx+1] = -D7
    
    #INTERIOR
    for k in range(1,N-1):
        D3 = dt * (Kv_n[k]/dz[k-1])
        D4 = Hz[k] + ( dt* (Kv_n[k]/dz[k-1])) + (dt*(Kv_n[k+1]/dz[k]))
        D5 = dt * (Kv_n[k+1]/dz[k])
        
        idx = k
        A[idx,idx] = D4
        A[idx,idx-1] = -D3
        A[idx,idx+1] = -D5

    #TOP
    D2 = Hz[-1] + (dt*(Kv_n[N-1]/dz[-1]))
    D1 = dt * (Kv_n[N-1]/dz[-1]) 
    
    idx = N-1
    A[idx,idx] = D2
    A[idx,idx-1] = -D1  

    ################################################


    #########################
    # SET UP R.H.S. VECTOR
    '''
    PLACE TOP AND BOTTOM
    OUTSIDE OF LOOP TO ALLOW
    FOR BCS IF NEEDED
    '''
    ##########################
    R[0] = (Hz[0] * v_n_1[0]) + F_n[0] 
    for k in range(1,N-1):
        R[k] = (Hz[k] * v_n_1[k]) + F_n[k]
    R[-1] = (Hz[-1] * v_n_1[-1])  + F_n[-1]

    
    ##############################
    # SOLVE
    #############################
    A = A.tocsr()
    X = spsolve(A,R)

    return X



#############################
# TIME STEP
############################
for n in range(1,nt):  
    if (n+1) % 10 == 0: 
       print ('Solving TD SYSTEM: ' + str( round((float(n+1)/nt)*100)) + '% complete')
    v_nt[n,:] = solve_TD_implicit(v_nt[n-1,:],Kv_nt[n,:],z_r,z_w,dt,F_nt[n,:])



#########################################
# WRITE TO NETCDF FILE
#######################################
# SWITCH TO NETCDF OUTPUT DIRECTORY
dir_name = 'nc_files'
if not os.path.exists(dir_name):
   print ('Creating directory: ' + dir_name)
   os.makedirs(dir_name)
print ('Moving into directory: ' + dir_name)
os.chdir(dir_name)

# DECLARE NETCDF FILE
out_file = Dataset(run_ID + '.nc','w')

#CREATE DIMENSIONS
[nt,N] = v_nt.shape
out_file.createDimension('time',nt)
out_file.createDimension('nz_rho',N)
out_file.createDimension('nz_w',N+1)


#SET GLOBAL ATTRIBUTES
out_file.title           = run_ID
out_file.description     = 'Temporal Diffusion Simulation'
out_file.dt              = dt
out_file.Kv_time_choice  = Kv_time_choice
out_file.v_IC_choice     = v_IC_choice


#DEFINE OUTPUT VARIALBES FOR NETCDF
z_r_out = out_file.createVariable('z_r',dtype('float64').char,('nz_rho'))
setattr(z_r_out,'long_name','z_r grid')

z_w_out = out_file.createVariable('z_w',dtype('float64').char,('nz_w'))
setattr(z_w_out,'long_name','z_w grid')


Kv_out = out_file.createVariable('Kv',dtype('float64').char,('time','nz_w'))
setattr(Kv_out, 'long_name','Vertical mixing (m^2/s)')

v_out = out_file.createVariable('v',dtype('float64').char,('time','nz_rho'))
setattr(v_out, 'long_name','velocity (m/s)')

F_out = out_file.createVariable('F',dtype('float64').char,('time','nz_rho'))
setattr(F_out, 'long_name','forcing term (m/s^2)')


#FILL VARIABLES
z_r_out[:] = z_r
z_w_out[:] = z_w
Kv_out[:,:] = Kv_nt
v_out[:,:] = v_nt
F_out[:,:] = F_nt

# CLOSE FILE
out_file.close()
print('Created and saved netcdf file: ' + run_ID + '.nc')

os.chdir('../')
#################################################################







