#include "cppdefs.h"
      subroutine sg_ubab
#ifdef SG_BBL96
!
!--------------------------------------------------------------------
!  This routine computes wave maximum bottom velocity and excursion
!  from input wind induced wave amplitude and period by solving the
!  linear wave theory dispersion relation for wave number.
!
!  On Input:   (Common block FBBLM in include file "bblm.h")
!
!     Awave    wind induced wave amplitude (m).
!     Pwave    wind induced wave period (s).
!
!  On Output:  (Common block FBBLM in include file "bblm.h")
!
!     Ub       wave maximum bottom horizontal velocity (m/s).
!     Ab       wave bottom excursion amplitude (m).
!
!  Reference:
!
!    Styles, R. and. S.M. Glenn, 1996:  Observation and modeling of
!      sediment transport events in the Middle Atlantic Bight,  8th
!      International conference on Physics of estuaries and coastal
!      seas. Submitted.
!
!--------------------------------------------------------------------
!
      implicit none
# include "param.h"
# include "pconst.h"
# include "bblm.h"
# include "grid.h"
# include "scalars.h"
# include "work.h"
 
      integer Kiter, i, j
      real  Kbh, KboKb0, Kconv, Kdelta
      real  Fwave(Lm,Mm), Kb(Lm,Mm), Kb0(Lm,Mm)
      equivalence (Fwave,a2d), (Kb,b2d), (Kb0,c2d)
 
!
! sg_Keps     Convergence criterion for the computation of bottom
!                  wavenumber via the Newton-Raphson method.
! sg_Kiter    Maximum number of iterations in the computation of
!                  bottom wavenumber.
 
      real sg_Keps
      integer sg_Kiter
      parameter (sg_Keps=1.e-4, sg_Kiter=10)
!
!  Compute wave angular frequency (1/s) at RHO-points.
!
      do j=1,Mm
        do i=1,Lm
          Fwave(i,j)=c2*pi/Pwave(i,j)
        enddo
      enddo
!
!  Compute bottom wavenumber, Kb.
!
!  Compute first guess for wavenumber, Kb0. Use deep water (Kb0*h>1)
!  and shallow water (Kb0*h<1) approximations.
!
      do j=1,Mm
        do i=1,Lm
          Kb0(i,j)=Fwave(i,j)*Fwave(i,j)/g
        enddo
      enddo
      do j=1,Mm
        do i=1,Lm
          if (Kb0(i,j)*h(i,j).lt.1.) then
            Kb(i,j)=Kb0(i,j)
          else
            Kb(i,j)=Fwave(i,j)/sqrt(g*h(i,j))
          endif
        enddo
      enddo
!
!  Compute bottom wave number via Newton-Raphson method.
!  Iterate until convergence.
!
      Kiter=0
      Kconv=sg_Keps
      do while ((Kiter.le.sg_Kiter).and.(Kconv.ge.sg_Keps))
        Kiter=Kiter+1
        do j=1,Mm
          do i=1,Lm
            Kbh=Kb(i,j)*h(i,j)
            KboKb0=Kb(i,j)/Kb0(i,j)
            Kdelta=(1.-KboKb0*tanh(Kbh))/(1.+Kbh*(KboKb0-1./KboKb0))
            Kconv=max(abs(Kdelta),Kconv)
            Kb(i,j)=Kb(i,j)*(1.+Kdelta)
          enddo
        enddo
      enddo
!
!--------------------------------------------------------------------
!  Compute wave maximum bottom velocity and wave bottom excursion
!  amplitude.
!--------------------------------------------------------------------
!
      do j=1,Mm
        do i=1,Lm
          Ab(i,j)=Awave(i,j)/sinh(Kb(i,j)*h(i,j))
          Ub(i,j)=Fwave(i,j)*Ab(i,j)
        enddo
      enddo
#endif SG_BBL96
      return
      end
 
