#include "cppdefs.h"
#ifdef MY25_MIXING
      subroutine my25_vmix (Kv,Kt,Ks,bvf)
!
!--------------------------------------------------------------------
!  This routine computes vertical mixing coefficients for momentum
!  and tracers using the Mellor and Yamada (1982) mixing level 2.5
!  scheme with modifications described in Galperin et al. (1988).
!
!  On Output:
!
!     Kv       vertical viscosity coefficient (m^2/s).
!     Kt       vertical diffusion coefficient for potential
!              temperature (m^2/s).
!     Ks       vertical diffusion coefficient for salinity (m^2/s).
!     bvf      Brunt-Vaisala frequency (1/s^2).
!
!  Calls:   bv_freq, my25_q
!
!  References:
!
!     Mellor, G.L. and T. Yamada, 1982:  Development of turbulence
!       closure model for geophysical fluid problems, Rev. Geophys.
!       Space Phys., 20, 851-875.
!     Galperin, B., L.H. Kantha, S. Hassid, and A.Rosati, 1988:  A
!       quasi-equilibrium  turbulent  energy model for geophysical
!       flows, J. Atmos. Sci., 45, 55-62.
!
!--------------------------------------------------------------------
!
      implicit none
# include "param.h"
# include "pconst.h"
# include "forces.h"
# include "grid.h"
# include "mixing.h"
# include "ocean.h"
# include "scalars.h"
# include "work.h"
 
 
      real my_A1, my_A2, my_B1, my_B2, my_C1, my_E1, my_E2,
     &                  my_Gh0, my_lmax, my_qmin, my_dtfac
      parameter(
     &     my_A1=0.92,    ! Turbulence closure coefficients
     &     my_A2=0.74,    ! associated with Mellor-Yamada (1982)
     &     my_B1=16.6,    ! level 2.5 model
     &     my_B2=10.1,
     &     my_C1=0.08,
     &     my_E1=1.8,
     &     my_E2=1.33,
 
     &     my_Gh0=0.028,  ! Lower bound on Galperin et
                          ! al. (1988) stability function.
 
     &     my_lmax=0.53,  ! Upper bound on the turbulent
                          ! length scale.
 
     &     my_qmin=1.0e-10, ! Lower bound on tubulent energy
                            ! "q2" and "q2l".
 
     &     my_dtfac=0.05) ! Asselin (1972) time filter coefficient.
 
!
      integer i, j, k
      real B1p2o3, E1o2, Sh1, Sh2, Sm1, Sm2, Sm3, eps
      real Ks(Lm,Mm,0:N), Kt(Lm,Mm,0:N), Kv(Lm,Mm,0:N), bvf(Lm,Mm,Nm)
      real    Gh(Lm,Mm), Kq(Lm,Mm), Qdiss(Lm,Mm,Nm), Qprod(Lm,Mm,Nm),
     &        Sh(Lm,Mm,Nm), Sm(Lm,Mm,Nm), bqflx(Lm,Mm), dudz(Lm,Mm),
     &        dvdz(Lm,Mm), ql(Lm,Mm,Nm), shear2(Lm,Mm,Nm),
     &        sqflx(Lm,Mm), Wscale(Lm,Mm)
      parameter (eps=1.0e-10)
      equivalence (Gh,f2d), (Kq,g2d), (Qdiss,a3d), (Qprod,b3d),
     &            (Sh,e3d), (Sm,f3d), (bqflx,d2d), (dudz,c2d),
     &            (dvdz,d2d), (ql,g3d), (shear2,c3d), (sqflx,c2d),
     &            (Wscale,e2d)
      save B1p2o3, E1o2, Sh1, Sh2, Sm1, Sm2, Sm3
!
      real  a1, a2, a3, a4, av2, av4
# include "avg.h"
!
!--------------------------------------------------------------------
c  On first pass, set parameters.
c
c  M-Y (1982):  A1=0.92,A2=0.74,B1=16.6,
!               B2=10.1,C1=0.08,E1=1.8,E2=1.33
!--------------------------------------------------------------------
c
c  Set coefficients used to compute stability functions for tracer
c  and momentum.
c
      if (iic.eq.ntstart) then
c
c  Set constant coefficients for stability functions.
c
        B1p2o3=my_B1**c2r3
        E1o2=0.5*my_E1
        Sh1=my_A2*(1.-(6.*my_A1/my_B1))
        Sh2=3.*my_A2*my_B2+18.*my_A1*my_A2
        Sm1=18.*my_A1*my_A1+c9*my_A1*my_A2
        Sm2=my_A1*(1.-(c3*my_C1)-(c6*my_A1/my_B1))
        Sm3=c9*my_A1*my_A2
c
c  Initialize two-time level indices for Leapfrog timestepping.
c
        mold=1
        mnew=2
c
c  Initialize turbulent energy variables.
c
        do k=0,N
          do j=0,M
            do i=0,L
              q2 (i,j,k,mold)=my_qmin
              q2 (i,j,k,mnew)=my_qmin
              q2l(i,j,k,mold)=my_qmin
              q2l(i,j,k,mnew)=my_qmin
            enddo
          enddo
        enddo
c
c  Initialize turbulent energy length scale.
c
        do k=1,Nm
          do j=1,Mm
            do i=1,Lm
              Lscale(i,j,k)=eps
            enddo
          enddo
        enddo
c
c  Initialix vertical mixing coefficients for turbulent energy.
c
        do k=0,N
          do j=1,Mm
            do i=1,Lm
              Akq(i,j,k)=Akq_bak
            enddo
          enddo
        enddo
      endif
c
!--------------------------------------------------------------------
c  Set parameters associated with Leapfrog time-stepping of the
c  turbulent energy equations.
!--------------------------------------------------------------------
c
      if (iic.eq.ntstart) then
        mstp=mold
      else
        mstp=mnew
      endif
      mrhs=mold
c
!--------------------------------------------------------------------
c  Compute shear and buoyant production of turbulent energy, Qprod:
c
! 2 * Hz * {Akv * [(du/ds)^2 + (dv/ds)^2] + Akt * [g/rho0 * drho/dz]}
!--------------------------------------------------------------------
c
c  Compute horizontal velocity shear squared (1/s^2) at horizontal
c  RHO-points and vertical W-points.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            dudz(i,j)=av2(u(i  ,j,k+1)-u(i  ,j,k),
     &                    u(i+1,j,k+1)-u(i+1,j,k))/
     &                (z_r(i,j,k+1)-z_r(i,j,k))
            dvdz(i,j)=av2(v(i,j  ,k+1)-v(i,j  ,k),
     &                    v(i,j+1,k+1)-v(i,j+1,k))/
     &                (z_r(i,j,k+1)-z_r(i,j,k))
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            shear2(i,j,k)=dudz(i,j)*dudz(i,j)+dvdz(i,j)*dvdz(i,j)
          enddo
        enddo
      enddo
c
c  Compute Brunt-Vaisala frequency (1/s^2).
c
      call bv_freq (bvf)
c
c  Compute shear and bouyant production of turbulent energy (m^3/s^3) at
c  horizontal RHO-points and vertical W-points.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            Qprod(i,j,k)=c2*av2(Hz(i,j,k),Hz(i,j,k+1))*
     &                   (Akv(i,j,k)*shear2(i,j,k)-
     &                    Akt(i,j,k,1)*bvf(i,j,k))
          enddo
        enddo
      enddo
!
!--------------------------------------------------------------------
!  Compute dissipation of turbulent energy (m^3/s^3), Qdiss.
!---------------------------------------------------------------------
!
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            Qdiss(i,j,k)=c2*av2(Hz(i,j,k),Hz(i,j,k+1))*
     &                   q2(i,j,k,mold)*sqrt(q2(i,j,k,mold))/
     &                   (my_B1*Lscale(i,j,k))
          enddo
        enddo
      enddo
!
!--------------------------------------------------------------------
!  Solve turbulent kinetic energy "q2" (m^2/s^2) equation.
!--------------------------------------------------------------------
!
!  Set surface boundary condition for [Kq/Hz d(q2)/d(s)] at
!  RHO-points (m^3/s^2).
!
      do j=1,Mm
        do i=1,Lm
          sqflx(i,j)=B1p2o3*Hz(i,j,N)*
     &               sqrt(av2(sustr(i,j),sustr(i+1,j))**2+
     &                    av2(svstr(i,j),svstr(i,j+1))**2)
        enddo
      enddo
c
c  Set bottom boundary condition for [Kq/Hz d(q2)/d(s)] at RHO-points
c  (m^3/s^2).
c
      do j=1,Mm
        do i=1,Lm
          bqflx(i,j)=B1p2o3*Hz(i,j,1)*
     &               sqrt(av2(bustr(i,j),bustr(i+1,j))**2+
     &                    av2(bvstr(i,j),bvstr(i,j+1))**2)
        enddo
      enddo
      call my25_q (q2,Qprod,Qdiss,sqflx,bqflx)
!
!--------------------------------------------------------------------
!  Solve turbulent kinetic energy times length scale "q2l" (m^3/s^2)
!  equation.
!--------------------------------------------------------------------
!
!  Scale shear and bouyant production of turbulent energy (m^4/s^3).
!
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            Qprod(i,j,k)=E1o2*Lscale(i,j,k)*Qprod(i,j,k)
          enddo
        enddo
      enddo
c
c  Scale dissipation of turbulent energy (m^4/s^3).
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            a2d(i,j)=(z_w(i,j,N)-z_r(i,j,k))*(z_r(i,j,k)-z_w(i,j,0))
     &                                      /(z_w(i,j,N)-z_r(i,j,0))
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            Wscale(i,j)=Lscale(i,j,k)/(vonKar*a2d(i,j))
            Qdiss(i,j,k)=0.5*Qdiss(i,j,k)*Lscale(i,j,k)
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            Qdiss(i,j,k)=(1.+my_E2*Wscale(i,j)*Wscale(i,j))
      &                                      *Qdiss(i,j,k)
          enddo
        enddo
      enddo
!
!  Set surface boundary condition for [Kq/Hz d(q2l)/d(s)] at
!  RHO-points (m^4/s^2)
!
      do j=1,Mm
        do i=1,Lm
          sqflx(i,j)=0.
        enddo
      enddo
c
c  Set bottom boundary condition for [Kq/Hz d(q2l)/d(s)].
c
      do j=1,Mm
        do i=1,Lm
          bqflx(i,j)=0.
        enddo
      enddo
      call my25_q (q2l,Qprod,Qdiss,sqflx,bqflx)
c
c-------------------------------------------------------------------
c  Compute turbulent length scale (m).
c-------------------------------------------------------------------
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            a2d(i,j)=my_lmax*sqrt(q2(i,j,k,mnew)/(max(eps,bvf(i,j,k))))
            b2d(i,j)=q2l(i,j,k,mnew)/q2(i,j,k,mnew)
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            Lscale(i,j,k)=min(a2d(i,j),b2d(i,j))
          enddo
        enddo
      enddo
c
c-------------------------------------------------------------------
c  Compute nondimensional stability functions for tracers (Sh) and
c  momentum (Sm).
c-------------------------------------------------------------------
c
      do k=1,Nm
c
c  Compute Galperin et al (1988) stability Gh function (nondimensional).
c
        do j=1,Mm
          do i=1,Lm
            a2d(i,j)=-bvf(i,j,k)*Lscale(i,j,k)*Lscale(i,j,k)/
     &               q2(i,j,k,mnew)
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            Gh(i,j)=min(my_Gh0,a2d(i,j))
          enddo
        enddo
c
c  Compute stability function for tracers (nondimensional).
c
        do j=1,Mm
          do i=1,Lm
            Sh(i,j,k)=Sh1/(1.-Sh2*Gh(i,j))
          enddo
        enddo
c
c  Compute stability function for momentum (nondimensional).
c
        do j=1,Mm
          do i=1,Lm
            Sm(i,j,k)=(Sm1*Sh(i,j,k)*Gh(i,j)+Sm2)/(1.-Sm3*Gh(i,j))
          enddo
        enddo
      enddo
c
c-------------------------------------------------------------------
c  Compute vertical mixing coefficients (m^2/s).
c--------------------------------------------------------------------
c
c  Get turbulent eddy kinetic energy times length scale (m^2/s).
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            ql(i,j,k)=sqrt(q2(i,j,k,mnew))*Lscale(i,j,k)
          enddo
        enddo
      enddo
c
c  Compute vertical mixing (m^2/s) coefficients: Kv, Kt, and Ks.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            Kv(i,j,k)=Akv_bak+ql(i,j,k)*Sm(i,j,k)
            Kt(i,j,k)=Akt_bak(1)+ql(i,j,k)*Sh(i,j,k)
            Ks(i,j,k)=Akt_bak(2)+ql(i,j,k)*Sh(i,j,k)
          enddo
        enddo
      enddo
c
c  Compute vertical mixing (m^2/s) coefficient for turbulent kinetic
c  energy: Akq.  Use a time filter to avoid temporal instability of the
c  implicit scheme. This is important when forcing varies abruptly in
c  time.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            Kq(i,j)=Akq_bak+vonKar*ql(i,j,k)*Sm(i,j,k)
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            Akq(i,j,k)=at1*Kq(i,j)+at2*Akq(i,j,k)
          enddo
        enddo
      enddo
c
c-------------------------------------------------------------------
c  Update indices for two-time level variables.
c-------------------------------------------------------------------
c
      mold=mnew
      mnew=3-mold
#else
      subroutine my25_vmix_empty
#endif /* MY25_MIXING */
      return
      end
 
