#include "cppdefs.h"
#ifdef MY2_MIXING
      subroutine my2_vmix (Kv,Kt,Ks,bvf)
!
!-------------------------------------------------------------------
!  This routine computes vertical mixing coefficients for momentum
!  and tracers using the  Mellor and Yamada (1974)  mixing level 2
!  scheme wich is based on  Richardson-number-dependent  stability
!  functions.
!
!  On Output:
!
!     Kv       vertical viscosity coefficient (m^2/s).
!     Kt       vertical diffusion coefficient for potential
!              temperature (m^2/s).
!     Ks       vertical diffusion coefficient for salinity (m^2/s).
!     bvf      Brunt-Vaisala frequency (1/sec).
!
!  Calls:   ri_number
!
!  References:
!
!     Mellor, G.L. and T. Yamada, 1974: A hierarchy of turbulence
!        closure models for planetary boundary layers, J. Atmosp.
!        Sciences, 31, 1791-1806.
!     Mellor, G.L.  and  P.A. Durbin,  1975:  The  structure  and
!        dynamics  of the  ocean surface  mixed layer,  J.  Phys.
!        Oceanog., 5, 718-728.
!-------------------------------------------------------------------
!
      implicit none
# include "param.h"
# include "pconst.h"
# include "grid.h"
# include "mixing.h"
# include "ocean.h"
# include "scalars.h"
# include "work.h"
 
 
      real my_A1, my_A2, my_B1, my_B2, my_C1, my_Ric, my_alpha,my_eps
      integer my_iter
      parameter (
     &    my_A1=0.78,     ! parameters associated with
     &    my_A2=0.79,     ! Mellor-Yamada (1974) level 2
     &    my_B1=15.0,     ! turbulence closure:
     &    my_B2=8.0,      ! Ref: M-Y (1974)
     &    my_C1=0.056,
     &    my_Ric=0.2281,
 
     &    my_alpha=0.01,  ! Empirical constant "alpha".
 
     &    my_eps=0.001,   ! Convergence criteria used to compute
                          ! the turbulent length scale iteratively.
 
     &    my_iter=5)      ! Number of iterations used to compute
                          ! the turbulent length scale.
 
 
 
 
      integer i, j, k
# ifdef MY2_ITERATE
      integer iter, kk
      real d_bot, d_sur, l_new, l_o, l_old, sum1, sum2
# endif
      real Ar0, Ar1, Ar2, Ar3, Sh1, Sh2, Sm1, Sm2, Sm3, eps, g1, g2,
     &        g3, g4
      real Ks(Lm,Mm,0:N),Kt(Lm,Mm,0:N), Kv(Lm,Mm,0:N), bvf(Lm,Mm,Nm),
     &     Rif(Lm,Mm,Nm),Rig(Lm,Mm,Nm),  Sh(Lm,Mm,Nm),  Sm(Lm,Mm,Nm),
     &      gamma(Lm,Mm), ql(Lm,Mm,Nm),shear2(Lm,Mm,Nm),tKE(Lm,Mm,Nm)
      equivalence (Rif,a3d), (Rig,a3d), (Sh,e3d), (Sm,f3d),
     &          (gamma,a2d),  (ql,g3d),          (tKE,c3d)
# ifdef MY2_ITERATE
      real Lscale(Lm,Mm,Nm)
      equivalence (Lscale,b3d)
# else
      real Lscale(Lm,Mm), sum_ql(Lm,Mm), sum_qlz(Lm,Mm)
      equivalence (Lscale,b2d), (sum_ql,c2d), (sum_qlz,d2d)
# endif
      parameter (eps=1.0e-10)
      save Ar0, Ar1, Ar2, Ar3, Sh1, Sh2, Sm1, Sm2, Sm3
c
c
      real a1, a2, a3, a4, av2, av4
# include "avg.h"
!
!  On first pass, set parameters.
!
!  M-Y (1974):  A1=0.78,A2=0.79,B1=15.0,B2= 8.0,C1=0.056,Ric=0.2281
!  M-Y (1982):  A1=0.92,A2=0.74,B1=16.6,B2=10.1,C1=0.08 ,Ric=0.1950
!--------------------------------------------------------------------
!
      if (iic.eq.ntstart) then
c
c  Set coefficients used to compute stability functions for tracer
c  and momentum.
c
        Sh1=my_A2-(c6*my_A1*my_A2)/my_B1
        Sh2=c3*my_A2*(my_B2+c6*my_A1)/my_B1
        Sm1=c3*my_A1*(r3-(c2*my_A1/my_B1)-my_C1)
        Sm2=c9*my_A1*(c2*my_A1+my_A2)/my_B2
        Sm3=c9*my_A1*my_A2/my_B1
c
c  Set coefficients used to convert gradient Richardson number to
c  flux Richardson number.
c
        g1=my_A1*(r3-(c2*my_A1/my_B1)-my_C1)
        g2=my_A1*(r3-my_C1+(c4*my_A1/my_B1)+(c3*my_A2/my_B1))
        g3=my_A2*(r3-(c2*my_A1/my_B1))
        g4=my_A2*(r3+(my_B2/my_B1)+(my_A1/my_B1))
        Ar0=0.5*g1/g2
        Ar1=0.5*g4/g2
        Ar2=2.*(g1/g4-2.*g2*g3/(g4*g4))
        Ar3=(g1*g1)/(g4*g4)
      endif
!
!--------------------------------------------------------------------
!  Compute gradient Richardson number, Brunt-Vaisala frequency, and
!  horizontal velocity shear squared.
!--------------------------------------------------------------------
!
      call ri_number (Rig,bvf,shear2)
c
c  Lower bound gradient Richardson number by its critical value "Ric".
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            Rig(i,j,k)=min(my_Ric,Rig(i,j,k))
          enddo
        enddo
      enddo
!
!--------------------------------------------------------------------
!  Convert gradient Richardson number to flux Richardson number (Rif).
!
!       Rif = Ar0 + Ar1*Rig + Ar1 * SQRT (Rig^2 + Ar2*Rig + Ar3)
!
!--------------------------------------------------------------------
!
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            Rif(i,j,k)=Ar0+
     &                 Ar1*Rig(i,j,k)-
     &                 Ar1*sqrt(Rig(i,j,k)*Rig(i,j,k)+
     &                          Ar2*Rig(i,j,k)+Ar3)
          enddo
        enddo
      enddo
!
!--------------------------------------------------------------------
!  Compute stability functions for tracers (Sh) and momentum (Sm) and
!  turbulent eddy kinetic energy, 1/2 (q/l)^2.
!-------------------------------------------------------------------
!
      do k=1,Nm
c
c  Compute gamma function, gamma=Rif/(1-Rif), which is the ratio of
c  negative buoyant production to total energy production.
c
        do j=1,Mm
          do i=1,Lm
            gamma(i,j)=Rif(i,j,k)/(1.-Rif(i,j,k))
          enddo
        enddo
c
c  Compute stability functions for tracers (Sh) and momentum (Sm).
c
c       Sh=Sh1-Sh2*gamma          Sm=Sh*(Sm1-Sm2*gamma)/(Sh+Sm3)
c
        do j=1,Mm
          do i=1,Lm
            Sh(i,j,k)=Sh1-Sh2*gamma(i,j)
            Sm(i,j,k)=Sh(i,j,k)*(Sm1-Sm2*gamma(i,j))/
     &                (Sh(i,j,k)+Sm3*gamma(i,j))
          enddo
        enddo
c
c  Compute turbulent kinetic energy [1/2 (q/l)^2]. Insure positive
c  values and take the squared root.
c
        do j=1,Mm
          do i=1,Lm
            tKE(i,j,k)=my_B1*(Sm(i,j,k)*shear2(i,j,k)-
     &                        Sh(i,j,k)*bvf(i,j,k))
            tKE(i,j,k)=sqrt(max(0.,tKE(i,j,k)))
          enddo
        enddo
      enddo
# ifdef MY2_ITERATE
!
!--------------------------------------------------------------------
!  Compute turbulent length scale.
!-------------------------------------------------------------------
!
      do j=1,Mm
        do i=1,Lm
c
c  Set initial length (l_o) scale to a fraction of the water column
c  depth.
c
          l_o=D(i,j,krhs)*ds+eps
c
c  Compute turbulent vertical length scale.
c
          do k=1,Nm
c
c  Compute length scale guess, l_old and l_new.
c
            d_sur=z_w(i,j,k)-z_w(i,j,k)
            d_bot=D(i,j,krhs)-z_w(i,j,k)
            l_old=l_o
            l_new=(vonKar*d_sur/(1.+vonKar*d_sur/l_o)+
     &             vonKar*d_bot/(1.+vonKar*d_bot/l_o))/l_o
c
c  Iterate until convergence or until maximum number of iterations
c  is reached.
 
            iter=0
            do while ((iter.le.my_iter).and.
     &                (abs(l_old-l_new).gt.my_eps))
              iter=iter+1
              l_old=l_new
c
c  Evaluate l_o = alpha * INTG[(z*q*l_g) ds]/INTG[(q*l_g) ds]
c
              sum1=0.
              sum2=0.
              do kk=1,Nm
                sum1=sum1-l_new*tKE(i,j,k)*z_w(i,j,k)*ds
                sum2=sum2+l_new*tKE(i,j,k)*ds
              enddo
              if (sum2.eq.0.) then
                l_o=eps
              else
                l_o=(my_alpha*sum1/sum2)+eps
              endif
c
c  Evaluate new length scale.
c
              l_new=(vonKar*d_sur/(1.+vonKar*d_sur/l_o)+
     &               vonKar*d_bot/(1.+vonKar*d_bot/l_o))/l_o
            enddo
c
c  Load length scale into "Lscale".
c
            Lscale(i,j,k)=l_new
          enddo
        enddo
      enddo
!
!--------------------------------------------------------------------
!  Compute vertical mixing coefficients.
!--------------------------------------------------------------------
!
c  Get turbulent eddy kinetic energy times length scale.  Multiply by
c  length scale squared.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            ql(i,j,k)=tKE(i,j,k)*Lscale(i,j,k)*Lscale(i,j,k)
          enddo
        enddo
      enddo
# else
!
!--------------------------------------------------------------------
c  Integrate turbulent eddy kinetic energy to find integral turbulent
c  length scale.  Store turbulent length scale in "tmp".
!--------------------------------------------------------------------
c
      do j=1,Mm
        do i=1,Lm
          sum_ql(i,j)=0.
          sum_qlz(i,j)=0.
        enddo
      enddo
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            sum_ql (i,j)=sum_ql (i,j)+tKE(i,j,k)*ds
            sum_qlz(i,j)=sum_qlz(i,j)+tKE(i,j,k)*ds*
     &                                av2(-z_r(i,j,k),-z_r(i,j,k+1))
          enddo
        enddo
      enddo
      do j=1,Mm
        do i=1,Lm
            Lscale(i,j)=my_alpha*sum_qlz(i,j)/(sum_ql(i,j)+eps)
        enddo
      enddo
c
!--------------------------------------------------------------------
c  Compute vertical mixing coefficients.
!--------------------------------------------------------------------
c
c  Get turbulent eddy kinetic energy times length scale.  Multiply by
c  length scale squared.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            ql(i,j,k)=tKE(i,j,k)*Lscale(i,j)*Lscale(i,j)
          enddo
        enddo
      enddo
# endif /* MY2_ITERATE */
c
c  Compute vertical mixing coefficients: Kv, Kt, and Ks.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            Kv(i,j,k)=Akv_bak+ql(i,j,k)*Sm(i,j,k)
            Kt(i,j,k)=Akt_bak(1)+ql(i,j,k)*Sh(i,j,k)
            Ks(i,j,k)=Akt_bak(2)+ql(i,j,k)*Sh(i,j,k)
          enddo
        enddo
      enddo
#else
      subroutine my2_vmix_empty
#endif /* MY2_MIXING */
      return
      end
 
 
