      subroutine spline (x,y,n, yp1,ypn,spval, y2)
!
! Given X, Y of length N containing a tabulated function,  Y=f(X),
! with the Xs  in ascending order,  and given values  Yp1 and  Ypn
! for the first derivative of the interpolating function at points
! 1 and N, respectively this routine returns an array Y2 of length
! N  which contains the  second  derivatives of the  interpolating
! function at the tabulated points X.  If Yp1 and/or Ypn are equal
! to  1.0E+30  or larger,  the routine  is  signalled  to  set the
! corresponding boundary condition for a natural spline, with zero
! second derivative on that boundary.
!
! Reference: Press, W.H, B.P. Flannery, S.A. Teukolsky, and 
!            W.T. Vetterling, 1986: Numerical Recipes, the art
!            of scientific computing. Cambridge University Press.
!
      implicit none
      integer n, i,k, nmax
      parameter (nmax=10000)
      real x(n),y(n),y2(n), yp1,ypn,spval, u(nmax), p,qn,sig,un  
      if (n.gt.nmax) then
        write(*,'(/2x,3A/16x,A,I6,3x,A,I6,A/)') 'ERROR: SPLINE: ',
     &      'Unsufficient setting of parameter ''nmax'' in file ',
     &      '''spline.F'',',  'nmax =', nmax,  'requested', n, '.'
        stop
      endif
!
! The lower boundary condition is set either to be "natural" or else
! to have a specified first derivative.
!
      if (yp1.eq.spval) then
        y2(1)=0.
        u(1)=0.
      else
        y2(1)=-0.5
        u(1)=(3./(x(2)-x(1)))*((y(2)-y(1))/(x(2)-x(1))-yp1)
      endif
!
! This is the decomposition loop of the tridiagonal algorithm. Y2
! and U are used for temporary storage of the decomposition factors.
!
      do i=2,n-1
        sig=(x(i)-x(i-1))/(x(i+1)-x(i-1))
        p=sig*y2(i-1)+2.
        y2(i)=(sig-1.)/p
        u(i)=( 6.*( (y(i+1)-y(i))/(x(i+1)-x(i))
     &             -(y(i)-y(i-1))/(x(i)-x(i-1))
     &          )/(x(i+1)-x(i-1))-sig*u(i-1))/p
      enddo
!
! The upper boundary condition is set either to be "natural" or else
! to have a specified first derivative.
!
      if (ypn.eq.spval) then
        qn=0.
        un=0.
      else
        qn=0.5
        un=(3./(x(n)-x(n-1)))*(ypn-(y(n)-y(n-1))/(x(n)-x(n-1)))
      endif
      y2(n)=(un-qn*u(n-1))/(qn*y2(n-1)+1.)
!
! This is the back-substitution loop of the tridiagonal algorithm.
!
      do k=n-1,1,-1
        y2(k)=y2(k)*y2(k+1)+u(k)
      enddo
      return
      end


      subroutine splint (x,y,y2,n,xx,yy,dydx)
!
! Given arrays X and Y of length N, which tabulate a function Y=f(X),
! with the Xs in ascending order, and given the array Y2 containing
! the second derivatives of the interpolating function at the
! tabulated points X as computed  by routine  SPLINE, and given
! a value  XX, compute a cubic-spline interpolated value YY.
!
! Reference: Press, W.H, B.P. Flannery, S.A. Teukolsky, and
!            W.T. Vetterling, 1986: Numerical Recipes, the art
!            of scientific computing. Cambridge University Press.
!
! Modified by H.G. Arango (1989) for output the first derivative
! DYDX at a given value XX.
!
! Copyright (c) 1996 Rutgers University
!
      implicit none
      integer n,   k, khi, klo
      real x(n),y(n),y2(n), xx,yy,dydx, h,a,b, cff, OneSixth,OneThird
      parameter (OneSixth=0.166666666666, OneThird=0.333333333333)

      klo=1                           ! Find the right place
      khi=n                           ! for XX n the table by
      do while (khi-klo .gt.1)        ! means of bisection.
        k=(khi+klo)/2
        if(x(k).gt.xx) then
          khi=k                       ! After this procedure
        else                          ! xx should lie between
          klo=k                       ! x(khi) and x(klo).
        endif
      enddo


      h=x(khi)-x(klo)
      if (h.ne.0.) then
        cff=1./h                      ! Compute interpolated value
        a=cff*(x(khi)-xx)             ! and derivative at point xx.
        b=cff*(xx-x(klo))

        yy=a*y(klo) + b*y(khi) + OneSixth*h*h*( (a*a*a-a)*y2(klo)
     &                                         +(b*b*b-b)*y2(khi))

        dydx=cff*(y(khi)-y(klo)) + 0.5*h*( (b*b-OneThird)*y2(khi)
     &                                    -(a*a-OneThird)*y2(klo))
        return
      else
        write(*,*) ' SPLINT: bad X input, they must be distinct.'
        stop
      endif
      end
