      subroutine spline (z,f,N, d1,dN,spval, d)
!
! Given function f(z) defined at non-uniformly spaced grid nodes
! z(k), k=1,..,N find set of values of first derivatives d(k) defined
! at the same grid nodes using cubic spline algorithm: essentially
! assumption that function is represented by cubic polynomials on
! each interval between pair of consecutive grid nodes, coefficients
! of which are expressed via values of function "f" and derivatives
! "d" at both ends of interval. Nodal derivatives d(k) are computed
! from the condition of continuity of the second derivative.
!
! input: z,f    a set of node coordinates and values of function
!        N      number of node points in the set
!        d1,dN  values of first derivative of the function at the
!                  lower and upper ends (they are may have meaningful
!                  values, or set to special value "spval" to signal
!                  that they should be rejected; an artificial
!                  "natural" boundary condition will be used instead.
!                  Basically this is assumption that the second
!                  derivative vanishes at the corresponding end of
!                  the grid.
!        spval  special value.
!
! output: d     array of first derivatives.          
!
      implicit none
      integer N, k, nmax
      parameter (nmax=100)
      real z(N),f(N), d1,dN,spval,  d(N), dz(N),dfz(N),CF(N), cff 

      if (N.gt.nmax) then
        write(*,'(/2x,2A/16x,A,I6,3x,A,I6,A/)') 'ERROR: setting of ',
     &       'parameter ''nmax'' in file ''spline.F'' is too small.',
     &       'nmax =', nmax,  'requested', N, '.'
        stop
      endif

      do k=2,N
        dz(k)=z(k)-z(k-1)
        dfz(k)=(f(k)-f(k-1))/dz(k)
      enddo

      if (d1.eq.spval) then
        CF(1)=0.5                ! "natural" boundary condition of
        d(1)=1.5*dfz(2)          ! vanishing second derivative at k=1
      else
        CF(1)=0.
        d(1)=d1                  ! <-- accept specified value
      endif
                                 !--> forward Gaussian elimination
      do k=2,N-1
        cff=1./(2.*dz(k)+(2.-CF(k-1))*dz(k+1))
        CF(k)=cff*dz(k)
        d(k)=cff*(3.*(dfz(k+1)*dz(k)+dfz(k)*dz(k+1))-dz(k+1)*d(k-1))
      enddo

      if (dN.eq.spval) then                    ! "natural" boundary
        d(N)=(3.*dfz(N)-d(N-1))/(2.-CF(N-1))   ! condition at k=N
      else
        d(N)=dN                  ! <-- accept specified value
      endif

      do k=N-1,1,-1              !--> backsubstitution
        d(k)=d(k)-CF(k)*d(k+1)
      enddo
      return
      end

      subroutine splint (z,f,d, N, zz,ff,dd)
!
! Spline interpolation: given array of grid nodes z(k), array of
! function values f(k) and array of first derivatives d(k), compute
! value "ff" of the function at specified point "zz", by finding
! appropriate grid interval to which "zz" belongs and assuming that
! function "f" is represented by cubic polynomial fit using the
! nearest available values and derivatives.
!
      implicit none
      integer N,  k,          klo,khi
      real z(N),f(N),d(N), zz,ff,dd, xi
#include "pconst.h"
                                 ! Find the right place for point
      khi=N                      ! "zz" among the grid levels z(k) 
      klo=1                      ! by means of bisection. After this 
      do while (khi-klo .gt.1)   ! procedure "zz" should lie between 
        k=(khi+klo)/2            ! z(klo) and z(khi), if "zz" falls 
        if (z(k).gt.zz) then     ! inside the whole interval z(1)... 
          khi=k                  ! z(N), or klo,khi are two nearest 
        else                     ! consecutive indices, 1,2 or N-1,N, 
          klo=k                  ! if zz<z(1) or zz>z(N). In these 
        endif                    ! two cases interpolation turns into 
      enddo                      ! extrapolation.

      xi=(zz-0.5*(z(khi)+z(klo)))/(z(khi)-z(klo))
      ff=0.5*(f(khi)+f(klo)) +(f(khi)-f(klo))*xi*(1.5-2.*xi*xi)
     &           +(xi*xi-0.25)*(z(khi)-z(klo))*(d(khi)*(xi+0.5)
     &                                        +d(klo)*(xi-0.5))

c      if ((z(N)-zz)*(zz-z(1)) .ge. 0.) then
c        do k=1,N-1
c          if ((z(k+1)-zz)*(zz-z(k)) .gt. 0.) ks=k
c        enddo
c
c        xi=(zz-0.5*(z(ks+1)+z(ks)))/(z(ks+1)-z(ks))
c        ff=0.5*(f(ks+1)+f(ks)) +(f(ks+1)-f(ks))*xi*(1.5-2.*xi*xi)
c     &             +(xi*xi-0.25)*(z(ks+1)-z(ks))*(d(ks+1)*(xi+0.5)
c     &                                            +d(ks)*(xi-0.5))
c      else
c        ff=spval1
c      endif

      return
      end
