      subroutine read_inputs (ibuff)
!
! Read input parameters from cnt/sec input script and defaults file,
! and set various parameters and switches for plotting. This version
! of the code allows flexible distribution of input variables between
! cnt/sec input file and defaults file. It is achieved by buffered
! reading, in which cnt/sec input is read first without actually
! interpreting it (it is simply saved into buffer as a set of
! character strings), then the name of default file is determined and
! content of the defaults file is added to the the buffer at the end. 
! After that the buffer is decoded second part first, so if the same 
! keyword is defined twice, once in cnt/sec input script, and once
! in the default file, the cnt/sec value overrides the default.
!
! This code consists of four segments:
!
! (I) Setting code internal default values (the respective
!     variables will retain these values, if neither defaults,
!     not cnt/sec input files specify them;
!
! (II) Reading both files into buffer with preliminary analysis,
!      including discarding all lines which do not start with a
!      keyword, and finding starting/ending indices of each word  
!      within each string. This operation essentially creates
!      unified buffer containing useful data from both files.
!
! (III) Decoding buffer.
!
! (IV)  Various checks for consistency and correctness.
!
! Arguments ibuff, buff are used as scratch variables only.
!
! Alexander Shchepetkin, January 26, 2002, Kaliningrad, Russia
!
      implicit none
      integer stdinp, maxstring, maxwords, inptmp 
      parameter (stdinp=5, maxstring=80, maxwords=32, inptmp=11)
      character*(maxstring) string, buff(1024)
      integer ibuff(2*maxwords+2,*), line,    indx,   i,j,
     &        is(maxwords),  nwords, ninput,  nboth,  lstr,
     &        ie(maxwords),  ierr,   ichk,    inperr, lenstr
      real pblat,ptlat, pllon,prlon, lon,lat, size,angle,ctr
      character infile*32, defaults*64, varids*64
      logical lswtch
#include "param.h"
#include "pconst.h"
#include "domdat.h"
#include "ncinfo.h"
#include "pltfld.h"
#include "pltio.h"
#include "pltlab.h"
#include "pltncar.h"
#include "cities.h"
#include "secpos.h"

      lstr=0
      call get_date (date_str) ! Set execution date label

!                            ! Reset all titles to blank strings:
! (I) Set default values:    ! if some of these are not specified in
!==== === ======= =======    ! the input script, or are commented
!                            ! out by '!', they will not appear on
      ntitles=0              ! the plot, and no space at the top of
      do i=1,4               ! the plot normally used to accomodate
        title(i)=' '         ! them will be left blank [see
      enddo                  ! computation of variable "TopMargin"
                             ! in viewport.F for more details.]
      magscale=1.000
      sdv_fac=0.
      geo_lab=.false.
c?    log_cnt=.false.
      lnozero=.false.

      logscale=.false.
      lcntlab=.false.
      cnt_cll=0
      lhiglow=.false.
c?    lmodday=.false.
      lframe=.false.
      ldomain=.false.
      ltmarks=.false.
      lclrbar=.false.
      allbarlab=.false.

      lvecrho=.false.
      lvcolor=.false.
      vec_vpo=0
      vec_amn=0.0007
      vec_vlc=0.
      vec_vfr=0.07
      vec_vrm=0.
      vec_mxx=0.85
      vec_mxy=-0.07

      secaxis=0
      coorscl=0
      hvelscl=0
      wvelscl=0
      zref=0.
c?    smthout=0
c?    nsappl=0
c?    nord=0

      frstd=0.              ! This default setting activates
      lastd=0.              ! plotting every available record.
      dskip=0.

      dstart=0
      st_year=0             ! Reset starting date of model calendar
      st_mon=0              ! (if these are found in the input file, 
      st_mday=0             ! model clock will be written as date.)

      nfields=0             ! number of fields to plot
      nlevels=1             ! number of levels (cnt/ccnt only)
      levs_flds=.false.     ! fields/levels sequence conrtol switch
      iref=0
      logscale=.false.
      do i=1,32             ! NOTE: For the purpose of
        fldid(i)=0          ! compatibility with old input
        fldlev(i)=0         ! files, the limiting bounds for
        pmax(i)=0.          ! primary and secondary fields, the 
        pmin(i)=0.          ! primary field contour interval, 
        cntval(i)=0.        ! overlay field IDs and levels are
                            ! first set to zero default values.
        idover(i)=0         ! The subsequent read statements
        levover(i)=0.       ! are error protected, so that in
        rmax(i)=0.          ! the case when there are not enough 
        rmin(i)=0.          ! values supplied for all fields,
      enddo                 ! the remaining values remain zeros.

      vec_lwd=1.            ! Default vector line width and scale,
      vec_scl=1.            ! maximum vector size in grid intervals.
      ivinc=1               ! Plot every vector, unless specified
      jvinc=1               ! otherwise.

      npage=1               ! number of plots per page
      row_column=.false.    ! column/raw plot arrangement switch
      keep_asp_ratio=.true. ! keep horizontal domain aspect ratio
      ipref=-1
      lview=.false.         ! switch to fill viewport with water
      lland=.false.         ! switch to fill land  
      lwater=.false.        ! switch to fill domain with water color 
      ifill=0               ! switch to fill area below water column 

      postscript=.false.    ! postscript/gmeta switch
      wrtclock=.false.
      wrtfield=.false.      ! switch to write bottom titles
      wrtrang=.false.       ! switch to write data range values
      wrtfile=.false.       ! switch to write primary file name
      wrtdate=.false.       ! switch to write current date

      proj='ME'             ! Set default spherical projection type,
      earthdat='        '   ! to Mercator, NCAR graphics dataset name 
      plon=spvgeo           ! to pre-1998 type, and projection pole  
      plat=spvgeo           ! coordinates. to indefinite.
      rota=spvgeo           ! Also set default satelite height to
      sath=1000.            ! a large number (view from infinity.)

      pblat=spvgeo          ! Set spherical map limits
      ptlat=spvgeo          ! to zeros in order to activate
      pllon=spvgeo          ! auto_limits algorithm in
      prlon=spvgeo          ! cnt_frame. 
      
      do i=1,2              ! Latutude/Longitude
        plim1(i)=spvgeo     ! geographical coordinates
        plim2(i)=spvgeo     ! of four corner points of
        plim3(i)=spvgeo     ! the desired portion of
        plim4(i)=spvgeo     ! spherical map projection.
      enddo

      isecpos=spvgeo        ! Variables controlling vertical 
      xsec1=spvgeo          ! crossections of csec/sec plots:
      ysec1=spvgeo          ! providing a set of meaningful
      xsec2=spvgeo          ! values is mandatory for them in 
      ysec2=spvgeo          ! the case of csec plot: their proper 
      sztop=0.              ! initialization is checked below and
      szbot=0.              ! if some of them remain spvgeo, the 
      dels=0.               ! execution of csec/sec is aborted.
      kms=0.
      ncl=1

      nhists=0              ! reset number of history files

      defaults='none'       ! This default setting is designed to
      varids='none'         ! speed up troubleshooting: if a file
      palette='none'        ! name is needed, but is not supplied 
      history(1)='none'     ! in the input, the error message of 
      reference(1)='none'   ! failed attempt to open this file
      grid='none'           ! contains thecorresponding variable. 
      coastline='none'      ! If name is specified more than once,
                            ! an error message will be issued.
!
! (II) Read cnt/sec input file first, perform preliminary analysis
! and save lines with keywords as a set of character strings in array
! "buff". For each string record its line number within the input
! file, number of words, and starting and ending indices of eaach
! word into array "ibuff". After it is done, value of variable
! "ninput" becomes the number of saved strings. NOTE that "ninput"
! is transparent input/output variable in the subroutine "read_buff":
! it is incremented by the number of strings recorded into "buff":
!
#ifdef VERBOSE
      write(*,'(/8x,A/)') 'Reading buffer...'
#endif
      ninput=0
      call read_buff (stdinp, ibuff, buff, ninput)
!
! Once input file is read, find the name of defaults file, check
! that it is unique, then open and read it, saving the strings into
! the same buffer array "buff": 
!
      inperr=0
      do j=1,ninput
        if (ibuff(4,j)-ibuff(3,j).eq.7) then 
          if (buff(j)(ibuff(3,j):ibuff(4,j)).eq.'defaults') then
            if (defaults.eq.'none') then
              defaults=buff(j)(ibuff(5,j):ibuff(6,j))
            else
              write(*,'(/1x,2A,I4,A)') 'ERROR: Duplicated name ',
     &                      'of defaults file on line', line, '.'
              inperr=inperr+1 
            endif
          endif
        endif
      enddo
      if (inperr.eq.0 .and. defaults.ne.'none') then
        lstr=lenstr(defaults)
        open (unit=inptmp, file=defaults(1:lstr), form='formatted',
     &                                         status='old', err=1)
        nboth=ninput
        call read_buff (inptmp, ibuff, buff, nboth)
        close (inptmp)
        goto 2
  1     write(*,'(/1x,A/)') 'ERROR: Cannot open defaults file.'
        stop
      else
        write(*,'(/1x,2A/)') 'ERROR: Cannot determine name of ',
     &                                         'defaults file.'
        stop
      endif
  2   continue
!
!                          At this moment "nboth" is the combined
! (III) Decode buffer:     number of saved strings from both files,
!====== ====== =======     input+defaults, while "ninput" is the
!                          number of strings from cnt/sec input file
! only. The following code segment decodes the strings according to
! the meaning of their keywords:
!
#ifdef VERBOSE
      write(*,'(/8x,A/)') 'Decoding buffer...'
#endif
      do indx=1,nboth                       ! Extract a character 
        j=ninput+indx                       ! string and its indices 
        if (j.gt.nboth) then                ! from buffer arrays. In
          j=j-nboth                         ! doing so, process the
          infile='cnt/sec input'            ! second part of buffer,
        else                                ! array first, i.e., with  
          infile=''/ /defaults(1:lstr)/ /'' ! indices greater than
        endif                               ! ninput. Once the end of
        line=ibuff(1,j)                     ! buffer is reached,  
        nwords=ibuff(2,j)                   ! proceed with the first 
        do i=1,nwords                       ! part, indx<=ninput. 
          is(i)=ibuff(2*i+1,j)              ! This order corresponds
          ie(i)=ibuff(2*i+2,j)              ! to processing defaults
        enddo                               ! file first and cnt/sec
        string=buff(j)                      ! input file second. 

!
! Decode input string data according to keyword.
!
        if (string(is(1):ie(1)).eq.'magscale') then
          read(string(is(2):ie(2)),*,err=99) magscale

        elseif (string(is(1):ie(1)).eq.'sdv_fac') then
          read (string(is(2):ie(2)),*,err=99) sdv_fac

        elseif (string(is(1):ie(1)).eq.'geo_lab') then
          if (string(is(2):is(2)).eq.'T') then
            geo_lab=.true.
          else
            geo_lab=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'lnozero') then
          if (string(is(2):is(2)).eq.'T') then
            lnozero=.true.
          else
            lnozero=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'lcntlab') then
          if (string(is(2):is(2)).eq.'T') then
            lcntlab=.true.
          else
            lcntlab=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'cnt_cll') then
          read (string(is(2):ie(2)),*,err=99) cnt_cll

        elseif (string(is(1):ie(1)).eq.'lhiglow') then
          if (string(is(2):is(2)).eq.'T') then
            lhiglow=.true.
          else
            lhiglow=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'ldomain') then
          if (string(is(2):is(2)).eq.'T') then
            ldomain=.true.
          else
            ldomain=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'lframe') then
          if (string(is(2):is(2)).eq.'T') then
            lframe=.true.
          else
            lframe=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'ltmarks') then
          if (string(is(2):is(2)).eq.'T') then
            ltmarks=.true.
          else
            ltmarks=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'lclrbar') then
          if (string(is(2):is(2)).eq.'T') then
            lclrbar=.true.
          else
            lclrbar=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'allbarlab') then
          if (string(is(2):is(2)).eq.'T') then 
            allbarlab=.true.  
          else
            allbarlab=.false. 
          endif


        elseif (string(is(1):ie(1)).eq.'lvecrho') then
          if (string(is(2):is(2)).eq.'T') then
            lvecrho=.true.
          else
            lvecrho=.false.
          endif
          

        elseif (string(is(1):ie(1)).eq.'lvcolor') then
          if (string(is(2):is(2)).eq.'T') then
            lvcolor=.true.
          else
            lvcolor=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'vec_vpo') then
          read (string(is(2):ie(2)),*,err=99) vec_vpo

        elseif (string(is(1):ie(1)).eq.'vec_amn') then
          read (string(is(2):ie(2)),*,err=99) vec_amn

        elseif (string(is(1):ie(1)).eq.'vec_vlc') then
          read (string(is(2):ie(2)),*,err=99) vec_vlc

        elseif (string(is(1):ie(1)).eq.'vec_vfr') then
          read (string(is(2):ie(2)),*,err=99) vec_vfr

        elseif (string(is(1):ie(1)).eq.'vec_vrm') then
          read (string(is(2):ie(2)),*,err=99) vec_vrm

        elseif (string(is(1):ie(1)).eq.'vec_mxx') then
          read (string(is(2):ie(2)),*,err=99) vec_mxx

        elseif (string(is(1):ie(1)).eq.'vec_mxy') then
          read (string(is(2):ie(2)),*,err=99) vec_mxy

        elseif (string(is(1):ie(1)).eq.'secaxis') then
          read (string(is(2):ie(2)),*,err=99) secaxis

        elseif (string(is(1):ie(1)).eq.'coorscl') then
          read (string(is(2):ie(2)),*,err=99) coorscl

        elseif (string(is(1):ie(1)).eq.'hvelscl') then
          read (string(is(2):ie(2)),*,err=99) hvelscl

        elseif (string(is(1):ie(1)).eq.'wvelscl') then
          read (string(is(2):ie(2)),*,err=99) wvelscl

        elseif (string(is(1):ie(1)).eq.'zref') then
          read (string(is(2):ie(2)),*,err=99) zref

c?      elseif (string(is(1):ie(1)).eq.'smthout') then
c?        read (string(is(2):ie(2)),*,err=99) smthout

c?      elseif (string(is(1):ie(1)).eq.'nsappl') then
c?        read (string(is(2):ie(2)),*,err=99) nsappl

c?      elseif (string(is(1):ie(1)).eq.'nord') then
c?        read (string(is(2):ie(2)),*,err=99) nord
!
! Read in contour interval matrix.
!
        elseif (string(is(1):ie(1)).eq.'matrix') then
c         read (string(is(2):ie(2)),*,err=99) iflds
c         read (string(is(3):ie(3)),*,err=99) ilevs
c         do i=0,iflds-1
c           read(inpcnt,*,err=99) j, (cont(j,k),k=1,ilevs)
c         enddo


!
! Read in plot titles and sort them according to their type:
! ending "c" means that the label will be centered, otherwise
! it is started from the respective left. Preceding letter, if
! any, controls placement of title around the plot: "b" means
! bottom (e.g., "...bc" means bottom-centered), "l" means left,
! "r" means right and nothing means top.
!  
        elseif (string(is(1):ie(1)) .eq. 'title' .or.
     &          string(is(1):ie(1)) .eq. 'titlec' .or.
     &          string(is(1):ie(1)) .eq. 'titlel' .or.
     &          string(is(1):ie(1)) .eq. 'titler'  .or.
     &          string(is(1):ie(1)) .eq. 'titleb'  .or.
     &          string(is(1):ie(1)) .eq. 'titlelc'  .or.
     &          string(is(1):ie(1)) .eq. 'titlerc'  .or.
     &          string(is(1):ie(1)) .eq. 'titlebc') then 

          if (ntitles.lt.maxtitles) then
            ntitles=ntitles+1
            title(ntitles)=string(is(2):ie(nwords))
            if (string(is(1):ie(1)) .eq. 'title') then 
              ititle(ntitles)=0
            elseif (string(is(1):ie(1)) .eq. 'titlec') then
              ititle(ntitles)=1
            elseif (string(is(1):ie(1)) .eq. 'titlel' ) then
              ititle(ntitles)=2
            elseif (string(is(1):ie(1)) .eq. 'titlelc') then
              ititle(ntitles)=3
            elseif (string(is(1):ie(1)) .eq. 'titleb' ) then
              ititle(ntitles)=4
            elseif (string(is(1):ie(1)) .eq. 'titlebc') then
              ititle(ntitles)=5
            elseif (string(is(1):ie(1)) .eq. 'titler' ) then
              ititle(ntitles)=6
            elseif (string(is(1):ie(1)) .eq. 'titlerc') then
              ititle(ntitles)=7
            endif
          else
            write(*,'(/A)') 'WARNING: Too many titles are specified.'
          endif
!
! Read in year and starting year day
!
        elseif (string(is(1):ie(1)).eq.'dstart:year:mon:mday') then
          read(string(is(2):ie(2)),*,err=99) dstart 
          read(string(is(3):ie(3)),*,err=99) st_year
          read(string(is(4):ie(4)),*,err=99) st_mon
          read(string(is(5):ie(5)),*,err=99) st_mday
!
! Plot timing control:  Read first, last days to plot and time
!----- ------ --------  interval between plots (normally in days).
! NOTE: if the interval is zero, the data is plotted  at their own
! time frequency (plot all available records); if negative, the
! data is plotted every other abs(int(dskip)) records.
!
        elseif (string(is(1):ie(1)).eq.'frstd:lastd:dskip') then
          read (string(is(2):ie(2)),*,err=99) frstd
          read (string(is(3):ie(3)),*,err=99) lastd
          read (string(is(4):ie(4)),*,err=99) dskip
!
! Primary field plotting control:
!-------- ----- -------- ---------
! Read in number of different fields to plot "nfields" and their
! identification numbers (fldid(1:nfields), which are then decoded
! by varids file);  Read number of different levels and/or depths to
! plot "nlevels". NOTE: if nlevels=0, then  plot all levels available
! for the requested field(s); Read fldlev(1:nlevels), which are then
! interpreted as depths (meters, if < 0); sigma levels (>0) and/or
! isopycnals (>1000). If the values are negative, it activates
! vertical interpolation to the requested depth. If >1000, it
! activates the vertical interpolation to requested isopycnal.
! Read minimum and maximum limit values for color palette and contour
! interval (default contour interval cntval=0.) this value is used
! only by monochrome version of the plot. Read switch "icnt" which
! controls overlaying contours on color plot (icnt=1 means do so,
! 0=no). Read iso-surface value to process "isoval".
!
        elseif (string(is(1):ie(1)).eq.'nfields') then
          read (string(is(2):ie(2)),*,err=99) nfields

        elseif (string(is(1):ie(1)).eq.'fldid') then
          do i=1,nfields
            read(string(is(i+1):ie(i+1)),*,err=99) fldid(i)
          enddo

        elseif (string(is(1):ie(1)).eq.'nlevels' .and.
     &                          plttyp.eq.cntplt) then
           read(string(is(2):ie(2)),*,err=99) nlevels
        elseif (string(is(1):ie(1)).eq.'fldlev' .and.
     &                          plttyp.eq.cntplt) then
          do i=1,nlevels
            read(string(is(i+1):ie(i+1)),*,err=99) fldlev(i)
          enddo

        elseif (string(is(1):ie(1)).eq.'levs_flds' .and.
     &                             plttyp.eq.cntplt) then
          if (string(is(2):ie(2)).eq.'T') then
            levs_flds=.true.
          else
            levs_flds=.true.
          endif

        elseif (string(is(1):ie(1)).eq.'pmin') then 
          do i=1,nfields*nlevels
            read(string(is(i+1):ie(i+1)),*,iostat=ierr) pmin(i)
          enddo
        elseif (string(is(1):ie(1)).eq.'pmax') then
          do i=1,nfields*nlevels
            read(string(is(i+1):ie(i+1)),*,iostat=ierr) pmax(i)
          enddo
        elseif (string(is(1):ie(1)).eq.'cint') then
          do i=1,nfields*nlevels
            read(string(is(i+1):ie(i+1)),*,iostat=ierr) cntval(i)
          enddo
        elseif (string(is(1):ie(1)) .eq. 'logscale') then
          if (string(is(2):is(2)) .eq. 'T') logscale=.true.

        elseif (string(is(1):ie(1)).eq.'icnt') then
          read (string(is(2):ie(2)),*,err=99) icnt

        elseif (string(is(1):ie(1)).eq.'isoval' .and.
     &                         plttyp.eq.cntplt) then
          read (string(is(2):ie(2)),*,err=99) isoval
!
! Secondary (reference, overlay) field plotting control:
!---------- ----------- -------- ----- -------- --------
! Read "iref" switch to plot secondary field:
!       [-1] overlay horizontal grid (cnt/ccnt only)
!        [0] no secondary or reference field to plot
!        [1] primary file overlay
!        [2] secondary file overlay
!        [3] primary - secondary (field subtaction)
!        [4] Day0 - DayN (field subtraction)
! Read overlay field identification; level of the field to
! overlay  (NOTE: if levover=0, then the level selected to be
! the same as for the current primary field); 
! Read in overlay field minimum and maximum limit values to contour.
! 
        elseif (string(is(1):ie(1)).eq.'iref') then
          read (string(is(2):ie(2)),*,err=99) iref

        elseif (string(is(1):ie(1)).eq.'idover') then
          do i=1,nfields
            read (string(is(i+1):ie(i+1)),*,iostat=ierr) idover(i)
          enddo

        elseif (string(is(1):ie(1)).eq.'levover' .and.
     &                           plttyp.eq.cntplt) then
          do i=1,nfields
            read(string(is(i+1):ie(i+1)),*,iostat=ierr) levover(i)
          enddo

        elseif (string(is(1):ie(1)).eq.'rmin') then
          do i=1,nfields*nlevels
            read(string(is(i+1):ie(i+1)),*,iostat=ierr) rmin(i)
          enddo

        elseif (string(is(1):ie(1)).eq.'rmax') then
          do i=1,nfields*nlevels
            read(string(is(i+1):ie(i+1)),*,iostat=ierr) rmax(i)
          enddo
!
! Spherical geometry projection control (ccnt spherical plot only):
!---------- -------- ---------- ------- ----- --------- ---- ------
! Read map projection PROJ type; projection center coordinates
! Plon,Plat = (longitude,latitude); rotation angle rota (positive
! clockwise); satellite hight altitude sath (measured in Earth
! radii); map limiting edges: pblat=south(bottom),
! ptlat=north(top), pllon=west(left), and prlon=east(right).
! Also read desired longitude/latitude grid spacing lgrid. 
! All lat/lon coordinates are measured in degrees. 
!
        elseif (string(is(1):ie(1)).eq.'proj' .and.
     &                        plttyp.eq.cntplt) then
           proj=string(is(2):ie(2))

        elseif (string(is(1):ie(1)).eq.'earthdat' .and.
     &                            plttyp.eq.cntplt) then
           earthdat=string(is(2):ie(2))

        elseif (string(is(1):ie(1)).eq.'plon:plat:rota'
     &                       .and. plttyp.eq.cntplt) then
          read (string(is(2):ie(2)),*,err=99) plon
          read (string(is(3):ie(3)),*,err=99) plat
          read (string(is(4):ie(4)),*,err=99) rota

        elseif (string(is(1):ie(1)).eq.'sath' .and.
     &                              plttyp.eq.cntplt) then
          read (string(is(2):ie(2)),*,err=99) sath

        elseif (string(is(1):ie(1)).eq.'pblat:ptlat' .and.
     &                              plttyp.eq.cntplt) then
          read (string(is(2):ie(2)),*,err=99) pblat
          read (string(is(3):ie(3)),*,err=99) ptlat

        elseif (string(is(1):ie(1)).eq.'pllon:prlon' .and.
     &                              plttyp.eq.cntplt) then
          read (string(is(2):ie(2)),*,err=99) pllon
          read (string(is(3):ie(3)),*,err=99) prlon

        elseif (string(is(1):ie(1)).eq.'plim12' .and.
     &                              plttyp.eq.cntplt) then
          read (string(is(2):ie(2)),*,err=99) plim1(1)
          read (string(is(3):ie(3)),*,err=99) plim1(2) 
          read (string(is(4):ie(4)),*,err=99) plim2(1)
          read (string(is(5):ie(5)),*,err=99) plim2(2)

        elseif (string(is(1):ie(1)).eq.'plim43' .and.
     &                              plttyp.eq.cntplt) then
          read (string(is(2):ie(2)),*,err=99) plim4(1)
          read (string(is(3):ie(3)),*,err=99) plim4(2)
          read (string(is(4):ie(4)),*,err=99) plim3(1)
          read (string(is(5):ie(5)),*,err=99) plim3(2)

        elseif (string(is(1):ie(1)).eq.'lgrid' .and.
     &                              plttyp.eq.cntplt) then
          read (string(is(2):ie(2)),*,err=99) lgrid
!
! Vertical crossection placement control (affects csec plot only).
!--------- ----------- --------- ------- -------- ---- ---- -------
! Read flag for extracting section coordinates isecpos (0 means grid
! units, 1 Cartesian (km), 2 Spherical, degrees); coordinates of the
! two points defining the section; horizontal grid spacing needed for
! positioning of section points between (xsec1,ysec1) and
! (xsec2,ysec2); section  minimum and maximun depths to consider
! sztop,szbot measured in meters; and, finally, number of refiment
! levels to interpolate section vertically between sztop and szbot)
!
        elseif (string(is(1):ie(1)).eq.'isecpos' .and.
     &                           plttyp.eq.secplt) then
          read (string(is(2):ie(2)),*,err=99) isecpos

        elseif (string(is(1):ie(1)).eq.'xsec1:ysec1' .and.
     &                              plttyp.eq.secplt) then
          read (string(is(2):ie(2)),*,err=99) xsec1
          read (string(is(3):ie(3)),*,err=99) ysec1

        elseif (string(is(1):ie(1)).eq.'xsec2:ysec2' .and.
     &                              plttyp.eq.secplt) then
          read (string(is(2):ie(2)),*,err=99) xsec2
          read (string(is(3):ie(3)),*,err=99) ysec2

        elseif (string(is(1):ie(1)).eq.'dels' .and.
     &                         plttyp.eq.secplt) then
          read (string(is(2):ie(2)),*,err=99) dels

        elseif (string(is(1):ie(1)).eq.'sztop:szbot' .and.
     &                              plttyp.eq.secplt) then
          read (string(is(2):ie(2)),*,err=99) sztop
          read (string(is(3):ie(3)),*,err=99) szbot

        elseif (string(is(1):ie(1)).eq.'kms' .and.
     &                        plttyp.eq.secplt) then
          read (string(is(2):ie(2)),*,err=99) kms
!
! Parameters controlling stylistic features of plotting:
!----------- ----------- --------- -------- -- ----------
! Read vector line width; vector length scale, vector grid sampling
! step in the X- and Y-directions, switch "lland" to color mask land
! (active for ccnt plot only, 0 = no land masking; 2 = mask land with
! brown color, 3 = with tan color); switch "ifill" to mask area below
! the water column (active for csec plot only, 0 = no, 1 = yes). Read
! switch for vertical interpolation scheme (0=linear, 1=cubic spline,
! check that the input value are permissible).
!
        elseif (string(is(1):ie(1)).eq.'vec_lwd' .and.
     &                             plttyp.eq.cntplt) then
          read (string(is(2):ie(2)),*,err=99) vec_lwd

        elseif (string(is(1):ie(1)).eq.'vec_scl' .and.
     &                              plttyp.eq.cntplt) then
          read (string(is(2):ie(2)),*,err=99) vec_scl

        elseif (string(is(1):ie(1)).eq.'ivinc:jvinc' .and.
     &                                  plttyp.eq.cntplt) then
          read (string(is(2):ie(2)),*,err=99) ivinc
          read (string(is(3):ie(3)),*,err=99) jvinc
          ivinc=abs(ivinc)
          jvinc=abs(jvinc)

        elseif (string(is(1):ie(1)).eq.'lview') then
          if (string(is(2):ie(2)).eq.'T') then
            lview=.true.
          else
            lview=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'lwater') then
          if (string(is(2):ie(2)).eq.'T') then
            lwater=.true.
          else
            lwater=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'lland') then
          if (string(is(2):ie(2)).eq.'T') then
            lland=.true.
          else
            lland=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'ifill' .and.
     &                         plttyp.eq.secplt) then
          read (string(is(2):ie(2)),*,err=99) ifill

        elseif (string(is(1):ie(1)).eq.'vintrp') then
          read (string(is(2):ie(2)),*,err=99) vintrp
          if (vintrp.lt.0 .or. vintrp.gt.1) then
            write(stdout,'(/1x,2A,I4/)') 'ERROR: illegal ',
     &             'interpolation scheme, vintrp =', vintrp
            stop
          endif
!
! General layout of the plot(s) on the page:
!-------- ------ -- --- ------- -- --- -----
! Read number of plots per page (see viewport.F). If npage is
! negative, cancel switch to preserve horizontal aspect ratio of the
! domain (this takes effect for nonspherical ccnt plots only; in the
! case of sperical geometry ccnt plots aspect ratio is preserved in
! any case regardless of sign of input value of npage).
! Read logical switches to activate writing of plot: (1) switch to
! generate a sequence of postscript files instead of NCAR gmeta;
! (2) switch read in positions from grid netCDF file; (3) switch to
! write header titles;  (4) bottom title; (5) writing information
! about the range values of the plotted data and the contour
! interval; (6) writing input filename; (7) current plotting date;
! (8) logical switch to plot coastlines.
!
        elseif (string(is(1):ie(1)).eq.'npage') then
           read(string(is(2):ie(2)),*,err=99) npage
           if (npage.lt.0) then
             npage=-npage
             keep_asp_ratio=.false.
           endif

        elseif (string(is(1):ie(1)).eq.'ipos') then
           read(string(is(2):ie(2)),*,err=99) ipref 

        elseif (string(is(1):ie(1)).eq.'row_column') then
          if (string(is(2):is(2)).eq.'T') then
            row_column=.true.
          else
            row_column=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'postscript') then
          if (string(is(2):is(2)).eq.'T') then
            postscript=.true.
          else
            postscript=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'wrtclock') then
          if (string(is(2):is(2)).eq.'T') then
            wrtclock=.true.
          else
            wrtclock=.false.
          endif
          write(*,*) 'wrtclock=', wrtclock

        elseif (string(is(1):ie(1)).eq.'wrtfield') then
          if (string(is(2):is(2)).eq.'T') then
            wrtfield=.true.
          else
            wrtfield=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'wrtrang') then
          if (string(is(2):is(2)).eq.'T') then
            wrtrang=.true.
          else
            wrtrang=.false.
          endif

        elseif (string(is(1):ie(1)).eq.'wrtfile') then
          if (string(is(2):is(2)).eq.'T') then
            wrtfile=.true.
          else
            wrtfile=.false. 
          endif

        elseif (string(is(1):ie(1)).eq.'wrtdate') then
          if (string(is(2):is(2)).eq.'T') then
            wrtdate=.true.
          else
            wrtdate=.false.
          endif
!
! Read names of files containing data: variable ids; color palette;
!----- ----- -- ----- ---------- ----- default plotting parameters;
! history and secondary netCDF files; grid netCDF file, and coast
! line data
!
        elseif (string(is(1):ie(1)) .eq. 'defaults') then

        elseif (string(is(1):ie(1)) .eq. 'varids') then
          if (varids(1:4) .eq. 'none') then
            varids=string(is(2):ie(2))
          else
            inperr=inperr+1
            write(*,'(/1x,2A,I4,A/)') 'ERROR: Duplicated name ',
     &                       'of varids file on line', line, '.'
          endif

        elseif (string(is(1):ie(1)) .eq. 'palette') then
          if (palette(1:4) .eq. 'none') then
            palette=string(is(2):ie(2))
          else
            inperr=inperr+1
            write(*,'(/1x,2A,I4,A/)') 'ERROR: Duplicated name ',
     &                      'of palette file on line', line, '.'
          endif

        elseif (string(is(1):ie(1)) .eq. 'history') then
          if (history(1)(1:4) .eq. 'none') then
            do i=2,nwords
              history(i-1)=string(is(i):ie(i))
            enddo
            nhists=nwords-1
          else
            inperr=inperr+1
            write(*,'(/1x,2A,I4,A/)') 'ERROR: Duplicated name ',
     &                        'of history file on line', line, '.'
          endif

        elseif (string(is(1):ie(1)) .eq. 'reference') then
          if (reference(1)(1:4) .eq. 'none') then
            reference(1)=string(is(2):ie(2))
            do i=2,nwords
              reference(i-1)=string(is(i):ie(i))
            enddo
          else
            inperr=inperr+1
            write(*,'(/1x,2A,I4,A/)') 'ERROR: Duplicated name ',
     &                    'of reference file on line', line, '.'
          endif

        elseif (string(is(1):ie(1)) .eq. 'grid') then
          if (grid(1:4) .eq. 'none') then
            grid=string(is(2):ie(2))
          else
            inperr=inperr+1
            write(*,'(/1x,2A,I4,A)') 'ERROR: Duplicated name ',
     &                          'of grid file on line', line, '.'
          endif

        elseif (string(is(1):ie(1)) .eq. 'coastline' .and.
     &                               plttyp.eq.cntplt) then
          if (coastline(1:4) .eq. 'none') then
            coastline=string(is(2):ie(2))
            lstr=ie(2)-is(2)+1 
            open (inpcst, file=coastline(1:lstr), status='old',
     &                                 form='formatted', err=3)
            goto 4
  3         write(*,'(/1x,4A/8x,A,I5,1x,A)')  'ERROR: Cannot ',
     &        'open coastline data file ''', coastline(1:lstr),
     &        '''',  'Check line', line, 'in cnt input script.'
            inperr=inperr+1
  4         continue 
          else
            inperr=inperr+1
            write(*,'(/1x,2A,I4,A)') 'ERROR: duplicated name ',
     &                    'of coastline file on line', line, '.'
            inperr=inperr+1
          endif
!
! Read  coordinates and names of geographical objects (e.g. cities)
! save them in to a special array for subsequent drawing. Each record
! in the file is expected to have form
!
!        city: lat lon size angle ctr 'Character String'
! where
!        lat,lon  -- are geographical coordinates of the object,
!                          (longitude and latitude in degrees);
!        size,angle,ctr -- size,angle and centering parameter
!                          of the desired label (the same as
!                          the last three arguments of plchhq)
!        'Character String' -- name of the object.
!
! NOTE: 1. It is not necessary to encase 'Character String' into
!          single or double quites, just leaving blank characters
!          is enough. The numbers may be separated with commas, or
!          just blank spaces;
!
!       2. The size is the size of letters as it is desired on
!          plot which occupies the whole page. If is automatically
!          rescaled later (multiplied by basesize, if the current
!          viewport is smaller that the whole page).
!
        elseif (string(is(1):ie(1)) .eq. 'city') then
          if (nwords.ge.7) then
            read (string(is(2):ie(2)),*,iostat=ichk) lat
            if (ichk.eq.0) then
              read (string(is(3):ie(3)),*,iostat=ichk) lon
              if (ichk.eq.0) then
                read (string(is(4):ie(4)),*,iostat=ichk) size
                if (ichk.eq.0) then
                  read (string(is(5):ie(5)),*,iostat=ichk) angle
                  if (ichk.eq.0) then
                    read (string(is(6):ie(6)),*,iostat=ichk) ctr
                    if (ichk.eq.0) then
                      lstr=ie(nwords)-is(7)+1
                      if (lstr.gt.0) then
                        if (ncities.lt.maxcities) then
                          ncities=ncities+1 
                          xcity(1,ncities)=lat
                          xcity(2,ncities)=lon
                          xcity(3,ncities)=size
                          xcity(4,ncities)=angle
                          xcity(5,ncities)=ctr
                          xcity(6,ncities)=float(lstr)
                          city(ncities)=string(is(7):ie(nwords))
                        else
                          write(*,'(/1x,3A,I4,1x,A)') 'WARNING:',
     &                       ' Number of cities exceeds allowed',
     &                              ' maximum of', maxcities, '.'
                        endif
                      endif
                    endif
                  endif
                endif
              endif
            endif
            if (ichk.ne.0) then
              write(*,'(/1x,2A,I4,1x,A)') 'WARNING: Cannon read',
     &         ' city string on line', line, 'in cnt input file.'
            endif
          else
            write(*,'(/1x,2A,I4,1x,A)') 'WARNING: Too few words',
     &      ' in city string on line', line, 'in cnt input file.'
          endif
        else
          write(*,'(/1x,3A)') 'WARNING: read_inputs: keyword ''',
     &               string(is(1):ie(1)), ''' is not recognized.'
        endif
      enddo  ! <-- while not_end_of_file
!
! (IV) Checking for missing mandatory input parameters:
!===== ======== === ======= ========= ===== ===========
!
      if (defaults(1:4) .eq. 'none') then
        write(*,'(/2(1x,A))') 'ERROR: No cnt default settings ',
     &           'file "defaults" is given in cnt input script.'
        inperr=inperr+1
      endif
      if (varids(1:4) .eq. 'none') then
        write(*,'(/2(1x,A))') 'ERROR: No variable ID "varids" file ',
     &                                'is given in cnt input script.'
        inperr=inperr+1
      endif
      if (history(1)(1:4) .eq. 'none') then
        write(*,'(/2(1x,A))') 'ERROR: No history netCDF file name ',
     &                           'is given in cnt/sec input script.'
        inperr=inperr+1
      endif
      if (inperr.ne.0) stop
!
! Read variable IDs
!
      call read_varids (inptmp, varids)
!
! If NLEVELS=0, set levels to plot.
!
      if (plttyp.eq.cntplt .and. nlevels.eq.0) then
        nlevels=Nr
        do i=1,Nr                 ! Set ALL levels to plot starting
          fldlev(i)=Nr+1-i        ! from above, if no non-zero value
        enddo                     ! for nlevels is specified.
      endif                       ! (this is default)

!
! The following code segment processess the list of history files in
! order to expand all wildcards, if any: Scan all filenames one-by-one,
! and for each which contains '*', '?', square [...] or curly {...}
! braces execute UNIX "ls" command (via system call) piping its output
! into temporal file. Once matching are found, reads their names and
! into "buff" and then replace name with wildcards with the list of
! actual names. At every occasion of this expansion, move the remaining
! unprocessed filenames to the end of the list and increase file
! counters accordingly.
!
#ifdef VERBOSE
      write(*,*) 'Original history list, nhists=', nhists
      do ihist=1,nhists
        write(*,*) history(ihist)(1:lenstr(history(ihist)))
      enddo
      write(*,*)
#endif
      ihist=0
      do while (ihist.lt.nhists)
        ihist=ihist+1
        lswtch=.false.
        lstr=lenstr(history(ihist))
        i=0
        ierr=0
        do while (i.lt.lstr .and. .not.lswtch)
          i=i+1
          if (history(ihist)(i:i).eq.'*' .or.
     &        history(ihist)(i:i).eq.'?') then
            lswtch=.true.
          elseif (history(ihist)(i:i).eq.'{') then
            do while (i.lt.lstr .and. .not.lswtch)
              i=i+1
              if (history(ihist)(i:i).eq.'}') lswtch=.true.
            enddo
            if (.not.lswtch) ierr=ierr+1 
          elseif (history(ihist)(i:i).eq.'[') then
            do while (i.lt.lstr .and. .not.lswtch)
              i=i+1
              if (history(ihist)(i:i).eq.']') lswtch=.true.
            enddo
            if (.not.lswtch) ierr=ierr+1 
          endif
        enddo 
        if (ierr.ne.0) then
          write(*,'(/8x,3A/)')  'ERROR: unmatched quote: ''',
     &                          history(ihist)(1:lstr), '''.'
          stop
        endif
        if (lswtch) then
          call system ('ls -1 '/ /history(ihist)/ /'> list.dat')
          open(unit=11, file='list.dat', form='formatted')
          ierr=0
          j=0 
          do while (ierr.eq.0)
            j=j+1
            buff(j)=' '
            read(11,'(A)',iostat=ierr,end=91) buff(j)
          enddo
  91      if (buff(j).eq.' ') j=j-1 
          close(unit=11, status='delete')
          if (j.gt.1) then
            do i=nhists,ihist+1,-1
              history(i+j-1)=history(i)
            enddo
            do i=1,j
              lstr=lenstr(buff(i))
              history(ihist+i-1)=buff(i)(1:lstr)
            enddo
            ihist=ihist+j-1
            nhists=nhists+j-1
          else
            lstr=lenstr(buff(1))
            history(ihist)=buff(1)(1:lstr)
          endif
        endif         !<-- lswtch
      enddo         !<-- ihist.lt.nhists 
!
! Verify that all history files from the list do exist:
!
      ierr=0
      do ihist=1,nhists
        lstr=lenstr(history(ihist))
        open(unit=11,file=history(ihist)(1:lstr),status='old',err=92)
        close(unit=11)
        goto 93  
  92    ierr=ierr+1
        write(*,'(/4x,3A/)') 'ERROR: Can not find history file ''',
     &                                history(ihist)(1:lstr), '''.'
  93    continue     
      enddo
#ifdef VERBOSE
      write(*,*) 'Expanded history list, nhists=', nhists
      do ihist=1,nhists
        write(*,*) history(ihist)(1:lenstr(history(ihist)))
      enddo
#endif
      if (ierr.ne.0) stop

!
! Reset file counter back to unity, check whethere there is a grid
! file given, and if not, and if there is nore than one history file,
! assume that the first one contain grid information. Once this is
! settled, open starting history and reference (if any) input netCDF
! files and get the necessary information for plotting.
!
      ihist=1
      if (nhists.gt.1. and. grid(1:4).eq.'none') grid=history(1)
      call opencdf (history(ihist), reference(ihist), grid)
!
! Checking that mandatory parameters are initialized: 
!========= ==== ========= ========== === ============
!
      ierr=0
      if (plttyp.eq.secplt) then
        ierr=0
        if (isecpos.eq.spvgeo) then
          write(*,*) 'ERROR: missing value for isecpos.'
          ierr=ierr+1
        endif
        if (xsec1.eq.spvgeo) then
          write(*,*) 'ERROR: missing value for xsec1.'
          ierr=ierr+1
        endif
        if (ysec1.eq.spvgeo) then
          write(*,*) 'ERROR: missing value for ysec1.'
          ierr=ierr+1
        endif
        if (xsec2.eq.spvgeo) then
          write(*,*) 'ERROR: missing value for xsec2.'
          ierr=ierr+1
        endif
        if (ysec2.eq.spvgeo) then
          write(*,*) 'ERROR: missing value for ysec2.'
          ierr=ierr+1
        endif
        if (dels.eq.0.) then
          write(*,*) 'ERROR: missing value for dels.'
          ierr=ierr+1
        endif
        if (kms.eq.0.) then
          write(*,*) 'ERROR: missing value for kms.'
          ierr=ierr+1
        endif
        if (ierr.ne.0) stop
      endif
!
! Report plotting parameters.
!======= ======== ===========
!
      write(stdout,'(/2x,A/,/6x,A,3I4/9x,A,2F10.3)')
     & 'Parameters:', 'number of points (Lr,Mr,Nr) =', Lr,Mr,Nr,
     & 'basin dimensions (Lx,Le) = ', xbasin*0.001, ybasin*0.001
      if (plttyp.eq.cntplt .and. spherical) then
        keep_asp_ratio=.true.
        if (proj.eq.'LC') then
          write(stdout,'(/6x,A)')
     &           'LC - Lambert conformal conic projection.'
        elseif (proj.eq.'ST') then
          write(stdout,'(/6x,A)')
     &             'ST - Stereographic azimuthal projection.'
        elseif (proj.eq.'OR') then
          write(stdout,'(/6x,A)')
     &              'OR - Orthographic azimuthal projection.'
        elseif (proj.eq.'LE') then
          write(stdout,'(/6x,A)')
     &        'LE - Lambert equal area azimuthal projection.'
        elseif (proj.eq.'GN') then
          write(stdout,'(/6x,A)')
     &                  'GN - Gnomonic azimuthal projection.'
        elseif (proj.eq.'AE') then
          write(stdout,'(/6x,A)')
     &              'AE - Azimuthal equaidistant projection.'
        elseif (proj.eq.'SV') then
          write(stdout,'(/6x,A)')
     &             'SV - Satelite view azimuthal projection.'
        elseif (proj.eq.'CE') then
          write(stdout,'(/6x,A)')
     &            'CE - Cylindrical Equaidistant projection.'
        elseif (proj.eq.'ME') then
          write(stdout,'(/6x,A)') 'ME - Mercator projection.'
        elseif (proj.eq.'MO') then
          write(stdout,'(/6x,A)')
     &          'MO - Mollweide-type Cylindrical projection.'
        else
          write(stdout,'(/1x,4A/8x,A/)')      'ERROR: illegal ',
     &       'projection type ''',proj,''' is specified in cnt/ccnt',
     &       'input file. Chose among LC ST OR LE GN AE SV CE ME MO.'
          ierr=ierr+1 
        endif

        if (prlon.ne.spvgeo .and. pllon.ne.spvgeo .and.
     &      ptlat.ne.spvgeo .and. pblat.ne.spvgeo)  then
          if (plim1(1).eq.spvgeo .and. plim1(2).eq.spvgeo .and.
     &        plim2(1).eq.spvgeo .and. plim2(2).eq.spvgeo  .and.
     &        plim3(1).eq.spvgeo .and. plim3(2).eq.spvgeo  .and.
     &        plim4(1).eq.spvgeo .and. plim4(2).eq.spvgeo) then
            plim1(1)=pblat
            plim1(2)=pllon          !   plim4 ---- plim3
            plim2(1)=pblat          !     |          |
            plim2(2)=prlon          !     |          |
            plim3(1)=ptlat          !     |          |
            plim3(2)=prlon          !     |          |
            plim4(1)=ptlat          !   plim1 ---- plim2 
            plim4(2)=pllon
          else
            write(stdout,'(/1x,2A/8x,2A/8x,A)') 'ERROR: Ambiguous ',
     &               'specification of spherical map limits. Either',
     &            '(pllon:prlon: and pblat:ptlat:) or (plim12: and ',
     &               'plim43:),', 'but not both, should be defined.'
            ierr=ierr+1
          endif                 !--> discard pllon,prlon,ptlat,pblat
        endif
        if (ierr.ne.0) stop

        if (plon.ne.spvgeo .and. plat.ne.spvgeo .and. rota.ne.spvgeo)
     &     write(stdout,'(11x,A,3F9.3)') 'plon, plat, rota =',
     &                                              plon, plat, rota

        if (plim1(1).ne.spvgeo .and. plim1(2).ne.spvgeo .and.
     &      plim2(1).ne.spvgeo .and. plim2(2).ne.spvgeo  .and.
     &      plim3(1).ne.spvgeo .and. plim3(2).ne.spvgeo  .and.
     &      plim4(1).ne.spvgeo .and. plim4(2).ne.spvgeo) then 
          write(stdout,'(11x,A,2x,A,2F9.3/24x,A,2F9.3)')
     &     'map limits:', 'Latitude ', min(plim1(1),plim2(1)),
     &                                 max(plim3(1),plim4(1)),
     &                    'Longitude', min(plim1(2),plim4(2)),
     &                                 max(plim2(2),plim3(2))
        endif
      endif
      if (ntitles.gt.0) then
        write(stdout,*) ' '
        do i=1,ntitles
          if (title(i)(1:1).ne.' ') write(stdout,'(A)') title(i) 
        enddo
      endif
      return

  99  lstr=lenstr(infile)
      write(stdout,'(/1x,4A,I4/8x,3A)') 'ERROR: Can not read ', 
     &         'keyword ''', string(is(1):ie(1)), ''' on line',
     &                         line, 'in ', infile(1:lstr), '.'
      stop
      end



      subroutine read_buff (iin, ibuff, buff, nlines)
!
! Read input file/standard input, perform preliminary analysis and
! save lines which start with valid keywords into character array
! "buff". For each recorded string also determine and save its line
! in the file from which it was read, number of words, starting and
! ending indices of each word. Put them into array "ibuff".
!
      implicit none
      integer iin, maxstring, maxwords, nlines
      parameter (maxstring=80, maxwords=32)
      integer ibuff(2*maxwords+2,*), is(maxwords),ie(maxwords), nwords,
     &        i,ierr, line, case_fold, nqts, ndqts, nbcs, nsbcs, ncbcs
      character*(maxstring) string, buff(*)
      character*1 tab, quote, double_quote
      logical not_end_of_file, blank

      tab=char(9)
      quote=char(39)
      double_quote=char(34)
      case_fold=ichar('a')-ichar('A')

!                                 ! Reset string to all blank
! Read input file:                ! characters, then read in the next
! ==== ===== =====                ! line from the input script file,
!                                 ! suppress all quotation marks and
      line=0                      ! commas, after which find starting
      not_end_of_file=.true.      ! and ending positions of all words
      do while (not_end_of_file)  ! and check whether the first word
        line=line+1               ! is a keyword, i.e. it ends with
        do i=1,maxstring          ! a column symbol (':'). If so,
          string(i:i)=' '         ! process the string further and
        enddo
        read(iin,'(A)',iostat=ierr,end=1) string
        goto 2
   1    not_end_of_file=.false.   ! interpret all words in the rest
   2    if (ierr.eq.0) then       ! of the string according to the
          blank=.true.            ! meaning of the keyword. If not,
          nwords=0                ! discard and read the next string.

          nqts=0            ! Reset counters of quotes, double quotes;
          ndqts=0           ! round, square and curly braces: this is
          nbcs=0            ! needed to distinguis which commas needs
          nsbcs=0           ! to be left along and which suppess: the 
          ncbcs=0           ! ones which are outside of everything.

          do i=1,maxstring
            if (string(i:i).eq.tab) then
              string(i:i)=' '
            elseif (string(i:i).eq.quote) then
              string(i:i)=' '
              nqts=1-nqts
            elseif (string(i:i).eq.double_quote) then
              string(i:i)=' '
              ndqts=1-ndqts
            elseif (string(i:i).eq.'(') then
              nbcs=nbcs+1
            elseif (string(i:i).eq.')') then
              nbcs=nbcs-1
            elseif (string(i:i).eq.'[') then
              nsbcs=nsbcs+1
            elseif (string(i:i).eq.']') then
              nsbcs=nsbcs-1
            elseif (string(i:i).eq.'{') then
              ncbcs=ncbcs+1
            elseif (string(i:i).eq.'}') then
              ncbcs=ncbcs-1
            elseif (string(i:i).eq.',' .and. nqts.eq.0 .and.
     &                      ndqts.eq.0 .and. nbcs.eq.0  .and.
     &                      nsbcs.eq.0 .and. ncbcs.eq.0) then
              string(i:i)=' '
            endif

            if (blank .and. string(i:i).ne.' ') then
              blank=.false.
              nwords=nwords+1     ! keyword meaning. If not, discard
              is(nwords)=i
            elseif (.not.blank .and. string(i:i).eq.' ') then
              blank=.true.
              ie(nwords)=i-1
            elseif (.not.blank .and. i.eq.maxstring) then
              ie(nwords)=i
            endif                 ! the string and read next line.
          enddo
          if (string(is(1):is(1)).ne.'!' .and.  ! Discard column (:)
     &        string(ie(1):ie(1)).eq.':') then  ! at the end of key-
            string(ie(1):ie(1))=' '             ! word and fold upper
            ie(1)=ie(1)-1                       ! and lower cases.
            do i=is(1),ie(1)
              if (string(i:i).ge.'A' .and. string(i:i).le.'Z') then
                string(i:i)=char(ichar(string(i:i))+case_fold)
              endif
            enddo

c**         write(*,*) '''', string(is(1):ie(nwords)), ''''

            if (string(is(1):ie(1)).eq.'end' .and.
     &                      ie(1)-is(1).eq.2) then
              not_end_of_file=.false.
            elseif (nwords.gt.1) then
              nlines=nlines+1                   ! It is established
              ibuff(1,nlines)=line              ! that the first word
              ibuff(2,nlines)=nwords            ! of current string
              do i=1,nwords                     ! is a valid keyword,
                ibuff(2*i+1,nlines)=is(i)       ! therefore save the
                ibuff(2*i+2,nlines)=ie(i)       ! whole string and
              enddo                             ! associated indices
              buff(nlines)=string               ! into buffer arrays.
            endif
          endif
        endif   ! <-- ierr.eq.0
      enddo
      return
      end

