      subroutine hindices (isec,jsec, dxi,deta, xsec,ysec,ims,
     &                                               x,y,im,jm)
!
! For each horizontal point of the cross-section (xsec(n),ysec(n))
! find its placement on the logically rectangular grid corresponding
! to physical curvilinear grid (x,y). This placement is specified by
! a pair of indices (isec,jsec) indicating that point (xsec,ysec)
! contains within the grid cell bounded by vertices with indices
! [isec:isec+1,jsec:jsec+1], and a pair of local fractional distances
! (dxi,deta) which have meaning of distances from the middle of that
! cell to the point (xsec,ysec) expressed in terms of fractions of
! the grid size in each direction. Literally, "continuous indices", 
!
!                isec+0.5+dxi    jsec+0.5+deta
! 
! are coordinates of point (xsec,ysec) in the transformed rectangular
! coordinate system corresponding to mapping of physical curvilinear
! grid onto rectangle of fortran indices. 
! 
! input:  xsec,ysec X,Y-coordinates of section points (km or degrees)
!            ims    number of section points in horizontal direction
!            x,y    X,Y-coordinates of nodes regular grid
!           im,jm   number of field points in the X- and Y-directions
!
! output: isec,jsec indices indicating that point (xsec,ysec) belongs
!                   to grid cell isec:isec+1,jsec:jsec+1 on x,y grid
!          dxi,deta fractional 2D coordinate displacements indicating
!                   exact position of point xsec,ysec relatively to
!                   the center of the cell isec:isec+1,jsec:jsec+1.
!
      implicit none
      integer ims, im,jm, isec(ims), jsec(ims)
      real xsec(ims),ysec(ims), dxi(ims),deta(ims), x(im,jm),y(im,jm)

      logical found, Try_Range
      integer is,js, imin,imax, jmin,jmax, n, iter
      real*8 xi,  xs, dx, xy, e11,e12, a11,a12, d1,  
     &       eta, ys, dy, yx, e21,e22, a21,a22, d2, cff

#define CHECKING
#ifdef CHECKING
      real dx0,dx1,dx2,dx3, dy0,dy1,dy2,dy3, s01,s12,s23,s30
#endif

!
! For each point on the cross-section xsec(n),ysec(n) check first
! whether it belongs to the whole domain... 
!
      do n=1,ims
        found=Try_Range(1,im,1,jm, xsec(n),ysec(n), x,y, im,jm) 
        if (found) then 
          imin=1          ! ...once it is established that the point
          imax=im         ! belongs to the whole domain, find the
          jmin=1          ! exact box to which it belongs by dividing
          jmax=jm         ! the domain by half and checking.

          do while (imax-imin.gt.1 .or. jmax-jmin.gt.1)
            if (imax-imin .gt. 1) then
              is=(imin+imax)/2
              found=Try_Range(imin,is, jmin,jmax, xsec(n), ysec(n),
     &                                            x,y, im,jm)
              if (found) then
                imax=is
              else
                imin=is
              endif
            endif
            if (jmax-jmin .gt. 1) then
              js=(jmin+jmax)/2
              found=Try_Range(imin,imax, jmin,js, xsec(n), ysec(n),
     &                                            x,y, im,jm)
              if (found) then
                jmax=js
              else
                jmin=js
              endif
            endif
          enddo

#ifdef CHECKING
!
! Check, that point (xsec(n),ysec(n)) indeed belongs to the grid box
! (imin:imin+1,jmin:jmin+1). Since the grid box is a CONVEX polygon,
! if the point (xsec(n),ysec(n)) is inside, then all consecutive
! pairs of corners of the grid box numbered as 0 --> 1 --> 2 --> 3
! --> 0 must be seen either left --> right  or  right --> left (all
! the same), which is equivalent to the mathematical condition that
! all cyclic vector products must be of the same sign. Faulure of
! this checking indicates logical algorithm error in either of
! Try_Range or logical function inside.
!
          dx0=x(imin,jmin )-xsec(n) 
          dx1=x(imin+1,jmin )-xsec(n)
          dx2=x(imin+1,jmin+1)-xsec(n)
          dx3=x(imin  ,jmin+1)-xsec(n)

          dy0=y(imin,jmin )-ysec(n)
          dy1=y(imin+1,jmin )-ysec(n)
          dy2=y(imin+1,jmin+1)-ysec(n)
          dy3=y(imin  ,jmin+1)-ysec(n)
                                            ! Compute vector products
          s01=dx0*dy1-dy0*dx1               ! s_ij=[dx_i,dx_j] where
          s12=dx1*dy2-dy1*dx2               ! indices i,j constitite 
          s23=dx2*dy3-dy2*dx3               ! cyclic chain, i.e., for 
          s30=dx3*dy0-dy3*dx0               ! each i, j=i+1 if i<3,
                                            ! or 0 of i=3.
          if (s01.ge.0. and. s12.ge.0. .and.
     &        s23.ge.0. and. s30.ge.0.) then       ! Check that all 
            isec(n)=imin                           ! four cyclic 
            jsec(n)=jmin                           ! vector products 
          elseif (s01.le.0. and. s12.le.0. .and.   ! have the same 
     &            s23.le.0. and. s30.le.0.) then   ! sign.
            isec(n)=imin
            jsec(n)=jmin
          else 
            write(*,'(/A/)') 'ERROR: Algorithm failure in hindices.'
            stop
          endif 
#else
          isec(n)=imin
          jsec(n)=jmin
#endif
!
! Determine local fractional coordinates "xi,eta" corresponding to
! the target point "xs,ys" on the grid x,y ["xi,eta" are defined in
! such a way that xi=eta=0 corresponds to the middle of the cell
! (is:is+1,js:js+1), while xi=+/-1/2; eta=+/-1/2 (any combination
! +/-) signs corresponds to four corner point of the cell. Inside
! the sell it is accumed that "x,y" are expressed via bi-linear
! functions of "xi,eta", where term proportional to xi*eta does not
! vanish because coordinate transformation may be at least weakly
! non-orthogonal due to discretization errors. The associated non-
! linear system is solved by iterative method of Newton.
!
          xs=xsec(n)      ! Load coordinates into temporal variables.
          ys=ysec(n)      ! Note that computations here are performed
          is=isec(n)      ! with double precision to ensure proper
          js=jsec(n)      ! convergence of iterative procedure.

          xy=x(is+1,js+1)-x(is,js+1)-x(is+1,js)+x(is,js)
          yx=y(is+1,js+1)-y(is,js+1)-y(is+1,js)+y(is,js)

          dx=xs-0.25D0*(x(is+1,js+1)+x(is,js+1)+x(is+1,js)+x(is,js))
          dy=ys-0.25D0*(y(is+1,js+1)+y(is,js+1)+y(is+1,js)+y(is,js))

          e11=0.5D0*(x(is+1,js+1)-x(is,js+1)+x(is+1,js)-x(is,js))
          e12=0.5D0*(x(is+1,js+1)+x(is,js+1)-x(is+1,js)-x(is,js))
          e21=0.5D0*(y(is+1,js+1)-y(is,js+1)+y(is+1,js)-y(is,js))
          e22=0.5D0*(y(is+1,js+1)+y(is,js+1)-y(is+1,js)-y(is,js))

                                        ! Coordinate transformation
          cff=1.D0/(e11*e22-e12*e21)    ! matrix
          xi=cff*(e22*dx-e12*dy)        !         e11 e12 
          eta=cff*(e11*dy-e21*dx)       !         e21 e22 
                                        !
          do iter=1,4                       ! contains derivatives
            d1=dx -e11*xi-e12*eta-xy*xi*eta ! of x,y with respect 
            d2=dy -e21*xi-e22*eta-yx*xi*eta ! to xi, eta. 
                                            ! Because the coordinates
            a11=e11+xy*eta              ! may be non-orthogonal (at
            a12=e12+xy*xi               ! least due to discretization
            a21=e21+yx*eta              ! errors), nonlinear system
            a22=e22+yx*xi               !
                                        ! e11*xi+e12*eta+xy*xi*eta=dx
            cff=1.D0/(a11*a22-a12*a21)  ! e21*xi+e22*eta+yx*xi*eta=dy
            xi=xi + cff*(a22*d1-a12*d2) !
            eta=eta+cff*(a11*d2-a21*d1) ! needs to be solved in order
          enddo                         ! to retain symmetry.
#if VERBOSE > 3
          if (abs(xi).gt.0.500001 .or. abs(eta).gt.0.500001) then
             write(*,'(/1x,2A,2I4,2F12.8/)') 'ERROR in hindices: ',
     &                  'point is outside grid cell:', is,js, xi,eta
          endif
#endif
          dxi(n)=xi    ! Copy fractional displacements
          deta(n)=eta  ! to storage arrays.
        else
          isec(n)=0    ! Set indices to zero to indicate that point
          jsec(n)=0    ! xsec(n),ysec(n) is outside the whole domain.
        endif
c**     write(*,'(1x,A,2I4,2F12.8)') 'hindices:', is,js, xi,eta
      enddo
      return
      end




      logical function Try_Range (imin,imax,jmin,jmax, xs,ys,
     &                                            x,y, im,jm)
!
! Find if the point (x,y) is inside the box defined by the
! corners (imin,jmin) and (imax,jmax).
!
! Input: imin,imax  indices specifying logically rectangular (but not
!        jmin,jmax  necessarily physically rectangular) region within
!                           the curvilinear grid specified by x,y; 
!          xs,ys    X,Y-coordinates of point to be tested
!           x,y     arrays of X,Y coordinates of nodes of regular 
!                                 two-dimensional curvilinear grid
!          im,jm    numbers of grid points in X- and Y-directions.
!
! Output: Try_Range Logical switch =.true. if the point (xs,ys) is
!                         inside the region specified by imin...jmax,
!                         or .false. if otherwise.
!
! Copyright (c) 1996 Rutgers University 
!
      implicit none
#include "param.h"
      logical inside
      integer imin,imax,jmin,jmax, im,jm, i,j, nb, shft 
      real xs,ys, x(im,jm),  y(im,jm), xb(NX),yb(NX)

      nb=2*(jmax-jmin + imax-imin) ! Check if allocated number of
      if (nb .lt. NX) then         ! points is large enough, then if 
        shft=1-imin                ! it is OK, define closed polygon. 
        do i=imin,imax-1           ! NOTE that the last point 
          xb(i+shft)=x(i,jmin)     ! (xb(nb),yb(nb)) does not repeat 
          yb(i+shft)=y(i,jmin)     ! (xb(1),yb(1)). Instead in the 
        enddo                      ! procedure inside it is implied 
        shft=1-jmin +imax-imin     ! that the closing segment is 
        do j=jmin,jmax-1           ! (xb(nb),yb(nb))-->(xb(1),yb(1)), 
          xb(j+shft)=x(imax,j)     ! in fact, logical function inside
          yb(j+shft)=y(imax,j)     ! sets xb(nb+1)=xb(1) and also 
        enddo                      !              yb(nb+1)=yb(1).   
        shft=1+jmax-jmin + 2*imax-imin
        do i=imax,imin+1,-1
          xb(shft-i)=x(i,jmax)
          yb(shft-i)=y(i,jmax)
        enddo
        shft=1+ 2*jmax-jmin + 2*(imax-imin) 
        do j=jmax,jmin+1,-1
          xb(shft-j)=x(imin,j)
          yb(shft-j)=y(imin,j)      ! Check if point (xs,ys)
        enddo                       ! is inside of the polygon.

        Try_Range=inside (xs,ys, xb,yb, nb)
      else
        write(*,'(/1x,2A/8x,A,I5,A/)') 'ERROR: Try_Range: ',
     &             'insufficient length of scratch arrays.',
     &             'parameter NX must be at least',  nb, '.' 
        stop
      endif
      return
      end


      logical function inside (xs,ys, xb,yb, nb)
!
! Check if point (xs,ys) is inside the closed polygon specified
! by vertices (xb(k),yb(k), k=1...nb).
!
! NOTE 1: if the point (xs,ys) falls exactly on the boundary of the
!         polygon, it still considered as it is INSIDE the polygon.
!
! NOTE 2: this code does not rely on the setting xb(nb)=xb(1) and
!         yb(nb)=yb(1); instead it is assumed that the last closing
!         segment is  (xb(nb),yb(nb)) --> (xb(1),yb(1)).
!
! Algorithm: the desision whether the point is inside or outside
! the polygon is done by counting the number of crossings of the
! ray (xs,ys) --> (xs,-infinity), hereafter called meridian, by the
! boundary of the polygon. In this counting procedure, crossing is
! counted as +2, the crossing happens from left to right as index
! along the contour increases and -2, if from right to left. If
! the counting adds up to zero, then the point is outside, otherwise
! it is either inside or on the boundary.
!
! This algorithm is a modified version of algorithm of Reid, 1969,
! where all crossings were counted positive and the decision is made
! based on whether the number of crossings is even or odd. The new
! algorithm may produce different result that the Reid's in cases
! where xs accidentally coinsides with one of the xb(k), k=1,...,nb.
! in the new algorithm in this case the crossing is counted as +1
! or -1, depending the sign of xb(k+1)-xb(k). Crossings are not
! counted if xs=xb(k)=xb(k+1).
! Therefore, if, for example, xs=xb(ks) and ys>yb(ks), then
! if xb(ks-1) < xb(ks) < xb(ks+1), the crossing is counted twice,
! but with weight +1 (for segments with k=ks-1 and k=ks). Similarly
! if xb(ks-1) > xb(ks) > xb(ks+1), the crossing is counted twice
! with weight -1 each time. If, on the other hand, the meridian is
! only touches the boundary, i.e., for example, xb(ks-1) < xb(ks)=xs
! and xb(ks+1) < xb(ks)=xs, then the crossing is couned as +1 for
! segment k=ks-1 and -1 for segment k=ks, which results in no
! crossing.
!
! Reference: Reid, C., 1969: A long way from Euclid. Oceanography
!                                                   EMR, page 174.
!
      implicit none
      integer nb,                       k, inc, crossings
      real xs,ys, xb(nb+1), yb(nb+1),   dx1,dx2, dxy
!
! New set of boundary points.  Find intersections.
!
#define BLOCKING
#ifdef BLOCKING
      integer chunk_size
      parameter (chunk_size=128)
      integer indx(chunk_size), m,n,kk
#endif

      crossings=0                    ! Reset counter of crossings
      xb(nb+1)=xb(1)                 ! and close the contour of the
      yb(nb+1)=yb(1)                 ! polygon.

#ifdef BLOCKING
      do kk=0,nb-1,chunk_size                ! Optimized version:
        n=0                                  ! First select indices
        do k=kk+1,min(kk+chunk_size,nb)      ! of segments where
          if ((xb(k+1)-xs)*(xs-xb(k)).ge.0.  ! xb(k) is different
     &         .and. xb(k).ne.xb(k+1)) then  ! from xb(k+1) and xs
            n=n+1                            ! falls between them...
            indx(n)=k ! SGI R1000: pipelined !      ...then further
          endif       ! unrolled twice, 14   ! investigate these
        enddo         ! cycles/2 iterations  ! segments in a separate
                                             ! loop. Doing it in two
        do m=1,n                             ! stages takes less time
          k=indx(m)                          ! because the first loop
#else
        do k=1,nb                            ! can be pipelined while
#endif
          if (xb(k).ne.xb(k+1)) then         ! the second loop is...
            dx1=xs-xb(k)
            dx2=xb(k+1)-xs
            dxy=dx2*(ys-yb(k))-dx1*(yb(k+1)-ys)
            inc=0
            if (xb(k).eq.xs .and. yb(k).eq.ys) then
              crossings=1
              goto 1
            elseif ((dx1.eq.0. .and. ys.ge.yb(k  ))  .or.
     &              (dx2.eq.0. .and. ys.ge.yb(k+1))) then
              inc=1
            elseif (dx1*dx2.gt.0. .and.          ! See below for
     &        (xb(k+1)-xb(k))*dxy.ge.0.) then    ! the explanation
              inc=2                              ! of this logical
            endif                                ! condition...
            if (xb(k+1).gt.xb(k)) then
              crossings=crossings+inc        ! ...expected to be
            else                             ! short, so that very
              crossings=crossings-inc        ! little time is spent
            endif                            ! here, despite the fact
          endif                              ! that there this loop
        enddo                                ! can not be pipelined.
#ifdef BLOCKING
      enddo
#endif

 1    if (crossings.eq.0) then
        inside=.false.
      else
        inside=.true.
      endif
      return
      end


!
! Explanation of the logical condition: Suppose that there are two
! points (x1,y1)=(xb(k),yb(k)) and (x2,y2)=(xb(k+1),yb(k+1)) such
! that either (x1 < xs < x2) or (x1 > xs > x2), therefore meridian
! x=xs intersects the segment (x1,y1) --> (x2,x2) and the ordinate
! of the point of intersection is
!
!                 y1*(x2-xs) + y2*(xs-x1)
!             y = -----------------------
!                         x2-x1
!
! The mathematical statement that point (xs,ys) either coinsides with
! the point of intersection or lies to the north from it (ys >= y) is
! therefore equivalent to the statement
!
!         ys*(x2-x1) >= y1*(x2-xs) + y2*(xs-x1)    if   x2-x1 > 0
! or
!         ys*(x2-x1) <= y1*(x2-xs) + y2*(xs-x1)    if   x2-x1 < 0
!
! which, after noting that  ys*(x2-x1) = ys*(x2-xs + xs-x1) may be
! rewritten as
!
!        (ys-y1)*(x2-xs) + (ys-y2)*(xs-x1) >= 0    if   x2-x1 > 0
! or
!        (ys-y1)*(x2-xs) + (ys-y2)*(xs-x1) <= 0    if   x2-x1 < 0
!
! and the both versions can be merged into essentially the condition
! that (ys-y1)*(x2-xs)+(ys-y2)*(xs-x1)  has the same sign as x2-x1,
! i.e. product of these two must be positive or zero.
!
!





