      subroutine get_3dfld (day, xpos,ypos,zpos, s1,s2,np,
     &                           igrid, file, idfld, level)
!
! Read in the requested 3D field from input NetCDF file and
! interpolates at the requested depth.
!
! Input:  day      Day to read (real).
!         file     NetCDF file to read (primary or secondary; integer)
!         idfld    Identification of field to read (integer)
!         level    Depth (>0), level (>0), or isopycnal (>1000) to
!                  process (real). If level=0, it process all
!                  available levels.
!
! Output: xpos     X-positions of the requested field (real array)
!         ypos     Y-positions of the requested field (real array)
!         ypos     Z-positions of the requested field (real array)
!         s1       Scalar or vector x-component field (real array)
!         s2       Vector y-component (real array)
!         np       Number of points in requested field (integer)
!         igrid    Requested file staggered C-grid type:
!                     igrid=0  => None.
!                     igrid=1  => density points.
!                     igrid=2  => streamfunction points.
!                     igrid=3  => u-velocity points.
!                     igrid=4  => v-velocity points.
!                     igrid=5  => w-velocity points.
!                     igrid=6  => w-velocity points, range 0:N.
!                     igrid=7  => interior density points.
!
      implicit none
#include "param.h"
      real*8 day
      integer np, file, idfld
      real xpos(NV), ypos(NV), zpos(NV), s1(NV), s2(NV), level

      logical gotangle, gotrho, gotsalt, gotzeta
      integer igrid,ixgrid,iygrid, i,j,k,n, ij, ijk, imx,jmx,imy,jmy,
     &        nc_fid, nc_gid, nc_vid, nc_vxid, nc_vyid, status, ifld,
     &        time_indx
      real  spv, xspv, yspv, idens, svel
      character*4 fldnam

#include "pconst.h"
#include "domdat.h"
#include "fields.h"
#include "fldsid.h"
#include "ncinfo.h"
#include "ndimen.h"
#include "pltfld.h"
#include "pltio.h"
#include "pltlab.h"
#include "pltncar.h"
#ifdef VERBOSE
      write(*,'(1x,A)') 'Enter get_3dfld...'
#endif
!
! Determine NetCDF ID for input file to read and time index of the
! field to read.
!
      gotangle=.false.
      gotrho=.false.
      gotsalt=.false.
      gotzeta=.false.
      isosurf=.false.
      if (file.eq.1) then
        nc_fid=ncid1
        nc_gid=ncgrd
        gotangle=gotangle1
        gotrho=gotrho1
        gotsalt=gotsalt1
        gotzeta=gotzeta1
        tindx=time_indx (day,time1,ntime1,gottime1)
      elseif (file.eq.2) then
        nc_fid=ncid2
        nc_gid=ncid2
        gotangle=gotangle2
        gotrho=gotrho2
        gotsalt=gotsalt2
        gotzeta=gotzeta2
        tindx=time_indx (day,time2,ntime2,gottime1)
      else
        write(stdout,'(/1x,2A,I3/)') 'ERROR: GET_3DFLD: Can not ',
     &                        'determine input file type ', file
        stop
      endif
!
! Determine field(s) type and get field(s) name.
! If reading data for cross-sections, and processing one vector
! component, modify local variable "status" to read both components
! so the vector component can be rotated normal to the cross-section.
!
      levless=is2d(idfld)
      fldlab=vlabel(idfld)
      if (plttyp.eq.secplt .and. (vcode(idfld).eq.'IDUTOT'  .or.
     &                            vcode(idfld).eq.'IDUCLI')) then
        ifld=idfld+2
        status=2
        fldnam1=vname1(ifld)
        fldnam2=vname2(ifld)
      elseif (plttyp.eq.secplt .and. (vcode(idfld).eq.'IDVTOT' .or.
     &                                vcode(idfld).eq.'IDVCLI')) then
        ifld=idfld+1
        status=2
        fldnam1=vname1(ifld)
        fldnam2=vname2(ifld)
      else
        ifld=idfld
        status=vtype(idfld)
        fldnam1=vname1(idfld)
        fldnam2=vname2(idfld)
      endif
#define idfld ILLEGAL
!
! Process 3D fields.
!======== == =======
!
      if (.not.levless) then
!
! 3D scalar field.
!
        if (status.eq.1) then
          call get_nc3dat (s1,NV, im,jm,km, nc_fid, tindx,
     &                      fldnam1,  igrid,  nc_vid, spv)
!
! Baroclinic velocity components.
!
          if (vcode(ifld).eq.'IDUCLI') then
            fldnam='ubar'
            call get_nc2dat (f5,NV,  im,jm, nc_fid, tindx,
     &                       fldnam, level, ixgrid, nc_vxid,xspv)
            do k=1,km
              do j=1,jm
                do i=1,im
                  ij=i+(j-1)*im
                  ijk=ij+(k-1)*im*jm
                  s1(ijk)=s1(ijk)-f5(ij)
                enddo
              enddo
            enddo
          elseif (vcode(ifld).eq.'IDVCLI') then
            fldnam='vbar'
            call get_nc2dat (f5,NV,  im,jm, nc_fid, tindx,
     &                       fldnam, level, ixgrid, nc_vxid,xspv)
            do k=1,km
              do j=1,jm
                do i=1,im
                  ij=i+(j-1)*im
                  ijk=ij+(k-1)*im*jm
                  s1(ijk)=s1(ijk)-f5(ij)
                enddo
              enddo
            enddo
          endif
          call mskscl (s1, im,jm,km, ifld, igrid, spval1,spval1)
!
! 3D vector field.
!
        elseif (status.eq.2 .or. status.eq.6) then
          call get_nc3dat (f5,NV,   imx,jmx,km, nc_fid, tindx,
     &                     fldnam1, ixgrid,     nc_vxid,  xspv)
          call get_nc3dat (f6,NV,   imy,jmy,km, nc_fid,  tindx,
     &                     fldnam2, iygrid,     nc_vyid, yspv)
!
! Compute baroclinic velocity components.
!
          if (vcode(ifld).eq.'IDCMAG'  .or.
     &        vcode(ifld).eq.'IDCVEC') then
            fldnam='ubar'
            call get_nc2dat (s1,NV,  imx,jmx,       nc_fid, tindx,
     &                       fldnam, level, ixgrid, nc_vxid, xspv)
            fldnam='vbar'
            call get_nc2dat (s2,NV,  imy,jmy,       nc_fid, tindx,
     &                       fldnam, level, iygrid, nc_vyid, yspv)
            do k=1,km
              do j=1,jmx
                do i=1,imx
                  ij=i+(j-1)*imx
                  ijk=ij+(k-1)*imx*jmx
                  f5(ijk)=f5(ijk)-s1(ij)
                enddo
              enddo
              do j=1,jmy
                do i=1,imy
                  ij=i+(j-1)*imy
                  ijk=ij+(k-1)*imy*jmy
                  f6(ijk)=f6(ijk)-s2(ij)
                enddo
              enddo
            enddo
          endif
!
! Compute vector components at interior RHO-points or PSI-points.
!
          call mskscl (f5, imx,jmx,km, ifld, ixgrid, xspv,0.)
          call mskscl (f6, imy,jmy,km, ifld, iygrid, yspv,0.)
          if ((ixgrid.eq.1).and.(iygrid.eq.1)) then
            igrid=1
            im=imx
            jm=jmx
            np=im*jm*km
            do ijk=1,np
              s1(ijk)=f5(ijk)
              s2(ijk)=f6(ijk)
            enddo
          else  
            if (lvecrho) then
              igrid=7
            else
              igrid=2
            endif
            call cvector (s1,s2,im,jm,km, f5,imx,jmx, f6,imy,jmy,
     &                             igrid,     nc_gid,   gotangle)
          endif
!
! Compute vector magnitude.
!
          if (vcode(ifld).eq.'IDTMAG'.or.vcode(ifld).eq.'IDBMAG'.or.
     &        vcode(ifld).eq.'IDCMAG'.or.vcode(ifld).eq.'IDSMFM'.or.
     &        vcode(ifld).eq.'IDSWDM') then
            do n=1,im*jm*km
              if (s1(n).ne.spval1 .and. s2(n).ne.spval1) then
                s1(n)=sqrt(s1(n)*s1(n)+s2(n)*s2(n))
              else
                s1(n)=spval1
              endif
            enddo
          endif
!
! Compute kinetic energy.
!
          if (vcode(ifld).eq.'IDKENG') then
            do n=1,im*jm*km
              if (s1(n).ne.spval1 .and. s2(n).ne.spval1) then
                s1(n)=0.5*(s1(n)*s1(n)+s2(n)*s2(n))/rho0
              else
                s1(n)=spval1
              endif
            enddo
          endif                           
!
! Secondary field derived from two primary fields.
!
        elseif (status.eq.3) then
          call get_nc3dat (f5,NV,   im,jm,km, nc_fid, tindx,
     &                     fldnam1, igrid,    nc_vxid, xspv)
          call get_nc3dat (f6,NV,   im,jm,km, nc_fid,  tindx,
     &                     fldnam2, igrid,    nc_vyid, yspv)
!
! Compute density anomaly from temperature and salinity.
!
          if (vcode(ifld).eq.'IDCDEN') then
            call get_nc3pos (z,NV,  im,jm,km, nc_gid, nc_fid,
     &                       tindx, igrid, igrid, gotzeta,file)
            do n=1,im*jm*km
              if (f5(n).ne.spval1 .and. f6(n).ne.spval1) then
                s1(n)=idens(f5(n),f6(n),z(n))
              else
                s1(n)=spval1
              endif
            enddo
!
! Compute sigma-t from temperature and salinity.
!
          elseif (vcode(ifld).eq.'IDSIGT') then
             do n=1,im*jm*km
              if (f5(n).ne.spval1 .and. f6(n).ne.spval1) then
                s1(n)=idens(f5(n),f6(n),0.)
              else
                s1(n)=spval1
              endif
            enddo
!
! Compute sound speed (m/s) from temperature and salinity.
!
          elseif (vcode(ifld).eq.'IDSNDS') then
            call get_nc3pos (z,NV,  im,jm,km, nc_gid,  nc_fid,
     &                       tindx, igrid, igrid, gotzeta,file)
            do n=1,im*jm*km
              if (f5(n).ne.spval1 .and. f6(n).ne.spval1) then
                s1(n)=svel(f6(n),f5(n),abs(z(n)))
              else
                s1(n)=spval1
              endif
            enddo
          endif
          call mskscl (s1, im,jm,km, ifld, igrid, spval1,spval1)
!
! Special fields.
!======== =======
!
        elseif (4.le.status .and. status.le.6) then
!
! Brunt-Vaisala frequency.
!
          if (vcode(ifld).eq.'IDBVFQ') then

       write(*,*) 'ifld =', ifld

            call get_bvf (s1, xpos,ypos,zpos, im,jm,km, igrid,nc_gid,
     &                    nc_fid, ifld,tindx, level, gotzeta,gotsalt,
     &                    gotrho, file)
!
! Horizontal divergence fields.
!
          elseif ((vcode(ifld).eq.'IDTVHD').or.
     &            (vcode(ifld).eq.'IDCVHD')) then
            call get_hdiv (s1,xpos,ypos,zpos,im,jm,km,igrid,nc_gid,
     &                     nc_fid,ifld,tindx,level,gotzeta,file)
!
! Relative vorticity fields.
!
          elseif (vcode(ifld).eq.'IDTVRV') then
            call get_rvor (s1,xpos,ypos,zpos,im,jm,km,igrid,nc_gid,
     &                     nc_fid,ifld,tindx,level,gotzeta,file)
!
! Potential vorticity
!
          elseif (vcode(ifld).eq.'IDPVOR') then
            call get_pv (s1,xpos,ypos,z,im,jm,km,igrid,nc_gid,nc_fid,
     &                 ifld,tindx,level,gotzeta,gotsalt,gotrho,file)
!
! Ertel potential vorticity.
!
          elseif (vcode(ifld).eq.'IDERPV') then
            call get_ertel (s1,xpos,ypos,zpos,im,jm,km,igrid,nc_gid,
     &                      nc_fid,ifld,tindx,level,gotzeta,file)
!
! Available potential energy.
!
          elseif (vcode(ifld).eq.'IDPENG') then
            call get_ape (s1,xpos,ypos,zpos,im,jm,km,igrid,nc_gid,
     &                    nc_fid,ifld,tindx,level,gotzeta,gotsalt,
     &                    gotrho,file)
!
! Dynamic pressure.
!
          elseif (vcode(ifld).eq.'IDDPRS') then
            call get_dpres (s1,xpos,ypos,z,im,jm,km,igrid,
     &                      nc_gid,nc_fid, ifld,tindx,level,
     &                      gotzeta, gotsalt, gotrho, file)
!
! Bernoulli function.
!
          elseif (vcode(ifld).eq.'IDBERN') then
            call get_bernoulli (s1,xpos,ypos,z,im,jm,km,igrid,nc_gid,
     &                          nc_fid,ifld,tindx,level,gotzeta,
     &                          gotsalt,gotrho,file)
!
! Gradient Richardson number.
!
          elseif (vcode(ifld).eq.'IDRIGN') then
            call get_rig (s1,xpos,ypos,zpos,im,jm,km,igrid,nc_gid,
     &                    nc_fid,ifld,tindx,level,gotzeta,gotsalt,
     &                    gotrho,file)
          endif
!
! If appropriate, scale and apply Land/Sea mask.
!
          call mskscl (s1, im,jm,km, ifld, igrid, spval1,spval1)
        else
          write(*,*) 'GET_3DFLD ERROR: unknown field type: ', status
          stop
        endif
!
! Read in field depths and horizontal positions.
!
        if (status.ge.1 .and. status.le.3 .or. status.eq.6) then
          call get_nc3pos (zpos,NV,im,jm,km,nc_gid,nc_fid,tindx,
     &                     igrid, igrid, gotzeta, file)
          call get_nc2pos (xpos,ypos,NV,im,jm,nc_gid,igrid)
        endif
!
! Set number of points in requested field.
!
        np=im*jm*km
      endif
#ifdef VERBOSE
      write(*,'(1x,A)') '...return from get_3dfld.'
#endif
      return
      end
