      subroutine get_2dfld (day, xpos,ypos, s1,s2, np,
     &                              file, idfld, level)
!
! Read in requested field, at the requested level and time from
! input NetCDF file.
!
! Input:  day      Day to read (real)
!         file     NetCDF file to read (primary or secondary; integer)
!         idfld    Identification of field to read (integer)
!         level    Depth (>0), level (>0), or isopycnal (>1000) to
!                  process (real).
!
! Output: xpos    X-positions of the requested field (real array)
!         ypos     Y-positions of the requested field (real array)
!         s1       Scalar or vector x-component field (real array)
!         s2       Vector y-component (real array)
!         np       Number of points in requested field (integer)
!
      implicit none
#include "param.h"
      real*8 day
      integer np, file, idfld
      real xpos(NV), ypos(NV), s1(NV), s2(NV), level

      logical gotangle, gotrho, gotsalt, gotzeta
      integer igrid,ixgrid,iygrid, imx,jmx, imy,jmy, ip,n, nc_fid,
     &        nc_gid, nc_vid, nc_vxid, nc_vyid, time_indx
      real spv, xspv,yspv,  idens,svel
      character*4 fldnam
#include "pconst.h"
#include "fields.h"
#include "fldsid.h"
#include "ncinfo.h"
#include "ndimen.h"
#include "pltfld.h"
#include "pltio.h"
#include "pltlab.h"
#include "pltncar.h"
#include "secbath.h"
#ifdef VERBOSE
      write(*,'(1x,A,1PE11.4, 2x,A,I3, 2x,A,I3, 2x,A,1PE10.3)')
     &          'Enter get_2dfld, day=', day,   'idfld=', idfld,
     &                'vtype=', vtype(idfld),   'level=', level
#endif
      
!
! Determine netCDF ID for input file to read and time index of the
! field to read.
!
      gotangle=.false.
      gotrho=.false.
      gotsalt=.false.
      gotzeta=.false.
      isosurf=.false.
      if (file.eq.1) then
        nc_fid=ncid1
        nc_gid=ncgrd
        gotangle=gotangle1
        gotrho=gotrho1
        gotsalt=gotsalt1
        gotzeta=gotzeta1
        tindx=time_indx (day,time1,ntime1,gottime1)
      elseif (file.eq.2) then
        nc_fid=ncid2
        nc_gid=ncid2
        gotangle=gotangle2
        gotrho=gotrho2
        gotsalt=gotsalt2
        gotzeta=gotzeta2
        tindx=time_indx (day,time2,ntime2,gottime2)
      else
        write(stdout,'(/1x,2A,I3/)') 'GET_2DFLD ERROR: Can not ',
     &                        'determine input file type ', file 
        stop
      endif
!
!  Process requested field.
!---------------------------
! First, determine the field type (1:scalar, 2:vector, 3:derived).
! Second, read field(s); if the requested field is secondary, then
! compute it from just read primary field(s).
! Third, mask/scale read field(s) if appropriate.
! And fourth, read field positions.  If "level" is positive, it
! processes fields at the requested level.  However, if "level"
! is negative, it processes  fields at the requested depth via
! interpolation.
!
!  Determine field(s) type and get field(s) name.
!
      levless=is2d(idfld)
      fldnam1=vname1(idfld)
      fldnam2=vname2(idfld)
      fldlab=vlabel(idfld)        
!
! Process field by levels.
!
      if (levless .or. (0..lt.level .and. level.lt.1000.)) then
!
! 2D scalar field.
!
        if (vtype(idfld).eq.1) then
!
! Compute bathymetry at PSI-points.
!
          if (vcode(idfld).eq.'IDBTHP') then
            call get_nc2dat (f5,NV, imx,jmx, nc_fid,tindx,
     &                    fldnam1,level, igrid,nc_vid,spv)
            im=imx-1
            jm=jmx-1
            igrid=2
            call cfield (s1,im,jm,1,f5,imx,jmx,1,igrid)
!
! Generic scalar fields.
!
          else
            call get_nc2dat (s1,NV, im,jm, nc_fid,tindx,
     &                     fldnam1,level,igrid,nc_vid,spv)
          endif
          call mskscl (s1, im,jm,1, idfld, igrid, spv,spval1)
!
! 2D vector field.
!
        elseif (vtype(idfld).eq.2 .or. vtype(idfld).eq.6) then
          call get_nc2dat (f5,NV, imx,jmx, nc_fid, tindx,  fldnam1,
     &                              level, ixgrid, nc_vxid,  xspv)
          call get_nc2dat (f6,NV, imy,jmy, nc_fid, tindx,  fldnam2,
     &                              level, iygrid, nc_vyid,   yspv)
!
! Compute baroclinic velocity components.
!
          if (vcode(idfld).eq.'IDCMAG' .or.
     &        vcode(idfld).eq.'IDCVEC') then
            fldnam='ubar'
            call get_nc2dat (s1,NV, imx,jmx, nc_fid, tindx, fldnam,
     &                                level, ixgrid, nc_vxid, xspv)
            fldnam='vbar'
            call get_nc2dat (s2,NV, imy,jmy, nc_fid, tindx, fldnam,
     &                                level, iygrid, nc_vyid, yspv)
            do n=1,imx*jmx
              f5(n)=f5(n)-s1(n)
            enddo
            do n=1,imy*jmy
              f6(n)=f6(n)-s2(n)
            enddo
          endif          
!
! Compute vector components at interior RHO-points or PSI-points.
!
          call mskscl (f5, imx,jmx,1, idfld, ixgrid, xspv,0.)
          call mskscl (f6, imy,jmy,1, idfld, iygrid, yspv,0.)
          if (ixgrid.eq.1 .and. iygrid.eq.1) then
            igrid=1
            im=imx
            jm=jmx
            np=im*jm
            do ip=1,np
              s1(ip)=f5(ip)
              s2(ip)=f6(ip)
            enddo
          else
            if (lvecrho) then
              igrid=7
            else
              igrid=2
            endif
            call cvector (s1,s2, im,jm,1, f5,imx,jmx, f6,imy,jmy,
     &                                   igrid, nc_gid, gotangle)
          endif
!
! Compute vector magnitude.
!
          if (vcode(idfld).eq.'IDTMAG' .or.
     &        vcode(idfld).eq.'IDBMAG'  .or.
     &        vcode(idfld).eq.'IDCMAG'  .or.
     &        vcode(idfld).eq.'IDSMFM'  .or.
     &        vcode(idfld).eq.'IDSWDM'  .or.
     &        vcode(idfld).eq.'IDSURM') then
            do n=1,im*jm
              if (s1(n).ne.spval1 .and. s2(n).ne.spval1) 
     &         s1(n)=sqrt(s1(n)*s1(n)+s2(n)*s2(n))
            enddo
          endif
!
! Secondary field derived from two primary fields.
!-------------------------------------------------
!
        elseif (vtype(idfld).eq.3) then
          call get_nc2dat (s1,NV,im,jm,nc_fid,tindx,fldnam1,level,
     &                     igrid,nc_vxid,xspv)
          call get_nc2dat (s2,NV,im,jm,nc_fid,tindx,fldnam2,level,
     &                     igrid,nc_vyid,yspv)
!
! Compute density anomaly from temperature and salinity.
!
          if (vcode(idfld).eq.'IDCDEN') then
            call get_nc3pos (z,NV,im,jm,Nr,nc_gid,nc_fid,tindx,igrid,
     &                       igrid,gotzeta,file)
            do n=1,im*jm
              ip=n+(int(level)-1)*im*jm
              if (s1(n).ne.spval1 .and. s2(n).ne.spval1) then
                s1(n)=idens(s1(n),s2(n),z(ip))
              else
                s1(n)=spval1
              endif
            enddo
!
! Compute Coriolis/Depth ratio (1/ms).
!
          elseif (vcode(idfld).eq.'IDFOHR') then
            do n=1,im*jm
              if (s1(n).ne.spval1 .and. s2(n).ne.spval1
     &                            .and. s2(n).ne.0.) then
                s1(n)=s1(n)/s2(n)
              else
                s1(n)=spval1
              endif
            enddo
!
! Compute local snow thickness (m).
!
          elseif (vcode(idfld).eq.'IDSHOA') then
            do n=1,im*jm
              if (s1(n).ne.spval1 .and. s2(n).ne.spval1
     &                            .and. s2(n).ne.0.) then
                s1(n)=s1(n)/s2(n)
              else
                s1(n)=spval1
              endif
            enddo
!
! Compute local ice thickness (m).
!
          elseif (vcode(idfld).eq.'IDIHOA') then
            do n=1,im*jm
              if (s1(n).ne.spval1 .and. s2(n).ne.spval1
     &                            .and. s2(n).ne.0.) then
                s1(n)=s1(n)/s2(n)
              else
                s1(n)=spval1
              endif
            enddo
!
! Compute sigma-t from temperature and salinity.
!
          elseif (vcode(idfld).eq.'IDSIGT') then
            do n=1,im*jm
              if (s1(n).ne.spval1 .and. s2(n).ne.spval1) then
                s1(n)=idens(s1(n),s2(n),0.)
              else
                s1(n)=spval1
              endif
            enddo
!
! Compute sound speed (m/s) from temperature and salinity.
!
          elseif (vcode(idfld).eq.'IDSNDS') then
            call get_nc3pos (z,NV,im,jm,Nr,nc_gid,nc_fid,tindx,
     &                       igrid, igrid,gotzeta,file)
            do n=1,im*jm
              if (s1(n).ne.spval1 .and. s2(n).ne.spval1) then
                ip=n+(int(level)-1)*im*jm
                s1(n)=svel(s2(n),s1(n),abs(z(ip)))
              else
                s1(n)=spval1
              endif
            enddo
          endif
          call mskscl (s1, im,jm,1, idfld, igrid, spval1,spval1)
!
! Special fields: vector or scalar.
!----------------------------------
!
        elseif (vtype(idfld).ge.4 .and. vtype(idfld).le.6) then
!
! Depths of requested field iso-surface: temperature, salinity,
! "in situ" density, sigma-t, or potential density.
!
          if (vcode(idfld).eq.'IDISOT' .or.
     &        vcode(idfld).eq.'IDISOH'  .or.
     &        vcode(idfld).eq.'IDISOR'  .or.
     &        vcode(idfld).eq.'IDISOD'  .or.
     &        vcode(idfld).eq.'IDISOS') then
            call get_isosur (s1,xpos,ypos,im,jm,igrid,nc_gid,nc_fid,
     &                       idfld,tindx,gotzeta,gotsalt,gotrho,file)
!
! Maximum entropy equilibrium transport.
!
          elseif (vcode(idfld).eq.'IDNPYT') then
            call get_entropy (s1, s2,f5, xpos,ypos, im,jm, igrid,
     &                                   nc_gid, idfld, gotangle)
!
! Maximum entropy equilibrium velocity.
!
          elseif (vcode(idfld).eq.'IDUNPY' .or.
     &            vcode(idfld).eq.'IDVNPY'  .or.
     &            vcode(idfld).eq.'IDNPYV'  .or.
     &            vcode(idfld).eq.'IDNPYM') then
            call get_entropy (f5, s1,s2, xpos,ypos, im,jm, igrid,
     &                                   nc_gid, idfld, gotangle)
!
! Joint Effect of Baroclinicity and bottom Relief (JEBAR).
!
          elseif (vcode(idfld).eq.'IDMHUJ' .or.
     &            vcode(idfld).eq.'IDMHVJ'  .or.
     &            vcode(idfld).eq.'IDMHJV'  .or.
     &            vcode(idfld).eq.'IDMHJM'  .or.
     &            vcode(idfld).eq.'IDRHUJ'  .or.
     &            vcode(idfld).eq.'IDRHVJ'  .or.
     &            vcode(idfld).eq.'IDRHJV'  .or.
     &            vcode(idfld).eq.'IDRHJM') then
            call get_jebar (s1,s2,xpos,ypos,z,im,jm,igrid,
     &                      nc_gid,nc_fid, idfld,tindx,
     &                      gotzeta,gotsalt,gotrho,gotangle, file)
!
! Brunt-Vaisala frequency.
!
          elseif (vcode(idfld).eq.'IDBVFQ') then
       write(*,*) 'BVF'
            call get_bvf (s1,xpos,ypos,z,im,jm,km, igrid,
     &                    nc_gid,nc_fid,idfld, tindx,level,
     &                    gotzeta,gotsalt,gotrho,file)
!
! Horizontal divergence fields.
!
          elseif (vcode(idfld).eq.'IDTVHD' .or.
     &            vcode(idfld).eq.'IDBVHD'  .or.
     &            vcode(idfld).eq.'IDCVHD'  .or.
     &            vcode(idfld).eq.'IDSWHD') then
            call get_hdiv (s1,xpos,ypos,z,im,jm,km,igrid,nc_gid,
     &                     nc_fid,idfld,tindx,level,gotzeta,file)
!
! Relative vorticity fields.
!
          elseif (vcode(idfld).eq.'IDTVRV' .or.
     &            vcode(idfld).eq.'IDBVRV'  .or.
     &            vcode(idfld).eq.'IDCVRV'  .or.
     &            vcode(idfld).eq.'IDSWRV') then
            call get_rvor (s1,xpos,ypos,z,im,jm,km,igrid,nc_gid,
     &                     nc_fid,idfld,tindx,level,gotzeta,file)
!
! Potential vorticity
!
          elseif (vcode(idfld).eq.'IDPVOR') then
            call get_pv (s1,xpos,ypos,z,im,jm,km,igrid,nc_gid,nc_fid,
     &                 idfld,tindx,level,gotzeta,gotsalt,gotrho,file)
!
! Ertel potential vorticity.
!
          elseif (vcode(idfld).eq.'IDERPV') then
            call get_ertel (s1,xpos,ypos,z,im,jm,km,igrid, nc_gid,
     &                      nc_fid,idfld,tindx,level,gotzeta,file)
!
! Available potential energy.
!
          elseif (vcode(idfld).eq.'IDPENG') then
            call get_ape (s1,xpos,ypos,z,im,jm,km,igrid,
     &                    nc_gid,nc_fid, idfld,tindx,level,
     &                    gotzeta, gotsalt,gotrho,file)
!
! Dynamic pressure.
!
          elseif (vcode(idfld).eq.'IDDPRS') then
            call get_dpres (s1,xpos,ypos,z,im,jm,km,igrid,nc_gid,
     &                      nc_fid, idfld,tindx,level,gotzeta,
     &                      gotsalt,gotrho, file)
!
! Bernoulli function.
!
          elseif (vcode(idfld).eq.'IDBERN') then
            call get_bernoulli (s1,xpos,ypos,z,im,jm,km,igrid,nc_gid,
     &                          nc_fid,idfld,tindx,level,gotzeta,
     &                          gotsalt,gotrho,file)
!
! Bathymetry slope or bathymetry r-factor.
!
          elseif (vcode(idfld).eq.'IDMHSL' .or.
     &            vcode(idfld).eq.'IDRHSL'  .or.
     &            vcode(idfld).eq.'IDMHRF'  .or.
     &            vcode(idfld).eq.'IDRHRF') then
            call get_hslope (s1,xpos,ypos,im,jm,igrid,nc_gid,
     &                           nc_fid, idfld,tindx,gotzeta)
!
! Gradient Richardson number.
!
          elseif (vcode(idfld).eq.'IDRIGN') then
            call get_rig (s1,xpos,ypos,z,im,jm,km,igrid,nc_gid,
     &                    nc_fid, idfld,tindx,level,gotzeta,
     &                                      gotsalt,gotrho,file)
!
! Surface wind stress curl.
!
          elseif (vcode(idfld).eq.'IDSWSC') then
            call get_wcurl (s1,xpos,ypos,im,jm,igrid,nc_gid,nc_fid,
     &                      idfld,tindx)
          endif
!
! If appropriate, scale and apply Land/Sea mask.
!
          call mskscl (s1, im,jm,1, idfld, igrid, spval1,spval1)
          if (vtype(idfld).eq.5 .or. vtype(idfld).eq.6) then 
            call mskscl (s2, im,jm,1, idfld, igrid, spval1,spval1)
          endif
        else
          write(*,*) 'ERROR: unknown field type.'
          stop
        endif
!
! Read in field positions.
!
        if (vtype(idfld).eq.1 .or. vtype(idfld).eq.2 .or.
     &      vtype(idfld).eq.3 .or. vtype(idfld).eq.6)  then
          call get_nc2pos ( xpos,ypos, NV, im,jm,nc_gid,igrid)
        endif
!
! Set number of points in requested field.
!
        np=im*jm
!
! Process field by depths or depth of isopycnal; intepolate
! vertically.
!
      elseif (level.le.0. .or. level.ge.1000.) then
        interpolate=.true.
        if (level.ge.1000.) then
          call zisopyc (day,ziso,s1,s2,file,level,idfld)
        else
          np=Lr*Mr
          do n=1,np
            depth(n)=level
          enddo
        endif
        call get_3dfld (day, xpos,ypos, z, s1,s2, np,
     &                     igrid, file, idfld, level)
        if (interpolate) then
          if (level.ge.1000.) then         
            call cfield (depth,im,jm,1,ziso,Lr,Mr,1,igrid)
          endif
          if (vtype(idfld).eq.2 .or. vtype(idfld).eq.5 .or.
     &                               vtype(idfld).eq.6) then
            do n=1,np
              f5(n)=s1(n)
              f6(n)=s2(n)
            enddo
            call xhslice (s1,depth,z,f5,im,jm,km,vintrp,spval1)
            call xhslice (s2,depth,z,f6,im,jm,km,vintrp,spval1)
          else
            do n=1,np
              f5(n)=s1(n)
            enddo
            call xhslice (s1,depth,z,f5,im,jm,km,vintrp,spval1)
          endif
        endif
      endif
#ifdef VERBOSE
      write(*,'(1x,A)') '...return from get_2dfld.' 
#endif
      return
      end
