      subroutine geodesic_dist (lon1,lat1,lon2,lat2,flag,gdist,galpha)
!
! Inverse, non-iterative solutions for distance and geodesic azimuth
! between two points on the ellipsoid (The Earth) from the equations,
! second order in spheroidal flattening, given by:
!
! Sodano , E.M., and T. Robinson, 1963: Direct and inverse solutions 
!   of geodesics, Army Map Service Technical Report No. 7, AD 657591.
!
! input:    Longitude is positive to the east and negative to the
!           west.  Latitude is positive to the north and negative
!           to the south.
!
!     LON1    Longitude point 1 (decimal degrees, real*8)
!     LAT1    Latitude  point 1 (decimal degrees, real*8)
!     LON2    Longitude point 2 (decimal degrees, real*8)
!     LAT2    Latitude  point 2 (decimal degrees, real*8)
!     FLAG    flag for distance units on output (integer)
!
! output:
!
!     GALPHA  Geodesic azimuth from point 1 to point 2 clockwise from
!             North (decimal degrees, real*8)
!     GDIST   Geodesic distance between point 1 and point 2 (real*8)
!
!             Units of distance
!
!               Flag    Units
!              ------  -------
!                 1     Meters
!                 2     Nautical Miles
!                 3     Feet
!                 4     Kilometers
!                 5     Statute Mile
!
      implicit none
      integer flag
      real*4 lon1,lat1,lon2,lat2, gdist,galpha
      real*8 a, adist, alpha, b, beta1, beta2, c, cott, ct, delta, dist,
     &       d_lat1, d_lat2, d_lon1, d_lon2,  l, lambda, m, q, r_lat1,
     &       r_lat2, sob, st, t, w, x, y, z
!
! Define parameters on first pass (SMIN: Ellipsoid semi-minor axis in
! meters; SMAJ: Ellipsoid semi-major axis in meters; F: spheroidal
! flattening).
!
      real*8 pi, deg2rad, rad2deg, smin, smaj, f 
      parameter (pi=3.14159265358979323846D0,  deg2rad=pi/180.D0,
     &           smin=6356750.52D0,            rad2deg=180.D0/pi,
     &           smaj=6378135.00D0,           f=(smaj-smin)/smaj)

      write(*,*) 'geodesic_dist'

!
! Return if zero distance.
!
      if (lon1.eq.lon2 .and. lat1.eq.lat2) then
        gdist=0.
        galpha=0.
        return
      endif
!
! Convert input data to double precision.
!
      d_lon1=dble(lon1)
      d_lat1=dble(lat1)
      d_lon2=dble(lon2)
      d_lat2=dble(lat2)
!
! Determine proper longitudinal shift.
!
      delta=d_lon2-d_lon1
      l=abs(delta)
      if (l .ge. 180.D0) l=360.D0-abs(d_lon1-d_lon2)
!                  
! Convert Decimal degrees to radians.
!
      r_lat1=d_lat1*deg2rad
      r_lat2=d_lat2*deg2rad
      l=l*deg2rad
!
! Calculate S/Bo subformulas.
!
      beta1=datan(dtan(r_lat1)*(1.D0-f))
      beta2=datan(dtan(r_lat2)*(1.D0-f))
      a=dsin(beta1)*dsin(beta2)
      b=dcos(beta1)*dcos(beta2)
      ct=a+b*dcos(l)
      st=dsqrt( ((dsin(l)*dcos(beta2))**2)
     &         +(((dsin(beta2)*dcos(beta1))
     &         -(dsin(beta1)*dcos(beta2)*dcos(l)))**2) )
      t=dasin(st)
      c=(b*dsin(l))/st
      m=1.D0-c*c
!
! Calculate S/Bo term.
!
      q=f+f*f
      z=0.5D0*f*f
      x=0.0625D0*f*f
      y=0.125D0*f*f
      w=0.25D0*f*f

      sob=((1.D0+q)*t)+(a*((q*st)-(z*(t*t)*(1.D0/dsin(t)))))+
     &(m*(((-q*0.5D0)*t)-((q*0.5D0)*st*ct)+(z*(t*t)*(1.D0/dtan(t)))))
     &   +((a**2)*(-z*st*ct))+
     &    ((m**2)*((x*t)+(x*st*ct)-(z*(t*t)*(1.D0/dtan(t)))-
     &     (y*st*(ct*ct*ct))))+
     &    ((a*m)*((z*(t*t)*(1.D0/dsin(t)))+(z*st*(ct*ct))))
!
! Compute geodesic azimuth from point 1 to point 2 clockwise from
! North, alpha.
!
      lambda=q*t+a*(-z*st-f*f*t*t/dsin(t))+
     &       m*(-5.D0*w*t+w*st*dcos(t)+f*f*t*t/dtan(t))
      lambda=c*lambda+l
      if (lambda .eq. 0.D0) then
        if (d_lat1.lt.d_lat2) alpha=0.D0
        if (d_lat1.gt.d_lat2) alpha=180.D0
        goto 10
      endif

      cott=( dcos(beta1)*dsin(beta2) - dsin(beta1)*dcos(beta2)
     &               *dcos(lambda) )/(dcos(beta2)*dsin(lambda))

      if (cott .eq. 0.D0) then
        alpha=90.D0
      else
        alpha=datan(1.D0/cott)*rad2deg
      endif
!
! Compute heading from point 1 to point 2 clockwise from north.
!
      if (delta .gt. 0.D0) then
        if (cott .gt. 0.D0) then
          alpha=alpha
        elseif (cott .lt. 0.D0) then
          alpha=180.D0+alpha
        endif
      endif
      if (delta .lt. 0.D0) then
        if (cott .lt. 0.D0) then
          alpha=180.D0-alpha
        elseif (cott .gt. 0.D0) then
          alpha=360.D0-alpha
        endif
      endif
!
! Calculate distance from point 1 to point 2.
!
  10  adist=sob*smin
!
! Check flag for proper output units.
!
      if (flag.eq.1) dist=adist                 ! meters
      if (flag.eq.2) dist=adist*5.396D-4        ! nautical miles
      if (flag.eq.3) dist=adist*3.281D0         ! feet
      if (flag.eq.4) dist=adist*1.D-3           ! kilometers
      if (flag.eq.5) dist=adist*6.214D-4        ! statute mile
!
! convert to single precision.
!
      gdist=dist
      galpha=alpha
      return
      end
