      subroutine bv_freq (bvf, t,s,z_w, im,jm,N)
!
! Compute Brunt-Vaisala frequency  (1/s^2) at horizontal RHO-points
! and vertical W-points,
!
!                   bvf = - g/rho0 d(rho)/d(z)
!
! The potential density difference is computed lowering/rising the
! water parcel above/below adiabatically at W-point depth "z_w".
!
! input: t,s    [potential] temperature (C) and salinity (PSU)
!        z_w    depth (m) at W-points.
!        im,jm  number of points in XI- and ETA-directions
!        N      number of points in vertical direction.
!
! output: bvf      Brunt-Vaisala frequency (1/s^2).
!
! Reference:
!
! << This equation of state formulation has been derived by Jackett
! and McDougall (1992), unpublished manuscript, CSIRO, Australia. It
! computes in-situ density anomaly as a function of potential
! temperature (Celsius) relative to the surface, salinity (PSU),
! and depth (meters).  It assumes  no  pressure  variation along
! geopotential  surfaces,  that  is,  depth  and  pressure  are
! interchangeable. >>
!                                          John Wilkin, 29 July 92
!
      implicit none
#include "domdat.h"

      integer im,jm,N, i,j,k 
      real bvf(0:im,0:jm,N-1), t(0:im,0:jm,N), s(0:im,0:jm,N),
     &     z_w(0:im,0:jm,0:N), rho1(0:im,N),  K0,K1,K2,
#define SPLIT_EOS
#ifdef SPLIT_EOS
     &                         qp1(0:im,N)
#else
     &                         K_up(0:im,N), K_dw(0:im,N)
#endif

      real r00,r01,r02,r03,r04,r05,    K00,K01,K02,K03,K04,  dr00,
     &     r10,r11,r12,r13,r14,        K10,K11,K12,K13,      Ts, Tt,
     &     rS0,rS1,rS2,                KS0,KS1,KS2,          sqrtTs,
     &     r20,                                              dpth,
     &             B00,B01,B02,B03,    E00,E01,E02,
     &             B10,B11,B12,        E10,E11,E12,          cff,
     &                 BS1,                                  cff1

      parameter(r00=999.842594,   r01=6.793952E-2,  r02=-9.095290E-3,
     &                            r03=1.001685E-4,  r04=-1.120083E-6,
     &                                              r05=6.536332E-9,
     &          r10=0.824493,     r11=-4.08990E-3,  r12=7.64380E-5,
     &                            r13=-8.24670E-7,  r14=5.38750E-9,
     &          rS0=-5.72466E-3,  rS1=1.02270E-4,   rS2=-1.65460E-6,
     &          r20=4.8314E-4,

     &          K00=19092.56,     K01=209.8925,     K02=-3.041638,
     &                            K03=-1.852732e-3, K04=-1.361629e-5,
     &          K10=104.4077,     K11=-6.500517,    K12=0.1553190,
     &                                              K13=2.326469e-4,
     &          KS0=-5.587545,    KS1=+0.7390729,   KS2=-1.909078e-2,


     &          B00=0.4721788,    B01=0.01028859,   B02=-2.512549e-4,
     &                                              B03=-5.939910e-7,
     &          B10=-0.01571896,  B11=-2.598241e-4, B12=7.267926e-6,
     &                            BS1=2.042967e-3,

     &          E00=+1.045941e-5, E01=-5.782165e-10,E02=+1.296821e-7,
     &          E10=-2.595994e-7, E11=-1.248266e-9, E12=-3.508914e-9)

      real g
      parameter (g=9.81)
      dr00=r00-rho0

      do j=0,jm
        do k=1,N
          do i=0,im
            Tt=t(i,j,k)
            Ts=s(i,j,k)
            sqrtTs=sqrt(Ts)

            rho1(i,k)=( dr00 +Tt*( r01+Tt*( r02+Tt*( r03+Tt*(
     &                                           r04+Tt*r05 ))))
     &                         +Ts*( r10+Tt*( r11+Tt*( r12+Tt*(
     &                                            r13+Tt*r14 )))
     &                              +sqrtTs*(rS0+Tt*(
     &                                   rS1+Tt*rS2 ))+Ts*r20 ))

            K0= Tt*( K01+Tt*( K02+Tt*( K03+Tt*K04 )))
     &         +Ts*( K10+Tt*( K11+Tt*( K12+Tt*K13 ))
     &              +sqrtTs*( KS0+Tt*( KS1+Tt*KS2 )))

#ifdef SPLIT_EOS
            qp1(i,k)=0.1*(K00*rho1(i,k)-rho0*K0)/(K00*(K00+K0))
#else 
            K1=B00+Tt*(B01+Tt*(B02+Tt*B03)) +Ts*( B10+Tt*( B11
     &                                    +Tt*B12 )+sqrtTs*BS1 )

            K2=E00+Tt*(E01+Tt*E02) +Ts*(E10+Tt*(E11+Tt*E12))

            dpth=z_w(i,j,N)-z_w(i,j,k)
            K_up(i,k)=K0+dpth*(K1+K2*dpth)
            dpth=z_w(i,j,N)-z_w(i,j,k-1)
            K_dw(i,k)=K0+dpth*(K1+K2*dpth)
#endif
          enddo
        enddo

        cff=g/rho0
        do k=1,N-1
          do i=0,im
#ifdef SPLIT_EOS
            bvf(i,j,k)=-cff*( rho1(i,k+1)-rho1(i,k)    ! Adiabatic
     &                        +(qp1(i,k+1)-qp1(i,k))   ! elementary
     &                      *(z_w(i,j,N)-z_w(i,j,k))   ! difference
     &           )/(0.5*(z_w(i,j,k+1)-z_w(i,j,k-1)))
#else
            cff1=0.1*(z_w(i,j,N)-z_w(i,j,k))
            bvf(i,j,k)=-cff*(   (rho1(i,k+1)-rho1(i,k))
     &                        *(K00+K_dw(i,k+1))*(K00+K_up(i,k))

     &                     -cff1*( rho0*(K_dw(i,k+1)-K_up(i,k))
     &                                +K00*(rho1(i,k+1)-rho1(i,k))

     &               +rho1(i,k+1)*K_dw(i,k+1) -rho1(i,k)*K_up(i,k)

     &          ) )/(  (K00+K_dw(i,k+1)-cff1)*(K00+K_up(i,k)-cff1)
     &                            *0.5*(z_w(i,j,k+1)-z_w(i,j,k-1))
     &                                                          )
#endif
          enddo
        enddo
      enddo
      return
      end
