#include "cppdefs.h"
#ifdef SMOLARKIEWICZ
      subroutine smol_adiff (trc,ta,ua,va,wa)
!
!--------------------------------------------------------------------
!  This subroutine computes the "anti-diffusion velocity" used to ===
!  supress the  numerical  diffusion  that is associated with the ===
!  upstream differencing operator for advection.  It is used when ===
!  the Smolarkiewicz advection scheme is activated.               ===
!                                                                 ===
!  On Input:                                                      ===
!                                                                 ===
!     trc       Non-advected tracer (scalar) field.               ===
!     ta        Advected tracer (scalar) field.                   ===
!     ua        Advection velocity component in XI-direction.     ===
!     va        Advection velocity component in ETA-direction.    ===
!     wa        Advection S-coordinate vertical velocity.         ===
!                                                                 ===
!  On Output:                                                     ===
!                                                                 ===
!     ua        Anti-diffusion velocity component in XI-direction. ==
!     va        Anti-diffusion velocity component in ETA-direction. =
!     wa        Anti-diffusion S-coordinate vertical velocity.    ===
!                                                                 ===
!  Calls:  u3dbc, v3dbc, w3dbc                                    ===
!                                                                 ===
!  The internal parameter EPS is used to insure zero anti-diffusion =
!  velocities when neighbor tracer values are all zero.            ==
!                                                                  ==
!--------------------------------------------------------------------
!
      implicit none
# include "param.h"
# include "pconst.h"
# include "grid.h"
# include "scalars.h"
# include "work.h"
 
      integer i, j, k
      real eps
      real ta(0:L,0:M,N), trc(0:L,0:M,N), ua(L,0:M,N), va(0:L,M,N),
     &      wa(0:L,0:M,0:N),
     &      Bdn(Lm,Mm,N), Bup(Lm,Mm,N), Tmax(Lm,Mm,N), Tmin(Lm,Mm,N),
     &        oTdTdx(Lm,Mm), udiff(Lm,Mm,N), uoTdTdx(Lm,Mm),
     &        oTdTde(Lm,Mm), vdiff(Lm,Mm,N), voTdTde(Lm,Mm),
     &        oTdTdz(Lm,Mm), wdiff(Lm,Mm,N), woTdTdz(Lm,Mm,N)
      equivalence (Bdn,f3d), (Bup,e3d), (Tmax,g3d), (Tmin,g3d),
     &            (oTdTdx,b2d), (udiff,e3d), (uoTdTdx,c2d),
     &            (oTdTde,b2d), (vdiff,f3d), (voTdTde,d2d),
     &            (oTdTdz,b2d), (wdiff,g3d), (woTdTdz,g3d)
      parameter (eps=1.0e-10)
 
      real a1, a2, a3, a4, av2, av4
# include "avg.h"
!
c  Compute anti-diffusion velocity component in the XI-direction.
c--------------------------------------------------------------------
c
c  Compute w/T d(T)/d(z) term at horizontal U-points and vertical
c  RHO-points.
c
      do k=1,Nm
        do j=1,Mm
          do i=2,Lm
            e3d(i,j,k)=av2(ta (i  ,j,k+1)-ta (i  ,j,k),
     &                     ta (i-1,j,k+1)-ta (i-1,j,k))/
     &                 av2(z_r(i  ,j,k+1)-z_r(i  ,j,k),
     &                     z_r(i-1,j,k+1)-z_r(i-1,j,k))
          enddo
        enddo
      enddo
      do j=1,Mm
        do i=2,Lm
          e3d(i,j,0)=0.
          e3d(i,j,N)=0.
        enddo
      enddo
      do k=1,N
        do j=1,Mm
          do i=2,Lm
            woTdTdz(i,j,k)=av4(wa(i,j,k-1),wa(i-1,j,k-1),
     &                         wa(i,j,k  ),wa(i-1,j,k  ))*
     &                     av2(e3d(i,j,k),e3d(i,j,k-1))/
     &                     (av2(ta(i,j,k),ta(i-1,j,k))+eps)
          enddo
        enddo
      enddo
c
c  Compute anti-diffusion velocity component in the XI-direction.
c
      do k=1,N
c
c  Compute 1/T d(T)/d(xi) term at horizontal U-points and vertical
c  RHO-points.
c
        do j=1,Mm
          do i=2,Lm
            a2d(i,j)=av2(pm(i,j),pm(i-1,j))*(ta(i,j,k)-ta(i-1,j,k))
          enddo
        enddo
        do j=1,Mm
          do i=2,Lm
            oTdTdx(i,j)=a2d(i,j)/(av2(ta(i,j,k),ta(i-1,j,k))+eps)
          enddo
        enddo
c
c  Compute v/T d(T)/d(eta) term at horizontal U-points and vertical
c  RHO-points.
c
        do j=1,M
          do i=1,L
            a2d(i,j)=av2(pn(i,j),pn(i,j-1))*(ta(i,j,k)-ta(i,j-1,k))
          enddo
        enddo
        do j=1,Mm
          do i=2,Lm
            voTdTde(i,j)=av4(va(i,j  ,k),va(i-1,j  ,k),
     &                       va(i,j+1,k),va(i-1,j+1,k))*
     &                   av4(a2d(i,j  ),a2d(i-1,j  ),
     &                       a2d(i,j+1),a2d(i-1,j+1))/
     &                   (av2(ta(i,j,k),ta(i-1,j,k))+eps)
          enddo
        enddo
 
c
c  Compute anti-diffusion velocity component in the XI-direction.
c
        do j=1,Mm
          do i=2,Lm
            udiff(i,j,k)=0.5*(abs(ua(i,j,k))/av2(pm(i,j),pm(i-1,j))-
     &                        dt*ua(i,j,k)*ua(i,j,k))*oTdTdx(i,j)-
     &                   0.5*dt*ua(i,j,k)*(voTdTde(i,j)+woTdTdz(i,j,k))
          enddo
        enddo
      enddo
c
c  Compute anti-diffusion velocity component in the ETA-direction.
c--------------------------------------------------------------------
c  Compute w/T d(T)/d(z) term at horizontal V-points and vertical
c  RHO-points.
c
      do k=1,Nm
        do j=2,Mm
          do i=1,Lm
            f3d(i,j,k)=av2(ta (i,j  ,k+1)-ta (i,j  ,k),
     &                     ta (i,j-1,k+1)-ta (i,j-1,k))/
     &                 av2(z_r(i,j  ,k+1)-z_r(i,j  ,k),
     &                     z_r(i,j-1,k+1)-z_r(i,j-1,k))
          enddo
        enddo
      enddo
      do j=2,Mm
        do i=1,Lm
          f3d(i,j,0)=f3d(i,j,1 )
          f3d(i,j,N)=f3d(i,j,Nm)
        enddo
      enddo
      do k=1,N
        do j=2,Mm
          do i=1,Lm
            woTdTdz(i,j,k)=av4(wa(i,j,k-1),wa(i,j-1,k-1),
     &                         wa(i,j,k  ),wa(i,j-1,k))*
     &                     av2(f3d(i,j,k),f3d(i,j,k-1))/
     &                     (av2(ta(i,j,k),ta(i,j-1,k))+eps)
          enddo
        enddo
      enddo
c
c  Compute anti-diffusion velocity component in the ETA-direction.
c
      do k=1,N
c
c  Compute u/T d(T)/d(xi) term at horizontal V-points and vertical
c  RHO-points.
c
        do j=1,Mm
          do i=1,L
            a2d(i,j)=av2(pm(i,j),pm(i-1,j))*(ta(i,j,k)-ta(i-1,j,k))
          enddo
        enddo
        do j=2,Mm
          do i=1,Lm
            uoTdTdx(i,j)=av4(ua(i,j  ,k),ua(i+1,j  ,k),
     &                       ua(i,j-1,k),ua(i+1,j-1,k))*
     &                   av4(a2d(i,j  ),a2d(i+1,j  ),
     &                       a2d(i,j-1),a2d(i+1,j-1))/
     &                   (av2(ta(i,j,k),ta(i,j-1,k))+eps)
          enddo
        enddo
c
c  Compute 1/T d(T)/d(eta) term at horizontal V-points and vertical
c  RHO-points.
c
        do j=1,Mm
          do i=1,Lm
            a2d(i,j)=av2(pn(i,j),pn(i,j-1))*(ta(i,j,k)-ta(i,j-1,k))
          enddo
        enddo
        do j=2,Mm
          do i=1,Lm
            oTdTde(i,j)=a2d(i,j)/(av2(ta(i,j,k),ta(i,j-1,k))+eps)
          enddo
        enddo
c
c  Compute anti-diffusion velocity in the ETA-direction.
c
        do j=2,Mm
          do i=1,Lm
            vdiff(i,j,k)=0.5*(abs(va(i,j,k))/av2(pn(i,j),pn(i,j-1))-
     &                       dt*va(i,j,k)*va(i,j,k))*oTdTde(i,j)-
     &                   0.5*dt*va(i,j,k)*(uoTdTdx(i,j)+woTdTdz(i,j,k))
          enddo
        enddo
      enddo
c
c  Compute anti-diffusion velocity component in the S-direction.
c--------------------------------------------------------------------
c
      do k=1,Nm
c
c  Compute u/T d(T)/d(xi) term at horizontal RHO-points and vertical
c  W-points.
c
        do j=1,Mm
          do i=1,L
            a2d(i,j)=av2(pm(i,j),pm(i-1,j))*
     &               av2(ta(i,j,k+1)-ta(i-1,j,k+1),
     &                   ta(i,j,k  )-ta(i-1,j,k  ))
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            uoTdTdx(i,j)=av4(ua(i,j,k+1),ua(i+1,j,k+1),
     &                       ua(i,j,k  ),ua(i+1,j,k  ))*
     &                   av2(a2d(i,j),a2d(i+1,j))/
     &                   (av2(ta(i,j,k+1),ta(i,j,k))+eps)
          enddo
        enddo
c
c  Compute v/T d(T)/d(eta) term at horizontal RHO-points and vertical
c  W-points.
c
        do j=1,M
          do i=1,Lm
            a2d(i,j)=av2(pn(i,j),pn(i,j-1))*
     &               av2(ta(i,j,k+1)-ta(i,j-1,k+1),
     &                   ta(i,j,k  )-ta(i,j-1,k  ))
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            voTdTde(i,j)=av4(va(i,j,k+1),va(i,j+1,k+1),
     &                       va(i,j,k  ),va(i,j+1,k  ))*
     &                   av2(a2d(i,j),a2d(i,j+1))/
     &                   (av2(ta(i,j,k+1),ta(i,j,k))+eps)
          enddo
        enddo
c
c  Compute 1/T d(T)/d(z) term at horizontal RHO-points and vertical
c  W-points.
c
        do j=1,Mm
          do i=1,Lm
            a2d(i,j)=(ta(i,j,k+1)-ta(i,j,k))/(z_r(i,j,k+1)-z_r(i,j,k))
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            oTdTdz(i,j)=a2d(i,j)/(av2(ta(i,j,k+1),ta(i,j,k))+eps)
          enddo
        enddo
c
c  Compute anti-diffusion velocity component in the S-direction.
c
        do j=1,Mm
          do i=1,Lm
            wdiff(i,j,k)=0.5*(abs(wa(i,j,k))*(z_r(i,j,k+1)-z_r(i,j,k))-
     &                       dt*wa(i,j,k)*wa(i,j,k))*oTdTdz(i,j)-
     &                   0.5*dt*wa(i,j,k)*(uoTdTdx(i,j)+voTdTde(i,j))
          enddo
        enddo
      enddo
c
c  Load anti-diffusion velocity components into (ua,va,wa) arrays.
c====================================================================
c
      do k=1,N
        do j=1,Mm
          do i=2,Lm
            ua(i,j,k)=udiff(i,j,k)
          enddo
        enddo
        do j=2,Mm
          do i=1,Lm
            va(i,j,k)=vdiff(i,j,k)
          enddo
        enddo
      enddo
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            wa(i,j,k)=wdiff(i,j,k)
          enddo
        enddo
      enddo
c
c  Supress false oscillations in the solution by imposing appropriate
c  limits on the transport fluxes. Compute the UP and DOWN beta-ratios
c  described in Smolarkiewicz and Grabowski (1990).
c====================================================================
c
c  Compute UP B-flux.
c--------------------------------------------------------------------
c
c  Find the maximum tracer value for each tracer cell.
c
      do k=1,N
        do j=1,Mm
          do i=1,Lm
            Tmax(i,j,k)=max(ta(i  ,j  ,k),trc(i  ,j  ,k),
     &                      ta(i-1,j  ,k),trc(i-1,j  ,k),
     &                      ta(i+1,j  ,k),trc(i+1,j  ,k),
     &                      ta(i  ,j-1,k),trc(i  ,j-1,k),
     &                      ta(i  ,j+1,k),trc(i  ,j+1,k))
          enddo
        enddo
      enddo
      do k=2,Nm
        do j=1,Mm
          do i=1,Lm
            Tmax(i,j,k)=max(Tmax(i,j,k),ta(i,j,k-1),trc(i,j,k-1),
     &                                  ta(i,j,k+1),trc(i,j,k+1))
          enddo
        enddo
      enddo
c
c  Evaluate the UP B-flux.
c
      do k=1,N
        do j=1,Mm
          do i=1,Lm
            e3d(i,j,k)=dt*pm(i,j)*(max(0.,ua(i  ,j,k))*ta(i-1,j,k)-
     &                             min(0.,ua(i+1,j,k))*ta(i+1,j,k))+
     &                 dt*pn(i,j)*(max(0.,va(i,j  ,k))*ta(i,j-1,k)-
     &                             min(0.,va(i,j+1,k))*ta(i,j+1,k))
          enddo
        enddo
      enddo
      do j=1,Mm
        do i=1,Lm
          e3d(i,j,1)=e3d(i,j,1)+dt*(max(0.,wa(i,j,0))*ta(i,j,1)-
     &                              min(0.,wa(i,j,1))*ta(i,j,2))/
     &                             (z_w(i,j,1)-z_w(i,j,0))
        enddo
      enddo
      do k=2,Nm
        do j=1,Mm
          do i=1,Lm
            e3d(i,j,k)=e3d(i,j,k)+dt*(max(0.,wa(i,j,k-1))*ta(i,j,k-1)
     &                               -min(0.,wa(i,j,k ))*ta(i,j,k+1))
     &                              /(z_w(i,j,k)-z_w(i,j,k-1))
          enddo
        enddo
      enddo
      do j=1,Mm
        do i=1,Lm
          e3d(i,j,N)=e3d(i,j,N)+dt*(max(0.,wa(i,j,Nm))*ta(i,j,Nm)
     &                             -min(0.,wa(i,j,N ))*ta(i,j,N ))
     &                            /(z_w(i,j,N)-z_w(i,j,Nm))
        enddo
      enddo
      do k=1,N
        do j=1,Mm
          do i=1,Lm
            Bup(i,j,k)=(Tmax(i,j,k)-ta(i,j,k))/(e3d(i,j,k)+eps)
          enddo
        enddo
      enddo
c
c  Compute DOWN B-flux.
c--------------------------------------------------------------------
c
c  Find the minimum tracer value for each tracer cell.
c
      do k=1,N
        do j=1,Mm
          do i=1,Lm
            Tmin(i,j,k)=min(ta(i  ,j  ,k),trc(i  ,j  ,k),
     &                      ta(i-1,j  ,k),trc(i-1,j  ,k),
     &                      ta(i+1,j  ,k),trc(i+1,j  ,k),
     &                      ta(i  ,j-1,k),trc(i  ,j-1,k),
     &                      ta(i  ,j+1,k),trc(i  ,j+1,k))
          enddo
        enddo
      enddo
      do k=2,Nm
        do j=1,Mm
          do i=1,Lm
            Tmin(i,j,k)=min(Tmin(i,j,k),ta(i,j,k-1),trc(i,j,k-1),
     &                                  ta(i,j,k+1),trc(i,j,k+1))
          enddo
        enddo
      enddo
c
c  Evaluate the DOWN B-flux.
c
      do k=1,N
        do j=1,Mm
          do i=1,Lm
            a2d(i,j)=dt*pm(i,j)*(max(0.,ua(i+1,j,k))*ta(i,j,k)-
     &                           min(0.,ua(i  ,j,k))*ta(i,j,k))+
     &               dt*pn(i,j)*(max(0.,va(i,j+1,k))*ta(i,j,k)-
     &                           min(0.,va(i,j  ,k))*ta(i,j,k))+
     &               dt*(max(0.,wa(i,j,k  ))*ta(i,j,k)-
     &                   min(0.,wa(i,j,k-1))*ta(i,j,k))/
     &                  (z_w(i,j,k)-z_w(i,j,k-1))
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            Bdn(i,j,k)=(ta(i,j,k)-Tmin(i,j,k))/(a2d(i,j)+eps)
          enddo
        enddo
      enddo
c
c  Calculate monotonic anti-diffusion velocities.
c--------------------------------------------------------------------
c
      do k=1,N
        do j=1,Mm
          do i=2,Lm
            ua(i,j,k)=min(1.,Bdn(i-1,j,k),Bup(i,j,k))*max(0.,ua(i,j,k))+
     &                min(1.,Bup(i-1,j,k),Bdn(i,j,k))*min(0.,ua(i,j,k))
          enddo
        enddo
        do j=2,Mm
          do i=1,Lm
            va(i,j,k)=min(1.,Bdn(i,j-1,k),Bup(i,j,k))*max(0.,va(i,j,k))+
     &                min(1.,Bup(i,j-1,k),Bdn(i,j,k))*min(0.,va(i,j,k))
          enddo
        enddo
      enddo
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            wa(i,j,k)=min(1.,Bdn(i,j,k),Bup(i,j,k+1))*max(0.,wa(i,j,k))+
     &                min(1.,Bup(i,j,k),Bdn(i,j,k+1))*min(0.,wa(i,j,k))
          enddo
        enddo
      enddo
# ifdef MASKING
c
c  Apply Land/Sea mask.
c====================================================================
c
      do k=1,N
        do j=1,Mm
          do i=2,Lm
            ua(i,j,k)=ua(i,j,k)*umask(i,j)
          enddo
        enddo
        do j=2,Mm
          do i=1,Lm
            va(i,j,k)=va(i,j,k)*vmask(i,j)
          enddo
        enddo
      enddo
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            wa(i,j,k)=wa(i,j,k)*rmask(i,j)
          enddo
        enddo
      enddo
# endif /* MASKING */
c
c  Set boundary conditions for anti-diffusion velocities.
c====================================================================
c
      call u3dbc (ua,ibcUa)
      call v3dbc (va,ibcVa)
      call w3dbc (wa,ibcWa)
#else
      subroutine smol_adiff
#endif /* SMOLARKIEWICZ */
      return
      end
 
 
