#include "cppdefs.h"
#ifdef MY25_MIXING
      subroutine my25_q (q,Qprod,Qdiss,sqflx,bqflx)
!
!-------------------------------------------------------------------
!  This routine solves the prognostic equation for turbulent energy
!  variables used in the  Mellor-Yamada (1982)  level 2.5 turbulent
!  closure.  The prognostic equation is time-stepped via a Leapfrog
!  scheme.  An Asselin type filter is applied to  supress the  time
!  splitting of the solution.
!
!  On Input:
!
!     q        Turbulent energy variable.
!     Qprod    Shear and buoyant production of turbulent energy.
!     Qdiss    Dissipation of turbulent energy.
!     sqflx    surface flux of turbulent kinetic energy.
!     bqflx    bottom flux of turbulent kinetic energy.
!
!  On Output:
!
!     q        New time-level turbulent kinetic energy variable.
!
!  Calls:    tridiagonal, t3dbc, w3dbc
!
!--------------------------------------------------------------------
!
      implicit none
# include "param.h"
# include "pconst.h"
# include "grid.h"
# include "mixing.h"
# include "ocean.h"
# include "scalars.h"
# include "work.h"
 
      integer i, j, k
      real cff,qdtfac
     &        Qdiss(Lm,Mm,Nm), Qprod(Lm,Mm,Nm), bqflx(Lm,Mm),
     &        q(0:L,0:M,0:N,2), sqflx(Lm,Mm)
      real
     &        AC(0:L,0:M,0:N), BC(0:L,0:M,0:N), CC(0:L,0:M,0:N),
     &        DC(0:L,0:M,0:N), X(0:L,0:M,0:N), diff(Lm,Mm,N),
     &        rq(Lm,Mm,N)
      equivalence (AC,c3d), (BC,f3d), (DC,e3d), (CC,h3d), (X,g3d),
     &            (diff,g3d), (rq,c3d)
# if defined Q_DIF2 || defined Q_DIF4
      real
     &        Qflux(L,M,N)
      equivalence (Qflux,e3d)
# endif /* Q_DIF2 || Q_DIF4 */
# ifdef Q_DIF4
      real
     &        LapQ(0:L,0:M,N)
      equivalence (LapQ,f3d)
# endif /* QDIF4 */
!
      real       a1, a2, a3, a4, av2, av4
# include "avg.h"
c
c  Compute right-hand-side term.  ===================================
c====================================================================
c
c  Initialize right-hand-side term.
c
      do k=1,N
        do j=1,Mm
          do i=1,Lm
            rq(i,j,k)=0.
          enddo
        enddo
      enddo
# ifdef Q_DIF2
c
c  Add in horizontal Laplacian mixing along constant S-surfaces.
c--------------------------------------------------------------------
c
c  Compute horizontal flux in the XI-direction.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,L
            Qflux(i,j,k)=pmon_u(i,j)*
     &                   av4(Hz(i,j,k  ),Hz(i-1,j,k  ),
     &                       Hz(i,j,k+1),Hz(i-1,j,k+1))*
     &                   (q(i,j,k,mrhs)-q(i-1,j,k,mrhs))
          enddo
        enddo
#  ifdef MASKING
c
c  Apply Land/Sea mask.
c
        do j=1,Mm
          do i=1,L
            Qflux(i,j,k)=Qflux(i,j,k)*umask(i,j)
          enddo
        enddo
#  endif /* MASKING */
      enddo
c
c  Add in horizontal diffusion (Q m^3/s) in the XI-direction.
!  Multiply by mixing coefficient.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            rq(i,j,k)=rq(i,j,k)+
     &                q2nu2*(Qflux(i+1,j,k)-Qflux(i,j,k))
          enddo
        enddo
      enddo
c
c  Compute horizontal flux in the ETA-direction.
c
      do k=1,Nm
        do j=1,M
          do i=1,Lm
            Qflux(i,j,k)=pnom_v(i,j)*
     &                   av4(Hz(i,j,k  ),Hz(i,j-1,k  ),
     &                       Hz(i,j,k+1),Hz(i,j-1,k+1))*
     &                   (q(i,j,k,mrhs)-q(i,j-1,k,mrhs))
          enddo
        enddo
#  ifdef MASKING
c
c  Apply Land/Sea mask.
c
        do j=1,M
          do i=1,Lm
            Qflux(i,j,k)=Qflux(i,j,k)*vmask(i,j)
          enddo
        enddo
#  endif /* MASKING */
      enddo
c
c  Add in horizontal diffusion (Q m^3/s) in the ETA-direction.
!   Multiply by mixing coefficient.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            rq(i,j,k)=rq(i,j,k)+
     &                q2nu2*(Qflux(i,j+1,k)-Qflux(i,j,k))
          enddo
        enddo
      enddo
# endif /* Q_DIF2 */
# ifdef Q_DIF4
c
c  Add in horizontal biharmonic diffusion along constant S-surfaces.
c  The biharmonic operator is computed by applying the Laplacian
c  operator twice.
c-------------------------------------------------------------------
c
c  Compute d/dXI term at U-points.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,L
            Qflux(i,j,k)=pmon_u(i,j)*
     &                   av4(Hz(i,j,k  ),Hz(i-1,j,k  ),
     &                       Hz(i,j,k+1),Hz(i-1,j,k+1))*
     &                   (q(i,j,k,mrhs)-q(i-1,j,k,mrhs))
          enddo
        enddo
#  ifdef MASKING
c
c  Apply Land/Sea mask.
c
        do j=1,Mm
          do i=1,L
            Qflux(i,j,k)=Qflux(i,j,k)*umask(i,j)
          enddo
        enddo
#  endif /* MASKING */
      enddo
c
c  Compute d/dXI [d/dXI] term at RHO-points.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            LapQ(i,j,k)=Qflux(i+1,j,k)-Qflux(i,j,k)
          enddo
        enddo
      enddo
c
c  Compute d/dETA term at V-points.
c
      do k=1,Nm
        do j=1,M
          do i=1,Lm
            Qflux(i,j,k)=pnom_v(i,j)*
     &                   av4(Hz(i,j,k  ),Hz(i,j-1,k  ),
     &                       Hz(i,j,k+1),Hz(i,j-1,k+1))*
     &                   (q(i,j,k,mrhs)-q(i,j-1,k,mrhs))
          enddo
        enddo
#  ifdef MASKING
c
c  Apply Land/Sea mask.
c
        do j=1,M
          do i=1,Lm
            Qflux(i,j,k)=Qflux(i,j,k)*vmask(i,j)
          enddo
        enddo
#  endif /* MASKING */
      enddo
c
c  Compute and add d/dETA [d/dETA] term at RHO-points.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            LapQ(i,j,k)=LapQ(i,j,k)+(Qflux(i,j+1,k)-Qflux(i,j,k))
          enddo
        enddo
      enddo
c
c  Multiply the first Laplacian by the metrics of the second
!  Laplacian.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            LapQ(i,j,k)=LapQ(i,j,k)*pm(i,j)*pn(i,j)/
     &                  av2(Hz(i,j,k),Hz(i,j,k+1))
          enddo
        enddo
      enddo
c
c  Apply boundary conditions to the first Laplacian.
c
      call t3dbc (LapQ,-1,ibcLapQ)
c
c  Compute d/dXI term at U-points.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,L
            Qflux(i,j,k)=pmon_u(i,j)*
     &                   av4(Hz(i,j,k  ),Hz(i-1,j,k  ),
     &                       Hz(i,j,k+1),Hz(i-1,j,k+1))*
     &                   (LapQ(i,j,k)-LapQ(i-1,j,k))
          enddo
        enddo
#  ifdef MASKING
c
c  Apply Land/Sea mask.
c
        do j=1,Mm
          do i=1,L
            Qflux(i,j,k)=Qflux(i,j,k)*umask(i,j)
          enddo
        enddo
#  endif /* MASKING */
      enddo
c
c  Compute d/dXI [d/dXI] term at RHO-points.  Add in horizontal
c  biharmonic diffusion (Q m^3/s) in the XI-direction. Multiply by
c  mixing coefficient.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            rq(i,j,k)=rq(i,j,k)-
     &                q2nu4*(Qflux(i+1,j,k)-Qflux(i,j,k))
          enddo
        enddo
      enddo
c
c  Compute d/dETA term at V-points.
c
      do k=1,Nm
        do j=1,M
          do i=1,Lm
            Qflux(i,j,k)=pnom_v(i,j)*
     &                   av4(Hz(i,j,k  ),Hz(i,j-1,k  ),
     &                       Hz(i,j,k+1),Hz(i,j-1,k+1))*
     &                   (LapQ(i,j,k)-LapQ(i,j-1,k))
          enddo
        enddo
#  ifdef MASKING
c
c  Apply Land/Sea mask.
c
        do j=1,M
          do i=1,Lm
            Qflux(i,j,k)=Qflux(i,j,k)*vmask(i,j)
          enddo
        enddo
#  endif /* MASKING */
      enddo
c
c  Compute d/dETA [d/dETA] term at RHO-points.  Add in horizontal
c  biharmonic diffusion (Q m^3/s) in the ETA-direction. Multiply by
c  mixing coefficient.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            rq(i,j,k)=rq(i,j,k)-
     &                q2nu4*(Qflux(i,j+1,k)-Qflux(i,j,k))
          enddo
        enddo
      enddo
# endif /* Q_DIF4 */
c
c  Add in horizontal advection using centered differences.
c-------------------------------------------------------------------
c
c  Add horizontal advection in the XI-direction.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,L
            a2d(i,j)=av2(Huon(i,j,k),Huon(i,j,k+1))*
     &               av2(q(i,j,k,mrhs),q(i-1,j,k,mrhs))
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            rq(i,j,k)=rq(i,j,k)-(a2d(i+1,j)-a2d(i,j))
          enddo
        enddo
      enddo
c
c  Add horizontal advection in the ETA-direction.
c
      do k=1,Nm
        do j=1,M
          do i=1,Lm
            a2d(i,j)=av2(Hvom(i,j,k),Hvom(i,j,k+1))*
     &               av2(q(i,j,k,mrhs),q(i,j-1,k,mrhs))
          enddo
        enddo
        do j=1,Mm
          do i=1,Lm
            rq(i,j,k)=rq(i,j,k)-(a2d(i,j+1)-a2d(i,j))
          enddo
        enddo
      enddo
c
c  Add in vertical advection.
c------------------------------------------------------------------
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            e3d(i,j,k)=q(i,j,k,mrhs)*w(i,j,k)
          enddo
        enddo
      enddo
      do j=1,Mm
        do i=1,Lm
          e3d(i,j,0)=0.
          e3d(i,j,N)=0.
        enddo
      enddo
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            rq(i,j,k)=rq(i,j,k)-(e3d(i,j,k+1)-e3d(i,j,k-1))*ods
          enddo
        enddo
      enddo
c
c  Time step turbulent energy equation.
c===================================================================
c
c  Compute coefficient [dt*Akq/(Hz*ds^2)] for vertical diffusion term at
c  horizontal and vertical RHO-points.
c
 
      if (iic.eq.ntstart) then
        qdtfac=1.
      else
        qdtfac=2.
      endif
      cff=qdtfac * dt/(ds*ds) * 0.5
      do k=1,N
        do j=1,Mm
          do i=1,Lm
            diff(i,j,k)=cff*(Akq(i,j,k)+Akq(i,j,k-1))/Hz(i,j,k)
          enddo
        enddo
      enddo
c
c  Load right-hand-side terms into matrix DC.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            DC(i,j,k)=av2(Hz(i,j,k),Hz(i,j,k+1))*q(i,j,k,mstp)+
     &                dt*qdtfac*pm(i,j)*pn(i,j)*rq(i,j,k)
          enddo
        enddo
      enddo
c
c  Add in shear and buoyant production of turbulent energy and
c  substract dissipation of turbulent energy into vector DC.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            DC(i,j,k)=DC(i,j,k)+dt*qdtfac*(Qprod(i,j,k)-Qdiss(i,j,k))
          enddo
        enddo
      enddo
c
c  Add in vertical diffusion term into vector DC.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            DC(i,j,k)=DC(i,j,k)+(1.-lambda)*
     &                (diff(i,j,k+1)*(q(i,j,k+1,mstp)-q(i,j,k  ,mstp))-
     &                 diff(i,j,k  )*(q(i,j,k  ,mstp)-q(i,j,k-1,mstp)))
          enddo
        enddo
      enddo
c
c  Add in bottom and surface flux into matrix DC.
c
      do j=1,Mm
        do i=1,Lm
          DC(i,j,0)=bqflx(i,j)
          DC(i,j,N)=sqflx(i,j)
        enddo
      enddo
c
c  Load tridiangle matrix coefficients for Q(k-1), Q(k), and Q(k+1) into
c  matrices AC, BC, CC, respectively.
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            AC(i,j,k)=-lambda*diff(i,j,k)
          enddo
        enddo
      enddo
      do j=1,Mm
        do i=1,Lm
          AC(i,j,0)=0.
          AC(i,j,N)=0.
        enddo
      enddo
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            BC(i,j,k)=av2(Hz(i,j,k),Hz(i,j,k+1))+
     &                lambda*(diff(i,j,k+1)+diff(i,j,k))
          enddo
        enddo
      enddo
      do j=1,Mm
        do i=1,Lm
          BC(i,j,0)=Hz(i,j,1)
          BC(i,j,N)=Hz(i,j,N)
        enddo
      enddo
c
      do k=1,Nm
        do j=1,Mm
          do i=1,Lm
            CC(i,j,k)=-lambda*diff(i,j,k+1)
          enddo
        enddo
      enddo
      do j=1,Mm
        do i=1,Lm
          CC(i,j,0)=0.
          CC(i,j,N)=0.
        enddo
      enddo
c
c  Solve tridiagonal system and update solution.
c
      call trisolver (1,Lm,1,Mm,0,N,AC,BC,CC,DC,X)
c
c  Insure positive turbulent energy.  Lower bound solution.
c
      do k=0,N
        do j=0,M
          do i=0,L
            X(i,j,k)=max(my_qmin,X(i,j,k))
          enddo
        enddo
      enddo
c
c  Suppress spliting of the Leapfrog solution a weak time filter
c  (Asselin, 1972).  Update solution.
c
      do k=0,N
        do j=0,M
          do i=0,L
            q(i,j,k,mold)=q(i,j,k,mold)+my_dtfac*
     &                    (X(i,j,k)-c2*q(i,j,k,mold)+q(i,j,k,mstp))
            q(i,j,k,mnew)=X(i,j,k)
          enddo
        enddo
      enddo
# ifdef MASKING
c
c  Apply Land/Sea mask.
c
      do k=0,N
        do j=0,M
          do i=0,L
            q(i,j,k,mold)=max(my_qmin,q(i,j,k,mold)*rmask(i,j))
            q(i,j,k,mnew)=max(my_qmin,q(i,j,k,mnew)*rmask(i,j))
          enddo
        enddo
      enddo
# endif /* MASKING */
c
c  Set lateral boundary conditions.
c
      call w3dbc (q(0,0,0,mold),ibcQ)
      call w3dbc (q(0,0,0,mnew),ibcQ)
#else
      subroutine my25_q_empty
#endif /* MY2_MIXING */
      return
      end
 
 
