#include "cppdefs.h"
      function wrt_info (ncid,ncname)
!
!=====================================================================
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine writes out information variables into requested      !
!  NetCDF file.                                                      !
!                                                                    !
!  On Input:                                                         !
!                                                                    !
!     ncid     NetCDF file ID (integer).                             !
!     ncname   NetCDF file name (character).                         !
!                                                                    !
!  On Output:                                                        !
!                                                                    !
!     wrt_info   Error flag (integer).                               !
!                                                                    !
!=====================================================================
!
      implicit none
#include "param.h"
#include "biology.h"
#include "boundary.h"
#include "grid.h"
#include "iounits.h"
#include "mask.h"
#include "mixing.h"
#include "ncparam.h"
#include "netcdf.inc"
#include "obs.h"
#include "scalars.h"
!
      INTEGER_TYPE
     &        k, ncid, lstr, varid
      INTEGER_TYPE
     &        nf_fwrite, lenstr, wrt_info
#if defined SOLVE3D
      INTEGER_TYPE
     &        itrc
# ifdef NUDGING_COFF
      INTEGER_TYPE
     &        count(2), start(2)
# endif
# ifdef TS_DIF4
      REAL_TYPE
     &        diff(NT)
# endif
      REAL_TYPE
     &        nudg(NT), sc(N)
#endif /* SOLVE3D */
      REAL_TYPE
     &        scale, wrk(NS)
      character*(*) ncname
      character*1   char1
!
!---------------------------------------------------------------------
!  Write out running parameters.
!---------------------------------------------------------------------
!
      lstr=lenstr(ncname)
!
!  Time stepping parameters.
!
      wrt_info=nf_inq_varid(ncid,'ntimes',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,ntimes)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'ntimes', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'ndtfast',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,ndtfast)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'ndtfast', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'dt',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,dt)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'dt', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'dtfast',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,dtfast)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'dtfast', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'dstart',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,dstart)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'dstart', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'nhis',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,nhis)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'nhis', ncname(1:lstr)
        exit_flag=3
        return
      endif

      if (ndefhis.gt.0) then
        wrt_info=nf_inq_varid(ncid,'ndefhis',varid)
        wrt_info=nf_put_var1_int(ncid,varid,1,ndefhis)
        if (wrt_info.ne.nf_noerr) then
          write(stdout,10) 'ndefhis', ncname(1:lstr)
          exit_flag=3
          return
        endif
      endif

      wrt_info=nf_inq_varid(ncid,'nrst',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,nrst)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'nrst', ncname(1:lstr)
        exit_flag=3
        return
      endif

#ifdef AVERAGES
      wrt_info=nf_inq_varid(ncid,'ntsavg',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,ntsavg)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'ntsavg', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'navg',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,navg)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'navg', ncname(1:lstr)
        exit_flag=3
        return
      endif

      if (ndefavg.gt.0) then
        wrt_info=nf_inq_varid(ncid,'ndefavg',varid)
        wrt_info=nf_put_var1_int(ncid,varid,1,ndefavg)
        if (wrt_info.ne.nf_noerr) then
          write(stdout,10) 'ndefavg', ncname(1:lstr)
          exit_flag=3
          return
        endif
      endif
#endif /* AVERAGES */

#ifdef STATIONS
      wrt_info=nf_inq_varid(ncid,'nsta',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,nsta)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'nsta', ncname(1:lstr)
        exit_flag=3
        return
      endif
#endif /* STATIONS */

#if defined MIX_ISO_TS
      wrt_info=nf_inq_varid(ncid,'nmix_en',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,nmix_en)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'nmix_en', ncname(1:lstr)
        exit_flag=3
        return
      endif
#endif /* MIX_ISO_TS */
!
!  Horizontal mixing coefficients.
!
#if defined SOLVE3D && defined TS_DIF2
      wrt_info=nf_inq_varid(ncid,'tnu2',varid)
      wrt_info=nf_put_vara_FTYPE(ncid,varid,1,NT,tnu2)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'tnu2', ncname(1:lstr)
        exit_flag=3
        return
      endif
#endif /* SOLVE3D && TS_DIF2 */

#if defined SOLVE3D && defined TS_DIF4
      wrt_info=nf_inq_varid(ncid,'tnu4',varid)
      do itrc=1,NT
        diff(itrc)=tnu4(itrc)*tnu4(itrc)
      enddo
      wrt_info=nf_put_vara_FTYPE(ncid,varid,1,NT,diff)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'tnu4', ncname(1:lstr)
        exit_flag=3
        return
      endif
#endif /* SOLVE3D && TS_DIF4 */

#ifdef UV_VIS2
      wrt_info=nf_inq_varid(ncid,'visc2',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,visc2)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'visc2', ncname(1:lstr)
        exit_flag=3
        return
      endif
#endif /* UV_VIS2 */

#ifdef UV_VIS4
      wrt_info=nf_inq_varid(ncid,'visc4',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,visc4*visc4)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'visc4', ncname(1:lstr)
        exit_flag=3
        return
      endif
#endif /* UV_VIS4 */

#if defined SOLVE3D && (defined MY25_MIXING || defined GLS_MIXING)
# ifdef TKE_DIF2
      wrt_info=nf_inq_varid(ncid,'tkenu2',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,tkenu2)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'tkenu2', ncname(1:lstr)
        exit_flag=3
        return
      endif
# endif /* TKE_DIF2 */

# ifdef TKE_DIF4
      wrt_info=nf_inq_varid(ncid,'tkenu4',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,tkenu4*tkenu4)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'tkenu4', ncname(1:lstr)
        exit_flag=3
        return
      endif
# endif /* TKE_DIF4 */
#endif /* SOLVE3D && (MY25_MIXING || GLS_MIXING) */
#ifdef SOLVE3D
!
!  Background vertical mixing coefficients.
!
      wrt_info=nf_inq_varid(ncid,'Akt_bak',varid)
      wrt_info=nf_put_vara_FTYPE(ncid,varid,1,NT,Akt_bak)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Akt_bak', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'Akv_bak',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,Akv_bak)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Akv_bak', ncname(1:lstr)
        exit_flag=3
        return
      endif

# if defined MY25_MIXING || defined GLS_MIXING
      wrt_info=nf_inq_varid(ncid,'Akk_bak',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,Akk_bak)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Akk_bak', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'Akp_bak',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,Akp_bak)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Akp_bak', ncname(1:lstr)
        exit_flag=3
        return
      endif
# endif /* MY25_MIXING || GLS_MIXING*/
#endif /* SOLVE3D */
!
!  Drag coefficients.
!
      wrt_info=nf_inq_varid(ncid,'rdrg',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,rdrg)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'rdrg', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'rdrg2',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,rdrg2)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'rdrg2', ncname(1:lstr)
        exit_flag=3
        return
      endif

#ifdef SOLVE3D
      wrt_info=nf_inq_varid(ncid,'Zob',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,Zob)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Zob', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'Zos',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,Zos)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Zos', ncname(1:lstr)
        exit_flag=3
        return
      endif
#endif /* SOLVE3D */
#if defined SOLVE3D && defined GLS_MIXING
!
!  Generic length-scale parameters.
!
      wrt_info=nf_inq_varid(ncid,'gls_p',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_p)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_p', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'gls_m',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_m)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_m', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'gls_n',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_n)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_n', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'gls_cmu0',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_cmu0)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_cmu0', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'gls_c1',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_c1)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_c1', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'gls_c2',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_c2)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_c2', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'gls_c3m',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_c3m)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_c3m', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'gls_c3p',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_c3p)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_c3p', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'gls_sigk',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_sigk)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_sigk', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'gls_sigp',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_sigp)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_sigp', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'gls_Kmin',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_Kmin)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_Kmin', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'gls_Pmin',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gls_Pmin)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gls_Pmin', ncname(1:lstr)
        exit_flag=3
        return
      endif
#endif /* SOLVE3D && GLS_MIXING */
!
!  Nudging inverse time scales used in various tasks.
!
      wrt_info=nf_inq_varid(ncid,'Znudg',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,Znudg/sec2day)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Znudg', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'M2nudg',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,M2nudg/sec2day)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'M2nudg', ncname(1:lstr)
        exit_flag=3
        return
      endif

#ifdef SOLVE3D
      wrt_info=nf_inq_varid(ncid,'M3nudg',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,M3nudg/sec2day)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'M3nudg', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'Tnudg',varid)
      do itrc=1,NT
        nudg(itrc)=Tnudg(itrc)/sec2day
      enddo
      wrt_info=nf_put_vara_FTYPE(ncid,varid,1,NT,nudg)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Tnudg', ncname(1:lstr)
        exit_flag=3
        return
      endif
#endif /* SOLVE3D */
#ifdef NUDGING
!
!  Nudging inverse time scales used in data assimilation.
!
      wrt_info=nf_inq_varid(ncid,'Znudass',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,Znudass/sec2day)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Znudass', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'M2nudass',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,M2nudass/sec2day)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'M2nudass', ncname(1:lstr)
        exit_flag=3
        return
      endif

# ifdef SOLVE3D
      wrt_info=nf_inq_varid(ncid,'M3nudass',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,M3nudass/sec2day)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'M3nudass', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'Tnudass',varid)
      do itrc=1,NT
        nudg(itrc)=Tnudass(itrc)/sec2day
      enddo
      wrt_info=nf_put_vara_FTYPE(ncid,varid,1,NT,nudg)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Tnudass', ncname(1:lstr)
        exit_flag=3
        return
      endif
# endif /* SOLVE3D */
#endif /* NUDGING */
#ifdef NUDGING_COFF
!
!  Open boundary nudging, inverse time scales.
!
      wrt_info=nf_inq_varid(ncid,'FSobc_in',varid)
      wrt_info=nf_put_vara_FTYPE(ncid,varid,1,4,FSobc_in)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'FSobc_in', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'FSobc_out',varid)
      wrt_info=nf_put_vara_FTYPE(ncid,varid,1,4,FSobc_out)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'FSobc_out', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'M2obc_in',varid)
      wrt_info=nf_put_vara_FTYPE(ncid,varid,1,4,M2obc_in)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'M2obc_in', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'M2obc_out',varid)
      wrt_info=nf_put_vara_FTYPE(ncid,varid,1,4,M2obc_out)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'M2obc_out', ncname(1:lstr)
        exit_flag=3
        return
      endif

# ifdef SOLVE3D
      start(1)=1
      count(1)=4
      start(2)=1
      count(2)=NT

      wrt_info=nf_inq_varid(ncid,'Tobc_in',varid)
      wrt_info=nf_put_vara_FTYPE(ncid,varid,start,count,Tobc_in)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Tobc_in', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'Tobc_out',varid)
      wrt_info=nf_put_vara_FTYPE(ncid,varid,start,count,Tobc_out)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Tobc_out', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'M3obc_in',varid)
      wrt_info=nf_put_vara_FTYPE(ncid,varid,1,4,M3obc_in)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'M3obc_in', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'M3obc_out',varid)
      wrt_info=nf_put_vara_FTYPE(ncid,varid,1,4,M3obc_out)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'M3obc_out', ncname(1:lstr)
        exit_flag=3
        return
      endif
# endif /* SOLVE3D */
#endif /* NUDGING_COFF */
!
!  Equation of State parameters.
!
      wrt_info=nf_inq_varid(ncid,'rho0',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,rho0)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'rho0', ncname(1:lstr)
        exit_flag=3
        return
      endif

#if defined SOLVE3D && !defined NONLIN_EOS
      wrt_info=nf_inq_varid(ncid,'R0',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,R0)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'R0', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'Tcoef',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,Tcoef)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Tcoef', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'Scoef',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,Scoef)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Scoef', ncname(1:lstr)
        exit_flag=3
        return
      endif
#endif /* SOLVE3D && !NONLIN_EOS */
#ifdef SOLVE3D
!
!  Various parameters.
!
# ifdef SMOLARKIEWICZ
      wrt_info=nf_inq_varid(ncid,'adv_ord',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,adv_ord)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'adv_ord', ncname(1:lstr)
        exit_flag=3
        return
      endif
# endif /*  SMOLARKIEWICZ */

# ifdef BODYFORCE
      wrt_info=nf_inq_varid(ncid,'levsfrc',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,levsfrc)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'levsfrc', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'levbfrc',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,levbfrc)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'levbfrc', ncname(1:lstr)
        exit_flag=3
        return
      endif
# endif /* BODYFORCE */
#endif /* SOLVE3D */
!
!  Slipperiness parameters.
!
      wrt_info=nf_inq_varid(ncid,'gamma2',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,gamma2)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'gamma2', ncname(1:lstr)
        exit_flag=3
        return
      endif

#if defined BIOLOGY && defined SOLVE3D
# ifdef BIO_FASHAM
!
!  Write out Fasham type biological module parameters.
!
      wrt_info=nf_inq_varid(ncid,'BioIter',varid)
      wrt_info=nf_put_var1_int(ncid,varid,1,BioIter)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'BioIter', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'AttSW',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,AttSW)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'AttSW', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'AttChl',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,AttChl)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'AttChl', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'PARfrac',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,PARfrac)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'PARfrac', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'K_NO3',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,K_NO3)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'K_NO3', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'K_NH4',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,K_NH4)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'K_NH4', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'K_Phy',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,K_Phy)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'K_Phy', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'PhyIS',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,PhyIS)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'PhyIS', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'PhyCN',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,PhyCN)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'PhyCN', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'PhyIP',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,PhyIP)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'PhyIP', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'PhyMR',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,PhyMR)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'PhyMR', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'ZooAE',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,ZooAE)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'ZooAE', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'ZooGR',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,ZooGR)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'ZooGR', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'ZooER',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,ZooER)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'ZooER', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'ZooMR',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,ZooMR)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'ZooMR', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'ChlMB',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,ChlMB)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'ChlMB', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'CPrMax',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,CPrMax)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'CPrMax', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'SDeBR',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,SDeBR)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'SDeBR', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'SDeAR',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,SDeAR)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'SDeAR', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'LDeRR',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,LDeRR)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'LDeRR', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'wSDe',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,wSDe)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'wSDe', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'wLDe',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,wLDe)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'wLDe', ncname(1:lstr)
        exit_flag=3
        return
      endif

# endif /* BIO_FASHAM */
#endif /* BIOLOGY && SOLVE3D */
!
!---------------------------------------------------------------------
!  Write out grid variables.
!---------------------------------------------------------------------
!
!  Grid type switch.
!
      wrt_info=nf_inq_varid(ncid,'spherical',varid)
      write(char1,'(l1)') spherical
      wrt_info=nf_put_var1_text(ncid,varid,1,char1)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'spherical', ncname(1:lstr)
        exit_flag=3
        return
      endif
!
!  Domain Length.
!
      wrt_info=nf_inq_varid(ncid,'xl',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,xl)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'xl', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'el',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,el)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'el', ncname(1:lstr)
        exit_flag=3
        return
      endif
#ifdef SOLVE3D
!
!  S-coordinate parameters.
!
      wrt_info=nf_inq_varid(ncid,'theta_s',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,theta_s)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'theta_s', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'theta_b',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,theta_b)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'theta_b', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'Tcline',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,Tcline)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Tcline', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'hc',varid)
      wrt_info=nf_put_var1_FTYPE(ncid,varid,1,hc)
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'hc', ncname(1:lstr)
        exit_flag=3
        return
      endif
!
!  S-coordinate non-dimensional independent variables.
!
      wrt_info=nf_inq_varid(ncid,'sc_r',varid)
      if (ncid.eq.nchisid) then
        do k=1,Nlev
          sc(k)=sc_r(Lev(k))
        enddo
        wrt_info=nf_put_vara_FTYPE(ncid,varid,1,Nlev,sc)
      else
        wrt_info=nf_put_vara_FTYPE(ncid,varid,1,N,sc_r)
      endif
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'sc_r', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'sc_w',varid)
      if (ncid.eq.nchisid) then
        do k=1,Nlev
          sc(k)=sc_w(Lev(k))
        enddo
        wrt_info=nf_put_vara_FTYPE(ncid,varid,1,Nlev,sc)
      elseif ((ncid.eq.ncstaid).or.(ncid.eq.ncfltid)) then
        wrt_info=nf_put_vara_FTYPE(ncid,varid,1,N,sc_w(1))
      else
        wrt_info=nf_put_vara_FTYPE(ncid,varid,1,Np,sc_w(0))
      endif
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'sc_w', ncname(1:lstr)
        exit_flag=3
        return
      endif
!
!  S-coordinate non-dimensional stretching curves.
!
      wrt_info=nf_inq_varid(ncid,'Cs_r',varid)
      if (ncid.eq.nchisid) then
        do k=1,Nlev
          sc(k)=Cs_r(Lev(k))
        enddo
        wrt_info=nf_put_vara_FTYPE(ncid,varid,1,Nlev,sc)
      else
        wrt_info=nf_put_vara_FTYPE(ncid,varid,1,N,Cs_r)
      endif
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Cs_r', ncname(1:lstr)
        exit_flag=3
        return
      endif

      wrt_info=nf_inq_varid(ncid,'Cs_w',varid)
      if (ncid.eq.nchisid) then
        do k=1,Nlev
          sc(k)=Cs_w(Lev(k))
        enddo
        wrt_info=nf_put_vara_FTYPE(ncid,varid,1,Nlev,sc)
      elseif ((ncid.eq.ncstaid).or.(ncid.eq.ncfltid)) then
        wrt_info=nf_put_vara_FTYPE(ncid,varid,1,N,Cs_w(1))
      else
        wrt_info=nf_put_vara_FTYPE(ncid,varid,1,Np,Cs_w(0))
      endif
      if (wrt_info.ne.nf_noerr) then
        write(stdout,10) 'Cs_w', ncname(1:lstr)
        exit_flag=3
        return
      endif
!
!  Output levels.
!
      if (ncid.eq.nchisid) then
        wrt_info=nf_inq_varid(ncid,'Lev',varid)
        wrt_info=nf_put_vara_int(ncid,varid,1,Nlev,Lev)
        if (wrt_info.ne.nf_noerr) then
          write(stdout,10) 'Lev', ncname(1:lstr)
          exit_flag=3
          return
        endif
      endif
#endif /* SOLVE3D */
!
!  User generic parameters.
!
      if (Nuser.gt.0) then
        wrt_info=nf_inq_varid(ncid,'user',varid)
        wrt_info=nf_put_vara_FTYPE(ncid,varid,1,Nuser,user)
        if (wrt_info.ne.nf_noerr) then
          write(stdout,10) 'user', ncname(1:lstr)
          exit_flag=3
          return
        endif
      endif
#ifdef STATIONS
!
!  IJ-index station positions.
!
      if (ncid.eq.ncstaid) then
        wrt_info=nf_inq_varid(ncid,'Ipos',varid)
        wrt_info=nf_put_vara_int(ncid,varid,1,nstation,ispos)
        if (wrt_info.ne.nf_noerr) then
          write(stdout,10) 'Ipos', ncname(1:lstr)
          exit_flag=3
          return
        endif

        wrt_info=nf_inq_varid(ncid,'Jpos',varid)
        wrt_info=nf_put_vara_int(ncid,varid,1,nstation,jspos)
        if (wrt_info.ne.nf_noerr) then
          write(stdout,10) 'Jpos', ncname(1:lstr)
          exit_flag=3
          return
        endif
      endif
#endif /* STATIONS */
#ifndef NO_WRITE_GRID
!
!  Bathymetry.
!
      if (ncid.ne.ncfltid) then
        scale=1.0_r8
        wrt_info=nf_inq_varid(ncid,'h',varid)
        if (ncid.eq.ncstaid) then
          do k=1,nstation
            wrk(k)=h(ispos(k),jspos(k))*scale
          enddo
          wrt_info=nf_put_vara_FTYPE(ncid,varid,1,nstation,wrk)
        else
          wrt_info=nf_fwrite(h(START_2D_ARRAY),scale,ncid,
     &                       varid,0,r2dvar)
        endif
        if (wrt_info.ne.nf_noerr) then
          write(stdout,10) 'h', ncname(1:lstr)
          exit_flag=3
          return
        endif
!
!  Coriolis parameter.
!
        if (ncid.ne.ncstaid) then
          scale=1.0_r8
          wrt_info=nf_inq_varid(ncid,'f',varid)
          wrt_info=nf_fwrite(f(START_2D_ARRAY),scale,ncid,
     &                       varid,0,r2dvar)
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'f', ncname(1:lstr)
            exit_flag=3
            return
          endif
        endif
!
!  Curvilinear transformation metrics.
!
        if (ncid.ne.ncstaid) then
          scale=1.0_r8
          wrt_info=nf_inq_varid(ncid,'pm',varid)
          wrt_info=nf_fwrite(pm(START_2D_ARRAY),scale,ncid,
     &                       varid,0,r2dvar)
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'pm', ncname(1:lstr)
            exit_flag=3
            return
          endif

          wrt_info=nf_inq_varid(ncid,'pn',varid)
          wrt_info=nf_fwrite(pn(START_2D_ARRAY),scale,ncid,
     &                       varid,0,r2dvar)
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'pn', ncname(1:lstr)
            exit_flag=3
            return
          endif
        endif
!
!  Longitude and latitude of RHO-points, if appropriate.
!
        if (spherical) then
          scale=1.0_r8
          wrt_info=nf_inq_varid(ncid,'lon_rho',varid)
          if (ncid.eq.ncstaid) then
            do k=1,nstation
              wrk(k)=lonr(ispos(k),jspos(k))*scale
            enddo
            wrt_info=nf_put_vara_FTYPE(ncid,varid,1,nstation,wrk)
          else
            wrt_info=nf_fwrite(lonr(START_2D_ARRAY),scale,ncid,
     &                         varid,0,r2dvar)
          endif
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'lon_rho', ncname(1:lstr)
            exit_flag=3
            return
          endif

          wrt_info=nf_inq_varid(ncid,'lat_rho',varid)
          if (ncid.eq.ncstaid) then
            do k=1,nstation
              wrk(k)=latr(ispos(k),jspos(k))*scale
            enddo
            wrt_info=nf_put_vara_FTYPE(ncid,varid,1,nstation,wrk)
          else
            wrt_info=nf_fwrite(latr(START_2D_ARRAY),scale,ncid,
     &                         varid,0,r2dvar)
          endif
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'lat_rho', ncname(1:lstr)
            exit_flag=3
            return
          endif
        else
          scale=1.0_r8
          wrt_info=nf_inq_varid(ncid,'x_rho',varid)
          if (ncid.eq.ncstaid) then
            do k=1,nstation
              wrk(k)=xr(ispos(k),jspos(k))*scale
            enddo
            wrt_info=nf_put_vara_FTYPE(ncid,varid,1,nstation,wrk)
          else
            wrt_info=nf_fwrite(xr(START_2D_ARRAY),scale,ncid,
     &                         varid,0,r2dvar)
          endif
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'x_rho', ncname(1:lstr)
            exit_flag=3
            return
          endif

          wrt_info=nf_inq_varid(ncid,'y_rho',varid)
          if (ncid.eq.ncstaid) then
            do k=1,nstation
              wrk(k)=yr(ispos(k),jspos(k))*scale
            enddo
            wrt_info=nf_put_vara_FTYPE(ncid,varid,1,nstation,wrk)
          else
            wrt_info=nf_fwrite(yr(START_2D_ARRAY),scale,ncid,
     &                         varid,0,r2dvar)
          endif
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'y_rho', ncname(1:lstr)
            exit_flag=3
            return
          endif
        endif
# ifdef CURVGRID
!
!  Angle between XI-axis and EAST at RHO-points, if appropriate.
!
        if (spherical) then
          scale=1.0_r8
          wrt_info=nf_inq_varid(ncid,'angle',varid)
          if (ncid.eq.ncstaid) then
            do k=1,nstation
              wrk(k)=angler(ispos(k),jspos(k))*scale
            enddo
            wrt_info=nf_put_vara_FTYPE(ncid,varid,1,nstation,wrk)
          else
            wrt_info=nf_fwrite(angler(START_2D_ARRAY),scale,ncid,
     &                         varid,0,r2dvar)
          endif
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'angle', ncname(1:lstr)
            exit_flag=3
            return
          endif
        endif
# endif /* CURVGRID */
# ifdef MASKING
!
!  Masking fields at RHO-, U-, V-points, and PSI-points.
!
        if (ncid.ne.ncstaid) then
          scale=1.0_r8
          wrt_info=nf_inq_varid(ncid,'mask_rho',varid)
          wrt_info=nf_fwrite(rmask(START_2D_ARRAY),scale,ncid,
     &                       varid,0,r2dvar)
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'mask_rho', ncname(1:lstr)
            exit_flag=3
            return
          endif

          wrt_info=nf_inq_varid(ncid,'mask_u',varid)
          wrt_info=nf_fwrite(umask(START_2D_ARRAY),scale,ncid,
     &                       varid,0,u2dvar)
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'mask_u', ncname(1:lstr)
            exit_flag=3
            return
          endif

          wrt_info=nf_inq_varid(ncid,'mask_v',varid)
          wrt_info=nf_fwrite(vmask(START_2D_ARRAY),scale,ncid,
     &                       varid,0,v2dvar)
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'mask_v', ncname(1:lstr)
            exit_flag=3
            return
          endif

          wrt_info=nf_inq_varid(ncid,'mask_psi',varid)
          wrt_info=nf_fwrite(pmask(START_2D_ARRAY),scale,ncid,
     &                       varid,0,p2dvar)
          if (wrt_info.ne.nf_noerr) then
            write(stdout,10) 'mask_psi', ncname(1:lstr)
            exit_flag=3
            return
          endif
        endif
# endif /* MASKING */
      endif
#endif /* !NO_WRITE_GRID */
!
  10  format(/,' WRT_INFO - error while writing variable: ',a,/,
     &       12x,'into NetCDF file: ',a)
      return
      end
