#include "cppdefs.h"
#if defined SOLVE3D && defined UV_VIS4 && defined MIX_GEO_UV
      subroutine uv3dmix4_geo (tile)
!
!=====================================================================
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This subroutine computes  biharmonic mixing of momentum,  rotated !
!  along geopotentials, from the horizontal divergence of the stress !
!  tensor.  A transverse isotropy is assumed so the stress tensor is !
!  splitted into vertical and horizontal subtensors.                 !
!                                                                    !
!  Reference:                                                        !
!                                                                    !
!      Wajsowicz, R.C, 1993: A consistent formulation of the         !
!         anisotropic stress tensor for use in models of the         !
!         large-scale ocean circulation, JCP, 105, 333-338.          !
!                                                                    !
!      Sadourny, R. and K. Maynard, 1997: Formulations of            !
!         lateral diffusion in geophysical fluid dynamics            !
!         models, In "Numerical Methods of Atmospheric and           !
!         Oceanic Modelling". Lin, Laprise, and Ritchie,             !
!         Eds., NRC Research Press, 547-556.                         !
!                                                                    !
!      Griffies, S.M. and R.W. Hallberg, 2000: Biharmonic            !
!         friction with a Smagorinsky-like viscosity for             !
!         use in large-scale eddy-permitting ocean models,           !
!         Monthly Weather Rev., 128, 8, 2935-2946.                   !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "scratch.h"
# include "tile.h"
!
# ifdef PROFILE
      call wclock_on (33)
# endif
      call uv3dmix4_geo_tile (Istr,Iend,Jstr,Jend,
     &                        A3d(1, 1),A3d(1, 2),A2d(1, 1),A2d(1, 2),
     &                        A2d(1, 3),A2d(1, 4),A2d(1, 5),A2d(1, 7),
     &                        A2d(1, 9),A2d(1,11),A2d(1,13),A2d(1,15),
     &                        A2d(1,17),A2d(1,19),A2d(1,21),A2d(1,23),
     &                        A2d(1,25),A2d(1,27))
# ifdef PROFILE
      call wclock_off (33)
# endif
      return
      end
!
!*********************************************************************
      subroutine uv3dmix4_geo_tile (Istr,Iend,Jstr,Jend,LapU,LapV,UFx,
     &                              UFe,VFx,VFe,UFs,VFs,dnUdx,dmUde,
     &                              dUdz,dnVdx,dmVde,dVdz,dZdx_r,
     &                              dZdx_p,dZde_r,dZde_p)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "mask.h"
# include "mixing.h"
# include "ocean.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k, k1, k2
      REAL_TYPE
     &        cff, cff1, cff2, cff3, cff4, cff5, cff6, cff7, cff8,
     &        dmUdz, dnUdz, dmVdz, dnVdz
      REAL_TYPE
     &        LapU(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &        LapV(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &           UFe(PRIVATE_2D_SCRATCH_ARRAY),
     &           VFe(PRIVATE_2D_SCRATCH_ARRAY),
     &           UFx(PRIVATE_2D_SCRATCH_ARRAY),
     &           VFx(PRIVATE_2D_SCRATCH_ARRAY),
     &           UFs(PRIVATE_2D_SCRATCH_ARRAY,2),
     &           VFs(PRIVATE_2D_SCRATCH_ARRAY,2),
     &         dmUde(PRIVATE_2D_SCRATCH_ARRAY,2),
     &         dmVde(PRIVATE_2D_SCRATCH_ARRAY,2),
     &         dnUdx(PRIVATE_2D_SCRATCH_ARRAY,2),
     &         dnVdx(PRIVATE_2D_SCRATCH_ARRAY,2),
     &          dUdz(PRIVATE_2D_SCRATCH_ARRAY,2),
     &          dVdz(PRIVATE_2D_SCRATCH_ARRAY,2),
     &        dZde_p(PRIVATE_2D_SCRATCH_ARRAY,2),
     &        dZde_r(PRIVATE_2D_SCRATCH_ARRAY,2),
     &        dZdx_p(PRIVATE_2D_SCRATCH_ARRAY,2),
     &        dZdx_r(PRIVATE_2D_SCRATCH_ARRAY,2)
!
# include "set_bounds.h"
# ifdef EW_PERIODIC
#  define IV_RANGE Istr-1,Iend+1
#  define IU_RANGE Istr-1,Iend+1
# else
#  define IV_RANGE MAX(1,Istr-1),MIN(Iend+1,Lm)
#  define IU_RANGE MAX(2,IstrU-1),MIN(Iend+1,Lm)
# endif
# ifdef NS_PERIODIC
#  define JU_RANGE Jstr-1,Jend+1
#  define JV_RANGE Jstr-1,Jend+1
# else
#  define JU_RANGE MAX(1,Jstr-1),MIN(Jend+1,Mm)
#  define JV_RANGE MAX(2,JstrV-1),MIN(Jend+1,Mm)
# endif
!
!--------------------------------------------------------------------
!  Compute horizontal biharmonic viscosity along geopotential
!  surfaces.  The biharmonic operator is computed by applying
!  the harmonic operator twice.
!--------------------------------------------------------------------
!
!  Compute horizontal and vertical gradients.  Notice the recursive
!  blocking sequence. It is assumed here that "visc4_r" and "visc4_p"
!  are the squared root of the biharmonic viscosity coefficient. For
!  momentum balance purposes, the thickness "Hz" appears only when
!  computing the second harmonic operator. The vertical placement of
!  the gradients is:
!
!    dZdx_r, dZde_r, dnUdx, dmVde(:,:,k1) k      rho-points
!    dZdx_r, dZde_r, dnUdx, dmVde(:,:,k2) k+1    rho-points
!    dZdx_p, dZde_p, dnVdx, dmUde(:,:,k1) k      psi-points
!    dZdx_p, dZde_p, dnVdx, dmUde(:,:,k2) k+1    psi-points
!                       UFs, dUdz(:,:,k1) k-1/2  WU-points
!                       UFs, dUdz(:,:,k2) k+1/2  WU-points
!                       VFs, dVdz(:,:,k1) k-1/2  WV-points
!                       VFs, dVdz(:,:,k2) k+1/2  WV-points
!
      k2=1
      do k=0,N
        k1=k2
        k2=3-k1
        if (k.lt.N) then
!
!  Compute slopes (nondimensional) at RHO- and PSI-points.
!
          do j=-1+JU_RANGE+1
            do i=IV_RANGE+1
              UFx(i,j)=0.5_r8*(pm(i-1,j)+pm(i,j))*
     &                 (z_r(i,j,k+1)-z_r(i-1,j,k+1))
# ifdef MASKING
     &                *umask(i,j)
# endif
            enddo
            do i=IV_RANGE
              dZdx_r(i,j,k2)=0.5_r8*(UFx(i,j)+UFx(i+1,j))
            enddo
          enddo
          do j=JU_RANGE+1
            do i=-1+IV_RANGE+1
              VFe(i,j)=0.5_r8*(pn(i,j-1)+pn(i,j))*
     &                 (z_r(i,j,k+1)-z_r(i,j-1,k+1))
# ifdef MASKING
     &                *vmask(i,j)
# endif
            enddo
            do i=IV_RANGE+1
              dZde_p(i,j,k2)=0.5_r8*(VFe(i-1,j)+VFe(i,j))
              dZdx_p(i,j,k2)=0.5_r8*(UFx(i,j-1)+UFx(i,j))
            enddo
          enddo
          do j=JU_RANGE
            do i=-1+IV_RANGE+1
              dZde_r(i,j,k2)=0.5_r8*(VFe(i,j)+VFe(i,j+1))
            enddo
          enddo
!
!  Compute momentum horizontal (1/m/s) and vertical (1/s) gradients.
!
          do j=-1+JV_RANGE
            do i=-1+IU_RANGE
              dnUdx(i,j,k2)=0.5_r8*pm(i,j)*
     &                      ((pn(i  ,j)+pn(i+1,j))*u(i+1,j,k+1,nrhs)-
     &                       (pn(i-1,j)+pn(i  ,j))*u(i  ,j,k+1,nrhs))
# ifdef MASKING
     &                     *rmask(i,j)
# endif
              dmVde(i,j,k2)=0.5_r8*pn(i,j)*
     &                      ((pm(i,j  )+pm(i,j+1))*v(i,j+1,k+1,nrhs)-
     &                       (pm(i,j-1)+pm(i,j  ))*v(i,j  ,k+1,nrhs))
# ifdef MASKING
     &                     *rmask(i,j)
# endif
            enddo
          enddo
          do j=JU_RANGE+1
            do i=IV_RANGE+1
              dmUde(i,j,k2)=0.125_r8*(pn(i-1,j  )+pn(i,j  )+
     &                                pn(i-1,j-1)+pn(i,j-1))*
     &                      ((pm(i-1,j  )+pm(i,j  ))*u(i,j  ,k+1,nrhs)-
     &                       (pm(i-1,j-1)+pm(i,j-1))*u(i,j-1,k+1,nrhs))
# ifdef MASKING
     &                     *pmask(i,j)
# endif
              dnVdx(i,j,k2)=0.125_r8*(pm(i-1,j  )+pm(i,j  )+
     &                                pm(i-1,j-1)+pm(i,j-1))*
     &                      ((pn(i  ,j-1)+pn(i  ,j))*v(i  ,j,k+1,nrhs)-
     &                       (pn(i-1,j-1)+pn(i-1,j))*v(i-1,j,k+1,nrhs))
# ifdef MASKING
     &                     *pmask(i,j)
# endif
            enddo
          enddo
        endif
        if ((k.eq.0).or.(k.eq.N)) then
          do j=JU_RANGE
            do i=-1+IU_RANGE+1
              dUdz(i,j,k2)=0.0_r8
              UFs(i,j,k2)=0.0_r8
            enddo
          enddo
          do j=-1+JV_RANGE+1
            do i=IV_RANGE
              dVdz(i,j,k2)=0.0_r8
              VFs(i,j,k2)=0.0_r8
            enddo
          enddo
        else
          do j=JU_RANGE
            do i=-1+IU_RANGE+1
              dUdz(i,j,k2)=(u(i,j,k+1,nrhs)-u(i,j,k,nrhs))/
     &                     (0.5_r8*(z_r(i-1,j,k+1)-z_r(i-1,j,k)+
     &                              z_r(i  ,j,k+1)-z_r(i  ,j,k)))
            enddo
          enddo
          do j=-1+JV_RANGE+1
            do i=IV_RANGE
              dVdz(i,j,k2)=(v(i,j,k+1,nrhs)-v(i,j,k,nrhs))/
     &                     (0.5_r8*(z_r(i,j-1,k+1)-z_r(i,j-1,k)+
     &                              z_r(i,j  ,k+1)-z_r(i,j  ,k)))
            enddo
          enddo
        endif
!
!  Compute components of the rotated viscous flux (m^4 s-^3/2) along
!  geopotential surfaces in the XI- and ETA-directions.
!
        if (k.gt.0) then
          do j=-1+JV_RANGE
            do i=-1+IU_RANGE
              cff=visc4_r(i,j)*
     &            (on_r(i,j)*(dnUdx(i,j,k1)-0.5_r8*pn(i,j)*
     &                        (MIN(dZdx_r(i,j,k1),0.0_r8)*
     &                             (dUdz(i,j,k1)+dUdz(i+1,j,k2))+
     &                         MAX(dZdx_r(i,j,k1),0.0_r8)*
     &                             (dUdz(i,j,k2)+dUdz(i+1,j,k1))))-
     &             om_r(i,j)*(dmVde(i,j,k1)-0.5_r8*pm(i,j)*
     &                        (MIN(dZde_r(i,j,k1),0.0_r8)*
     &                             (dVdz(i,j,k1)+dVdz(i,j+1,k2))+
     &                         MAX(dZde_r(i,j,k1),0.0_r8)*
     &                             (dVdz(i,j,k2)+dVdz(i,j+1,k1)))))
# ifdef MASKING
     &           *rmask(i,j)
# endif
              UFx(i,j)=on_r(i,j)*on_r(i,j)*cff
              VFe(i,j)=om_r(i,j)*om_r(i,j)*cff
            enddo
          enddo
          do j=JU_RANGE+1
            do i=IV_RANGE+1
              cff=visc4_p(i,j)*
     &            (on_p(i,j)*(dnVdx(i,j,k1)-
     &                        0.125_r8*(pn(i-1,j-1)+pn(i-1,j)+
     &                                  pn(i  ,j-1)+pn(i  ,j))*
     &                        (MIN(dZdx_p(i,j,k1),0.0_r8)*
     &                             (dVdz(i-1,j,k1)+dVdz(i,j,k2))+
     &                         MAX(dZdx_p(i,j,k1),0.0_r8)*
     &                             (dVdz(i-1,j,k2)+dVdz(i,j,k1))))+
     &             om_p(i,j)*(dmUde(i,j,k1)-
     &                        0.125_r8*(pm(i-1,j-1)+pm(i-1,j)+
     &                                  pm(i  ,j-1)+pm(i  ,j))*
     &                        (MIN(dZde_p(i,j,k1),0.0_r8)*
     &                             (dUdz(i,j-1,k1)+dUdz(i,j,k2))+
     &                         MAX(dZde_p(i,j,k1),0.0_r8)*
     &                             (dUdz(i,j-1,k2)+dUdz(i,j,k1)))))
# ifdef MASKING
     &           *pmask(i,j)
# endif
              UFe(i,j)=om_p(i,j)*om_p(i,j)*cff
              VFx(i,j)=on_p(i,j)*on_p(i,j)*cff
            enddo
          enddo
!
!  Compute vertical flux (m^2 s^-3/2) due to sloping terrain-following
!  surfaces.
!
          if (k.lt.N) then
            do j=JU_RANGE
              do i=IU_RANGE
                cff=0.5_r8*(pn(i-1,j)+pn(i,j))
                dnUdz=cff*dUdz(i,j,k2)
                dnVdz=cff*0.25_r8*(dVdz(i-1,j+1,k2)+dVdz(i,j+1,k2)+
     &                             dVdz(i-1,j  ,k2)+dVdz(i,j  ,k2))
                cff=0.5_r8*(pm(i-1,j)+pm(i,j))
                dmUdz=cff*dUdz(i,j,k2)
                dmVdz=cff*0.25_r8*(dVdz(i-1,j+1,k2)+dVdz(i,j+1,k2)+
     &                             dVdz(i-1,j  ,k2)+dVdz(i,j  ,k2))
                cff1=MIN(dZdx_r(i-1,j,k1),0.0_r8)
                cff2=MIN(dZdx_r(i  ,j,k2),0.0_r8)
                cff3=MAX(dZdx_r(i-1,j,k2),0.0_r8)
                cff4=MAX(dZdx_r(i  ,j,k1),0.0_r8)
                UFs(i,j,k2)=on_u(i,j)*
     &                      0.25_r8*(visc4_r(i-1,j)+visc4_r(i,j))*
     &                      (cff1*(cff1*dnUdz-dnUdx(i-1,j,k1))+
     &                       cff2*(cff2*dnUdz-dnUdx(i  ,j,k2))+
     &                       cff3*(cff3*dnUdz-dnUdx(i-1,j,k2))+
     &                       cff4*(cff4*dnUdz-dnUdx(i  ,j,k1)))
                cff1=MIN(dZde_p(i,j  ,k1),0.0_r8)
                cff2=MIN(dZde_p(i,j+1,k2),0.0_r8)
                cff3=MAX(dZde_p(i,j  ,k2),0.0_r8)
                cff4=MAX(dZde_p(i,j+1,k1),0.0_r8)
                UFs(i,j,k2)=UFs(i,j,k2)+om_u(i,j)*
     &                      0.25_r8*(visc4_r(i-1,j)+visc4_r(i,j))*
     &                      (cff1*(cff1*dmUdz-dmUde(i,j  ,k1))+
     &                       cff2*(cff2*dmUdz-dmUde(i,j+1,k2))+
     &                       cff3*(cff3*dmUdz-dmUde(i,j  ,k2))+
     &                       cff4*(cff4*dmUdz-dmUde(i,j+1,k1)))
                cff1=MIN(dZde_p(i,j  ,k1),0.0_r8)
                cff2=MIN(dZde_p(i,j+1,k2),0.0_r8)
                cff3=MAX(dZde_p(i,j  ,k2),0.0_r8)
                cff4=MAX(dZde_p(i,j+1,k1),0.0_r8)
                cff5=MIN(dZdx_p(i,j  ,k1),0.0_r8)
                cff6=MIN(dZdx_p(i,j+1,k2),0.0_r8)
                cff7=MAX(dZdx_p(i,j  ,k2),0.0_r8)
                cff8=MAX(dZdx_p(i,j+1,k1),0.0_r8)
                UFs(i,j,k2)=UFs(i,j,k2)+on_u(i,j)*
     &                      0.25_r8*(visc4_r(i-1,j)+visc4_r(i,j))*
     &                      (cff1*(cff5*dnVdz-dnVdx(i,j  ,k1))+
     &                       cff2*(cff6*dnVdz-dnVdx(i,j+1,k2))+
     &                       cff3*(cff7*dnVdz-dnVdx(i,j  ,k2))+
     &                       cff4*(cff8*dnVdz-dnVdx(i,j+1,k1)))
                cff1=MIN(dZdx_r(i-1,j,k1),0.0_r8)
                cff2=MIN(dZdx_r(i  ,j,k2),0.0_r8)
                cff3=MAX(dZdx_r(i-1,j,k2),0.0_r8)
                cff4=MAX(dZdx_r(i  ,j,k1),0.0_r8)
                cff5=MIN(dZde_r(i-1,j,k1),0.0_r8)
                cff6=MIN(dZde_r(i  ,j,k2),0.0_r8)
                cff7=MAX(dZde_r(i-1,j,k2),0.0_r8)
                cff8=MAX(dZde_r(i  ,j,k1),0.0_r8)
                UFs(i,j,k2)=UFs(i,j,k2)-om_u(i,j)*
     &                      0.25_r8*(visc4_r(i-1,j)+visc4_r(i,j))*
     &                      (cff1*(cff5*dmVdz-dmVde(i-1,j,k1))+
     &                       cff2*(cff6*dmVdz-dmVde(i  ,j,k2))+
     &                       cff3*(cff7*dmVdz-dmVde(i-1,j,k2))+
     &                       cff4*(cff8*dmVdz-dmVde(i  ,j,k1)))
              enddo
            enddo
!
            do j=JV_RANGE
              do i=IV_RANGE
                cff=0.5_r8*(pn(i,j-1)+pn(i,j))
                dnUdz=cff*0.25_r8*(dUdz(i,j  ,k2)+dUdz(i+1,j  ,k2)+
     &                             dUdz(i,j-1,k2)+dUdz(i+1,j-1,k2))
                dnVdz=cff*dVdz(i,j,k2)
                cff=0.5_r8*(pm(i,j-1)+pm(i,j))
                dmUdz=cff*0.25_r8*(dUdz(i,j  ,k2)+dUdz(i+1,j  ,k2)+
     &                             dUdz(i,j-1,k2)+dUdz(i+1,j-1,k2))
                dmVdz=cff*dVdz(i,j,k2)
                cff1=MIN(dZdx_p(i  ,j,k1),0.0_r8)
                cff2=MIN(dZdx_p(i+1,j,k2),0.0_r8)
                cff3=MAX(dZdx_p(i  ,j,k2),0.0_r8)
                cff4=MAX(dZdx_p(i+1,j,k1),0.0_r8)
                VFs(i,j,k2)=on_v(i,j)*
     &                      0.25_r8*(visc4_r(i,j-1)+visc4_r(i,j))*
     &                      (cff1*(cff1*dnVdz-dnVdx(i  ,j,k1))+
     &                       cff2*(cff2*dnVdz-dnVdx(i+1,j,k2))+
     &                       cff3*(cff3*dnVdz-dnVdx(i  ,j,k2))+
     &                       cff4*(cff4*dnVdz-dnVdx(i+1,j,k1)))
                cff1=MIN(dZde_r(i,j-1,k1),0.0_r8)
                cff2=MIN(dZde_r(i,j  ,k2),0.0_r8)
                cff3=MAX(dZde_r(i,j-1,k2),0.0_r8)
                cff4=MAX(dZde_r(i,j  ,k1),0.0_r8)
                VFs(i,j,k2)=VFs(i,j,k2)+om_v(i,j)*
     &                      0.25_r8*(visc4_r(i,j-1)+visc4_r(i,j))*
     &                      (cff1*(cff1*dmVdz-dmVde(i,j-1,k1))+
     &                       cff2*(cff2*dmVdz-dmVde(i,j  ,k2))+
     &                       cff3*(cff3*dmVdz-dmVde(i,j-1,k2))+
     &                       cff4*(cff4*dmVdz-dmVde(i,j  ,k1)))
                cff1=MIN(dZde_r(i,j-1,k1),0.0_r8)
                cff2=MIN(dZde_r(i,j  ,k2),0.0_r8)
                cff3=MAX(dZde_r(i,j-1,k2),0.0_r8)
                cff4=MAX(dZde_r(i,j  ,k1),0.0_r8)
                cff5=MIN(dZdx_r(i,j-1,k1),0.0_r8)
                cff6=MIN(dZdx_r(i,j  ,k2),0.0_r8)
                cff7=MAX(dZdx_r(i,j-1,k2),0.0_r8)
                cff8=MAX(dZdx_r(i,j  ,k1),0.0_r8)
                VFs(i,j,k2)=VFs(i,j,k2)-on_v(i,j)*
     &                      0.25_r8*(visc4_r(i,j-1)+visc4_r(i,j))*
     &                      (cff1*(cff5*dnUdz-dnUdx(i,j-1,k1))+
     &                       cff2*(cff6*dnUdz-dnUdx(i,j  ,k2))+
     &                       cff3*(cff7*dnUdz-dnUdx(i,j-1,k2))+
     &                       cff4*(cff8*dnUdz-dnUdx(i,j  ,k1)))
                cff1=MIN(dZdx_p(i  ,j,k1),0.0_r8)
                cff2=MIN(dZdx_p(i+1,j,k2),0.0_r8)
                cff3=MAX(dZdx_p(i  ,j,k2),0.0_r8)
                cff4=MAX(dZdx_p(i+1,j,k1),0.0_r8)
                cff5=MIN(dZde_p(i  ,j,k1),0.0_r8)
                cff6=MIN(dZde_p(i+1,j,k2),0.0_r8)
                cff7=MAX(dZde_p(i  ,j,k2),0.0_r8)
                cff8=MAX(dZde_p(i+1,j,k1),0.0_r8)
                VFs(i,j,k2)=VFs(i,j,k2)+om_v(i,j)*
     &                      0.25_r8*(visc4_r(i,j-1)+visc4_r(i,j))*
     &                      (cff1*(cff5*dmUdz-dmUde(i  ,j,k1))+
     &                       cff2*(cff6*dmUdz-dmUde(i+1,j,k2))+
     &                       cff3*(cff7*dmUdz-dmUde(i  ,j,k2))+
     &                       cff4*(cff8*dmUdz-dmUde(i+1,j,k1)))
              enddo
            enddo
          endif
!
! Compute first harmonic operator (m s^-3/2).
!
          do j=JU_RANGE
            do i=IU_RANGE
              LapU(i,j,k)=(0.125_r8*
     &                     (pm(i-1,j)+pm(i,j))*(pn(i-1,j)+pn(i,j))*
     &                     ((pn(i-1,j)+pn(i,j))*(UFx(i  ,j)-
     &                                           UFx(i-1,j))+
     &                      (pm(i-1,j)+pm(i,j))*(UFe(i,j+1)-
     &                                           UFe(i,j  )))+
     &                     (UFs(i,j,k2)-UFs(i,j,k1))/
     &                     (0.5_r8*(Hz(i-1,j,k)+Hz(i,j,k))))
# ifdef MASKING
     &                   *umask(i,j)
# endif
            enddo
          enddo
          do j=JV_RANGE
            do i=IV_RANGE
              LapV(i,j,k)=(0.125_r8*
     &                     (pm(i,j)+pm(i,j-1))*(pn(i,j)+pn(i,j-1))*
     &                     ((pn(i,j-1)+pn(i,j))*(VFx(i+1,j)-
     &                                           VFx(i  ,j))-
     &                      (pm(i,j-1)+pm(i,j))*(VFe(i,j  )-
     &                                           VFe(i,j-1)))+
     &                     (VFs(i,j,k2)-VFs(i,j,k1))/
     &                     (0.5_r8*(Hz(i,j-1,k)+Hz(i,j,k))))
# ifdef MASKING
     &                   *vmask(i,j)
# endif
            enddo
          enddo
        endif
      enddo
!
!  Apply boundary conditions (except periodic; closed or gradient)
!  to the first harmonic operator.
!
# ifndef EW_PERIODIC
      if (WESTERN_EDGE) then
        do k=1,N
          do j=JU_RANGE
#  ifdef WESTERN_WALL
            LapU(IstrU-1,j,k)=0.0_r8
#  else
            LapU(IstrU-1,j,k)=LapU(IstrU,j,k)
#  endif
          enddo
          do j=JV_RANGE
#  ifdef WESTERN_WALL
            LapV(Istr-1,j,k)=gamma2*LapV(Istr,j,k)
#  else
            LapV(Istr-1,j,k)=0.0_r8
#  endif
          enddo
        enddo
      endif
      if (EASTERN_EDGE) then
        do k=1,N
          do j=JU_RANGE
#  ifdef EASTERN_WALL
            LapU(Iend+1,j,k)=0.0_r8
#  else
            LapU(Iend+1,j,k)=LapU(Iend,j,k)
#  endif
          enddo
          do j=JV_RANGE
#  ifdef EASTERN_WALL
            LapV(Iend+1,j,k)=gamma2*LapV(Iend,j,k)
#  else
            LapV(Iend+1,j,k)=0.0_r8
#  endif
          enddo
        enddo
      endif
# endif /* !EW_PERIODIC */
# ifndef NS_PERIODIC
      if (SOUTHERN_EDGE) then
        do k=1,N
          do i=IU_RANGE
#  ifdef SOUTHERN_WALL
            LapU(i,Jstr-1,k)=gamma2*LapU(i,Jstr,k)
#  else
            LapU(i,Jstr-1,k)=0.0_r8
#  endif
          enddo
          do i=IV_RANGE
#  ifdef SOUTHERN_WALL
            LapV(i,JstrV-1,k)=0.0_r8
#  else
            LapV(i,JstrV-1,k)=LapV(i,JstrV,k)
#  endif
          enddo
        enddo
      endif
      if (NORTHERN_EDGE) then
        do k=1,N
          do i=IU_RANGE
#  ifdef NORTHERN_WALL
            LapU(i,Jend+1,k)=gamma2*LapU(i,Jend,k)
#  else
            LapU(i,Jend+1,k)=0.0_r8
#  endif
          enddo
          do i=IV_RANGE
#  ifdef NORTHERN_WALL
            LapV(i,Jend+1,k)=0.0_r8
#  else
            LapV(i,Jend+1,k)=LapV(i,Jend,k)
#  endif
          enddo
        enddo
      endif
# endif /* !NS_PERIODIC */
# if !defined EW_PERIODIC && !defined NS_PERIODIC
      if (SOUTHERN_EDGE .and. WESTERN_EDGE) then
        do k=1,N
          LapU(1,0,k)=0.5_r8*(LapU(2,0,k)+LapU(1,1,k))
          LapV(0,1,k)=0.5_r8*(LapV(0,2,k)+LapV(1,1,k))
        enddo
      endif
      if (SOUTHERN_EDGE .and. EASTERN_EDGE) then
        do k=1,N
          LapU(L,0,k)=0.5_r8*(LapU(Lm,0,k)+LapU(L,1,k))
          LapV(L,1,k)=0.5_r8*(LapV(Lm,1,k)+LapV(L,2,k))
        enddo
      endif
      if (NORTHERN_EDGE .and. WESTERN_EDGE) then
        do k=1,N
          LapU(1,M,k)=0.5_r8*(LapU(2,M,k)+LapU(1,Mm,k))
          LapV(0,M,k)=0.5_r8*(LapV(1,M,k)+LapV(0,Mm,k))
        enddo
      endif
      if (NORTHERN_EDGE .and. EASTERN_EDGE) then
        do k=1,N
          LapU(L,M,k)=0.5_r8*(LapU(Lm,M,k)+LapU(L,Mm,k))
          LapV(L,M,k)=0.5_r8*(LapV(Lm,M,k)+LapV(L,Mm,k))
        enddo
      endif
# endif /* !EW_PERIODIC && !NS_PERIODIC */
!
!  Compute horizontal and vertical gradients associated with the
!  second rotated harmonic operator.
!
      k2=1
      do k=0,N
        k1=k2
        k2=3-k1
        if (k.lt.N) then
!
!  Compute slopes (nondimensional) at RHO- and PSI-points.
!
          do j=-1+JU_RANGE+1
            do i=IV_RANGE+1
              UFx(i,j)=0.5_r8*(pm(i-1,j)+pm(i,j))*
     &                 (z_r(i,j,k+1)-z_r(i-1,j,k+1))
# ifdef MASKING
     &                *umask(i,j)
# endif
            enddo
            do i=IV_RANGE
              dZdx_r(i,j,k2)=0.5_r8*(UFx(i,j)+UFx(i+1,j))
            enddo
          enddo
          do j=JU_RANGE+1
            do i=-1+IV_RANGE+1
              VFe(i,j)=0.5_r8*(pn(i,j-1)+pn(i,j))*
     &                 (z_r(i,j,k+1)-z_r(i,j-1,k+1))
# ifdef MASKING
     &                *vmask(i,j)
# endif
            enddo
            do i=IV_RANGE+1
              dZde_p(i,j,k2)=0.5_r8*(VFe(i-1,j)+VFe(i,j))
              dZdx_p(i,j,k2)=0.5_r8*(UFx(i,j-1)+UFx(i,j))
            enddo
          enddo
          do j=JU_RANGE
            do i=-1+IV_RANGE+1
              dZde_r(i,j,k2)=0.5_r8*(VFe(i,j)+VFe(i,j+1))
            enddo
          enddo
!
!  Compute momentum horizontal (m^-1 s^-3/2) and vertical (s^-3/2)
!  gradients.
!
          do j=JstrV-1,Jend
            do i=IstrU-1,Iend
              dnUdx(i,j,k2)=0.5_r8*pm(i,j)*
     &                      ((pn(i  ,j)+pn(i+1,j))*LapU(i+1,j,k+1)-
     &                       (pn(i-1,j)+pn(i  ,j))*LapU(i  ,j,k+1))
# ifdef MASKING
     &                     *rmask(i,j)
# endif
              dmVde(i,j,k2)=0.5_r8*pn(i,j)*
     &                      ((pm(i,j  )+pm(i,j+1))*LapV(i,j+1,k+1)-
     &                       (pm(i,j-1)+pm(i,j  ))*LapV(i,j  ,k+1))
# ifdef MASKING
     &                     *rmask(i,j)
# endif
            enddo
          enddo
          do j=Jstr,Jend+1
            do i=Istr,Iend+1
              dmUde(i,j,k2)=0.125_r8*(pn(i-1,j  )+pn(i,j  )+
     &                                pn(i-1,j-1)+pn(i,j-1))*
     &                      ((pm(i-1,j  )+pm(i,j  ))*LapU(i,j  ,k+1)-
     &                       (pm(i-1,j-1)+pm(i,j-1))*LapU(i,j-1,k+1))
# ifdef MASKING
     &                     *pmask(i,j)
# endif
              dnVdx(i,j,k2)=0.125_r8*(pm(i-1,j  )+pm(i,j  )+
     &                                pm(i-1,j-1)+pm(i,j-1))*
     &                      ((pn(i  ,j-1)+pn(i  ,j))*LapV(i  ,j,k+1)-
     &                       (pn(i-1,j-1)+pn(i-1,j))*LapV(i-1,j,k+1))
# ifdef MASKING
     &                     *pmask(i,j)
# endif
            enddo
          enddo
        endif
        if ((k.eq.0).or.(k.eq.N)) then
          do j=Jstr-1,Jend+1
            do i=IstrU-1,Iend+1
              dUdz(i,j,k2)=0.0_r8
              UFs(i,j,k2)=0.0_r8
            enddo
          enddo
          do j=JstrV-1,Jend+1
            do i=Istr-1,Iend+1
              dVdz(i,j,k2)=0.0_r8
              VFs(i,j,k2)=0.0_r8
            enddo
          enddo
        else
          do j=Jstr-1,Jend+1
            do i=IstrU-1,Iend+1
              dUdz(i,j,k2)=(LapU(i,j,k+1)-LapU(i,j,k))/
     &                     (0.5_r8*(z_r(i-1,j,k+1)-z_r(i-1,j,k)+
     &                              z_r(i  ,j,k+1)-z_r(i  ,j,k)))
            enddo
          enddo
          do j=JstrV-1,Jend+1
            do i=Istr-1,Iend+1
              dVdz(i,j,k2)=(LapV(i,j,k+1)-LapV(i,j,k))/
     &                     (0.5_r8*(z_r(i,j-1,k+1)-z_r(i,j-1,k)+
     &                              z_r(i,j  ,k+1)-z_r(i,j  ,k)))
            enddo
          enddo
        endif
!
!  Compute components of the rotated viscous flux (m5/s2) along
!  geopotential surfaces in the XI- and ETA-directions.
!
        if (k.gt.0) then
          do j=JstrV-1,Jend
            do i=IstrU-1,Iend
              cff=visc4_r(i,j)*Hz(i,j,k)*
     &            (on_r(i,j)*(dnUdx(i,j,k1)-0.5_r8*pn(i,j)*
     &                        (MIN(dZdx_r(i,j,k1),0.0_r8)*
     &                             (dUdz(i,j,k1)+dUdz(i+1,j,k2))+
     &                         MAX(dZdx_r(i,j,k1),0.0_r8)*
     &                             (dUdz(i,j,k2)+dUdz(i+1,j,k1))))-
     &             om_r(i,j)*(dmVde(i,j,k1)-0.5_r8*pm(i,j)*
     &                        (MIN(dZde_r(i,j,k1),0.0_r8)*
     &                             (dVdz(i,j,k1)+dVdz(i,j+1,k2))+
     &                         MAX(dZde_r(i,j,k1),0.0_r8)*
     &                             (dVdz(i,j,k2)+dVdz(i,j+1,k1)))))
# ifdef MASKING
     &           *rmask(i,j)
# endif
              UFx(i,j)=on_r(i,j)*on_r(i,j)*cff
              VFe(i,j)=om_r(i,j)*om_r(i,j)*cff
            enddo
          enddo
          do j=Jstr,Jend+1
            do i=Istr,Iend+1
              cff=visc4_p(i,j)*0.25_r8*(Hz(i-1,j  ,k)+Hz(i,j  ,k)+
     &                                  Hz(i-1,j-1,k)+Hz(i,j-1,k))*
     &            (on_p(i,j)*(dnVdx(i,j,k1)-
     &                        0.125_r8*(pn(i-1,j-1)+pn(i-1,j)+
     &                                  pn(i  ,j-1)+pn(i  ,j))*
     &                        (MIN(dZdx_p(i,j,k1),0.0_r8)*
     &                             (dVdz(i-1,j,k1)+dVdz(i,j,k2))+
     &                         MAX(dZdx_p(i,j,k1),0.0_r8)*
     &                             (dVdz(i-1,j,k2)+dVdz(i,j,k1))))+
     &             om_p(i,j)*(dmUde(i,j,k1)-
     &                        0.125_r8*(pm(i-1,j-1)+pm(i-1,j)+
     &                                  pm(i  ,j-1)+pm(i  ,j))*
     &                        (MIN(dZde_p(i,j,k1),0.0_r8)*
     &                             (dUdz(i,j-1,k1)+dUdz(i,j,k2))+
     &                         MAX(dZde_p(i,j,k1),0.0_r8)*
     &                             (dUdz(i,j-1,k2)+dUdz(i,j,k1)))))
# ifdef MASKING
     &           *pmask(i,j)
# endif
              UFe(i,j)=om_p(i,j)*om_p(i,j)*cff
              VFx(i,j)=on_p(i,j)*on_p(i,j)*cff
            enddo
          enddo
!
!  Compute vertical flux (m2/s2) due to sloping terrain-following
!  surfaces.
!
          if (k.lt.N) then
            do j=Jstr,Jend
              do i=IstrU,Iend
                cff=0.5_r8*(pn(i-1,j)+pn(i,j))
                dnUdz=cff*dUdz(i,j,k2)
                dnVdz=cff*0.25_r8*(dVdz(i-1,j+1,k2)+dVdz(i,j+1,k2)+
     &                             dVdz(i-1,j  ,k2)+dVdz(i,j  ,k2))
                cff=0.5_r8*(pm(i-1,j)+pm(i,j))
                dmUdz=cff*dUdz(i,j,k2)
                dmVdz=cff*0.25_r8*(dVdz(i-1,j+1,k2)+dVdz(i,j+1,k2)+
     &                             dVdz(i-1,j  ,k2)+dVdz(i,j  ,k2))
                cff1=MIN(dZdx_r(i-1,j,k1),0.0_r8)
                cff2=MIN(dZdx_r(i  ,j,k2),0.0_r8)
                cff3=MAX(dZdx_r(i-1,j,k2),0.0_r8)
                cff4=MAX(dZdx_r(i  ,j,k1),0.0_r8)
                UFs(i,j,k2)=on_u(i,j)*
     &                      0.25_r8*(visc4_r(i-1,j)+visc4_r(i,j))*
     &                      (cff1*(cff1*dnUdz-dnUdx(i-1,j,k1))+
     &                       cff2*(cff2*dnUdz-dnUdx(i  ,j,k2))+
     &                       cff3*(cff3*dnUdz-dnUdx(i-1,j,k2))+
     &                       cff4*(cff4*dnUdz-dnUdx(i  ,j,k1)))
                cff1=MIN(dZde_p(i,j  ,k1),0.0_r8)
                cff2=MIN(dZde_p(i,j+1,k2),0.0_r8)
                cff3=MAX(dZde_p(i,j  ,k2),0.0_r8)
                cff4=MAX(dZde_p(i,j+1,k1),0.0_r8)
                UFs(i,j,k2)=UFs(i,j,k2)+om_u(i,j)*
     &                      0.25_r8*(visc4_r(i-1,j)+visc4_r(i,j))*
     &                      (cff1*(cff1*dmUdz-dmUde(i,j  ,k1))+
     &                       cff2*(cff2*dmUdz-dmUde(i,j+1,k2))+
     &                       cff3*(cff3*dmUdz-dmUde(i,j  ,k2))+
     &                       cff4*(cff4*dmUdz-dmUde(i,j+1,k1)))
                cff1=MIN(dZde_p(i,j  ,k1),0.0_r8)
                cff2=MIN(dZde_p(i,j+1,k2),0.0_r8)
                cff3=MAX(dZde_p(i,j  ,k2),0.0_r8)
                cff4=MAX(dZde_p(i,j+1,k1),0.0_r8)
                cff5=MIN(dZdx_p(i,j  ,k1),0.0_r8)
                cff6=MIN(dZdx_p(i,j+1,k2),0.0_r8)
                cff7=MAX(dZdx_p(i,j  ,k2),0.0_r8)
                cff8=MAX(dZdx_p(i,j+1,k1),0.0_r8)
                UFs(i,j,k2)=UFs(i,j,k2)+on_u(i,j)*
     &                      0.25_r8*(visc4_r(i-1,j)+visc4_r(i,j))*
     &                      (cff1*(cff5*dnVdz-dnVdx(i,j  ,k1))+
     &                       cff2*(cff6*dnVdz-dnVdx(i,j+1,k2))+
     &                       cff3*(cff7*dnVdz-dnVdx(i,j  ,k2))+
     &                       cff4*(cff8*dnVdz-dnVdx(i,j+1,k1)))
                cff1=MIN(dZdx_r(i-1,j,k1),0.0_r8)
                cff2=MIN(dZdx_r(i  ,j,k2),0.0_r8)
                cff3=MAX(dZdx_r(i-1,j,k2),0.0_r8)
                cff4=MAX(dZdx_r(i  ,j,k1),0.0_r8)
                cff5=MIN(dZde_r(i-1,j,k1),0.0_r8)
                cff6=MIN(dZde_r(i  ,j,k2),0.0_r8)
                cff7=MAX(dZde_r(i-1,j,k2),0.0_r8)
                cff8=MAX(dZde_r(i  ,j,k1),0.0_r8)
                UFs(i,j,k2)=UFs(i,j,k2)-om_u(i,j)*
     &                      0.25_r8*(visc4_r(i-1,j)+visc4_r(i,j))*
     &                      (cff1*(cff5*dmVdz-dmVde(i-1,j,k1))+
     &                       cff2*(cff6*dmVdz-dmVde(i  ,j,k2))+
     &                       cff3*(cff7*dmVdz-dmVde(i-1,j,k2))+
     &                       cff4*(cff8*dmVdz-dmVde(i  ,j,k1)))
              enddo
            enddo
!
            do j=JstrV,Jend
              do i=Istr,Iend
                cff=0.5_r8*(pn(i,j-1)+pn(i,j))
                dnUdz=cff*0.25_r8*(dUdz(i,j  ,k2)+dUdz(i+1,j  ,k2)+
     &                             dUdz(i,j-1,k2)+dUdz(i+1,j-1,k2))
                dnVdz=cff*dVdz(i,j,k2)
                cff=0.5_r8*(pm(i,j-1)+pm(i,j))
                dmUdz=cff*0.25_r8*(dUdz(i,j  ,k2)+dUdz(i+1,j  ,k2)+
     &                             dUdz(i,j-1,k2)+dUdz(i+1,j-1,k2))
                dmVdz=cff*dVdz(i,j,k2)
                cff1=MIN(dZdx_p(i  ,j,k1),0.0_r8)
                cff2=MIN(dZdx_p(i+1,j,k2),0.0_r8)
                cff3=MAX(dZdx_p(i  ,j,k2),0.0_r8)
                cff4=MAX(dZdx_p(i+1,j,k1),0.0_r8)
                VFs(i,j,k2)=on_v(i,j)*
     &                      0.25_r8*(visc4_r(i,j-1)+visc4_r(i,j))*
     &                      (cff1*(cff1*dnVdz-dnVdx(i  ,j,k1))+
     &                       cff2*(cff2*dnVdz-dnVdx(i+1,j,k2))+
     &                       cff3*(cff3*dnVdz-dnVdx(i  ,j,k2))+
     &                       cff4*(cff4*dnVdz-dnVdx(i+1,j,k1)))
                cff1=MIN(dZde_r(i,j-1,k1),0.0_r8)
                cff2=MIN(dZde_r(i,j  ,k2),0.0_r8)
                cff3=MAX(dZde_r(i,j-1,k2),0.0_r8)
                cff4=MAX(dZde_r(i,j  ,k1),0.0_r8)
                VFs(i,j,k2)=VFs(i,j,k2)+om_v(i,j)*
     &                      0.25_r8*(visc4_r(i,j-1)+visc4_r(i,j))*
     &                      (cff1*(cff1*dmVdz-dmVde(i,j-1,k1))+
     &                       cff2*(cff2*dmVdz-dmVde(i,j  ,k2))+
     &                       cff3*(cff3*dmVdz-dmVde(i,j-1,k2))+
     &                       cff4*(cff4*dmVdz-dmVde(i,j  ,k1)))
                cff1=MIN(dZde_r(i,j-1,k1),0.0_r8)
                cff2=MIN(dZde_r(i,j  ,k2),0.0_r8)
                cff3=MAX(dZde_r(i,j-1,k2),0.0_r8)
                cff4=MAX(dZde_r(i,j  ,k1),0.0_r8)
                cff5=MIN(dZdx_r(i,j-1,k1),0.0_r8)
                cff6=MIN(dZdx_r(i,j  ,k2),0.0_r8)
                cff7=MAX(dZdx_r(i,j-1,k2),0.0_r8)
                cff8=MAX(dZdx_r(i,j  ,k1),0.0_r8)
                VFs(i,j,k2)=VFs(i,j,k2)-on_v(i,j)*
     &                      0.25_r8*(visc4_r(i,j-1)+visc4_r(i,j))*
     &                      (cff1*(cff5*dnUdz-dnUdx(i,j-1,k1))+
     &                       cff2*(cff6*dnUdz-dnUdx(i,j  ,k2))+
     &                       cff3*(cff7*dnUdz-dnUdx(i,j-1,k2))+
     &                       cff4*(cff8*dnUdz-dnUdx(i,j  ,k1)))
                cff1=MIN(dZdx_p(i  ,j,k1),0.0_r8)
                cff2=MIN(dZdx_p(i+1,j,k2),0.0_r8)
                cff3=MAX(dZdx_p(i  ,j,k2),0.0_r8)
                cff4=MAX(dZdx_p(i+1,j,k1),0.0_r8)
                cff5=MIN(dZde_p(i  ,j,k1),0.0_r8)
                cff6=MIN(dZde_p(i+1,j,k2),0.0_r8)
                cff7=MAX(dZde_p(i  ,j,k2),0.0_r8)
                cff8=MAX(dZde_p(i+1,j,k1),0.0_r8)
                VFs(i,j,k2)=VFs(i,j,k2)+om_v(i,j)*
     &                      0.25_r8*(visc4_r(i,j-1)+visc4_r(i,j))*
     &                      (cff1*(cff5*dmUdz-dmUde(i  ,j,k1))+
     &                       cff2*(cff6*dmUdz-dmUde(i+1,j,k2))+
     &                       cff3*(cff7*dmUdz-dmUde(i  ,j,k2))+
     &                       cff4*(cff8*dmUdz-dmUde(i+1,j,k1)))
              enddo
            enddo
          endif
!
! Time-step biharmonic, geopotential viscosity term. Notice that
! momentum at this stage is HzU and HzV and has m2/s units.
!
          cff=dt*0.125_r8
          do j=Jstr,Jend
            do i=IstrU,Iend
              u(i,j,k,nnew)=u(i,j,k,nnew)-
     &                      (cff*(pm(i-1,j)+pm(i,j))*
     &                           (pn(i-1,j)+pn(i,j))*
     &                           ((pn(i-1,j)+pn(i,j))*(UFx(i  ,j)-
     &                                                 UFx(i-1,j))+
     &                            (pm(i-1,j)+pm(i,j))*(UFe(i,j+1)-
     &                                                 UFe(i,j  )))+
     &                       dt*(UFs(i,j,k2)-UFs(i,j,k1)))
            enddo
          enddo
          do j=JstrV,Jend
            do i=Istr,Iend
              v(i,j,k,nnew)=v(i,j,k,nnew)-
     &                      (cff*(pm(i,j)+pm(i,j-1))*
     &                           (pn(i,j)+pn(i,j-1))*
     &                           ((pn(i,j-1)+pn(i,j))*(VFx(i+1,j)-
     &                                                 VFx(i  ,j))-
     &                            (pm(i,j-1)+pm(i,j))*(VFe(i,j  )-
     &                                                 VFe(i,j-1)))+
     &                       dt*(VFs(i,j,k2)-VFs(i,j,k1)))
            enddo
          enddo
        endif
      enddo
# undef IU_RANGE
# undef IV_RANGE
# undef JU_RANGE
# undef JV_RANGE
#else
      subroutine uv3dmix4_geo
#endif
      return
      end
