#include "cppdefs.h"
#undef DEBUG
      subroutine step2d (tile)
!
!======================================== Alexander F. Shchepetkin ===
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine performs a fast (predictor or corrector) time-step   !
!  for the free-surface  and 2D momentum equations.  The predictor   !
!  step is  Leap-Frog  whereas the  corrector step is trapezoidal,   !
!  Adams-Moulton.  If applicable, it also calculates time averages   !
!  over all  fast-time  steps to damp high frequency signal.         !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
#include "param.h"
#include "scratch.h"
#include "tile.h"
!
#ifdef PROFILE
      call wclock_on (9)
#endif
      call step2d_tile (Istr,Iend,Jstr,Jend) !,
c     &                  A2d(1, 1),A2d(1, 2),A2d(1, 3),A2d(1, 4),
c     &                  A2d(1, 5),A2d(1, 6),A2d(1, 7),A2d(1, 8),
c     &                  A2d(1, 9),A2d(1,10),A2d(1,11),A2d(1,12),
c     &                  A2d(1,13),A2d(1,14),A2d(1,15))
#ifdef PROFILE
      call wclock_off (9)
#endif
      return
      end
!
!*********************************************************************
      subroutine step2d_tile (Istr,Iend,Jstr,Jend) !,zeta_new,Dnew,
c     &                        rhs_zeta,rhs_ubar,rhs_vbar,DUon,DVom,
c     &                        Drhs,Drhs_p,UFe,UFx,VFe,VFx,grad,Dgrad)
!*********************************************************************
!
      implicit none
#include "param.h"
#include "clima.h"
#include "coupling.h"
#include "forces.h"
#include "grid.h"
#include "mask.h"
#include "mixing.h"
#include "ocean.h"
#include "scalars.h"
#include "sources.h"
!
      logical CORRECTOR_2D_STEP
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, ptsk
#ifdef UV_PSOURCE
      INTEGER_TYPE
     &        is
#endif
      REAL_TYPE
     &        cff, cff1, cff2, cff3, cff4, cff5, fac
      REAL_TYPE
     &           Dgrad(PRIVATE_2D_SCRATCH_ARRAY),
     &            Dnew(PRIVATE_2D_SCRATCH_ARRAY),
     &            Drhs(PRIVATE_2D_SCRATCH_ARRAY),
     &          Drhs_p(PRIVATE_2D_SCRATCH_ARRAY),
     &            DUon(PRIVATE_2D_SCRATCH_ARRAY),
     &            DVom(PRIVATE_2D_SCRATCH_ARRAY),
     &             UFe(PRIVATE_2D_SCRATCH_ARRAY),
     &             UFx(PRIVATE_2D_SCRATCH_ARRAY),
     &             VFe(PRIVATE_2D_SCRATCH_ARRAY),
     &             VFx(PRIVATE_2D_SCRATCH_ARRAY),
     &            grad(PRIVATE_2D_SCRATCH_ARRAY),
     &        rhs_ubar(PRIVATE_2D_SCRATCH_ARRAY),
     &        rhs_vbar(PRIVATE_2D_SCRATCH_ARRAY),
     &        rhs_zeta(PRIVATE_2D_SCRATCH_ARRAY),
     &        zeta_new(PRIVATE_2D_SCRATCH_ARRAY)
!
#include "set_bounds.h"
!
      ptsk=3-kstp
      CORRECTOR_2D_STEP=.not.PREDICTOR_2D_STEP
!
!---------------------------------------------------------------------
!  Compute total depth (m) and vertically integrated mass fluxes.
!---------------------------------------------------------------------
!
#ifdef EW_PERIODIC
# define I_RANGE IstrU-1,Iend+1
#else
# define I_RANGE MAX(2,IstrU-1),MIN(Iend+1,Lm)
#endif
#ifdef NS_PERIODIC
# define J_RANGE JstrV-1,Jend+1
#else
# define J_RANGE MAX(2,JstrV-1),MIN(Jend+1,Mm)
#endif
      do j=-2+J_RANGE+1
        do i=-2+I_RANGE+1
          Drhs(i,j)=zeta(i,j,krhs)+h(i,j)
        enddo
      enddo
      do j=-2+J_RANGE+1
        do i=-1+I_RANGE+1
          DUon(i,j)=0.5_r8*(Drhs(i,j)+Drhs(i-1,j))*ubar(i,j,krhs)*
     &              on_u(i,j)
        enddo
      enddo
      do j=-1+J_RANGE+1
        do i=-2+I_RANGE+1
          DVom(i,j)=0.5_r8*(Drhs(i,j)+Drhs(i,j-1))*vbar(i,j,krhs)*
     &              om_v(i,j)
        enddo
      enddo
#undef I_RANGE
#undef J_RANGE
#ifdef OBC_VOLCONS
!
! Set vertically integrated mass fluxes DUon and DVom along the open
! boundaries in such a way that the integral volume is conserved.
!
      call set_DUV_bc_tile (Istr,Iend,Jstr,Jend,Drhs,DUon,DVom)
#endif
#ifdef SOLVE3D
!
!---------------------------------------------------------------------
!  Compute time averaged fields over all short time-steps.
!---------------------------------------------------------------------
!
      if (PREDICTOR_2D_STEP) then
        if (FIRST_2D_STEP) then
!
!  Reset arrays for 2D fields averaged within the short time-steps.
!
          cff2=(-1.0_r8/12.0_r8)*weight(2,iif+1)
          do j=JstrR,JendR
            do i=IstrR,IendR
              Zt_avg1(i,j)=0.0_r8
            enddo
            do i=Istr,IendR
              DU_avg1(i,j)=0.0_r8
              DU_avg2(i,j)=cff2*DUon(i,j)
            enddo
          enddo
          do j=Jstr,JendR
            do i=IstrR,IendR
              DV_avg1(i,j)=0.0_r8
              DV_avg2(i,j)=cff2*DVom(i,j)
            enddo
          enddo
        else
!
!  Accumulate field averages of previous time-step after they are
!  computed in the previous corrector step, updated their boundaries,
!  and synchronized.
!
          cff1=weight(1,iif-1)
          cff2=(8.0_r8/12.0_r8)*weight(2,iif  )-
     &         (1.0_r8/12.0_r8)*weight(2,iif+1)
          do j=JstrR,JendR
            do i=IstrR,IendR
              Zt_avg1(i,j)=Zt_avg1(i,j)+cff1*zeta(i,j,krhs)
            enddo
            do i=Istr,IendR
              DU_avg1(i,j)=DU_avg1(i,j)+cff1*DUon(i,j)
              DU_avg2(i,j)=DU_avg2(i,j)+cff2*DUon(i,j)
            enddo
          enddo
          do j=Jstr,JendR
            do i=IstrR,IendR
              DV_avg1(i,j)=DV_avg1(i,j)+cff1*DVom(i,j)
              DV_avg2(i,j)=DV_avg2(i,j)+cff2*DVom(i,j)
            enddo
          enddo
        endif
      else
        if (FIRST_2D_STEP) then
          cff2=weight(2,iif)
        else
          cff2=(5.0_r8/12.0_r8)*weight(2,iif)
        endif
        do j=JstrR,JendR
          do i=Istr,IendR
            DU_avg2(i,j)=DU_avg2(i,j)+cff2*DUon(i,j)
          enddo
        enddo
        do j=Jstr,JendR
          do i=IstrR,IendR
            DV_avg2(i,j)=DV_avg2(i,j)+cff2*DVom(i,j)
          enddo
        enddo
      endif
!
!  After all fast time steps are completed, recompute S-coordinate
!  surfaces according to the new free surface field.
!
      if ((iif.eq.(nfast+1)).and.PREDICTOR_2D_STEP) then
        call set_depth_tile (Istr,Iend,Jstr,Jend)
      endif
#endif /* SOLVE3D */
!
!  Do not perform the actual time stepping during the auxiliary
!  (nfast+1) time step.
!
      if (iif.gt.nfast) return
!
!=====================================================================
!  Time step free-surface equation.
!=====================================================================
!
!  During the first time-step, the predictor step is Forward-Euler
!  and the corrector step is Backward-Euler. Otherwise, the predictor
!  step is Leap-frog and the corrector step is Adams-Moulton.
!
#define zwrk UFx
#define gzeta UFe
#define gzeta2 VFx
#define gzetaSA VFe
!
      if (FIRST_2D_STEP) then
        cff1=dtfast
        do j=JstrV-1,Jend
          do i=IstrU-1,Iend
            rhs_zeta(i,j)=DUon(i,j)-DUon(i+1,j)+
     &                    DVom(i,j)-DVom(i,j+1)
#ifdef ZCLM_NUDGING
     &                   +Znudgcof(i,j)*
     &                    (ssh(i,j)-zeta(i,j,krhs))*omn(i,j)
#endif
            zeta_new(i,j)=zeta(i,j,kstp)+
     &                    pm(i,j)*pn(i,j)*cff1*rhs_zeta(i,j)
#ifdef MASKING
            zeta_new(i,j)=zeta_new(i,j)*rmask(i,j)
#endif
            Dnew(i,j)=zeta_new(i,j)+h(i,j)
!
            zwrk(i,j)=0.5_r8*(zeta(i,j,kstp)+zeta_new(i,j))
#if defined VAR_RHO_2D && defined SOLVE3D
            cff=1000.0_r8/rho0
            gzeta(i,j)=(cff+rhoS(i,j))*zwrk(i,j)
            gzeta2(i,j)=gzeta(i,j)*zwrk(i,j)
            gzetaSA(i,j)=zwrk(i,j)*(rhoS(i,j)-rhoA(i,j))
#else
            gzeta(i,j)=zwrk(i,j)
            gzeta2(i,j)=zwrk(i,j)*zwrk(i,j)
#endif
          enddo
        enddo
      elseif (PREDICTOR_2D_STEP) then
        cff1=2.0_r8*dtfast
        cff4=4.0_r8/25.0_r8
        cff5=1.0_r8-2.0_r8*cff4
        do j=JstrV-1,Jend
          do i=IstrU-1,Iend
            rhs_zeta(i,j)=DUon(i,j)-DUon(i+1,j)+
     &                    DVom(i,j)-DVom(i,j+1)
#ifdef ZCLM_NUDGING
     &                   +Znudgcof(i,j)*
     &                    (ssh(i,j)-zeta(i,j,krhs))*omn(i,j)
#endif
            zeta_new(i,j)=zeta(i,j,kstp)+
     &                    pm(i,j)*pn(i,j)*cff1*rhs_zeta(i,j)
#ifdef MASKING
            zeta_new(i,j)=zeta_new(i,j)*rmask(i,j)
#endif
            Dnew(i,j)=zeta_new(i,j)+h(i,j)
!
            zwrk(i,j)=cff5*zeta(i,j,krhs)+
     &                cff4*(zeta(i,j,kstp)+zeta_new(i,j))
#if defined VAR_RHO_2D && defined SOLVE3D
            cff=1000.0_r8/rho0
            gzeta(i,j)=(cff+rhoS(i,j))*zwrk(i,j)
            gzeta2(i,j)=gzeta(i,j)*zwrk(i,j)
            gzetaSA(i,j)=zwrk(i,j)*(rhoS(i,j)-rhoA(i,j))
#else
            gzeta(i,j)=zwrk(i,j)
            gzeta2(i,j)=zwrk(i,j)*zwrk(i,j)
#endif
          enddo
        enddo
      elseif (CORRECTOR_2D_STEP) then
        cff1=dtfast*5.0_r8/12.0_r8
        cff2=dtfast*8.0_r8/12.0_r8
        cff3=dtfast*1.0_r8/12.0_r8
        cff4=2.0_r8/5.0_r8
        cff5=1.0_r8-cff4
        do j=JstrV-1,Jend
          do i=IstrU-1,Iend
            cff=cff1*(DUon(i,j)-DUon(i+1,j)+
     &                DVom(i,j)-DVom(i,j+1))
#ifdef ZCLM_NUDGING
     &         +Znudgcof(i,j)*
     &          (ssh(i,j)-zeta(i,j,krhs))*omn(i,j)
#endif
            zeta_new(i,j)=zeta(i,j,kstp)+
     &                    pm(i,j)*pn(i,j)*(cff+
     &                                     cff2*rzeta(i,j,kstp)-
     &                                     cff3*rzeta(i,j,ptsk))
#ifdef MASKING
            zeta_new(i,j)=zeta_new(i,j)*rmask(i,j)
#endif
            Dnew(i,j)=zeta_new(i,j)+h(i,j)
!
            zwrk(i,j)=cff5*zeta_new(i,j)+cff4*zeta(i,j,krhs)
#if defined VAR_RHO_2D && defined SOLVE3D
            cff=1000.0_r8/rho0
            gzeta(i,j)=(cff+rhoS(i,j))*zwrk(i,j)
            gzeta2(i,j)=gzeta(i,j)*zwrk(i,j)
            gzetaSA(i,j)=zwrk(i,j)*(rhoS(i,j)-rhoA(i,j))
#else
            gzeta(i,j)=zwrk(i,j)
            gzeta2(i,j)=zwrk(i,j)*zwrk(i,j)
#endif
          enddo
        enddo
      endif
!
!  Load new free-surface values into shared array at both predictor
!  and corrector steps.
!
      do j=Jstr,Jend
        do i=Istr,Iend
          zeta(i,j,knew)=zeta_new(i,j)
        enddo
      enddo
!
!  If predictor step, load right-side-term into shared array.
!
      if (PREDICTOR_2D_STEP) then
#ifdef EW_PERIODIC
# define I_RANGE MAX(IstrR,0),Iend
#else
# define I_RANGE Istr,Iend
#endif
#ifdef NS_PERIODIC
# define J_RANGE MAX(JstrR,0),Jend
#else
# define J_RANGE Jstr,Jend
#endif
        do j=J_RANGE
          do i=I_RANGE
            rzeta(i,j,krhs)=rhs_zeta(i,j)
          enddo
        enddo
#undef I_RANGE
#undef J_RANGE
#undef rhs_zeta
      endif
!
! Set free-surface lateral boundary conditions.
!
      call zetabc_tile (Istr,Iend,Jstr,Jend,grad)
#if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        zeta(START_2D_ARRAY,knew))
#endif
!
!=====================================================================
!  Compute right-hand-side for the 2D momentum equations.
!=====================================================================
!
!---------------------------------------------------------------------
! Compute pressure gradient terms
!---------------------------------------------------------------------
!
      cff1=0.5_r8*g
      cff2=1.0/3.0_r8
      do j=Jstr,Jend
        do i=IstrU,Iend
          rhs_ubar(i,j)=cff1*on_u(i,j)*
     &                  ((h(i-1,j)+h(i,j))*
     &                   (gzeta(i-1,j)-gzeta(i,j))+
#if defined VAR_RHO_2D && defined SOLVE3D
     &                   (h(i-1,j)-h(i,j))*
     &                   (gzetaSA(i-1,j)+gzetaSA(i,j)+
     &                    cff2*(rhoA(i-1,j)-rhoA(i,j))*
     &                         (zwrk(i-1,j)-zwrk(i,j)))+
#endif
     &                   (gzeta2(i-1,j)-gzeta2(i,j)))
        enddo
      enddo
      do j=JstrV,Jend
        do i=Istr,Iend
          rhs_vbar(i,j)=cff1*om_v(i,j)*
     &                  ((h(i,j-1)+h(i,j))*
     &                   (gzeta(i,j-1)-gzeta(i,j))+
#if defined VAR_RHO_2D && defined SOLVE3D
     &                   (h(i,j)-h(i,j-1))*
     &                   (gzetaSA(i,j-1)+gzetaSA(i,j)+
     &                    cff2*(rhoA(i,j-1)-rhoA(i,j))*
     &                         (zwrk(i,j-1)-zwrk(i,j)))+
#endif
     &                   (gzeta2(i,j-1)-gzeta2(i,j)))
        enddo
      enddo
#undef zwrk
#undef gzeta
#undef gzeta2
#undef gzetaSA

#ifdef SOLVE3D
# define UV_C2ADVECTION
# undef UV_VIS2
# undef UV_VIS4
#endif


#ifdef UV_ADV
!
!---------------------------------------------------------------------
!  Add in horizontal advection of momentum.
!---------------------------------------------------------------------
!

# if defined UV_C2ADVECTION
!
!  Second-order, centered differences advection.
!
      do j=Jstr,Jend
        do i=IstrU-1,Iend
          UFx(i,j)=0.25_r8*(DUon(i,j)+DUon(i+1,j))*
     &                     (ubar(i,j,krhs)+ubar(i+1,j,krhs))
        enddo
      enddo
      do j=Jstr,Jend+1
        do i=IstrU,Iend
          UFe(i,j)=0.25_r8*(DVom(i,j)+DVom(i-1,j))*
     &                     (ubar(i,j,krhs)+ubar(i,j-1,krhs))
        enddo
      enddo
      do j=JstrV,Jend
        do i=Istr,Iend+1
          VFx(i,j)=0.25_r8*(DUon(i,j)+DUon(i,j-1))*
     &                     (vbar(i,j,krhs)+vbar(i-1,j,krhs))
        enddo
      enddo
      do j=JstrV-1,Jend
        do i=Istr,Iend
          VFe(i,j)=0.25_r8*(DVom(i,j)+DVom(i,j+1))*
     &                     (vbar(i,j,krhs)+vbar(i,j+1,krhs))
        enddo
      enddo
# else
!
!  Fourth-order, centered differences advection.
!
#  ifdef EW_PERIODIC
#   define IU_RANGE IstrU-1,Iend+1
#  else
#   define IU_RANGE MAX(IstrU-1,2),MIN(Iend+1,Lm)
#  endif
      do j=Jstr,Jend
        do i=IU_RANGE
          grad (i,j)=ubar(i-1,j,krhs)-2.0_r8*ubar(i,j,krhs)+
     &               ubar(i+1,j,krhs)
          Dgrad(i,j)=DUon(i-1,j)-2.0_r8*DUon(i,j)+DUon(i+1,j)
        enddo
      enddo
#  undef IU_RANGE
#  ifndef EW_PERIODIC
      if (WESTERN_EDGE) then
        do j=Jstr,Jend
          grad (Istr,j)=grad (Istr+1,j)
          Dgrad(Istr,j)=Dgrad(Istr+1,j)
        enddo
      endif
      if (EASTERN_EDGE) then
        do j=Jstr,Jend
          grad (Iend+1,j)=grad (Iend,j)
          Dgrad(Iend+1,j)=Dgrad(Iend,j)
        enddo
      endif
#  endif /* !EW_PERIODIC */
      cff=1.0_r8/6.0_r8
      do j=Jstr,Jend
        do i=IstrU-1,Iend
          UFx(i,j)=0.25_r8*(ubar(i,j,krhs)+ubar(i+1,j,krhs)-
     &                      cff*(grad (i,j)+grad (i+1,j)))*
     &                     (DUon(i,j)+DUon(i+1,j)-
     &                      cff*(Dgrad(i,j)+Dgrad(i+1,j)))
        enddo
      enddo
#  ifdef NS_PERIODIC
#   define JU_RANGE Jstr-1,Jend+1
#  else
#   define JU_RANGE MAX(Jstr-1,1),MIN(Jend+1,Mm)
#  endif
      do j=JU_RANGE
        do i=IstrU,Iend
          grad(i,j)=ubar(i,j-1,krhs)-2.0_r8*ubar(i,j,krhs)+
     &              ubar(i,j+1,krhs)
        enddo
      enddo
#  undef JU_RANGE
#  ifndef NS_PERIODIC
      if (SOUTHERN_EDGE) then
        do i=IstrU,Iend
          grad(i,Jstr-1)=grad(i,Jstr)
        enddo
      endif
      if (NORTHERN_EDGE) then
        do i=IstrU,Iend
          grad(i,Jend+1)=grad(i,Jend)
        enddo
      endif
#  endif
      do j=Jstr,Jend+1
        do i=IstrU-1,Iend
          Dgrad(i,j)=DVom(i-1,j)-2.0_r8*DVom(i,j)+DVom(i+1,j)
        enddo
      enddo
      cff=1.0_r8/6.0_r8
      do j=Jstr,Jend+1
        do i=IstrU,Iend
          UFe(i,j)=0.25_r8*(ubar(i,j,krhs)+ubar(i,j-1,krhs)-
     &                      cff*(grad (i,j)+grad (i,j-1)))*
     &                     (DVom(i,j)+DVom(i-1,j)-
     &                      cff*(Dgrad(i,j)+Dgrad(i-1,j)))
        enddo
      enddo
#  ifdef EW_PERIODIC
#   define IV_RANGE Istr-1,Iend+1
#  else
#   define IV_RANGE MAX(Istr-1,1),MIN(Iend+1,Lm)
#  endif
      do j=JstrV,Jend
        do i=IV_RANGE
          grad(i,j)=vbar(i-1,j,krhs)-2.0_r8*vbar(i,j,krhs)+
     &              vbar(i+1,j,krhs)
        enddo
      enddo
#  undef IV_RANGE
#  ifndef EW_PERIODIC
      if (WESTERN_EDGE) then
        do j=JstrV,Jend
          grad(Istr-1,j)=grad(Istr,j)
        enddo
      endif
      if (EASTERN_EDGE) then
        do j=JstrV,Jend
          grad(Iend+1,j)=grad(Iend,j)
        enddo
      endif
#  endif
      do j=JstrV-1,Jend
        do i=Istr,Iend+1
          Dgrad(i,j)=DUon(i,j-1)-2.0_r8*DUon(i,j)+DUon(i,j+1)
        enddo
      enddo
      cff=1.0_r8/6.0_r8
      do j=JstrV,Jend
        do i=Istr,Iend+1
          VFx(i,j)=0.25_r8*(vbar(i,j,krhs)+vbar(i-1,j,krhs)-
     &                      cff*(grad (i,j)+grad (i-1,j)))*
     &                     (DUon(i,j)+DUon(i,j-1)-
     &                      cff*(Dgrad(i,j)+Dgrad(i,j-1)))
        enddo
      enddo
#  ifdef NS_PERIODIC
#   define JV_RANGE JstrV-1,Jend+1
#  else
#   define JV_RANGE MAX(JstrV-1,2),MIN(Jend+1,Mm)
#  endif
      do j=JV_RANGE
        do i=Istr,Iend
          grad(i,j)=vbar(i,j-1,krhs)-2.0_r8*vbar(i,j,krhs)+
     &              vbar(i,j+1,krhs)
          Dgrad(i,j)=DVom(i,j-1)-2.0_r8*DVom(i,j)+DVom(i,j+1)
        enddo
      enddo
#  undef JV_RANGE
#  ifndef NS_PERIODIC
      if (SOUTHERN_EDGE) then
        do i=Istr,Iend
          grad (i,Jstr)=grad (i,Jstr+1)
          Dgrad(i,Jstr)=Dgrad(i,Jstr+1)
        enddo
      endif
      if (NORTHERN_EDGE) then
        do i=Istr,Iend
          grad (i,Jend+1)=grad (i,Jend)
          Dgrad(i,Jend+1)=Dgrad(i,Jend)
        enddo
      endif
#  endif
      cff=1.0_r8/6.0_r8
      do j=JstrV-1,Jend
        do i=Istr,Iend
          VFe(i,j)=0.25_r8*(vbar(i,j,krhs)+vbar(i,j+1,krhs)-
     &                      cff*(grad (i,j)+grad (i,j+1)))*
     &                     (DVom(i,j)+DVom(i,j+1)-
     &                      cff*(Dgrad(i,j)+Dgrad(i,j+1)))
        enddo
      enddo
# endif
      do j=Jstr,Jend
        do i=IstrU,Iend
          rhs_ubar(i,j)=rhs_ubar(i,j)-
     &                  (UFx(i,j)-UFx(i-1,j))-(UFe(i,j+1)-UFe(i,j))
        enddo
      enddo
      do j=JstrV,Jend
        do i=Istr,Iend
          rhs_vbar(i,j)=rhs_vbar(i,j)-
     &                  (VFx(i+1,j)-VFx(i,j))-(VFe(i,j)-VFe(i,j-1))
        enddo
      enddo
#endif /* UV_ADV */
#if defined UV_COR || (defined CURVGRID && defined UV_ADV)
!
!---------------------------------------------------------------------
!  Add in Coriolis and curvilinear transformation terms, if any.
!---------------------------------------------------------------------
!
      do j=JstrV-1,Jend
        do i=IstrU-1,Iend
          cff=0.5_r8*Drhs(i,j)*(
# ifdef UV_COR
     &        fomn(i,j)
# endif
# if (defined CURVGRID && defined UV_ADV)
     &       +0.5_r8*((vbar(i,j,krhs)+vbar(i  ,j+1,krhs))*dndx(i,j)-
     &                (ubar(i,j,krhs)+ubar(i+1,j  ,krhs))*dmde(i,j))
# endif
     &        )
          UFx(i,j)=cff*(vbar(i,j,krhs)+vbar(i  ,j+1,krhs))
          VFe(i,j)=cff*(ubar(i,j,krhs)+ubar(i+1,j  ,krhs))
        enddo
      enddo
      do j=Jstr,Jend
        do i=IstrU,Iend
          rhs_ubar(i,j)=rhs_ubar(i,j)+0.5_r8*(UFx(i,j)+UFx(i-1,j))
        enddo
      enddo
      do j=JstrV,Jend
        do i=Istr,Iend
          rhs_vbar(i,j)=rhs_vbar(i,j)-0.5_r8*(VFe(i,j)+VFe(i,j-1))
        enddo
      enddo
#endif /* UV_COR || (CURVGRID && UV_ADV) */
#if defined UV_VIS2 || defined UV_VIS4
!
!---------------------------------------------------------------------
!  If horizontal mixing, compute total depth at PSI-points
!---------------------------------------------------------------------
!
# ifdef UV_VIS4
#  ifdef EW_PERIODIC
#   define IV_RANGE Istr-1,Iend+1
#   define IU_RANGE Istr-1,Iend+1
#  else
#   define IV_RANGE MAX(1,Istr-1),MIN(Iend+1,Lm)
#   define IU_RANGE MAX(2,IstrU-1),MIN(Iend+1,Lm)
#  endif
#  ifdef NS_PERIODIC
#   define JU_RANGE Jstr-1,Jend+1
#   define JV_RANGE Jstr-1,Jend+1
#  else
#   define JU_RANGE MAX(1,Jstr-1),MIN(Jend+1,Mm)
#   define JV_RANGE MAX(2,JstrV-1),MIN(Jend+1,Mm)
#  endif
!
      do j=JU_RANGE+1
        do i=IV_RANGE+1
# else
      do j=Jstr,Jend+1
        do i=Istr,Iend+1
# endif
          Drhs_p(i,j)=0.25_r8*(Drhs(i,j  )+Drhs(i-1,j  )+
     &                         Drhs(i,j-1)+Drhs(i-1,j-1))
        enddo
      enddo
#endif /* UV_VIS2 || UV_VIS4 */
#ifdef UV_VIS2
!
!---------------------------------------------------------------------
!  Add in horizontal harmonic viscosity.
!---------------------------------------------------------------------
!
!  Compute flux-components of the horizontal divergence of the stress
!  tensor (m5/s2) in XI- and ETA-directions.
!
      do j=JstrV-1,Jend
        do i=IstrU-1,Iend
          cff=visc2_r(i,j)*Drhs(i,j)*0.5_r8*
     &        (pmon_r(i,j)*
     &         ((pn(i  ,j)+pn(i+1,j))*ubar(i+1,j,krhs)-
     &          (pn(i-1,j)+pn(i  ,j))*ubar(i  ,j,krhs))-
     &         pnom_r(i,j)*
     &         ((pm(i,j  )+pm(i,j+1))*vbar(i,j+1,krhs)-
     &          (pm(i,j-1)+pm(i,j  ))*vbar(i,j  ,krhs)))
          UFx(i,j)=on_r(i,j)*on_r(i,j)*cff
          VFe(i,j)=om_r(i,j)*om_r(i,j)*cff
        enddo
      enddo
      do j=Jstr,Jend+1
        do i=Istr,Iend+1
          cff=visc2_p(i,j)*Drhs_p(i,j)*0.5_r8*
     &        (pmon_p(i,j)*
     &         ((pn(i  ,j-1)+pn(i  ,j))*vbar(i  ,j,krhs)-
     &          (pn(i-1,j-1)+pn(i-1,j))*vbar(i-1,j,krhs))+
     &         pnom_p(i,j)*
     &         ((pm(i-1,j  )+pm(i,j  ))*ubar(i,j  ,krhs)-
     &          (pm(i-1,j-1)+pm(i,j-1))*ubar(i,j-1,krhs)))
#  ifdef MASKING
     &       *pmask(i,j)
#  endif /* MASKING */
          UFe(i,j)=om_p(i,j)*om_p(i,j)*cff
          VFx(i,j)=on_p(i,j)*on_p(i,j)*cff
        enddo
      enddo
!
!  Add in harmonic viscosity.
!
      do j=Jstr,Jend
        do i=IstrU,Iend
          rhs_ubar(i,j)=rhs_ubar(i,j)+0.5_r8*
     &                  ((pn(i-1,j)+pn(i,j))*(UFx(i,j  )-UFx(i-1,j))+
     &                   (pm(i-1,j)+pm(i,j))*(UFe(i,j+1)-UFe(i  ,j)))
        enddo
      enddo
      do j=JstrV,Jend
        do i=Istr,Iend
          rhs_vbar(i,j)=rhs_vbar(i,j)+0.5_r8*
     &                  ((pn(i,j-1)+pn(i,j))*(VFx(i+1,j)-VFx(i,j  ))-
     &                   (pm(i,j-1)+pm(i,j))*(VFe(i  ,j)-VFe(i,j-1)))
        enddo
      enddo
#endif /* UV_VIS2 */
#ifdef UV_VIS4
!
!---------------------------------------------------------------------
!  Add in horizontal biharmonic viscosity. The biharmonic operator
!  is computed by applying the harmonic operator twice.
!---------------------------------------------------------------------
!
!  Compute flux-components of the horizontal divergence of the stress
!  tensor (m4 s^-3/2) in XI- and ETA-directions.  It is assumed here
!  that "visc4_r" and "visc4_p" are the squared root of the biharmonic
!  viscosity coefficient.  For momentum balance purposes, the total
!  thickness "D" appears only when computing the second harmonic
!  operator.
!
      do j=-1+JV_RANGE
        do i=-1+IU_RANGE
          cff=visc4_r(i,j)*0.5_r8*
     &        (pmon_r(i,j)*
     &         ((pn(i  ,j)+pn(i+1,j))*ubar(i+1,j,krhs)-
     &          (pn(i-1,j)+pn(i  ,j))*ubar(i  ,j,krhs))-
     &         pnom_r(i,j)*
     &         ((pm(i,j  )+pm(i,j+1))*vbar(i,j+1,krhs)-
     &          (pm(i,j-1)+pm(i,j  ))*vbar(i,j  ,krhs)))
          UFx(i,j)=on_r(i,j)*on_r(i,j)*cff
          VFe(i,j)=om_r(i,j)*om_r(i,j)*cff
        enddo
      enddo
      do j=JU_RANGE+1
        do i=IV_RANGE+1
          cff=visc4_p(i,j)*0.5_r8*
     &        (pmon_p(i,j)*
     &         ((pn(i  ,j-1)+pn(i  ,j))*vbar(i  ,j,krhs)-
     &          (pn(i-1,j-1)+pn(i-1,j))*vbar(i-1,j,krhs))+
     &         pnom_p(i,j)*
     &         ((pm(i-1,j  )+pm(i,j  ))*ubar(i,j  ,krhs)-
     &          (pm(i-1,j-1)+pm(i,j-1))*ubar(i,j-1,krhs)))
#  ifdef MASKING
     &       *pmask(i,j)
#  endif /* MASKING */
          UFe(i,j)=om_p(i,j)*om_p(i,j)*cff
          VFx(i,j)=on_p(i,j)*on_p(i,j)*cff
        enddo
      enddo
!
!  Compute first harmonic operator (m s^-3/2).
!
# define LapU DUon
# define LapV DVom
!
      do j=JU_RANGE
        do i=IU_RANGE
          LapU(i,j)=0.125_r8*
     &              (pm(i-1,j)+pm(i,j))*(pn(i-1,j)+pn(i,j))*
     &              ((pn(i-1,j)+pn(i,j))*(UFx(i,j  )-UFx(i-1,j))+
     &               (pm(i-1,j)+pm(i,j))*(UFe(i,j+1)-UFe(i  ,j)))
        enddo
      enddo
      do j=JV_RANGE
        do i=IV_RANGE
          LapV(i,j)=0.125_r8*
     &              (pm(i,j)+pm(i,j-1))*(pn(i,j)+pn(i,j-1))*
     &              ((pn(i,j-1)+pn(i,j))*(VFx(i+1,j)-VFx(i,j  ))-
     &               (pm(i,j-1)+pm(i,j))*(VFe(i  ,j)-VFe(i,j-1)))
        enddo
      enddo
!
!  Apply boundary conditions (other than periodic) to the first
!  harmonic operator. These are gradient or closed (free slip or
!  no slip) boundary conditions.
!
# ifndef EW_PERIODIC
      if (WESTERN_EDGE) then
        do j=JU_RANGE
#  ifdef WESTERN_WALL
          LapU(IstrU-1,j)=0.0_r8
#  else
          LapU(IstrU-1,j)=LapU(IstrU,j)
#  endif
        enddo
        do j=JV_RANGE
#  ifdef WESTERN_WALL
          LapV(Istr-1,j)=gamma2*LapV(Istr,j)
#  else
          LapV(Istr-1,j)=0.0_r8
#  endif
        enddo
      endif
      if (EASTERN_EDGE) then
        do j=JU_RANGE
#  ifdef EASTERN_WALL
          LapU(Iend+1,j)=0.0_r8
#  else
          LapU(Iend+1,j)=LapU(Iend,j)
#  endif
        enddo
        do j=JV_RANGE
#  ifdef EASTERN_WALL
          LapV(Iend+1,j)=gamma2*LapV(Iend,j)
#  else
          LapV(Iend+1,j)=0.0_r8
#  endif
        enddo
      endif
# endif /* !EW_PERIODIC */
# ifndef NS_PERIODIC
      if (SOUTHERN_EDGE) then
        do i=IU_RANGE
#  ifdef SOUTHERN_WALL
          LapU(i,Jstr-1)=gamma2*LapU(i,Jstr)
#  else
          LapU(i,Jstr-1)=0.0_r8
#  endif
        enddo
        do i=IV_RANGE
#  ifdef SOUTHERN_WALL
          LapV(i,JstrV-1)=0.0_r8
#  else
          LapV(i,JstrV-1)=LapV(i,JstrV)
#  endif
        enddo
      endif
      if (NORTHERN_EDGE) then
        do i=IU_RANGE
#  ifdef NORTHERN_WALL
          LapU(i,Jend+1)=gamma2*LapU(i,Jend)
#  else
          LapU(i,Jend+1)=0.0_r8
#  endif
        enddo
        do i=IV_RANGE
#  ifdef NORTHERN_WALL
          LapV(i,Jend+1)=0.0_r8
#  else
          LapV(i,Jend+1)=LapV(i,Jend)
#  endif
        enddo
      endif
# endif /* !NS_PERIODIC */
# if !defined EW_PERIODIC && !defined NS_PERIODIC
      if (SOUTHERN_EDGE .and. WESTERN_EDGE) then
        LapU(1,0)=0.5_r8*(LapU(2,0)+LapU(1,1))
        LapV(0,1)=0.5_r8*(LapV(0,2)+LapV(1,1))
      endif
      if (SOUTHERN_EDGE .and. EASTERN_EDGE) then
        LapU(L,0)=0.5_r8*(LapU(Lm,0)+LapU(L,1))
        LapV(L,1)=0.5_r8*(LapV(Lm,1)+LapV(L,2))
      endif
      if (NORTHERN_EDGE .and. WESTERN_EDGE) then
        LapU(1,M)=0.5_r8*(LapU(2,M)+LapU(1,Mm))
        LapV(0,M)=0.5_r8*(LapV(1,M)+LapV(0,Mm))
      endif
      if (NORTHERN_EDGE .and. EASTERN_EDGE) then
        LapU(L,M)=0.5_r8*(LapU(Lm,M)+LapU(L,Mm))
        LapV(L,M)=0.5_r8*(LapV(Lm,M)+LapV(L,Mm))
      endif
# endif
# undef IU_RANGE
# undef IV_RANGE
# undef JU_RANGE
# undef JV_RANGE
!
!  Compute flux-components of the horizontal divergence of the
!  biharmonic stress tensor (m4/s2) in XI- and ETA-directions.
!
      do j=JstrV-1,Jend
        do i=IstrU-1,Iend
          cff=visc4_r(i,j)*Drhs(i,j)*0.5_r8*
     &        (pmon_r(i,j)*
     &         ((pn(i  ,j)+pn(i+1,j))*LapU(i+1,j)-
     &          (pn(i-1,j)+pn(i  ,j))*LapU(i  ,j))-
     &         pnom_r(i,j)*
     &         ((pm(i,j  )+pm(i,j+1))*LapV(i,j+1)-
     &          (pm(i,j-1)+pm(i,j  ))*LapV(i,j  )))
          UFx(i,j)=on_r(i,j)*on_r(i,j)*cff
          VFe(i,j)=om_r(i,j)*om_r(i,j)*cff
        enddo
      enddo
      do j=Jstr,Jend+1
        do i=Istr,Iend+1
          cff=visc4_p(i,j)*Drhs_p(i,j)*0.5_r8*
     &        (pmon_p(i,j)*
     &         ((pn(i  ,j-1)+pn(i  ,j))*LapV(i  ,j)-
     &          (pn(i-1,j-1)+pn(i-1,j))*LapV(i-1,j))+
     &         pnom_p(i,j)*
     &         ((pm(i-1,j  )+pm(i,j  ))*LapU(i,j  )-
     &          (pm(i-1,j-1)+pm(i,j-1))*LapU(i,j-1)))
#  ifdef MASKING
     &       *pmask(i,j)
#  endif /* MASKING */
          UFe(i,j)=om_p(i,j)*om_p(i,j)*cff
          VFx(i,j)=on_p(i,j)*on_p(i,j)*cff
        enddo
      enddo
# undef LapV
# undef LapU
!
!  Add in biharmonic viscosity.
!
      do j=Jstr,Jend
        do i=IstrU,Iend
          rhs_ubar(i,j)=rhs_ubar(i,j)-0.5_r8*
     &                  ((pn(i-1,j)+pn(i,j))*(UFx(i,j  )-UFx(i-1,j))+
     &                   (pm(i-1,j)+pm(i,j))*(UFe(i,j+1)-UFe(i  ,j)))
        enddo
      enddo
      do j=JstrV,Jend
        do i=Istr,Iend
          rhs_vbar(i,j)=rhs_vbar(i,j)-0.5_r8*
     &                  ((pn(i,j-1)+pn(i,j))*(VFx(i+1,j)-VFx(i,j  ))-
     &                   (pm(i,j-1)+pm(i,j))*(VFe(i  ,j)-VFe(i,j-1)))
        enddo
      enddo
#endif /* UV_VIS4 */
!
!---------------------------------------------------------------------
!  Add in linear and/or quadratic bottom stress.
!---------------------------------------------------------------------
!
      if (rdrg2.gt.0.0_r8) then
        do j=Jstr,Jend
          do i=IstrU,Iend
            cff1=0.25_r8*(vbar(i  ,j,krhs)+vbar(i  ,j+1,krhs)+
     &                    vbar(i-1,j,krhs)+vbar(i-1,j+1,krhs))
            cff=SQRT(ubar(i,j,krhs)*ubar(i,j,krhs)+cff1*cff1)
            rhs_ubar(i,j)=rhs_ubar(i,j)-
     &                    ubar(i,j,krhs)*(rdrg+rdrg2*cff)*
     &                    om_u(i,j)*on_u(i,j)
          enddo
        enddo
        do j=JstrV,Jend
          do i=Istr,Iend
            cff1=0.25_r8*(ubar(i,j  ,krhs)+ubar(i+1,j  ,krhs)+
     &                    ubar(i,j-1,krhs)+ubar(i+1,j-1,krhs))
            cff=SQRT(cff1*cff1+vbar(i,j,krhs)*vbar(i,j,krhs))
            rhs_vbar(i,j)=rhs_vbar(i,j)-
     &                    vbar(i,j,krhs)*(rdrg+rdrg2*cff)*
     &                    om_v(i,j)*on_v(i,j)
          enddo
        enddo
      elseif (rdrg.gt.0.0_r8) then
        do j=Jstr,Jend
          do i=IstrU,Iend
            cff=rdrg*om_u(i,j)*on_u(i,j)
            rhs_ubar(i,j)=rhs_ubar(i,j)-
     &                    cff*ubar(i,j,krhs)
          enddo
        enddo
        do j=JstrV,Jend
          do i=Istr,Iend
            cff=rdrg*om_v(i,j)*on_v(i,j)
            rhs_vbar(i,j)=rhs_vbar(i,j)-
     &                    cff*vbar(i,j,krhs)
          enddo
        enddo
      endif
#ifdef M2CLM_NUDGING
!
!---------------------------------------------------------------------
!  Add in nudging of 2D momentum climatology.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=IstrU,Iend
          rhs_ubar(i,j)=rhs_ubar(i,j)+
     &                  0.25_r8*(M2nudgcof(i-1,j)+M2nudgcof(i,j))*
     &                  (Drhs(i-1,j)+Drhs(i,j))*
     &                  (ubarclm(i,j)-ubar(i,j,krhs))*
     &                  om_u(i,j)*on_u(i,j)
        enddo
      enddo
      do j=JstrV,Jend
        do i=Istr,Iend
          rhs_vbar(i,j)=rhs_vbar(i,j)+
     &                  0.25_r8*(M2nudgcof(i,j-1)+M2nudgcof(i,j))*
     &                  (Drhs(i,j-1)+Drhs(i,j))*
     &                  (vbarclm(i,j)-vbar(i,j,krhs))*
     &                  om_v(i,j)*on_v(i,j)
        enddo
      enddo
#endif
#ifdef SOLVE3D
!
!---------------------------------------------------------------------
!  Coupling between 2D and 3D equations.
!---------------------------------------------------------------------
!
!  Before the predictor step of the first barotropic time-step,
!  arrays "rufrc" and "rvfrc" contain the vertical integrals of
!  the 3D right-hand-side terms for momentum equations (including
!  surface and bottom stresses, if so prescribed).
!
!  Convert them into forcing terms by subtracting the fast time
!  "rhs_ubar" and "rhs_vbar" from them; Also, immediately apply
!  these forcing terms "rhs_ubar" and "rhs_vbar".
!
!  From now on, these newly computed forcing terms will remain
!  constant during the fast time stepping and will added to
!  "rhs_ubar" and "rhs_vbar" during all subsequent time steps.
!
      if (FIRST_2D_STEP .and. PREDICTOR_2D_STEP) then
        if (iic.eq.ntfirst) then
          do j=Jstr,Jend
            do i=IstrU,Iend
              rufrc(i,j)=rufrc(i,j)-rhs_ubar(i,j)
              rhs_ubar(i,j)=rhs_ubar(i,j)+rufrc(i,j)
              ru(i,j,0,nstp)=rufrc(i,j)
            enddo
          enddo
          do j=JstrV,Jend
            do i=Istr,Iend
              rvfrc(i,j)=rvfrc(i,j)-rhs_vbar(i,j)
              rhs_vbar(i,j)=rhs_vbar(i,j)+rvfrc(i,j)
              rv(i,j,0,nstp)=rvfrc(i,j)
            enddo
          enddo
        elseif (iic.eq.(ntfirst+1)) then
          do j=Jstr,Jend
            do i=IstrU,Iend
              rufrc(i,j)=rufrc(i,j)-rhs_ubar(i,j)
              rhs_ubar(i,j)=rhs_ubar(i,j)+
     &                      1.5_r8*rufrc(i,j)-0.5_r8*ru(i,j,0,nnew)
              ru(i,j,0,nstp)=rufrc(i,j)
            enddo
          enddo
          do j=JstrV,Jend
            do i=Istr,Iend
              rvfrc(i,j)=rvfrc(i,j)-rhs_vbar(i,j)
              rhs_vbar(i,j)=rhs_vbar(i,j)+
     &                      1.5_r8*rvfrc(i,j)-0.5_r8*rv(i,j,0,nnew)
              rv(i,j,0,nstp)=rvfrc(i,j)
            enddo
          enddo
        else
          cff1=23.0_r8/12.0_r8
          cff2=16.0_r8/12.0_r8
          cff3= 5.0_r8/12.0_r8
          do j=Jstr,Jend
            do i=IstrU,Iend
              rufrc(i,j)=rufrc(i,j)-rhs_ubar(i,j)
              rhs_ubar(i,j)=rhs_ubar(i,j)+
     &                      cff1*rufrc(i,j)-
     &                      cff2*ru(i,j,0,nnew)+
     &                      cff3*ru(i,j,0,nstp)
              ru(i,j,0,nstp)=rufrc(i,j)
            enddo
          enddo
          do j=JstrV,Jend
            do i=Istr,Iend
              rvfrc(i,j)=rvfrc(i,j)-rhs_vbar(i,j)
              rhs_vbar(i,j)=rhs_vbar(i,j)+
     &                      cff1*rvfrc(i,j)-
     &                      cff2*rv(i,j,0,nnew)+
     &                      cff3*rv(i,j,0,nstp)
              rv(i,j,0,nstp)=rvfrc(i,j)
            enddo
          enddo
        endif
      else
        do j=Jstr,Jend
          do i=IstrU,Iend
            rhs_ubar(i,j)=rhs_ubar(i,j)+rufrc(i,j)
          enddo
        enddo
        do j=JstrV,Jend
          do i=Istr,Iend
            rhs_vbar(i,j)=rhs_vbar(i,j)+rvfrc(i,j)
          enddo
        enddo
      endif
#else
!
!---------------------------------------------------------------------
!  Add in surface momentum stress.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=IstrU,Iend
          rhs_ubar(i,j)=rhs_ubar(i,j)+
     &                  sustr(i,j)*om_u(i,j)*on_u(i,j)
        enddo
      enddo
      do j=JstrV,Jend
        do i=Istr,Iend
          rhs_vbar(i,j)=rhs_vbar(i,j)+
     &                  svstr(i,j)*om_v(i,j)*on_v(i,j)
        enddo
      enddo
#endif  /* SOLVE3D */
!
!=====================================================================
!  Time step 2D momentum equations.
!=====================================================================
!
!  Compute total water column depth.
!
#define Dstp DUon
      do j=JstrV-1,Jend
        do i=IstrU-1,Iend
          Dstp(i,j)=zeta(i,j,kstp)+h(i,j)
        enddo
      enddo
!
!  During the first time-step, the predictor step is Forward-Euler
!  and the corrector step is Backward-Euler. Otherwise, the predictor
!  step is Leap-frog and the corrector step is Adams-Moulton.
!
      if (FIRST_2D_STEP) then
        cff1=0.5_r8*dtfast
        do j=Jstr,Jend
          do i=IstrU,Iend
            cff=(pm(i,j)+pm(i-1,j))*(pn(i,j)+pn(i-1,j))
            fac=1.0_r8/(Dnew(i,j)+Dnew(i-1,j))
            ubar(i,j,knew)=(ubar(i,j,kstp)*(Dstp(i,j)+Dstp(i-1,j))+
     &                      cff*cff1*rhs_ubar(i,j))*fac
#ifdef MASKING
     &                    *umask(i,j)
#endif
          enddo
        enddo
        do j=JstrV,Jend
          do i=Istr,Iend
            cff=(pm(i,j)+pm(i,j-1))*(pn(i,j)+pn(i,j-1))
            fac=1.0_r8/(Dnew(i,j)+Dnew(i,j-1))
            vbar(i,j,knew)=(vbar(i,j,kstp)*(Dstp(i,j)+Dstp(i,j-1))+
     &                      cff*cff1*rhs_vbar(i,j))*fac
#ifdef MASKING
     &                    *vmask(i,j)
#endif
          enddo
        enddo
      elseif (PREDICTOR_2D_STEP) then
        cff1=dtfast
        do j=Jstr,Jend
          do i=IstrU,Iend
            cff=(pm(i,j)+pm(i-1,j))*(pn(i,j)+pn(i-1,j))
            fac=1.0_r8/(Dnew(i,j)+Dnew(i-1,j))
            ubar(i,j,knew)=(ubar(i,j,kstp)*(Dstp(i,j)+Dstp(i-1,j))+
     &                      cff*cff1*rhs_ubar(i,j))*fac
#ifdef MASKING
     &                    *umask(i,j)
#endif
          enddo
        enddo
        do j=JstrV,Jend
          do i=Istr,Iend
            cff=(pm(i,j)+pm(i,j-1))*(pn(i,j)+pn(i,j-1))
            fac=1.0_r8/(Dnew(i,j)+Dnew(i,j-1))
            vbar(i,j,knew)=(vbar(i,j,kstp)*(Dstp(i,j)+Dstp(i,j-1))+
     &                      cff*cff1*rhs_vbar(i,j))*fac
#ifdef MASKING
     &                    *vmask(i,j)
#endif
          enddo
        enddo
      elseif (CORRECTOR_2D_STEP) then
        cff1=0.5_r8*dtfast*5.0_r8/12.0_r8
        cff2=0.5_r8*dtfast*8.0_r8/12.0_r8
        cff3=0.5_r8*dtfast*1.0_r8/12.0_r8
        do j=Jstr,Jend
          do i=IstrU,Iend
            cff=(pm(i,j)+pm(i-1,j))*(pn(i,j)+pn(i-1,j))
            fac=1.0_r8/(Dnew(i,j)+Dnew(i-1,j))
            ubar(i,j,knew)=(ubar(i,j,kstp)*(Dstp(i,j)+Dstp(i-1,j))+
     &                      cff*(cff1*rhs_ubar(i,j)+
     &                           cff2*rubar(i,j,kstp)-
     &                           cff3*rubar(i,j,ptsk)))*fac
#ifdef MASKING
     &                    *umask(i,j)
#endif
          enddo
        enddo
        do j=JstrV,Jend
          do i=Istr,Iend
            cff=(pm(i,j)+pm(i,j-1))*(pn(i,j)+pn(i,j-1))
            fac=1.0_r8/(Dnew(i,j)+Dnew(i,j-1))
            vbar(i,j,knew)=(vbar(i,j,kstp)*(Dstp(i,j)+Dstp(i,j-1))+
     &                      cff*(cff1*rhs_vbar(i,j)+
     &                           cff2*rvbar(i,j,kstp)-
     &                           cff3*rvbar(i,j,ptsk)))*fac
#ifdef MASKING
     &                    *vmask(i,j)
#endif
          enddo
        enddo
      endif
#undef Dstp
!
!  If predictor step, load right-side-term into shared arrays for
!  future use during the subsequent corrector step.
!
      if (PREDICTOR_2D_STEP) then
        do j=Jstr,Jend
          do i=IstrU,Iend
            rubar(i,j,krhs)=rhs_ubar(i,j)
          enddo
        enddo
        do j=JstrV,Jend
          do i=Istr,Iend
            rvbar(i,j,krhs)=rhs_vbar(i,j)
          enddo
        enddo
      endif
!
!---------------------------------------------------------------------
!  Apply lateral boundary conditions.
!---------------------------------------------------------------------
!
      call u2dbc_tile (Istr,Iend,Jstr,Jend,UFx)
      call v2dbc_tile (Istr,Iend,Jstr,Jend,UFx)
#ifdef OBC_VOLCONS
      call obc_flux_tile (Istr,Iend,Jstr,Jend)
#endif
#if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_u2d_tile (Istr,Iend,Jstr,Jend,
     &                        ubar(START_2D_ARRAY,knew))
      call exchange_v2d_tile (Istr,Iend,Jstr,Jend,
     &                        vbar(START_2D_ARRAY,knew))
#endif
#ifdef UV_PSOURCE
!
!---------------------------------------------------------------------
!  Apply mass point sources.
!---------------------------------------------------------------------
!
      do j=Jstr-1,Jend+1
        do i=Istr-1,Iend+1
          Dnew(i,j)=zeta(i,j,knew)+h(i,j)
        enddo
      enddo
      do is=1,Nsrc
        i=Isrc(is)
        j=Jsrc(is)
        if (INT(Dsrc(is)).eq.0) then
          if (((Istr.le.i).and.(i.le.Iend+1)).and.
     &        ((Jstr.le.j).and.(j.le.Jend))) then
            cff=1.0_r8/(on_u(i,j)*0.5_r8*(Dnew(i-1,j)+Dnew(i,j)))
            ubar(i,j,knew)=Qbar(is)*cff
          endif
        else
          if (((Istr.le.i).and.(i.le.Iend)).and.
     &        ((Jstr.le.j).and.(j.le.Jend+1))) then
            cff=1.0_r8/(om_v(i,j)*0.5_r8*(Dnew(i,j-1)+Dnew(i,j)))
            vbar(i,j,knew)=Qbar(is)*cff
          endif
        endif
      enddo
#endif /* UV_PSOURCE */
      return
      end
