#include "cppdefs.h"
#ifdef SG_BBL
# undef SG_LOGINT
      subroutine bbl (tile)
!
!================================================== Richard Styles ===
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine compute bottom stresses for the case when the wave   !
!  solution in the wave boundary layer is based on a  2-layer eddy   !
!  viscosity that is linear increasing above Zo and constant above   !
!  Z1.                                                               !
!                                                                    !
!  Reference:                                                        !
!                                                                    !
!  Styles, R. and S.M. glenn,  2000: Modeling stratified wave and    !
!    current bottom boundary layers in the continental shelf, JGR,   !
!    105, 24119-24139.                                               !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "scratch.h"
# include "tile.h"
!
      call bbl_tile (Istr,Iend,Jstr,Jend,
     &               A2d(1, 1),A2d(1, 2),A2d(1, 3),A2d(1, 4),
     &               A2d(1, 5),A2d(1, 6),A2d(1, 7),A2d(1, 8),
     &               A2d(1, 9),A2d(1,10),A2d(1,11),A2d(1,12),
     &               A2d(1,13),A2d(1,14),A2d(1,15))
      return
      end
!
!*********************************************************************
      subroutine bbl_tile (Istr,Iend,Jstr,Jend,Ab,Ub,Zr,Ur,Vr,Umag,
     &                     phic,phicw,Tauc,Tauw,u100,rheight,rlength,
     &                     znot,znotc)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "bbl.h"
# include "forces.h"
# include "grid.h"
# include "ocean.h"
# include "scalars.h"
!
      logical ITERATE
      INTEGER_TYPE
     &        Iend, Istr, Iter, Jend, Jstr, i, j
# ifdef SG_LOGINT
      INTEGER_TYPE
     &        k
      REAL_TYPE
     &        fac, fac1, fac2, z1, z2
# endif
      REAL_TYPE
     &        Fwave, Kb, Kbh, KboKb0, Kb0, Kdelta, Ucur, Ustr, Vcur,
     &        anglec, anglew, cff1, cff2, og, sg_ab, sg_abokb, sg_a1,
     &        sg_b1, sg_chi, sg_c1, sg_dd, sg_epsilon, sg_eta, sg_fofa,
     &        sg_fofb, sg_fofc, sg_fwm, sg_kbs, sg_lambda, sg_mu,
     &        sg_phicw, sg_ro, sg_row, sg_shdnrm, sg_shld, sg_shldcr,
     &        sg_scf, sg_ss, sg_star, sg_ub, sg_ubokur, sg_ubouc,
     &        sg_ubouwm, sg_ur, sg_ustarc, sg_ustarcw, sg_ustarwm,
     &        sg_znot, sg_znotp, sg_zr, sg_zrozn, sg_z1, sg_z1ozn,
     &        sg_z2, twopi
      REAL_TYPE
     &              Ab(PRIVATE_2D_SCRATCH_ARRAY),
     &            Tauc(PRIVATE_2D_SCRATCH_ARRAY),
     &            Tauw(PRIVATE_2D_SCRATCH_ARRAY),
     &              Ub(PRIVATE_2D_SCRATCH_ARRAY),
     &            Umag(PRIVATE_2D_SCRATCH_ARRAY),
     &              Ur(PRIVATE_2D_SCRATCH_ARRAY),
     &              Vr(PRIVATE_2D_SCRATCH_ARRAY),
     &              Zr(PRIVATE_2D_SCRATCH_ARRAY),
     &            phic(PRIVATE_2D_SCRATCH_ARRAY),
     &           phicw(PRIVATE_2D_SCRATCH_ARRAY),
     &         rheight(PRIVATE_2D_SCRATCH_ARRAY),
     &         rlength(PRIVATE_2D_SCRATCH_ARRAY),
     &            u100(PRIVATE_2D_SCRATCH_ARRAY),
     &            znot(PRIVATE_2D_SCRATCH_ARRAY),
     &           znotc(PRIVATE_2D_SCRATCH_ARRAY)
!
#include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Initalize to default values.
!---------------------------------------------------------------------
!
      do j=JstrV-1,Jend
        do i=IstrU-1,Iend
          Tauc(i,j)=0.0_r8
          Tauw(i,j)=0.0_r8
          u100(i,j)=0.0_r8
          rheight(i,j)=0.0_r8
          rlength(i,j)=0.0_r8
          znot(i,j)=sg_znotdef
          znotc(i,j)=0.0_r8
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Set currents above bed.
!---------------------------------------------------------------------
!
      do j=JstrV-1,Jend+1
        do i=IstrU-1,Iend+1
          Zr(i,j)=z_r(i,j,1)-z_w(i,j,0)
          Ur(i,j)=u(i,j,1,nrhs)
          Vr(i,j)=v(i,j,1,nrhs)
# ifdef SG_LOGINT
!
!  If current height is less than z1ur, interpolate logarithmically
!  to z1ur.
!
          if (Zr(i,j).lt.sg_z1min) then
            do k=2,N
              z1=z_r(i,j,k-1)-z_w(i,j,0)
              z2=z_r(i,j,k  )-z_w(i,j,0)
              if ((z1.lt.sg_z1min).and.(sg_z1min.lt.z2)) then
                fac=1.0_r8/LOG(z2/z1)
                fac1=fac*LOG(z2/sg_z1min)
                fac2=fac*LOG(sg_z1min/z1)
                Ur(i,j)=fac1*u(i,j,k-1,nrhs)+fac2*u(i,j,k,nrhs)
                Vr(i,j)=fac1*v(i,j,k-1,nrhs)+fac2*v(i,j,k,nrhs)
                Zr(i,j)=sg_z1min
              endif
            enddo
          endif
#  endif
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute bed wave orbital velocity (m/s) and excursion amplitude
!  (m) from wind-induced waves.  Use linear wave theory dispersion
!  relation for wave number.
!---------------------------------------------------------------------
!
      twopi=2.0_r8*pi
      og=1.0_r8/g
      do j=JstrV-1,Jend
        do i=IstrU-1,Iend
!
!  Compute first guess for wavenumber, Kb0.  Use deep water (Kb0*h>1)
!  and shallow water (Kb0*H<1) approximations.
!
          Fwave=twopi/Pwave(i,j)
          Kb0=Fwave*Fwave*og
          if (Kb0*h(i,j).ge.1.0_r8) then
            Kb=Kb0
          else
            Kb=Fwave/SQRT(g*h(i,j))
          endif
!
!  Compute bottom wave number via Newton-Raphson method.
!
          do Iter=1,sg_n
            if (ITERATE) then
              Kbh=Kb*h(i,j)
              KboKb0=Kb/Kb0
              Kdelta=(1.0_r8-KboKb0*TANH(Kbh))/
     &               (1.0_r8+Kbh*(KboKb0-1.0_r8/KboKb0))
              ITERATE=ABS(Kb*Kdelta) .ge. sg_tol
              Kb=Kb*(1.0_r8+Kdelta)
            endif
          enddo
!
!  Compute bed wave orbital velocity and excursion amplitude.
!
          Ab(i,j)=Awave(i,j)/SINH(Kb*h(i,j))
          Ub(i,j)=Fwave*Ab(i,j)
!
!  Compute bottom current magnitude at RHO-points.
!
          Ucur=0.5_r8*(Ur(i,j)+Ur(i+1,j))
          Vcur=0.5_r8*(Vr(i,j)+Vr(i,j+1))
          Umag(i,j)=SQRT(Ucur*Ucur+Vcur*Vcur)
!
!  Compute angle between currents and waves (radians)
!
          phic(i,j)=ATAN2(Vcur,Ucur)
          phicw(i,j)=Dwave(i,j)-phic(i,j)
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Set default logarithmic profile.
!---------------------------------------------------------------------
!
      do j=JstrV-1,Jend
        do i=IstrU-1,Iend
          if (Umag(i,j).gt.0.0_r8) then
            Ustr=MIN(sg_ustarcdef,Umag(i,j)*vonKar/
     &                            LOG(Zr(i,j)/sg_znotdef))
            Tauc(i,j)=Ustr*Ustr
          endif
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Wave-current interaction case.
!---------------------------------------------------------------------
!
      do j=JstrV-1,Jend
        do i=IstrU-1,Iend
          sg_dd=Ssize(i,j)
          sg_ss=Sdens(i,j)/(rho(i,j,1)+1000.0_r8)
          sg_ab=Ab(i,j)
          sg_ub=Ub(i,j)
          sg_phicw=phicw(i,j)
          sg_ur=Umag(i,j)
          sg_zr=Zr(i,j)
!
!  Compute hydraulic roughness "Znot" (m), ripple height "eta" (m),
!  and ripple length "lambda" (m).
!
# ifdef SG_ZNOT
          sg_star=sg_dd/(4.0_r8*sg_nu)*SQRT((sg_ss-1.0_r8)*sg_g*sg_dd)
! 
!  Compute critical shield parameter based on grain diameter.
!  (sg_scf is a correction factor).
!
          sg_scf=1.0_r8
          if (sg_star.le.1.5_r8) then
            sg_shldcr=sg_scf*0.0932_r8*sg_star**(-0.707_r8)
          elseif ((1.5_r8.lt.sg_star).and.(sg_star.lt.4.0_r8)) then
            sg_shldcr=sg_scf*0.0848_r8*sg_star**(-0.473_r8)
          elseif ((4.0_r8.le.sg_star).and.(sg_star.lt.10.0_r8)) then
            sg_shldcr=sg_scf*0.0680_r8*sg_star**(-0.314_r8)
          elseif ((10.0_r8.le.sg_star).and.(sg_star.lt.34.0_r8)) then
            sg_shldcr=sg_scf*0.033_r8
          elseif ((34.0_r8.le.sg_star).and.(sg_star.lt.270.0_r8)) then
            sg_shldcr=sg_scf*0.0134_r8*sg_star**(0.255_r8)
          else
            sg_shldcr=sg_scf*0.056_r8
          endif
!
!  Calculate skin friction shear stress based on Ole Madsen (1994)
!  empirical formula. Check initiation of sediment motion criteria,
!  to see if we compute sg_znot based on the wave-formed ripples.
!  If the skin friction calculation indicates that sediment is NOT
!  in motion, the ripple model is invalid and take the default value,
!  sg_znotdef.
!
          sg_abokb=sg_ab/sg_dd
          if (sg_abokb.le.100.0_r8) then
            sg_fwm=EXP(7.02_r8*sg_abokb**(-0.078_r8)-8.82_r8)
          else
            sg_fwm=EXP(5.61_r8*sg_abokb**(-0.109_r8)-7.30_r8)
          endif
          sg_ustarwm=SQRT(0.5_r8*sg_fwm)*sg_ub
          sg_shdnrm=(sg_ss-1.0_r8)*sg_dd*sg_g 
          sg_shld=sg_ustarwm*sg_ustarwm/sg_shdnrm
          if ((sg_shld/sg_shldcr).le.1.0_r8) then
            sg_znot=sg_znotdef
            sg_eta=0.0_r8
            sg_lambda=0.0_r8
          else
!
!  Calculate ripple height and length and bottom roughness
!
            sg_chi=4.0_r8*sg_nu*sg_ub*sg_ub/
     &             (sg_dd*((sg_ss-1.0_r8)*sg_g*sg_dd)**1.5_r8)
            if (sg_chi.le.2.0_r8) then 
              sg_eta=sg_ab*0.30_r8*sg_chi**(-0.39_r8)
              sg_lambda=sg_ab*1.96_r8*sg_chi**(-0.28_r8)
            else
              sg_eta=sg_ab*0.45_r8*sg_chi**(-0.99_r8)
              sg_lambda=sg_ab*2.71_r8*sg_chi**(-0.75_r8)
            endif
            sg_kbs=sg_ab*0.0655_r8*
     &             (sg_ub*sg_ub/((sg_ss-1.0_r8)*sg_g*sg_ab))**1.4_r8
            sg_znot=(sg_dd+2.3_r8*sg_eta+sg_kbs)/30.0_r8
          endif
# else
          sg_znot=sg_znotdef
          sg_chi=4.0_r8*sg_nu*sg_ub*sg_ub/
     &           (sg_dd*((sg_ss-1.0_r8)*sg_g*sg_dd)**1.5_r8)
          if (sg_chi.le.2.0_r8) then
            sg_eta=sg_ab*0.32_r8*sg_chi**(-0.34_r8)
            sg_lambda=sg_ab*2.04_r8*sg_chi**(-0.23_r8)
          else
            sg_eta=sg_ab*0.52_r8*sg_chi**(-1.01_r8)
            sg_lambda=sg_ab*2.7_r8*sg_chi**(-0.78_r8)
          endif
# endif
          znot(i,j)=sg_znot
          rheight(i,j)=sg_eta
          rlength(i,j)=sg_lambda
!
!  Compute only when nonzero currents and waves.
!
          sg_zrozn=sg_zr/sg_znot
          if ((sg_ur.gt.0.0_r8).and.(sg_ub.gt.0.0_r8).and.
     &        (sg_zrozn.gt.1.0_r8)) then
!
! Compute bottom stress based on ripple roughness.
!
            sg_ubokur=sg_ub/(sg_kappa*sg_ur)
            sg_row=sg_ab/sg_znot
            sg_a1=1.0_e8-6
            call sg_bstress (sg_row,sg_zrozn,sg_phicw,sg_ubokur,
     &                       sg_a1,sg_mu,sg_epsilon,sg_ro,sg_fofa)
            sg_abokb=sg_ab/(30.0_r8*sg_znot)
            if (sg_abokb.le.100.0_r8) then
              sg_fwm=EXP(-8.82_r8+7.02_r8*sg_abokb**(-0.078_r8))
            else
              sg_fwm=EXP(-7.30_r8+5.61_r8*sg_abokb**(-0.109_r8))
            endif
            sg_ubouwm=SQRT(2.0_r8/sg_fwm)
!
!  Determine the maximum ratio of wave over combined shear stresses,
!  sg_ubouwm (ub/ustarwm).
!
            call sg_purewave (sg_row,sg_ubouwm,sg_znotp,sg_ro)
!
!  Set initial guess of the ratio of wave over shear stress, sg_c1
!  (ub/ustarc).
!
            sg_b1=sg_ubouwm
            sg_fofb=-sg_fofa
            sg_c1=0.5_r8*(sg_a1+sg_b1)
            call sg_bstress (sg_row,sg_zrozn,sg_phicw,sg_ubokur,
     &                       sg_c1,sg_mu,sg_epsilon,sg_ro,sg_fofc)
!
!  Solve PDE via bi-section method.
!
            ITERATE=.true.
            do Iter=1,sg_n
              if (ITERATE) then
                if ((sg_fofb*sg_fofc).lt.0.0_r8) then
                  sg_a1=sg_c1
                else
                  sg_b1=sg_c1
                endif
                sg_c1=0.5_r8*(sg_a1+sg_b1)
                call sg_bstress (sg_row,sg_zrozn,sg_phicw,sg_ubokur,
     &                           sg_c1,sg_mu,sg_epsilon,sg_ro,sg_fofc)
                ITERATE=(sg_b1-sg_c1) .ge. sg_tol
                if (ITERATE) Iconv(i,j)=Iter
              endif
            enddo
            sg_ubouc=sg_c1
!
!  Compute bottom shear stress magnitude (m/s).
!
            sg_ustarcw=sg_ub/sg_ubouc
            sg_ustarwm=sg_mu*sg_ustarcw
            sg_ustarc=MIN(sg_ustarcdef,sg_epsilon*sg_ustarcw)
            Tauc(i,j)=sg_ustarc*sg_ustarc
            Tauw(i,j)=sg_ustarwm*sg_ustarwm
!
!  Compute apparent hydraulic roughness (m).
!
            if (sg_epsilon.gt.0.0_r8) then
              sg_z1=sg_alpha*sg_kappa*sg_ab/sg_ubouc
              sg_z2=sg_z1/sg_epsilon
              sg_z1ozn=sg_z1/sg_znot
              znotc(i,j)=sg_z2*
     &                   EXP(-(1.0_r8-sg_epsilon+
     &                         sg_epsilon*LOG(sg_z1ozn)))
!
!  Compute mean (m/s) current at 100 cm above the bottom.
!
              if (sg_z100.gt.sg_z2) then
                u100(i,j)=sg_ustarc*
     &                    (LOG(sg_z100/sg_z2)+1.0_r8-sg_epsilon+
     &                    sg_epsilon*LOG(sg_z1ozn))/sg_kappa
              elseif ((sg_z100.le.sg_z2).and.(sg_zr.gt.sg_z1)) then
                u100(i,j)=sg_ustarc*sg_epsilon*
     &                    (sg_z100/sg_z1-1.0_r8+LOG(sg_z1ozn))/sg_kappa
              else
                u100(i,j)=sg_ustarc*sg_epsilon*
     &                    LOG(sg_z100/sg_znot)/sg_kappa
              endif
            endif
          endif
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute kinematic bottom stress components due current and wind-
!  induced waves.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=IstrU,Iend
          cff1=0.5_r8*(Tauc(i-1,j)+Tauc(i,j))
          cff2=0.5_r8*(Tauw(i-1,j)+Tauw(i,j))
!!        anglec=COS(0.5_r8*(phic(i-1,j)+phic(i,j)))
          anglec=Ur(i,j)/(0.5*(Umag(i-1,j)+Umag(i,j)+1.0_e8-10))
          anglew=COS(0.5_r8*(Dwave(i-1,j)+Dwave(i,j)))
          bustr (i,j)=cff1*anglec
          bustrw(i,j)=cff2*anglew
          Ubed(i,j)=Ub(i,j)*anglew
!!        Ubot(i,j)=u100(i,j)*anglec
          Ubot(i,j)=Ur(i,j)
        enddo
      enddo
      do j=JstrV,Jend
        do i=Istr,Iend
          cff1=0.5_r8*(Tauc(i,j-1)+Tauc(i,j))
          cff2=0.5_r8*(Tauw(i,j-1)+Tauw(i,j))
!!        anglec=SIN(0.5_r8*(phic(i,j-1)+phic(i,j)))
          anglec=Vr(i,j)/(0.5_r8*(Umag(i,j-1)+Umag(i,j)+1.0_e8-10))
          anglew=SIN(0.5_r8*(Dwave(i,j-1)+Dwave(i,j)))
          bvstr (i,j)=cff1*anglec
          bvstrw(i,j)=cff2*anglew
          Vbed(i,j)=Ub(i,j)*anglew
!!        Vbot(i,j)=u100(i,j)*anglec
          Vbot(i,j)=Vr(i,j)
        enddo
      enddo
!
!  Apply periodic or gradient boundary conditions for output
!  purposes only.
!
      call fill_u2d_tile (Istr,Iend,Jstr,Jend,bustr(START_2D_ARRAY))
      call fill_v2d_tile (Istr,Iend,Jstr,Jend,bvstr(START_2D_ARRAY))
      call fill_u2d_tile (Istr,Iend,Jstr,Jend,bustrw(START_2D_ARRAY))
      call fill_v2d_tile (Istr,Iend,Jstr,Jend,bvstrw(START_2D_ARRAY))
      call fill_u2d_tile (Istr,Iend,Jstr,Jend,Ubed(START_2D_ARRAY))
      call fill_v2d_tile (Istr,Iend,Jstr,Jend,Vbed(START_2D_ARRAY))
      call fill_u2d_tile (Istr,Iend,Jstr,Jend,Ubot(START_2D_ARRAY))
      call fill_v2d_tile (Istr,Iend,Jstr,Jend,Vbot(START_2D_ARRAY))
!
!---------------------------------------------------------------------
!  Load closure state variables for output purposes.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
          Abed(i,j)=Ab(i,j)
          Hripple(i,j)=rheight(i,j)
          Lripple(i,j)=rlength(i,j)
          Zbnot(i,j)=znot(i,j)
          Zbnotc(i,j)=znotc(i,j)
        enddo
      enddo
!
!  Apply periodic or gradient boundary conditions for output
!  purposes only.
!
      call fill_r2d_tile (Istr,Iend,Jstr,Jend,Abed(START_2D_ARRAY))
      call fill_r2d_tile (Istr,Iend,Jstr,Jend,Hripple(START_2D_ARRAY))
      call fill_r2d_tile (Istr,Iend,Jstr,Jend,Lripple(START_2D_ARRAY))
      call fill_r2d_tile (Istr,Iend,Jstr,Jend,Zbnot(START_2D_ARRAY))
      call fill_r2d_tile (Istr,Iend,Jstr,Jend,Zbnotc(START_2D_ARRAY))
# undef DEBUG
# ifdef DEBUG
      i=1
      j=106
      write(6,"(1pe23.16,2x,'sg_ab')") Ab(i,j)
      write(6,"(1pe23.16,2x,'sg_ub')") Ub(i,j)
      write(6,"(1pe23.16,2x,'sg_ur')") Ur(i,j)
      write(6,"(1pe23.16,2x,'sg_zr')") Zr(i,j)
      write(6,"(1pe23.16,2x,'sg_phicw')") phicw(i,j)
      write(6,"(1pe23.16,2x,'sg_rhow')") rho(i,j,1)+1000.0_r8
      write(6,*) ' '
      write(6,"('shld    = ',1pe15.8)") wrk3(i,j)
      write(6,"('mu      = ',1pe15.8)") wrk1(i,j)
      write(6,"('eps     = ',1pe15.8)") wrk2(i,j)
      write(6,"('Iters   = ',i4)") Iconv(i,j)
      write(6,"('Znot    = ',1pe15.8)") znot(i,j)
      write(6,"('Znotc   = ',1pe15.8)") znotc(i,j)
      write(6,"('Ustarc  = ',1pe15.8)") SQRT(Tauc(i,j))
      write(6,"('Ustarc2 = ',1pe15.8)") Tauc(i,j)
      write(6,"('Ustarwm = ',1pe15.8)") SQRT(Tauw(i,j))
      write(6,"('u100    = ',1pe15.8)") u100(i,j)
      write(6,"('eta     = ',1pe15.8)") Hripple(i,j)
      write(6,"('lambda  = ',1pe15.8)") Lripple(i,j)
# endif
      return
      end
      subroutine sg_bstress (sg_row,sg_zrozn,sg_phicw,sg_ubokur,
     &                       sg_ubouc,sg_mu,sg_epsilon,sg_ro,sg_fofx)
!
!=====================================================================
!                                                                    !
!  This routine computes bottom stresses via bottom boundary layer   !
!  formulation of Styles and Glenn (1999).                           !
!                                                                    !
!  On Input:                                                         !
!                                                                    !
!     sg_row      Ratio of wave excursion amplitude over roughness.  !
!     sg_zrozn    Ratio of height of current over roughness.         !
!     sg_phiwc    Angle between wave and currents (radians).         !
!     sg_ubokur   Ratio of wave over current velocity:               !
!                   ub/(vonKar*ur)                                   !
!     sg_ubouc    Ratio of bed wave orbital over bottom shear stress !
!                   (ub/ustarc), first guess.                        !
!                                                                    !
!  On Output:                                                        !
!                                                                    !
!     sg_ubouc    Ratio of bed wave orbital over bottom shear stress !
!                   (ub/ustarc), iterated value.                     !
!     sg_mu       Ratio between wave and current bottom shear        !
!                   stresses (ustarwm/ustarc).                       !
!     sg_epsilon  Ratio between combined (wave and current) and      !
!                   current bottom shear stresses (ustarc/ustarcw).  !
!     sg_ro       Internal friction Rossby number:                   !
!                   ustarc/(omega*znot)                              !
!     sg_fofx     Root of PDE used for convergence.                  !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "bbl.h"
!
      logical ITERATE
      INTEGER_TYPE
     &        Iter
      REAL_TYPE
     &        sg_epsilon, sg_fofx, sg_phicw, sg_ro, sg_row, sg_mu,
     &        sg_ubokur, sg_ubouc, sg_zrozn
      REAL_TYPE
     &        cff, sg_bei, sg_beip, sg_ber, sg_berp, sg_cosphi,
     &        sg_eps2, sg_kei, sg_keip, sg_ker, sg_kerp, sg_mu2,
     &        sg_phi, sg_ror, sg_x, sg_z2p, sg_znotp, sg_zroz1,
     &        sg_zroz2, sg_z1ozn, sg_z2ozn
      COMPLEX_TYPE
     &        sg_argi, sg_bnot, sg_bnotp, sg_b1, sg_b1p, sg_gammai,
     &        sg_knot, sg_knotp, sg_k1, sg_k1p, sg_ll, sg_nn
!
!---------------------------------------------------------------------
!  Compute bottom stresses.
!---------------------------------------------------------------------
!
!  Compute nondimensional bottom wave shear, phi.  Iterate to make
!  sure that there is an upper limit in "ubouc".  It usually requires
!  only one pass.
!
      ITERATE=.true.
      do Iter=1,sg_n
        if (ITERATE) then
          sg_ro=sg_row/sg_ubouc
          sg_znotp=1.0_r8/(sg_kappa*sg_ro)
          if ((sg_z1p/sg_znotp).gt.1.0_r8) then
            sg_x=2.0_r8*SQRT(sg_znotp)
            if (sg_x.le.8.0_r8) then
              call sg_kelvin8m (sg_x,sg_pi,sg_ber,sg_bei,sg_ker,
     &                          sg_kei,sg_berp,sg_beip,sg_kerp,
     &                          sg_keip)
            else
              call sg_kelvin8p (sg_x,sg_pi,sg_ker,sg_kei,sg_ber,
     &                          sg_bei,sg_kerp,sg_keip,sg_berp,
     &                          sg_beip)
            endif
            cff=1.0_r8/SQRT(sg_znotp)
            sg_bnot =CMPLX(sg_ber,sg_bei)
            sg_knot =CMPLX(sg_ker,sg_kei)
            sg_bnotp=CMPLX(sg_berp,sg_beip)*cff
            sg_knotp=CMPLX(sg_kerp,sg_keip)*cff
!
            sg_x=2.0_r8*SQRT(sg_z1p)
            if (sg_x.le.8.0_r8) then
              call sg_kelvin8m (sg_x,sg_pi,sg_ber,sg_bei,sg_ker,
     &                          sg_kei,sg_berp,sg_beip,sg_kerp,
     &                          sg_keip)
            else
              call sg_kelvin8p (sg_x,sg_pi,sg_ker,sg_kei,sg_ber,
     &                          sg_bei,sg_kerp,sg_keip,sg_berp,
     &                          sg_beip)
            endif
            cff=1.0_r8/SQRT(sg_z1p)
            sg_b1 =CMPLX(sg_ber,sg_bei)
            sg_k1 =CMPLX(sg_ker,sg_kei)
            sg_b1p=CMPLX(sg_berp,sg_beip)*cff
            sg_k1p=CMPLX(sg_kerp,sg_keip)*cff
!
            sg_ll=sg_mp*sg_b1+sg_b1p
            sg_nn=sg_mp*sg_k1+sg_k1p
            sg_argi=sg_bnotp*sg_nn/(sg_bnot*sg_nn-sg_knot*sg_ll)+
     &              sg_knotp*sg_ll/(sg_knot*sg_ll-sg_bnot*sg_nn)
            sg_gammai=-sg_kappa*sg_znotp*sg_argi
            sg_phi=CABS(sg_gammai)
          else
            sg_gammai=-sg_kappa*sg_z1p*sg_mp
            sg_phi=CABS(sg_gammai)
          endif
!
          if (sg_ubouc.gt.(1.0_r8/sg_phi)) then
            sg_ubouc=1.0_r8/sg_phi
          else
            ITERATE=.false.
          endif
        endif
      enddo
!
!  Compute ratio of wave over current bottom shear stresses.
!
      sg_mu=SQRT(sg_ubouc*sg_phi)
!
!  Compute ratio of current over combined bottom shear stresses.
!
      if (sg_mu.eq.1.0_r8) then
        sg_epsilon=0.0_r8
      else
        sg_mu2=sg_mu*sg_mu
        sg_cosphi=ABS(COS(sg_phicw))
        sg_eps2=-sg_mu2*sg_cosphi+
     &          SQRT(1.0_r8+sg_mu2*sg_mu2*(sg_cosphi*sg_cosphi-1.0_r8))
        sg_epsilon=SQRT(sg_eps2)
      endif
!
!  Determine root of PDE used for convergence.
!
      if (sg_epsilon.ne.0.0_r8) then
        sg_z2p=sg_z1p/sg_epsilon
        sg_ror=sg_ro/sg_zrozn
        sg_zroz1=1.0_r8/(sg_alpha*sg_kappa*sg_ror)
        sg_zroz2=sg_epsilon*sg_zroz1
        sg_z1ozn=sg_alpha*sg_kappa*sg_ro
        sg_z2ozn=sg_z1ozn/sg_epsilon
!
        if ((sg_zroz2.gt.1.0_r8).and.(sg_z1ozn.gt.1.0_r8)) then
          sg_fofx=-sg_ubouc+sg_ubokur*sg_epsilon*
     &                      (LOG(sg_zroz2)+1.0_r8-sg_epsilon+
     &                       sg_epsilon*LOG(sg_z1ozn))
        elseif ((sg_zroz2.le.1.0_r8).and.(sg_zroz1.gt.1.0_r8).and.
     &          (sg_z1ozn.gt.1.0_r8)) then
          sg_fofx=-sg_ubouc+sg_ubokur*sg_epsilon*sg_epsilon*
     &                      (sg_zroz1-1.0_r8+LOG(sg_z1ozn))
        elseif ((sg_zroz1.le.1.0_r8).and.(sg_z1ozn.gt.1.0_r8)) then
          sg_fofx=-sg_ubouc+sg_ubokur*sg_epsilon*sg_epsilon*
     &                      LOG(sg_zrozn)
        elseif ((sg_zroz2.gt.1.0_r8).and.(sg_z1ozn.le.1.0_r8).and.
     &          (sg_z2ozn.gt.1.0_r8)) then
          sg_fofx=-sg_ubouc+sg_ubokur*sg_epsilon*
     &                      (LOG(sg_zroz2)+1.0_r8-1.0_r8/sg_z2ozn)
        elseif ((sg_zroz2.le.1.0_r8).and.(sg_zroz1.gt.1.0_r8).and.
     &          (sg_z1ozn.le.1.0_r8).and.(sg_z2ozn.gt.1.0_r8)) then
          sg_fofx=-sg_ubouc+sg_ubokur*sg_epsilon*sg_epsilon*
     &                      (sg_zroz1-1.0_r8/sg_z1ozn)
        elseif ((sg_zroz2.gt.1.0_r8).and.(sg_z2ozn.le.1.0_r8)) then
          sg_fofx=-sg_ubouc+sg_ubokur*sg_epsilon*LOG(sg_zrozn)
        endif
      endif
      return
      end
      subroutine  sg_purewave (sg_row,sg_ubouwm,sg_znotp,sg_ro)
!
!=====================================================================
!                                                                    !
!  This routine determines the maximum ratio of waves over combined  !
!  bottom shear stress.                                              !
!                                                                    !
!  On Input:                                                         !
!                                                                    !
!     sg_row      Ratio of wave excursion amplitude over roughness.  !
!                                                                    !
!  On Output:                                                        !
!                                                                    !
!     sg_ubouwm   Maximum ratio of waves over combined bottom shear  !
!                   stress.                                          !
!     sg_znotp    Ratio of hydraulic roughness over scaled height    !
!                   of bottom boundary layer.                        !
!     sg_ro       Internal friction Rossby number:                   !
!                   ustarc/(omega*znot)                              !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "bbl.h"
!
      INTEGER_TYPE
     &        Iter
      REAL_TYPE
     &        sg_ro, sg_row, sg_ubouwm, sg_znotp
      REAL_TYPE
     &        cff, sg_bei, sg_beip, sg_ber, sg_berp, sg_kei, sg_keip,
     &        sg_ker, sg_kerp, sg_phi, sg_ubouwmn, sg_x
      COMPLEX_TYPE
     &        sg_argi, sg_bnot, sg_bnotp, sg_b1, sg_b1p, sg_gammai,
     &        sg_knot, sg_knotp, sg_k1, sg_k1p, sg_ll, sg_nn
!
!---------------------------------------------------------------------
!  Compute wind-induced wave stress.
!---------------------------------------------------------------------
!
      do Iter=1,sg_n
        sg_ro=sg_row/sg_ubouwm
        sg_znotp=1.0_r8/(sg_kappa*sg_ro)
        if (sg_z1p/sg_znotp.gt.1.0_r8) then
          sg_x=2.0_r8*SQRT(sg_znotp)
          if (sg_x.le.8.0_r8) then
            call sg_kelvin8m (sg_x,sg_pi,sg_ber,sg_bei,sg_ker,sg_kei,
     &                        sg_berp,sg_beip,sg_kerp,sg_keip)
          else
            call sg_kelvin8p (sg_x,sg_pi,sg_ker,sg_kei,sg_ber,sg_bei,
     &                        sg_kerp,sg_keip,sg_berp,sg_beip)
          endif
          cff=1.0_r8/SQRT(sg_znotp)
          sg_bnot =CMPLX(sg_ber,sg_bei)
          sg_knot =CMPLX(sg_ker,sg_kei)
          sg_bnotp=CMPLX(sg_berp,sg_beip)*cff
          sg_knotp=CMPLX(sg_kerp,sg_keip)*cff
!
          sg_x=2.0*SQRT(sg_z1p)
          if (sg_x.le.8.0_r8) then
            call sg_kelvin8m (sg_x,sg_pi,sg_ber,sg_bei,sg_ker,sg_kei,
     &                        sg_berp,sg_beip,sg_kerp,sg_keip)
          else
            call sg_kelvin8p (sg_x,sg_pi,sg_ker,sg_kei,sg_ber,sg_bei,
     &                        sg_kerp,sg_keip,sg_berp,sg_beip)
          endif
          cff=1.0_r8/SQRT(sg_z1p)
          sg_b1 =CMPLX(sg_ber,sg_bei)
          sg_k1 =CMPLX(sg_ker,sg_kei)
          sg_b1p=CMPLX(sg_berp,sg_beip)*cff
          sg_k1p=CMPLX(sg_kerp,sg_keip)*cff
!
          sg_ll=sg_mp*sg_b1+sg_b1p
          sg_nn=sg_mp*sg_k1+sg_k1p
          sg_argi=sg_bnotp*sg_nn/(sg_bnot*sg_nn-sg_knot*sg_ll)+
     &            sg_knotp*sg_ll/(sg_knot*sg_ll-sg_bnot*sg_nn)
          sg_gammai=-sg_kappa*sg_znotp*sg_argi
          sg_phi=CABS(sg_gammai)
        else
          sg_gammai=-sg_kappa*sg_z1p*sg_mp
          sg_phi=CABS(sg_gammai)
        endif
!
        sg_ubouwmn=1.0_r8/sg_phi
        if (abs((sg_ubouwmn-sg_ubouwm)/sg_ubouwmn).le.sg_tol) then
          sg_ubouwm=sg_ubouwmn
          return
        else
          sg_ubouwm=sg_ubouwmn
        endif
      enddo
      return
      end
      subroutine sg_kelvin8m (x,pi,ber,bei,ker,kei,berp,beip,kerp,
     &                        keip)
!
!=====================================================================
!                                                                    !
! This rotuine computes the Kelvin functions for arguments less      !
! than eight.                                                        !
!                                                                    !
!=====================================================================
!
      implicit none
!
      INTEGER_TYPE
     &        i
      REAL_TYPE
     &        bei, beip, ber, berp, cff, kei, keip, ker, kerp, pi, x,
     &        xhalf
      REAL_TYPE
     &        xp(28)
!
!---------------------------------------------------------------------
!  Compute Kelvin functions.
!---------------------------------------------------------------------
!
      cff=0.125_r8*x
      xp(1)=cff
      do i=2,28
        xp(i)=xp(i-1)*cff
      enddo
      xhalf=0.5_r8*x
!
      ber=1.0_r8-
     &    64.0_r8*xp(4)+113.77777774_r8*xp(8)-
     &    32.36345652_r8*xp(12)+2.64191397_r8*xp(16)-
     &    0.08349609_r8*xp(20)+0.00122552_r8*xp(24)-
     &    0.00000901_r8*xp(28)
      bei=16.0_r8*xp(2)-113.77777774_r8*xp(6)+
     &    72.81777742*xp(10)-10.56765779_r8*xp(14)+
     &    0.52185615_r8*xp(18)-0.01103667_r8*xp(22)+
     &    0.00011346*xp(26)
!
      ker=-ber*LOG(xhalf)+0.25_r8*pi*bei-
     &    0.57721566_r8-59.05819744*xp(4)+
     &    171.36272133_r8*xp(8)-60.60977451_r8*xp(12)+
     &    5.65539121_r8*xp(16)-0.19636347_r8*xp(20)+
     &    0.00309699_r8*xp(24)-0.00002458_r8*xp(28)
      kei=-bei*LOG(xhalf)-0.25_r8*pi*ber+
     &    6.76454936_r8*xp(2)-142.91827687_r8*xp(6)+
     &    124.23569650_r8*xp(10)-21.30060904_r8*xp(14)+
     &    1.17509064_r8*xp(18)-0.02695875_r8*xp(22)+
     &    0.00029532_r8*xp(26)
!
      berp=x*(-4.0_r8*xp(2)+14.22222222_r8*xp(6)-
     &        6.06814810_r8*xp(10)+0.66047849_r8*xp(14)-
     &        0.02609253_r8*xp(18)+0.00045957_r8*xp(22)-
     &        0.00000394_r8*xp(26))
      beip=x*(0.5_r8-10.66666666_r8*xp(4)+11.37777772_r8*xp(8)-
     &        2.31167514_r8*xp(12)+0.14677204_r8*xp(16)-
     &        0.00379386_r8*xp(20)+0.00004609_r8*xp(24))
!
      kerp=-berp*LOG(xhalf)-ber/x+0.25*pi*beip+
     &     x*(-3.69113734_r8*xp(2)+21.42034017_r8*xp(6)-
     &        11.36433272_r8*xp(10)+1.41384780_r8*xp(14)-
     &        0.06136358_r8*xp(18)+0.00116137_r8*xp(22)-
     &        0.00001075*xp(26))
      keip=-beip*LOG(xhalf)-bei/x-0.25_r8*pi*berp+
     &     x*(0.21139217_r8-13.39858846_r8*xp(4)+
     &        19.41182758_r8*xp(8)-4.65950823_r8*xp(12)+
     &        0.33049424_r8*xp(16)-0.00926707_r8*xp(20)+
     &        0.00011997_r8*xp(24))
      return
      end
      subroutine sg_kelvin8p (x,pi,ker,kei,ber,bei,kerp,keip,berp,
     &                        beip)
!
!=====================================================================
!                                                                    !
! This rotuine computes the Kelvin functions for arguments greater   !
! than eight.                                                        !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "bbl.h"
!
      INTEGER_TYPE
     &        i
      REAL_TYPE
     &          bei, beip, ber, berp, cff, kei, keip, ker, kerp, pi, x
      REAL_TYPE
     &          xm(6), xp(6)
      COMPLEX_TYPE
     &          argm, argp, fofx, gofx, phim, phip, thetam, thetap
!
!---------------------------------------------------------------------
!  Compute Kelvin functions.
!---------------------------------------------------------------------
!
      cff=8.0_r8/x
      xp(1)=cff
      xm(1)=-cff
      do i=2,6
        xp(i)=xp(i-1)*cff
        xm(i)=-xm(i-1)*cff
      enddo
!
      thetap=CMPLX(0.0_r8,-0.3926991_r8)+
     &       CMPLX(0.0110486_r8,-0.0110485_r8)*xp(1)+
     &       CMPLX(0.0_r8,-0.0009765_r8)*xp(2)+
     &       CMPLX(-0.0000906_r8,-0.0000901_r8)*xp(3)+
     &       CMPLX(-0.0000252_r8,0.0_r8)*xp(4)+
     &       CMPLX(-0.0000034_r8,0.0000051_r8)*xp(5)+
     &       CMPLX(0.0000006,0.0000019)*xp(6)
      thetam=CMPLX(0.0_r8,-0.3926991_r8)+
     &       CMPLX(0.0110486_r8,-0.0110485_r8)*xm(1)+
     &       CMPLX(0.0_r8,-0.0009765_r8)*xm(2)+
     &       CMPLX(-0.0000906_r8,-0.0000901_r8)*xm(3)+
     &       CMPLX(-0.0000252_r8,0.0_r8)*xm(4)+
     &       CMPLX(-0.0000034_r8,0.0000051_r8)*xm(5)+
     &       CMPLX(0.0000006_r8,0.0000019_r8)*xm(6)
!
      phip=CMPLX(0.7071068_r8,0.7071068_r8)+
     &     CMPLX(-0.0625001_r8,-0.0000001_r8)*xp(1)+
     &     CMPLX(-0.0013813_r8,0.0013811_r8)*xp(2)+
     &     CMPLX(0.0000005_r8,0.0002452_r8)*xp(3)+
     &     CMPLX(0.0000346_r8,0.0000338_r8)*xp(4)+
     &     CMPLX(0.0000117_r8,-0.0000024_r8)*xp(5)+
     &     CMPLX(0.0000016_r8,-0.0000032_r8)*xp(6)
      phim=CMPLX(0.7071068_r8,0.7071068_r8)+
     &     CMPLX(-0.0625001_r8,-0.0000001_r8)*xm(1)+
     &     CMPLX(-0.0013813_r8,0.0013811_r8)*xm(2)+
     &     CMPLX(0.0000005_r8,0.0002452_r8)*xm(3)+
     &     CMPLX(0.0000346_r8,0.0000338_r8)*xm(4)+
     &     CMPLX(0.0000117_r8,-0.0000024_r8)*xm(5)+
     &     CMPLX(0.0000016_r8,-0.0000032_r8)*xm(6)
!
      cff=x/SQRT(2.0_r8)
      argm=-cff*CMPLX(1.0_r8,1.0_r8)+thetam
      fofx=SQRT(pi/(2.0_r8*x))*CEXP(argm)
      ker=REAL(fofx)
      kei=IMAG(fofx)
!
      argp=cff*CMPLX(1.0_r8,1.0_r8)+thetap
      gofx=1.0_r8/SQRT(2.0_r8*pi*x)*CEXP(argp)
      ber=REAL(gofx)-kei/pi
      bei=IMAG(gofx)+ker/pi
!
      kerp=REAL(-fofx*phim)
      keip=IMAG(-fofx*phim)
!
      berp=REAL(gofx*phip)-keip/pi
      beip=IMAG(gofx*phip)+kerp/pi
#else
      subroutine sg_bbl
#endif
      return
      end
