#include "cppdefs.h"
      subroutine set_weights
# ifdef SOLVE3D
!
!======================================== Alexander F. Shchepetkin ===
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine sets the weigth functions for the time averaging of  !
!  2D fields over all short time-steps.                              !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "iounits.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        i, j, iter

# ifdef POWER_LAW
      REAL_TYPE
     &        alpha, beta, gamma, scale
# elif defined COSINE2
      REAL_TYPE
     &     arg
# endif
      REAL_TYPE
     &     cff, cff1, cff2, shift, wsum
!
!=======================================================================
!  Compute time-averaging filter for barotropic fields.
!=======================================================================
!
!  Initialize both sets of weights to zero.
!
      nfast=0
      do i=1,2*ndtfast
        weight(1,i)=0.0_r8
        weight(2,i)=0.0_r8
      enddo

# ifdef POWER_LAW
!
!-----------------------------------------------------------------------
!  Power-law shape filters.
!-----------------------------------------------------------------------
!
!  The power-law shape filters are given by:
!
!     f(xi)=xi^alpha*(1-xi^beta)-gamma*xi 
!
!  where xi=scale*i/ndtfast; and scale, alpha, beta, gamma, and
!  normalization are chosen to yield the correct zeroth-order
!  (normalization), first-order (consistency), and second-order moments,
!  resulting in overall second-order temporal accuracy for time-averaged
!  barotropic motions resolved by baroclinic time step.
!
!  Possible settiings of parameters to yield the second-order accuracy:
!
!     alpha    beta   gamma
!     ---------------------
!      2.0     1.0    0.169      The problem here is setting gamma.
!      2.0     2.0    0.234      The values of gamma here are understood
!      2.0     3.0    0.266      as the maximum value allowed. They are
!      2.0     4.0    0.284      computed using a Newton iteration
!      2.0     5.0    0.296      scheme.
!      2.0     6.0    0.304
!      2.0     8.0    0.314
!
      alpha=2.0_r8
      beta=2.0_r8
      gamma=0.2_r8

      scale=(alpha+1.0_r8)*(alpha+beta+1.0_r8)/
     &      ((alpha+2.0_r8)*(alpha+beta+2.0_r8)*FLOAT(ndtfast))
!
!  Find center of gravity of the primary weighting shape function and
!  iteratively adjust "scale" to place the  centroid exactly at
!  "ndtfast".
!
      do iter=1,16
        nfast=0
        do i=1,2*ndtfast
          cff=scale*FLOAT(i)
          weight(1,i)=cff**alpha-cff**(alpha+beta)-gamma*cff
          if ((i.gt.ndtfast).and.(weight(1,i).lt.0.0_r8)) then
            weight(1,i)=0.0_r8
          else
            nfast=i
          endif
        enddo
        wsum=0.0_r8
        shift=0.0_r8
        do i=1,nfast
          wsum=wsum+weight(1,i)
          shift=shift+weight(1,i)*FLOAT(i)
        enddo
        scale=scale*shift/(wsum*FLOAT(ndtfast))
      enddo

# elif defined COSINE2
!
!-----------------------------------------------------------------------
!  Cosine-squared shaped filter.
!-----------------------------------------------------------------------
!
      cff=pi/FLOAT(ndtfast)
      do i=1,2*ndtfast
        arg=cff*float(i-ndtfast)
        if ((2.0_r8*ABS(arg)).lt.pi) then
          weight(1,i)=(COS(arg))**2
          nfast=i
        endif
      enddo
# endif
!
!-----------------------------------------------------------------------
!  Post-processing of primary weights.
!-----------------------------------------------------------------------
!
!  Although it is assumed that the initial settings of the primary
!  weights has its center of gravity "reasonably close" to NDTFAST,
!  it may be not so according to the discrete rules of integration.
!  The following procedure is designed to put the center of gravity
!  exactly to NDTFAST by computing mismatch (NDTFAST-shift) and
!  applying basically an upstream advection of weights to eliminate
!  the mismatch iteratively. Once this procedure is complete primary
!  weights are normalized.
!
!  Find center of gravity of the primary weights and subsequently
!  calculate the mismatch to be compensated.
!
      do iter=1,ndtfast
        wsum=0.0_r8
        shift=0.0_r8
        do i=1,nfast
          wsum=wsum+weight(1,i)
          shift=shift+FLOAT(i)*weight(1,i)
        enddo
        shift=shift/wsum
        cff=FLOAT(ndtfast)-shift
!
!  Apply advection step using either whole, or fractional shifts.
!  Notice that none of the four loops here is reversible.
!
        if (cff.gt.1.0_r8) then
          nfast=nfast+1
          do i=nfast,2,-1
            weight(1,i)=weight(1,i-1)
          enddo
          weight(1,1)=0.0_r8
        elseif (cff.gt.0.0_r8) then
          wsum=1.0_r8-cff
          do i=nfast,2,-1
            weight(1,i)=wsum*weight(1,i)+cff*weight(1,i-1)
          enddo
          weight(1,1)=wsum*weight(1,1)
        elseif (cff.lt.-1.0_r8) then
          nfast=nfast-1
          do i=1,nfast,+1
            weight(1,i)=weight(1,i+1)
          enddo
          weight(1,nfast+1)=0.0_r8
         elseif (cff.lt.0.0_r8) then
          wsum=1.0_r8+cff
          do i=1,nfast-1,+1
            weight(1,i)=wsum*weight(1,i)-cff*weight(1,i+1)
          enddo
          weight(1,nfast)=wsum*weight(1,nfast)
        endif
      enddo
!
!  Set SECONDARY weights assuming that backward Euler time step is used
!  for free surface.  Notice that array weight(2,i) is assumed to have
!  all-zero status at entry in this segment of code.
!
      do j=1,nfast
        cff=weight(1,j)
        do i=1,j
          weight(2,i)=weight(2,i)+cff
        enddo
      enddo
!
!  Normalize both set of weights.
!
      wsum=0.0_r8
      cff=0.0_r8
      do i=1,nfast
        wsum=wsum+weight(1,i)
        cff=cff+weight(2,i)
      enddo
      wsum=1.0_r8/wsum
      cff=1.0_r8/cff
      DO i=1,nfast
        weight(1,i)=wsum*weight(1,i)
        weight(2,i)=cff*weight(2,i)
      enddo
!
!  Report weights.
!
      write(stdout,10) ndtfast, nfast
      cff=0.0_r8
      cff1=0.0_r8
      cff2=0.0_r8
      wsum=0.0_r8
      shift=0.0_r8
      do i=1,nfast
        cff=cff+weight(1,i)
        cff1=cff1+weight(1,i)*FLOAT(i)
        cff2=cff2+weight(1,i)*FLOAT(i*i)
        wsum=wsum+weight(2,i)
        shift=shift+FLOAT(i)*weight(2,i)
        write(stdout,20) i, weight(1,i), weight(2,i), cff, wsum
      enddo
      cff1=cff1/FLOAT(ndtfast)
      cff2=cff2/(FLOAT(ndtfast)*FLOAT(ndtfast))
      shift=(shift-0.5_r8)/FLOAT(ndtfast)
      write(stdout,30) ndtfast, nfast, FLOAT(nfast)/FLOAT(ndtfast)
# ifdef POWER_LAW
      write(stdout,40) cff1, cff2, shift, cff, wsum
      if (cff2.lt.1.0001_r8) write (stdout,50)
# endif
!
  10  format(/,'Time Splitting Weights: ndtfast = ',i3,4x,'nfast = ',
     &       i3,/,/,4x,'Primary',12x,'Secondary',12x,
     &       'Accumulated to Current Step',/)
  20  format(i3,4f19.16)
  30  format(/,1x,'ndtfast, nfast = ',2i4,3x,'nfast/ndtfast = ',f7.5)
# ifdef POWER_LAW
  40  format(/,1x,'Centers of gravity and integrals ',
     &        '(values must be 1, 1, approx 1/2, 1, 1):',/,/,3x,5F15.12)
  50  format(/,' WARNING: unstable weights, reduce parameter gamma.',/)
# endif
#endif
      return
      end
