#include "cppdefs.h"
#ifdef SEDIMENT
      subroutine sediment (tile)
!
!================================================== John C. Warner ===
!  Copyright (c) 2002 Rutgers                                        !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This  routine computes the sediment sources and sinks and adds    !
!  then the global sediment tracer fields. Currently, it computes    !
!  the following terms:                                              !
!                                                                    !  
!  * Vertical settling of sediment in the water column.              !
!  * Erosive and depositional flux interactions of sediment          !
!    between water column and the bed.                               !
!                                                                    !  
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call sediment_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine sediment_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "bbl.h"
# include "forces.h"
# include "grid.h"
# include "mask.h"
# include "ocean.h"
# include "scalars.h"
# include "sediment.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Iter, Jend, Jstr, i, ised, j, k
      REAL_TYPE
     &        cff, cff1, cff2, aR, aL, cu, dep_flux, dtsed, ero_flux,
     &        tau_water, tot_flux
      REAL_TYPE
     &        FC(0:N), SED(NST,N), SED_bak(NST,N), dSED(0:N), oHz(N)
!
# include "set_bounds.h"
# ifdef PROFILE
!
!---------------------------------------------------------------------
!  Turn on sediment module time wall clock.
!---------------------------------------------------------------------
!
      call wclock_on (16)
# endif
!
!=====================================================================
!  Add sediment Source/Sink terms.
!=====================================================================
!
      dtsed=dt/FLOAT(SedIter)
      do j=Jstr,Jend
        do i=Istr,Iend
!
!  Compute inverse thicknesses.
!
          do k=1,N
            oHz(k)=1.0_r8/Hz(i,j,k)
!!          oHz(k)=1.0_r8
          enddo
!
!  Extract sediment variables from tracer arrays, place them into
!  scratch arrays, and restrict their values to be positive definite.
!  Here, the tracer are assume to be mutiplied by the thicknesses
!  (m Tunits).  This depends on where the call to this module is
!  placed in "step3d_t".
!
          do ised=1,NST
            do k=1,N
              SED(ised,k)=MAX(t(i,j,k,nnew,idsed(ised))*oHz(k),
     &                        0.0_r8)
              SED_bak(ised,k)=SED(ised,k)
            enddo
          enddo
!
!  Start internal iterations of sediment source/sink terms.
!
          do Iter=1,SedIter
            do ised=1,NST

!
!---------------------------------------------------------------------
!  Compute sediment vertical settling term:  Vertical advection
!  algorithm based on monotonic, continuous conservative parabolic
!  splines.
!---------------------------------------------------------------------
!
!  Construct parabolic splines:  Compute vertical derivatives for
!  SED at W-points.  Neumann boundary conditions are assumed on top
!  and bottom.
!
              FC(0)=0.0_r8
              dSED(0)=0.0_r8
              do k=1,N-1
                cff=1.0_r8/(2.0_r8*Hz(i,j,k+1)+
     &                      Hz(i,j,k)*(2.0_r8-FC(k-1)))
                FC(k)=cff*Hz(i,j,k+1)
                dSED(k)=cff*(6.0_r8*(SED(ised,k+1)-SED(ised,k))-
     &                       Hz(i,j,k)*dSED(k-1))
              enddo
              dSED(N)=0.0_r8
              do k=N-1,1,-1
                dSED(k)=dSED(k)-FC(k)*dSED(k+1)
              enddo
!
!  Convert vertical derivatives "dSED" into field values at grid box
!  interfaces assuming parabolic profiles within each grid box.
!  Restrict these values to lie between bounds determined from
!  box-averaged values of grid boxes adjscent from above and
!  below. This restriction is part of the PPM-like monotonization
!  procedure.
!
              cff=1.0_r8/3.0_r8
              dSED(0)=SED(ised,1)   ! -cff*Hz(i,j,1)*(dSED(0)+0.5_r8*dSED(1))
              dSED(N)=SED(ised,N)   ! +cff*Hz(i,j,N)*(dSED(N)+0.5_r8*dSED(N-1))
              do k=2,N
                dSED(k-1)=SED(ised,k)-
     &                    cff*Hz(i,j,k)*(0.5_r8*dSED(k)+dSED(k-1))
                dSED(k-1)=MAX(dSED(k-1),
     &                        MIN(SED(ised,k-1),SED(ised,k)))
                dSED(k-1)=MIN(dSED(k-1),
     &                        MAX(SED(ised,k-1),SED(ised,k)))
              enddo
!
!  Convert "dSED" into flux-integrated values;  complete PPM flux
!  limiting.  This procedure starts from assigning left and right
!  (aR,aL) values of the interpolating parabolae, then the
!  monotonicity conditions are checked and aL, aR are modified to
!  fit.
!
              do k=1,N
                FC(k)=dtsed/Hz(i,j,k)
                aR=dSED(k)
                aL=dSED(k-1)
                cff1=(aR-aL)*6.0_r8*(SED(ised,k)-0.5_r8*(aR+aL))
                cff2=(aR-aL)**2
                if ((aR-SED(ised,k))*(SED(ised,k)-aL).lt.0.0_r8) then
                  aL=SED(ised,k)
                  aR=SED(ised,k)
                elseif (cff1.gt.cff2) then
                  aL=3.0_r8*SED(ised,k)-2.0_r8*aR
                elseif (cff1.lt.-cff2) then
                  aR=3.0_r8*SED(ised,k)-2.0_r8*aL
                endif
                cu=Wsed(ised)*FC(k)
                dSED(k-1)=SED(ised,k)-(1.0_r8-cu)*(0.5_r8*(aR-aL)-
     &                    (0.5_r8*(aR+aL)-SED(ised,k))*
     &                    (1.0_r8-2.0_r8*cu))
              enddo
              dSED(N)=0.0_r8
!
!  Apply vertical fluxes.  Add back into bottom level what was settled
!  out of bottom level to conserve mass in the water column.
!
              do k=1,N
                SED(ised,k)=SED(ised,k)+
     &                      Wsed(ised)*FC(k)*(dSED(k)-dSED(k-1))
              enddo
              SED(ised,1)=SED(ised,1)+Wsed(ised)*FC(1)*dSED(0)
!          
!---------------------------------------------------------------------
!  Sediment resuspension or deposition.
!---------------------------------------------------------------------
!
!  Compute shear of water on the bed.
!
# ifdef BBL
              tau_water=SQRT((0.5_r8*(bustrw(i,j)+bustrw(i+1,j)))**2+
     &                       (0.5_r8*(bvstrw(i,j)+bvstrw(i,j+1)))**2)
# else
              tau_water=SQRT((0.5_r8*(bustr(i,j)+bustr(i+1,j)))**2+
     &                       (0.5_r8*(bvstr(i,j)+bvstr(i,j+1)))**2)
# endif
!
!  Compute upward flux (kg; positive) of sediment due to surface
!  erosion. Notice conversion from kg/m2/s to kg.
!
              ero_flux=0.0_r8
              if (bed(i,j,1,ithck).gt.0.0_r8) then
                ero_flux=dtsed*on_r(i,j)*om_r(i,j)*
     &                   Erate(ised)*(1.0_r8-poros(ised))*
     &                   (tau_water-tau_ce(ised))/tau_ce(ised)
                ero_flux=MAX(ero_flux,0.0_r8)
              endif
!
!  Compute downward flux (kg; negative) of sediment due to deposition
!  from settling velocity. Notice conversion m/s mg/l to kg.
!
              dep_flux=-dtsed*om_r(i,j)*on_r(i,j)*0.001_r8*
     &                 Wsed(ised)*SED(ised,1)
              dep_flux=MIN(dep_flux,0.0_r8)
!
!  Compute total net flux (kg).
!
              tot_flux=ero_flux+dep_flux
!
!  Add sediment resuspension or deposition contribution to bottom
!  level.  Notice conversion to mg/l.
!
              SED(ised,1)=(SED(ised,1)+
     &                     tot_flux*1000.0_r8/
     &                     (om_r(i,j)*on_r(i,j)*Hz(i,j,1)))
#   ifdef MASKING
     &                   *rmask(i,j)
#   endif
!
!  Compute new sediment top layer thickness (m).
!
              bed(i,j,1,ithck)=(bed(i,j,1,ithck)-
     &                          tot_flux/(Srho(ised)*
     &                                    (1.0_r8-poros(ised))*
     &                                    om_r(i,j)*on_r(i,j)))
#   ifdef MASKING
     &                        *rmask(i,j)
#   endif
!
!---------------------------------------------------------------------
!  Update global tracer variables (m Tunits).
!---------------------------------------------------------------------
!
              do k=1,N
                t(i,j,k,nnew,idsed(ised))=t(i,j,k,nnew,idsed(ised))+
     &                                    (SED(ised,k)-
     &                                     SED_bak(ised,k))/oHz(k)
              enddo
            enddo
          enddo
        enddo
      enddo
# ifdef PROFILE
!
!---------------------------------------------------------------------
!  Turn off sediment module time wall clock.
!---------------------------------------------------------------------
!
      call wclock_off (16)
# endif
#else
      subroutine sediment
#endif
      return
      end
