#include "cppdefs.h"
#ifdef SOLVE3D
      subroutine rho_eos (tile)
!
!=====================================================================
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine computes  "in situ" density and other associated     !
!  quantitites as a function of potential temperature,  salinity,    !
!  and pressure from a polynomial expression (Jackett & McDougall,   !
!  1992). The polynomial expression was found from fitting to 248    !
!  values  in the  oceanographic  ranges of  salinity,  potential    !
!  temperature,  and pressure.  It  assumes no pressure variation    !
!  along geopotential surfaces, that is, depth (meters; negative)    !
!  and pressure (dbar; assumed negative here) are interchangeable.   !
!                                                                    !
!  Check Values: (T=3 C, S=35.5 PSU, Z=-5000 m)                      !
!                                                                    !
!     alpha = 2.1014611551470d-04 (1/Celsius)                        !
!     beta  = 7.2575037309946d-04 (1/PSU)                            !
!     gamma = 3.9684764511766d-06 (1/Pa)                             !
!     den   = 1050.3639165364     (kg/m3)                            !
!     den1  = 1028.2845117925     (kg/m3)                            !
!     sound = 1548.8815240223     (m/s)                              !
!     bulk  = 23786.056026320     (Pa)                               !
!                                                                    !
!  Reference:                                                        !
!                                                                    !
!  Jackett, D. R. and T. J. McDougall, 1995, Minimal Adjustment of   !
!    Hydrostatic Profiles to Achieve Static Stability, J. of Atmos.  !
!    and Oceanic Techn., vol. 12, pp. 381-389.                       !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "scratch.h"
# include "tile.h"
!
# ifdef PROFILE
      call wclock_on (14)
# endif
# ifdef NONLIN_EOS
      call rho_eos_tile (Istr,Iend,Jstr,Jend,
     &                   A2d(1,1),A2d(1,2),A2d(1,3),A2d(1,4),
     &                   A2d(1,5),A2d(1,6),A2d(1,7),A2d(1,8),
     &                   A2d(1,9))
# else
      call rho_eos_tile (Istr,Iend,Jstr,Jend)
# endif
# ifdef PROFILE
      call wclock_off (14)
# endif
      return
      end
# ifdef NONLIN_EOS
!
!*********************************************************************
      subroutine rho_eos_tile (Istr,Iend,Jstr,Jend,den,den1,bulk,
     &                         bulk0,bulk1,bulk2,Scof,Tcof,wrk)
!*********************************************************************
!
      implicit none
#  include "param.h"
#  include "coupling.h"
#  include "grid.h"
#  include "ocean.h"
#  include "mask.h"
#  include "mixing.h"
#  include "eos_coef.h"
#  include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
      REAL_TYPE
     &        Tp, Tpr10, Ts, Tt, sqrtTs
      REAL_TYPE
     &        cff(23)
#  ifdef VAR_RHO_2D
      REAL_TYPE
     &        cff1, cff2
#  endif
#  ifdef BV_FREQUENCY
      REAL_TYPE
     &        bulk_dn, bulk_up, den_dn, den_up
#  endif
#  if defined LMD_SKPP || defined LMD_BKPP || defined BULK_FLUXES
      REAL_TYPE
     &        DbulkDS, DbulkDT, Dden1DS, Dden1DT
#  endif
#  ifdef MIX_ISO_TS
      REAL_TYPE
     &        DbulkDP, rhogamma, sound2
#  endif
      REAL_TYPE
     &         Scof(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &         Tcof(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &         bulk(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &        bulk0(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &        bulk1(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &        bulk2(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &          den(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &         den1(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &          wrk(PRIVATE_1D_SCRATCH_ARRAY,0:N)
!
#  include "set_bounds.h"
!
!=====================================================================
!  Nonlinear equation of state.  Notice that this equation of state
!  is only valid for potential temperature range of -2C to 40C and
!  a salinity range of 0 PSU to 42 PSU.
!=====================================================================
!
      do j=JstrR,JendR
        do k=1,N
          do i=IstrR,IendR
            Tt=MAX(-2.0_r8,t(i,j,k,nrhs,itemp))
#  ifdef SALINITY
            Ts=MAX(0.0_r8,t(i,j,k,nrhs,isalt))
            sqrtTs=SQRT(Ts)
#  else
            Ts=0.0_r8
            sqrtTs=0.0_r8
#  endif
            Tp=z_r(i,j,k)
            Tpr10=0.1_r8*Tp
!
!---------------------------------------------------------------------
!  Compute density (kg/m3) at standard one atmosphere pressure.
!---------------------------------------------------------------------
!
            cff(1)=Q00+Tt*(Q01+Tt*(Q02+Tt*(Q03+Tt*(Q04+Tt*Q05))))
            cff(2)=U00+Tt*(U01+Tt*(U02+Tt*(U03+Tt*U04)))
            cff(3)=V00+Tt*(V01+Tt*V02)
!
            den1(i,k)=cff(1)+Ts*(cff(2)+sqrtTs*cff(3)+Ts*W00)
!
!---------------------------------------------------------------------
!  Compute secant bulk modulus (bulk = bulk0 - bulk1*z + bulk2*z*z).
!---------------------------------------------------------------------
!
            cff(4)=A00+Tt*(A01+Tt*(A02+Tt*(A03+Tt*A04)))
            cff(5)=B00+Tt*(B01+Tt*(B02+Tt*B03))
            cff(6)=D00+Tt*(D01+Tt*D02)
            cff(7)=E00+Tt*(E01+Tt*(E02+Tt*E03))
            cff(8)=F00+Tt*(F01+Tt*F02)
            cff(9)=G01+Tt*(G02+Tt*G03)
            cff(10)=H00+Tt*(H01+Tt*H02)
!
            bulk0(i,k)=cff(4)+Ts*(cff(5)+sqrtTs*cff(6))
            bulk1(i,k)=cff(7)+Ts*(cff(8)+sqrtTs*G00)
            bulk2(i,k)=cff(9)+Ts*cff(10)
!
            bulk(i,k)=bulk0(i,k)-Tp*(bulk1(i,k)-Tp*bulk2(i,k))
!
!---------------------------------------------------------------------
!  Compute "in situ" density anomaly (kg/m3).
!---------------------------------------------------------------------
!
            den(i,k)=(den1(i,k)*bulk(i,k))/(bulk(i,k)+Tpr10)
            rho(i,j,k)=(den(i,k)-1000.0_r8)
!!          rho(i,j,k)=(den1(i,k)-1000.0_r8)
#  ifdef MASKING
     &                *rmask(i,j)
#  endif
#  if defined LMD_SKPP || defined LMD_BKPP
!
!---------------------------------------------------------------------
!  Compute "in situ" density anomaly (kg/m3) referenced to the
!  surface.
!---------------------------------------------------------------------
!
            pden(i,j,k)=(den1(i,k)-1000.0_r8)
#   ifdef MASKING
     &                 *rmask(i,j)
#   endif
#  endif /* LMD_SKPP || LMD_BKPP */
          enddo
        enddo
#  ifdef VAR_RHO_2D
!
!---------------------------------------------------------------------
!  Compute vertical averaged density and density perturbation used
!  in barotropic pressure gradient.
!---------------------------------------------------------------------
!
        do i=IstrR,IendR
          cff1=rho(i,j,N)*Hz(i,j,N)
          rhoS(i,j)=0.5_r8*cff1*Hz(i,j,N)
          rhoA(i,j)=cff1
        enddo
        do k=N-1,1,-1
          do i=IstrR,IendR
            cff1=rho(i,j,k)*Hz(i,j,k)
            rhoS(i,j)=rhoS(i,j)+Hz(i,j,k)*(rhoA(i,j)+0.5_r8*cff1)
            rhoA(i,j)=rhoA(i,j)+cff1
          enddo
        enddo
        cff2=1.0_r8/rho0
        do i=IstrR,IendR
          cff1=1.0_r8/(z_w(i,j,N)-z_w(i,j,0))
          rhoA(i,j)=cff2*cff1*rhoA(i,j)
          rhoS(i,j)=2.0_r8*cff1*cff1*cff2*rhoS(i,j)
        enddo
#  endif /* VAR_RHO_2D */
#  ifdef BV_FREQUENCY
!
!---------------------------------------------------------------------
!  Compute Brunt-Vaisala frequency (1/s2) at horizontal RHO-points
!  and vertical W-points:
!
!                  bvf = - g/rho d(rho)/d(z).
!
!  The density anomaly difference is computed by lowering/rising the
!  water parcel above/below adiabatically at W-point depth "z_w".
!---------------------------------------------------------------------
!
        do k=1,Nm
          do i=IstrR,IendR
            bulk_up=bulk0(i,k+1)-
     &              z_w(i,j,k)*(bulk1(i,k+1)-
     &                          bulk2(i,k+1)*z_w(i,j,k))
            bulk_dn=bulk0(i,k  )-
     &              z_w(i,j,k)*(bulk1(i,k  )-
     &                          bulk2(i,k  )*z_w(i,j,k))
            den_up=(den1(i,k+1)*bulk_up)/(bulk_up+0.1_r8*z_w(i,j,k))
            den_dn=(den1(i,k  )*bulk_dn)/(bulk_dn+0.1_r8*z_w(i,j,k))
            bvf(i,j,k)=-g*(den_up-den_dn)/
     &                 (0.5_r8*(den_up+den_dn)*
     &                  (z_r(i,j,k+1)-z_r(i,j,k)))
          enddo
        enddo
        do i=IstrR,IendR
!!        bvf(i,j,0)=bvf(i,j,1)
!!        bvf(i,j,N)=bvf(i,j,Nm)
          bvf(i,j,0)=0.0_r8
          bvf(i,j,N)=0.0_r8
        enddo
#  endif /* BV_FREQUENCY */
#  if defined LMD_SKPP || defined LMD_BKPP || defined BULK_FLUXES
!
!---------------------------------------------------------------------
!  Compute thermal expansion (1/Celsius) and saline contraction
!  (1/PSU) coefficients.
!---------------------------------------------------------------------
!
#   ifdef LMD_DDMIX
        do k=1,N
#   else
        k=N
#   endif /* LMD_DDMIX */
          do i=IstrR,IendR
            Tt=MAX(-2.0_r8,t(i,j,k,nrhs,itemp))
#  ifdef SALINITY
            Ts=MAX(0.0_r8,t(i,j,k,nrhs,isalt))
            sqrtTs=SQRT(Ts)
#   else
            Ts=0.0_r8
            sqrtTs=0.0_r8
#   endif
            Tp=z_r(i,j,k)
            Tpr10=0.1_r8*Tp
!
!  Compute d(den1)/d(S) and d(den1)/d(T) derivatives.
!
            cff(2)=U00+Tt*(U01+Tt*(U02+Tt*(U03+Tt*U04)))
            cff(3)=V00+Tt*(V01+Tt*V02)
            cff(11)=Q01+Tt*(2.0_r8*Q02+
     &                      Tt*(3.0_r8*Q03+
     &                          Tt*(4.0_r8*Q04+Tt*5.0_r8*Q05)))
            cff(12)=U01+Tt*(2.0_r8*U02+
     &                      Tt*(3.0_r8*U03+Tt*4.0_r8*U04))
            cff(13)=V01+Tt*2.0_r8*V02
!
            Dden1DS=cff(2)+1.5_r8*cff(3)*sqrtTs+2.0_r8*W00*Ts
            Dden1DT=cff(11)+Ts*(cff(12)+sqrtTs*cff(13))
!
!  Compute d(bulk)/d(S) and d(bulk)/d(T) derivatives.
!
            cff(5)=B00+Tt*(B01+Tt*(B02+Tt*B03))
            cff(6)=D00+Tt*(D01+Tt*D02)
            cff(8)=F00+Tt*(F01+Tt*F02)
            cff(10)=H00+Tt*(H01+Tt*H02)
            cff(14)=A01+Tt*(2.0_r8*A02+Tt*(3.0_r8*A03+Tt*4.0_r8*A04))
            cff(15)=B01+Tt*(2.0_r8*B02+Tt*3.0_r8*B03)
            cff(16)=D01+Tt*2.0_r8*D02
            cff(17)=E01+Tt*(2.0_r8*E02+Tt*3.0_r8*E03)
            cff(18)=F01+Tt*2.0_r8*F02
            cff(19)=G02+Tt*2.0_r8*G03
            cff(20)=H01+Tt*2.0_r8*H02
!
            DbulkDS=cff(5)+sqrtTs*1.5_r8*cff(6)-
     &              Tp*(cff(8)+sqrtTs*1.5_r8*G00-Tp*cff(10))
            DbulkDT=cff(14)+Ts*(cff(15)+sqrtTs*cff(16))-
     &              Tp*(cff(17)+Ts*cff(18)-Tp*(cff(19)+Ts*cff(20)))
!
!  Compute thermal expansion and saline contraction coefficients.
!
            cff(21)=Tpr10*den1(i,k)
            cff(22)=bulk(i,k)*(bulk(i,k)+Tpr10)
            wrk(i,k)=den(i,k)*(bulk(i,k)+Tpr10)*(bulk(i,k)+Tpr10)
            Tcof(i,k)=-(DbulkDT*cff(21)+Dden1DT*cff(22))
            Scof(i,k)= (DbulkDS*cff(21)+Dden1DS*cff(22))
#   ifdef LMD_DDMIX
            alfaobeta(i,j,k)=Tcof(i,k)/Scof(i,k)
#   endif /* LMD_DDMIX */
          enddo
          if (k.eq.N) then
            do i=IstrR,IendR
              alpha(i,j)=Tcof(i,N)/wrk(i,N)
              beta (i,j)=Scof(i,N)/wrk(i,N)
            enddo
          endif
#   ifdef LMD_DDMIX
        enddo
#   endif /* LMD_DDMIX */
#  endif /* LMD_SKPP || LMD_BKPP || BULK_FLUXES */
#  if defined MIX_ISO_TS || defined GENT_McWILLIAMS
!
!---------------------------------------------------------------------
!  Compute factor to convert isopycnal, "in situ" density slopes to
!  to isopycnal, neutral slopes:
!
!    neutral = [1 + g^2 / (c^2 * bvf)];    c^2 = 10000/(rho*gamma)
!
!  were "c" is the sound speed (m/s) and "gamma" is the adiabatic and
!  isentropic compressibility coefficient (1/Pa).
!---------------------------------------------------------------------
!
        do k=1,N
          do i=IstrR,IendR
            Tp=z_r(i,j,k)
            Tpr10=0.1_r8*Tp
            cff(23)=(bulk(i,k)+Tpr10)*(bulk(i,k)+Tpr10)
            DbulkDP=-bulk1(i,k)+Tp*2.0_r8*bulk2(i,k)
            rhogamma=ABS(0.1_r8*den1(i,k)*
     &                   ((Tp*DbulkDP-bulk(i,k))/cff(23)))
            sound2=10000.0_r8/rhogamma
            neutral(i,j,k)=1.0_r8+g*g/(sound2*bvf(i,j,k))
          enddo
        enddo
#  endif /* MIX_ISO_TS || GENT_McWILLIAMS */
      enddo
# endif /* NONLIN_EOS */
# ifndef NONLIN_EOS
!
!*********************************************************************
      subroutine rho_eos_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
#  include "param.h"
#  include "coupling.h"
#  include "grid.h"
#  include "mask.h"
#  include "mixing.h"
#  include "ocean.h"
#  include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
#  ifdef VAR_RHO_2D
      REAL_TYPE
     &        cff1, cff2
#  endif
#  ifdef LMD_DDMIX
      REAL_TYPE
     &        cff
#  endif
!
#  include "set_bounds.h"
!
!=====================================================================
!  Linear equation of state.
!=====================================================================
!
!---------------------------------------------------------------------
!  Compute "in situ" density anomaly (kg/m3).
!---------------------------------------------------------------------
!
      do j=JstrR,JendR
        do k=1,N
          do i=IstrR,IendR
            rho(i,j,k)=R0+Tcoef*t(i,j,k,nrhs,itemp)
#  ifdef SALINITY
     &                   +Scoef*t(i,j,k,nrhs,isalt)
#  endif
#  ifdef MASKING
            rho(i,j,k)=rho(i,j,k)*rmask(i,j)
#  endif
#  if defined LMD_SKPP || defined LMD_BKPP
            pden(i,j,k)=rho(i,j,k)
#  endif
          enddo
        enddo
#  ifdef VAR_RHO_2D
!
!---------------------------------------------------------------------
!  Compute vertical averaged density and density perturbation used
!  in barotropic pressure gradient.
!---------------------------------------------------------------------
!
        do i=IstrR,IendR
          cff1=rho(i,j,N)*Hz(i,j,N)
          rhoS(i,j)=0.5_r8*cff1*Hz(i,j,N)
          rhoA(i,j)=cff1
        enddo
        do k=N-1,1,-1
          do i=IstrR,IendR
            cff1=rho(i,j,k)*Hz(i,j,k)
            rhoS(i,j)=rhoS(i,j)+Hz(i,j,k)*(rhoA(i,j)+0.5_r8*cff1)
            rhoA(i,j)=rhoA(i,j)+cff1
          enddo
        enddo
        cff2=1.0_r8/rho0
        do i=IstrR,IendR
          cff1=1.0_r8/(z_w(i,j,N)-z_w(i,j,0))
          rhoA(i,j)=cff2*cff1*rhoA(i,j)
          rhoS(i,j)=2.0_r8*cff1*cff1*cff2*rhoS(i,j)
        enddo
#  endif /* VAR_RHO_2D */
#  ifdef BV_FREQUENCY
!
!---------------------------------------------------------------------
!  Compute Brunt-Vaisala frequency (1/s2) at horizontal RHO-points
!  and vertical W-points.
!---------------------------------------------------------------------
!
        do k=1,Nm
          do i=IstrR,IendR
            bvf(i,j,k)=-gorho0*(rho(i,j,k+1)-rho(i,j,k))/
     &                         (z_r(i,j,k+1)-z_r(i,j,k))
          enddo
        enddo
#  endif /* BV_FREQUENCY */
#  if defined LMD_SKPP || defined LMD_BKPP || defined BULK_FLUXES
!
!---------------------------------------------------------------------
!  Compute thermal expansion (1/Celsius) and saline contraction
!  (1/PSU) coefficients.
!---------------------------------------------------------------------
!
        do i=IstrR,IendR
          alpha(i,j)=ABS(Tcoef)
#   ifdef SALINITY
          beta(i,j)=ABS(Scoef)
#   else
          beta(i,j)=0.0_r8
#   endif
        enddo
#   ifdef LMD_DDMIX
!
!  Compute ratio of thermal expansion and saline contraction
!  coefficients.
!
        if (Scoef.eq.0.0_r8) then
          cff=1.0_r8
        else
          cff=1.0_r8/Scoef
        endif
        do k=1,N
          do i=IstrR,IendR
            alfaobeta(i,j,k)=cff*Tcoef
          enddo
        enddo
#   endif /* LMD_DDMIX */
#  endif /* LMD_SKPP || LMD_BKPP || BULK_FLUXES */
      enddo
# endif /* !NONLIN_EOS */
#else
      subroutine rho_eos
#endif
      return
      end
