#include "cppdefs.h"
#ifdef SOLVE3D
# undef NEUMANN
      subroutine prsgrd (tile)
!
!======================================== Alexander F. Shchepetkin ===
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!=====================================================================
!                                                                    !
!  This subroutine evaluates the baroclinic,  hydrostatic pressure   !
!  gradient term using a  finite-volume  pressure Jacobian scheme.   !
!  The scheme is based on local, conservative, limited-oscillation   !
!  vertical quartic polynomial reconstruction of density field and   !
!  subsequent  projection fits of the derivatives of  density into   !
!  isosurface of vertical coordinate.  The monotonicity constraint   !
!  uses a PPM-style limitting algorithm.                             !
!                                                                    !
!  The pressure gradient terms (m4/s2) are loaded into right-hand-   !
!  side arrays "ru" and "rv".                                        !
!                                                                    !
!  Reference:                                                        !
!                                                                    !
!    Shchepetkin A.F and J.C. McWilliams, 2001:  A method for        !
!      computing horizontal pressure gradient force in an ocean      !
!      model with non-aligned vertical coordinate.  DRAFT            !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "scratch.h"
# include "tile.h"
!
# ifdef PROFILE
      call wclock_on (23)
# endif
      call prsgrd_tile (Istr,Iend,Jstr,Jend,
     &                  A3d(1,1),A3d(1,2),A3d(1,5),A2d(1,1),
     &                  A2d(1,2),A2d(1,3),A2d(1,4), A2d(1,5))
# ifdef PROFILE
      call wclock_off (23)
# endif
      return
      end
!
!*********************************************************************
      subroutine prsgrd_tile (Istr,Iend,Jstr,Jend,P,r,FX,FC,aR,dR,
     &                        aL,dL)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "ocean.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
      REAL_TYPE
     &        cff, cff1, cff2, cffL, cffR, deltaL, deltaR, dh, dP,
     &        eps, rr
      REAL_TYPE
     &       FX(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &        P(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &        r(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &       FC(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &       aL(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &       aR(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &       dL(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &       dR(PRIVATE_1D_SCRATCH_ARRAY,0:N)
      parameter (eps=1.0_e8-8)
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Finite-volume pressure gradient force algorithm.
!---------------------------------------------------------------------
!
      cff2=1.0_r8/6.0_r8
      do j=JstrV-2,Jend+1
        do k=N-1,1,-1
          do i=IstrU-2,Iend+1
            FC(i,k)=(rho(i,j,k+1)-rho(i,j,k))/(Hz(i,j,k+1)+Hz(i,j,k))
          enddo
        enddo
!
!  Parabolic WENO reconstruction of density field. Compute left and
!  right side limits aL and aR for the density assuming monotonized
!  parabolic distributions within each grid box.  Also compute dL and
!  dR which are used as a measure of quadratic variation during
!  subsquent WENO reconciliation of side limits.
!
        do k=2,N-1
          do i=IstrU-2,Iend+1
            deltaR=Hz(i,j,k)*FC(i,k)
            deltaL=Hz(i,j,k)*FC(i,k-1)
            if ((deltaR*deltaL).lt.0.0_r8) then
              deltaR=0.0_r8
              deltaL=0.0_r8
            endif
            cff=Hz(i,j,k-1)+2.0_r8*Hz(i,j,k)+Hz(i,j,k+1)
            cffR=cff*FC(i,k)
            cffL=cff*FC(i,k-1)
            if (ABS(deltaR).gt.ABS(cffL)) deltaR=cffL
            if (ABS(deltaL).gt.ABS(cffR)) deltaL=cffR
            cff=(deltaR-deltaL)/(Hz(i,j,k-1)+Hz(i,j,k)+Hz(i,j,k+1))
            deltaR=deltaR-cff*Hz(i,j,k+1)
            deltaL=deltaL+cff*Hz(i,j,k-1)
            aR(i,k)=rho(i,j,k)+deltaR
            aL(i,k)=rho(i,j,k)-deltaL
            dR(i,k)=(2.0_r8*deltaR-deltaL)**2
            dL(i,k)=(2.0_r8*deltaL-deltaR)**2
          enddo
        enddo
!
        do i=IstrU-2,Iend+1
          aL(i,N)=aR(i,N-1)
          aR(i,N)=2.0_r8*rho(i,j,N)-aL(i,N)
          dR(i,N)=(2.0_r8*aR(i,N)+aL(i,N)-3.0_r8*rho(i,j,N))**2
          dL(i,N)=(3.0_r8*rho(i,j,N)-2.0_r8*aL(i,N)-aR(i,N))**2
          aR(i,1)=aL(i,2)
          aL(i,1)=2.0_r8*rho(i,j,1)-aR(i,1)
          dR(i,1)=(2.0_r8*aR(i,1)+aL(i,1)-3.0_r8*rho(i,j,1))**2
          dL(i,1)=(3.0_r8*rho(i,j,1)-2.0_r8*aL(i,1)-aR(i,1))**2
        enddo
!
        do k=1,N-1
          do i=IstrU-2,Iend+1
             deltaL=MAX(dL(i,k  ),eps)
             deltaR=MAX(dR(i,k+1),eps)
             r(i,j,k)=(deltaR*aR(i,k)+deltaL*aL(i,k+1))/
     &                (deltaR+deltaL)
          enddo
        enddo
!
        do i=IstrU-2,Iend+1
# ifdef NEUMANN
          r(i,j,N)=1.5_r8*rho(i,j,N)-0.5_r8*r(i,j,N-1)
          r(i,j,0)=1.5_r8*rho(i,j,1)-0.5_r8*r(i,j,1  )
# else
          r(i,j,N)=2.0_r8*rho(i,j,N)-r(i,j,N-1)
          r(i,j,0)=2.0_r8*rho(i,j,1)-r(i,j,1  )
# endif
        enddo
!
!  Compute pressure (P) and lateral pressure force (FX). Initialize
!  pressure at the free-surface as zero
!
        do i=IstrU-2,Iend+1
          P(i,j,N)=0.0_r8
        enddo
        do k=N,1,-1
          do i=IstrU-2,Iend+1
            P(i,j,k-1)=P(i,j,k)+Hz(i,j,k)*rho(i,j,k)
            deltaR=r(i,j,k)-rho(i,j,k)
            deltaL=rho(i,j,k)-r(i,j,k-1)
            if ((deltaR*deltaL).lt.0.0_r8) then
              rr=0.0_r8
            elseif (ABS(deltaR).gt.(2.0_r8*ABS(deltaL))) then
              rr=3.0_r8*deltaL
            elseif (ABS(deltaL).gt.(2.0_r8*ABS(deltaR))) then
              rr=3.0_r8*deltaR
            else
              rr=deltaR+deltaL
            endif
            FX(i,j,k)=0.5_r8*Hz(i,j,k)*
     &                (P(i,j,k)+P(i,j,k-1)+cff2*rr*Hz(i,j,k))
          enddo
        enddo
!
!  Compute net pressure gradient forces in the XI-directions.
!  Set pressure at free-surface as zero.
!
        if ((j.ge.Jstr).and.(j.le.Jend)) then
          do i=IstrU-1,Iend+1
            FC(i,N)=0.0_r8
          enddo
          do k=N,1,-1
            do i=IstrU-1,Iend+1
              dP=P(i-1,j,k-1)-P(i,j,k-1)
              dh=z_w(i,j,k-1)-z_w(i-1,j,k-1)
              deltaR=dh*r(i,j,k-1)-dP
              deltaL=dP-dh*r(i-1,j,k-1)
              if ((deltaR*deltaL).lt.0.0_r8) then
                rr=0.0_r8
              elseif (ABS(deltaR).gt.(2.0_r8*ABS(deltaL))) then
                rr=3.0_r8*deltaL
              elseif (ABS(deltaL).gt.(2.0_r8*ABS(deltaR))) then
                rr=3.0_r8*deltaR
              else
                rr=deltaR+deltaL
              endif
              FC(i,k-1)=0.5_r8*dh*(P(i,j,k-1)+P(i-1,j,k-1)+cff2*rr)
              ru(i,j,k,nrhs)=2.0_r8*(FX(i-1,j,k)-FX(i,j,k)+
     &                               FC(i,k)-FC(i,k-1))/
     &                       (Hz(i-1,j,k)+Hz(i,j,k))
# ifdef MASKING
     &                      *umask(i,j)
# endif
            enddo
          enddo
        endif
!
!  Compute net pressure gradient forces in the ETA-directions.
!  Set pressure at free-surface as zero.
!
        if (j.ge.JstrV-1) then
          do i=Istr,Iend
            FC(i,N)=0.0_r8
          enddo
          do k=N,1,-1
            do i=Istr,Iend
              dP=P(i,j-1,k-1)-P(i,j,k-1)
              dh=z_w(i,j,k-1)-z_w(i,j-1,k-1)
              deltaR=dh*r(i,j,k-1)-dP
              deltaL=dP-dh*r(i,j-1,k-1)
              if ((deltaR*deltaL).lt.0.0_r8) then
                rr=0.0_r8
              elseif (ABS(deltaR).gt.(2.0_r8*ABS(deltaL))) then
                rr=3.0_r8*deltaL
              elseif (ABS(deltaL).gt.(2.0_r8*ABS(deltaR))) then
                rr=3.0_r8*deltaR
              else
                rr=deltaR+deltaL
              endif
              FC(i,k-1)=0.5_r8*dh*(P(i,j,k-1)+P(i,j-1,k-1)+cff2*rr)
              rv(i,j,k,nrhs)=2.0_r8*(FX(i,j-1,k)-FX(i,j,k)+
     &                               FC(i,k)-FC(i,k-1))/
     &                       (Hz(i,j-1,k)+Hz(i,j,k))
# ifdef MASKING
     &                      *vmask(i,j)
# endif
            enddo
          enddo
        endif
      enddo
!
      rr=g/(24.0_r8*rho0)
      cff=0.5_r8*g
      cff1=0.5_r8*g/rho0
      do j=Jstr,Jend
        do k=N-1,1,-1
          do i=IstrU,Iend
            dh=rr*(z_w(i,j,k)-z_w(i-1,j,k))
            FC(i,k)=MAX(dh,0.0_r8)*
     &                 (ru(i,j,k+1,nrhs)+ru(i+1,j,k  ,nrhs)-
     &                  ru(i,j,k  ,nrhs)-ru(i-1,j,k+1,nrhs))+
     &              MIN(dh,0.0_r8)*
     &                 (ru(i,j,k  ,nrhs)+ru(i+1,j,k+1,nrhs)-
     &                  ru(i,j,k+1,nrhs)-ru(i-1,j,k  ,nrhs))
          enddo
        enddo
        do i=IstrU,Iend
          FC(i,N)=0.0_r8
          dh=rr*(z_w(i,j,0)-z_w(i-1,j,0))
          FC(i,0)=MAX(dh,0.0_r8)*
     &               (ru(i  ,j,1,nrhs)-ru(i-1,j,1,nrhs))+
     &            MIN(dh,0.0_r8)*
     &               (ru(i+1,j,1,nrhs)-ru(i  ,j,1,nrhs))
        enddo
        do k=1,N
          do i=IstrU,Iend
            ru(i,j,k,nrhs)=(cff*(z_w(i-1,j,N)-z_w(i,j,N))+
     &                      cff1*ru(i,j,k,nrhs))*
     &                     (Hz(i-1,j,k)+Hz(i,j,k))*on_u(i,j)+
     &                     (FC(i,k)-FC(i,k-1))*on_u(i,j)
          enddo
        enddo
      enddo
!
      do j=JstrV,Jend
        do k=N-1,1,-1
          do i=Istr,Iend
            dh=rr*(z_w(i,j,k)-z_w(i,j-1,k))
            FX(i,j,k)=MAX(dh,0.0_r8)*
     &                   (rv(i,j,k+1,nrhs)+rv(i+1,j  ,k  ,nrhs)-
     &                    rv(i,j,k  ,nrhs)-rv(i  ,j-1,k+1,nrhs))+
     &                MIN(dh,0.0_r8)*
     &                   (rv(i,j,k  ,nrhs)+rv(i+1,j  ,k+1,nrhs)-
     &                    rv(i,j,k+1,nrhs)-rv(i  ,j-1,k  ,nrhs))
          enddo
        enddo
        do i=Istr,Iend
          FX(i,j,N)=0.0_r8
          dh=rr*(z_w(i,j,0)-z_w(i,j-1,0))
          FX(i,j,0)=MAX(dh,0.0_r8)*
     &                 (rv(i  ,j,1,nrhs)-rv(i,j-1,1,nrhs))+
     &              MIN(dh,0.0_r8)*
     &                 (rv(i+1,j,1,nrhs)-rv(i,j  ,1,nrhs))
        enddo
      enddo
      do j=JstrV,Jend
        do k=1,N
          do i=Istr,Iend
            rv(i,j,k,nrhs)=(cff*(z_w(i,j-1,N)-z_w(i,j,N))+
     &                      cff1*rv(i,j,k,nrhs))*
     &                     (Hz(i,j-1,k)+Hz(i,j,k))*om_v(i,j)+
     &                     (FX(i,j,k)-FX(i,j,k-1))*om_v(i,j)
          enddo
        enddo
      enddo
      return
      end
#else
      subroutine prsgrd
      return
      end
#endif
