#include "cppdefs.h"
#ifdef SOLVE3D
# define NEUMANN
      subroutine pre_step3d (tile)
!
!======================================== Alexander F. Shchepetkin ===
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This subroutine initialize computations for new time step of the  !
!  3D primitive variables.                                           !
!                                                                    !
!  Both n-1 and n-2 time-step contributions of the  Adams/Bashforth  !
!  scheme are added here to u and v at time index "nnew", since the  !
!  right-hand-side  arrays ru and rv at  n-2  will be overwriten in  !
!  subsequent calls to routines within the 3D engine.                !
!                                                                    !
!  It also computes the time  "n"  vertical viscosity and diffusion  !
!  contributions of the Crank-Nicholson implicit scheme because the  !
!  thicknesses "Hz" will be overwriten at the end of the  2D engine  !
!  (barotropic mode) computations.                                   !
!                                                                    !
!  The actual time step will be carried out in routines "step3d_uv"  !
!  and "step3d_t".                                                   !
!                                                                    !
!====================================================================!
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "scratch.h"
# include "tile.h"
!
# ifdef PROFILE
      call wclock_on (22)
# endif
      call pre_step3d_tile (Istr,Iend,Jstr,Jend,
     &                      A2d(1,1),A2d(1,2),A2d(1,1),A2d(1,2),
     &                      A2d(1,3),A2d(1,4),A2d(1,5),A3d(1,1))
# ifdef PROFILE
      call wclock_off (22)
# endif
      return
      end
!
!*********************************************************************
      subroutine pre_step3d_tile (Istr,Iend,Jstr,Jend,FC,DC,FX,FE,CF,
     &                            grad,curv,swdk)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "forces.h"
# include "grid.h"
# include "mask.h"
# include "mixing.h"
# include "ocean.h"
# include "scalars.h"
# include "sources.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, indx, itrc, j, k, ltrc
# ifdef TS_PSOURCE
      INTEGER_TYPE
     &        is
# endif
!
      REAL_TYPE
     &        Gamma, cff, cff1, cff2, cff3, eps
      REAL_TYPE
     &          CF(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &          DC(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &          FC(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &          FE(PRIVATE_2D_SCRATCH_ARRAY),
     &          FX(PRIVATE_2D_SCRATCH_ARRAY),
     &        curv(PRIVATE_2D_SCRATCH_ARRAY),
     &        grad(PRIVATE_2D_SCRATCH_ARRAY),
     &        swdk(PRIVATE_2D_SCRATCH_ARRAY,0:N)
      parameter (Gamma=1.0_r8/6.0_r8, eps=1.0_e8-16)
!
# include "set_bounds.h"
# ifndef DIAGNOSTIC
!
!=====================================================================
!  Tracer equation(s).
!=====================================================================
!
#  ifdef SOLAR_SOURCE
!
!  Compute fraction of the solar shortwave radiation, "swdk"
!  (at vertical W-points) penetrating water column.
!
      do k=1,Nm
        do j=Jstr,Jend
          do i=Istr,Iend
            FX(i,j)=z_w(i,j,N)-z_w(i,j,k)
          enddo
        enddo
        call lmd_swfrac_tile (Istr,Iend,Jstr,Jend,-1.0_r8,FX,FE)
        do j=Jstr,Jend
          do i=Istr,Iend
             swdk(i,j,k)=FE(i,j)
          enddo
        enddo
      enddo
#  endif /* SOLAR_SOURCE */
!
!---------------------------------------------------------------------
!  Compute intermediate tracer at n+1/2 time-step, t(i,j,k,3,itrc).
!---------------------------------------------------------------------
!
!  Compute time rate of change of intermediate tracer due to
!  horizontal advection.
!
      do itrc=1,NT
        do k=1,N
#  ifdef TS_C2HADVECTION
          do j=Jstr,Jend
            do i=Istr,Iend+1
              FX(i,j)=Huon(i,j,k)*
     &                0.5_r8*(t(i-1,j,k,nstp,itrc)+t(i,j,k,nstp,itrc))
            enddo
          enddo
          do j=Jstr,Jend+1
            do i=Istr,Iend
              FE(i,j)=Hvom(i,j,k)*
     &                0.5_r8*(t(i,j-1,k,nstp,itrc)+t(i,j,k,nstp,itrc))
            enddo
          enddo
#  else
#   ifdef EW_PERIODIC
#    define I_RANGE Istr-1,Iend+2
#   else
#    define I_RANGE MAX(Istr-1,1),MIN(Iend+2,Lm+1)
#   endif
          do j=Jstr,Jend
            do i=I_RANGE
              FX(i,j)=(t(i,j,k,nstp,itrc)-t(i-1,j,k,nstp,itrc))
#   ifdef MASKING
     &               *umask(i,j)
#   endif
            enddo
          enddo
#   undef I_RANGE
#   ifndef EW_PERIODIC
          if (WESTERN_EDGE) then
            do j=Jstr,Jend
              FX(Istr-1,j)=FX(Istr,j)
            enddo
          endif
          if (EASTERN_EDGE) then
            do j=Jstr,Jend
              FX(Iend+2,j)=FX(Iend+1,j)
            enddo
          endif
#   endif
!
          do j=Jstr,Jend
            do i=Istr-1,Iend+1
#   if defined TS_U3HADVECTION
              curv(i,j)=FX(i+1,j)-FX(i,j)
#   elif defined TS_A4HADVECTION
              cff=2.0_r8*FX(i+1,j)*FX(i,j)
              if (cff.gt.eps) then
                grad(i,j)=cff/(FX(i+1,j)+FX(i,j))
              else
                grad(i,j)=0.0_r8
              endif
#   else
              grad(i,j)=0.5_r8*(FX(i+1,j)+FX(i,j))
#   endif
            enddo
          enddo
!
          cff1=1.0_r8/6.0_r8
          cff2=1.0_r8/3.0_r8
          do j=Jstr,Jend
            do i=Istr,Iend+1
#   ifdef TS_U3HADVECTION
              FX(i,j)=Huon(i,j,k)*0.5_r8*
     &                (t(i-1,j,k,nstp,itrc)+t(i,j,k,nstp,itrc))-
     &                cff1*(curv(i-1,j)*MAX(Huon(i,j,k),0.0_r8)+
     &                      curv(i  ,j)*MIN(Huon(i,j,k),0.0_r8))
#   else
              FX(i,j)=Huon(i,j,k)*0.5_r8*
     &                (t(i-1,j,k,nstp,itrc)+t(i,j,k,nstp,itrc)-
     &                 cff2*(grad(i,j)-grad(i-1,j)))
#   endif
            enddo
          enddo
!
#   ifdef NS_PERIODIC
#    define J_RANGE Jstr-1,Jend+2
#   else
#    define J_RANGE MAX(Jstr-1,1),MIN(Jend+2,Mm+1)
#   endif
          do j=J_RANGE
            do i=Istr,Iend
              FE(i,j)=(t(i,j,k,nstp,itrc)-t(i,j-1,k,nstp,itrc))
#   ifdef MASKING
     &               *vmask(i,j)
#   endif
            enddo
          enddo
#   undef J_RANGE
#   ifndef NS_PERIODIC
          if (SOUTHERN_EDGE) then
            do i=Istr,Iend
              FE(i,Jstr-1)=FE(i,Jstr)
            enddo
          endif
          if (NORTHERN_EDGE) then
            do i=Istr,Iend
              FE(i,Jend+2)=FE(i,Jend+1)
            enddo
          endif
#   endif
!
          do j=Jstr-1,Jend+1
            do i=Istr,Iend
#   if defined TS_U3HADVECTION
              curv(i,j)=FE(i,j+1)-FE(i,j)
#   elif defined TS_A4HADVECTION
              cff=2.0_r8*FE(i,j+1)*FE(i,j)
              if (cff.gt.eps) then
                grad(i,j)=cff/(FE(i,j+1)+FE(i,j))
              else
                grad(i,j)=0.0_r8
              endif
#   else
              grad(i,j)=0.5_r8*(FE(i,j+1)+FE(i,j))
#   endif
            enddo
          enddo
!
          cff1=1.0_r8/6.0_r8
          cff2=1.0_r8/3.0_r8
          do j=Jstr,Jend+1
            do i=Istr,Iend
#   ifdef TS_U3HADVECTION
              FE(i,j)=Hvom(i,j,k)*0.5_r8*
     &                (t(i,j-1,k,nstp,itrc)+t(i,j,k,nstp,itrc))-
     &                cff1*(curv(i,j-1)*MAX(Hvom(i,j,k),0.0_r8)+
     &                      curv(i,j  )*MIN(Hvom(i,j,k),0.0_r8))
#   else
              FE(i,j)=Hvom(i,j,k)*0.5_r8*
     &                (t(i,j-1,k,nstp,itrc)+t(i,j,k,nstp,itrc)-
     &                 cff2*(grad(i,j)-grad(i,j-1)))
#   endif
            enddo
          enddo
#  endif
#  ifdef TS_PSOURCE
!
!  Apply tracers point sources to the horizontal advection terms.
!
          do is=1,Nsrc
            i=Isrc(is)
            j=Jsrc(is)
            if ((INT(Dsrc(is)).eq.0).and.(Lsrc(is,itrc))) then
              if (((Istr.le.i).and.(i.le.Iend+1)).and.
     &            ((Jstr.le.j).and.(j.le.Jend))) then
                FX(i,j)=Huon(i,j,k)*Tsrc(is,k,itrc)
              endif
            elseif ((INT(Dsrc(is)).eq.1).and.(Lsrc(is,itrc))) then
              if (((Istr.le.i).and.(i.le.Iend)).and.
     &            ((Jstr.le.j).and.(j.le.Jend+1))) then
                FE(i,j)=Hvom(i,j,k)*Tsrc(is,k,itrc)
              endif
            endif
          enddo
#  endif /* TS_PSOURCE */
!
!  Time-step horizontal advection (m Tunits).
!
          if (iic.eq.ntfirst) then
            cff=0.5_r8*dt
            cff1=1.0_r8
            cff2=0.0_r8
          else
            cff=(1.0_r8-Gamma)*dt
            cff1=0.5_r8+Gamma
            cff2=0.5_r8-Gamma
          endif
          do j=Jstr,Jend
            do i=Istr,Iend
              t(i,j,k,3,itrc)=Hz(i,j,k)*(cff1*t(i,j,k,nstp,itrc)+
     &                                   cff2*t(i,j,k,nnew,itrc))-
     &                        cff*pm(i,j)*pn(i,j)*
     &                        (FX(i+1,j)-FX(i,j)+FE(i,j+1)-FE(i,j))
            enddo
          enddo
        enddo
      enddo
!
!  Compute artificial continuity equation (same for all tracers).
!  Notice that the J-loop is pipelined from here to the end.
!
      do j=Jstr,Jend
        if (j.le.Jend) then
          if (iic.eq.ntfirst) then
            cff=0.5_r8*dt
          else
            cff=(1.0_r8-Gamma)*dt
          endif
          do k=1,N
            do i=Istr,Iend
              DC(i,k)=1.0_r8/(Hz(i,j,k)-cff*pm(i,j)*pn(i,j)*
     &                        (Huon(i+1,j,k)-Huon(i,j,k)+
     &                         Hvom(i,j+1,k)-Hvom(i,j,k)+
     &                        (W(i,j,k)-W(i,j,k-1))))
            enddo
          enddo
!
!---------------------------------------------------------------------
!  Compute time rate of change of intermediate tracer due to vertical
!  advection.  Impose artificial continuity equation.
!---------------------------------------------------------------------
!
          do itrc=1,NT
#  ifdef TS_SVADVECTION
!
!  Build conservative parabolic splines for the vertical derivatives
!  "FC" of the tracer.  Then, the interfacial "FC" values are
!  converted to vertical advective fluxes.
!
            do i=Istr,Iend
#   ifdef NEUMANN
              FC(i,0)=1.5_r8*t(i,j,1,nstp,itrc)
              CF(i,1)=0.5_r8
#   else
              FC(i,0)=2.0_r8*t(i,j,1,nstp,itrc)
              CF(i,1)=1.0_r8
#   endif
            enddo
            do k=1,N-1
              do i=Istr,Iend
                cff=1.0_r8/(2.0_r8*Hz(i,j,k)+
     &                      Hz(i,j,k+1)*(2.0_r8-CF(i,k)))
                CF(i,k+1)=cff*Hz(i,j,k)
                FC(i,k)=cff*(3.0_r8*(Hz(i,j,k  )*t(i,j,k+1,nstp,itrc)+
     &                               Hz(i,j,k+1)*t(i,j,k  ,nstp,itrc))-
     &                       Hz(i,j,k+1)*FC(i,k-1))
              enddo
            enddo
            do i=Istr,Iend
#   ifdef NEUMANN
              FC(i,N)=(3.0_r8*t(i,j,N,nstp,itrc)-FC(i,N-1))/
     &                (2.0_r8-CF(i,N))
#   else
              FC(i,N)=(2.0_r8*t(i,j,N,nstp,itrc)-FC(i,N-1))/
     &                (1.0_r8-CF(i,N))
#   endif
            enddo
            do k=N-1,0,-1
              do i=Istr,Iend
                FC(i,k)=FC(i,k)-CF(i,k+1)*FC(i,k+1)
                FC(i,k+1)=W(i,j,k+1)*FC(i,k+1)
              enddo
            enddo
            do i=Istr,Iend
              FC(i,N)=0.0_r8
              FC(i,0)=0.0_r8
            enddo
#  elif defined TS_A4VADVECTION
            do k=1,N-1
              do i=Istr,Iend
                FC(i,k)=t(i,j,k+1,nstp,itrc)-t(i,j,k,nstp,itrc)
              enddo
            enddo
            do i=Istr,Iend
              FC(i,0)=FC(i,1)
              FC(i,N)=FC(i,N-1)
            enddo
            do k=1,N
              do i=Istr,Iend
                cff=2.0_r8*FC(i,k)*FC(i,k-1)
                if (cff.gt.eps) then
                  CF(i,k)=cff/(FC(i,k)+FC(i,k-1))
                else
                  CF(i,k)=0.0_r8
                endif
              enddo
            enddo
            cff1=1.0_r8/3.0_r8
            do k=1,N-1
              do i=Istr,Iend
                FC(i,k)=W(i,j,k)*
     &                  0.5_r8*(t(i,j,k  ,nstp,itrc)+
     &                          t(i,j,k+1,nstp,itrc)-
     &                          cff1*(CF(i,k+1)-CF(i,k)))
              enddo
            enddo
            do i=Istr,Iend
              FC(i,0)=0.0_r8
              FC(i,N)=0.0_r8
            enddo
#  elif defined TS_C2VADVECTION
            do k=1,Nm
              do i=Istr,Iend
                FC(i,k)=W(i,j,k)*
     &                  0.5_r8*(t(i,j,k  ,nstp,itrc)+
     &                          t(i,j,k+1,nstp,itrc))
              enddo
            enddo
            do i=Istr,Iend
              FC(i,0)=0.0_r8
              FC(i,N)=0.0_r8
            enddo
#  else
            cff1=0.5_r8
            cff2=7.0_r8/12.0_r8
            cff3=1.0_r8/12.0_r8
            do k=2,N-2
              do i=Istr,Iend
                FC(i,k)=W(i,j,k)*(cff2*(t(i,j,k  ,nstp,itrc)+
     &                                  t(i,j,k+1,nstp,itrc))-
     &                            cff3*(t(i,j,k-1,nstp,itrc)+
     &                                  t(i,j,k+2,nstp,itrc)))
              enddo
            enddo
            do i=Istr,Iend
              FC(i,0  )=0.0_r8
              FC(i,1  )=W(i,j,1  )*(cff1*t(i,j,1  ,nstp,itrc)+
     &                              cff2*t(i,j,2  ,nstp,itrc)-
     &                              cff3*t(i,j,3  ,nstp,itrc))
              FC(i,N-1)=W(i,j,N-1)*(cff1*t(i,j,N  ,nstp,itrc)+
     &                              cff2*t(i,j,N-1,nstp,itrc)-
     &                              cff3*t(i,j,N-2,nstp,itrc))
              FC(i,N  )=0.0_r8
            enddo
#  endif
!
! Time-step vertical advection of tracers (Tunits).
!
            if (iic.eq.ntfirst) then
              cff=0.5_r8*dt
            else
              cff=(1.0_r8-Gamma)*dt
            endif
            do k=1,N
              do i=Istr,Iend
                t(i,j,k,3,itrc)=DC(i,k)*
     &                          (t(i,j,k,3,itrc)-
     &                           cff*pm(i,j)*pn(i,j)*
     &                           (FC(i,k)-FC(i,k-1)))
              enddo
            enddo
          enddo
!
!---------------------------------------------------------------------
!  Start computation of tracers at n+1 time-step, t(i,j,k,nnew,itrc).
!---------------------------------------------------------------------
!
!  Compute vertical diffusive fluxes "FC" of the tracer fields at
!  current time step n, and at horizontal RHO-points and vertical
!  W-points.
!
          cff3=dt*(1.0_r8-lambda)
          do itrc=1,NT
            ltrc=MIN(NAT,itrc)
            do k=1,Nm
              do i=Istr,Iend
                FC(i,k)=cff3*Akt(i,j,k,ltrc)*
     &                  (t(i,j,k+1,nstp,itrc)-t(i,j,k,nstp,itrc))/
     &                  (z_r(i,j,k+1)-z_r(i,j,k))
#  ifdef LMD_NONLOCAL
!
!  Add in the nonlocal transport flux for unstable (convective)
!  forcing conditions into matrix DC when using the Large et al.
!  KPP scheme.
!
     &                 -dt*Akt(i,j,k,ltrc)*ghats(i,j,k,ltrc)
#  endif
              enddo
            enddo
#  ifdef SOLAR_SOURCE
!
!  Add in incoming solar radiation at interior W-points using decay
!  decay penetration function based on Jerlow water type.
!
            if (itrc.eq.itemp) then
              do k=1,Nm
                do i=Istr,Iend
                  FC(i,k)=FC(i,k)+dt*srflx(i,j)*swdk(i,j,k)
                enddo
              enddo
            endif
#  endif
!
!  Apply bottom and surface tracer flux conditions.
!
            do i=Istr,Iend
              FC(i,0)=dt*btflx(i,j,itrc)
              FC(i,N)=dt*stflx(i,j,itrc)
            enddo
!
!  Compute new tracer field (m Tunits).
!
            do k=1,N
              do i=Istr,Iend
                t(i,j,k,nnew,itrc)=Hz(i,j,k)*t(i,j,k,nstp,itrc)+
     &                             FC(i,k)-FC(i,k-1)
              enddo
            enddo
          enddo
        endif
# else /* DIAGNOSTIC */
      do j=Jstr,Jend
# endif /* !DIAGNOSTIC */
!
!=====================================================================
!  3D momentum equation in the XI-direction.
!=====================================================================
!
!  Compute U-component viscous vertical momentum fluxes "FC" at
!  current time-step n, and at horizontal U-points and vertical
!  W-points.
!
        if (j.le.Jend) then
          cff3=dt*(1.0_r8-lambda)
          do k=1,Nm
            do i=IstrU,Iend
              FC(i,k)=cff3*(u(i,j,k+1,nstp)-u(i,j,k,nstp))*
     &                (Akv(i,j,k)+Akv(i-1,j,k))/
     &                (z_r(i,j,k+1)+z_r(i-1,j,k+1)-
     &                 z_r(i,j,k  )-z_r(i-1,j,k  ))
            enddo
          enddo
!
!  Apply surface and bottom stresses, if so is prescribed.
!
          do i=IstrU,Iend
# ifdef BODYFORCE
            FC(i,0)=0.0_r8
            FC(i,N)=0.0_r8
# else
            FC(i,0)=dt*bustr(i,j)
            FC(i,N)=dt*sustr(i,j)
# endif
          enddo
!
!  Compute new U-momentum (m m/s).
!
          cff=dt*0.25_r8
          do i=IstrU,Iend
            DC(i,0)=cff*(pm(i,j)+pm(i-1,j))*(pn(i,j)+pn(i-1,j))
          enddo
          indx=3-nrhs
          if (iic.eq.ntfirst) then
            do k=1,N
              do i=IstrU,Iend
                u(i,j,k,nnew)=u(i,j,k,nstp)*
     &                        0.5_r8*(Hz(i,j,k)+Hz(i-1,j,k))+
     &                        FC(i,k)-FC(i,k-1)
              enddo
            enddo
          elseif (iic.eq.(ntfirst+1)) then
            do k=1,N
              do i=IstrU,Iend
                u(i,j,k,nnew)=u(i,j,k,nstp)*
     &                        0.5_r8*(Hz(i,j,k)+Hz(i-1,j,k))-
     &                        0.5_r8*DC(i,0)*ru(i,j,k,indx)+
     &                        FC(i,k)-FC(i,k-1)
              enddo
            enddo
          else
            cff1= 5.0_r8/12.0_r8
            cff2=16.0_r8/12.0_r8
            do k=1,N
              do i=IstrU,Iend
                u(i,j,k,nnew)=u(i,j,k,nstp)*
     &                        0.5_r8*(Hz(i,j,k)+Hz(i-1,j,k))+
     &                        DC(i,0)*(cff1*ru(i,j,k,nrhs)-
     &                                 cff2*ru(i,j,k,indx))+
     &                        FC(i,k)-FC(i,k-1)
              enddo
            enddo
          endif
        endif
!
!=====================================================================
!  3D momentum equation in the ETA-direction.
!=====================================================================
!
!  Compute V-component viscous vertical momentum fluxes "FC" at
!  current time-step n, and at horizontal V-points and vertical
!  W-points.
!
        if (j.ge.JstrV) then
          cff3=dt*(1.0_r8-lambda)
          do k=1,Nm
            do i=Istr,Iend
              FC(i,k)=cff3*(v(i,j,k+1,nstp)-v(i,j,k,nstp))*
     &                (Akv(i,j,k)+Akv(i,j-1,k))/
     &                (z_r(i,j,k+1)+z_r(i,j-1,k+1)-
     &                 z_r(i,j,k  )-z_r(i,j-1,k  ))
            enddo
          enddo
!
!  Apply surface and bottom stresses, if so is prescribed.
!
          do i=Istr,Iend
# ifdef BODYFORCE
            FC(i,0)=0.0_r8
            FC(i,N)=0.0_r8
# else
            FC(i,0)=dt*bvstr(i,j)
            FC(i,N)=dt*svstr(i,j)
# endif
          enddo
!
!  Compute new V-momentum (m m/s).
!
          cff=dt*0.25_r8
          do i=Istr,Iend
            DC(i,0)=cff*(pm(i,j)+pm(i,j-1))*(pn(i,j)+pn(i,j-1))
          enddo
          if (iic.eq.ntfirst) then
            do k=1,N
              do i=Istr,Iend
                v(i,j,k,nnew)=v(i,j,k,nstp)*
     &                        0.5_r8*(Hz(i,j,k)+Hz(i,j-1,k))+
     &                        FC(i,k)-FC(i,k-1)
              enddo
            enddo
          elseif (iic.eq.(ntfirst+1)) then
            do k=1,N
              do i=Istr,Iend
                v(i,j,k,nnew)=v(i,j,k,nstp)*
     &                        0.5_r8*(Hz(i,j,k)+Hz(i,j-1,k))-
     &                        0.5_r8*DC(i,0)*rv(i,j,k,indx)+
     &                        FC(i,k)-FC(i,k-1)
              enddo
            enddo
          else
            cff1= 5.0_r8/12.0_r8
            cff2=16.0_r8/12.0_r8
            do k=1,N
              do i=Istr,Iend
                v(i,j,k,nnew)=v(i,j,k,nstp)*
     &                        0.5_r8*(Hz(i,j,k)+Hz(i,j-1,k))+
     &                        DC(i,0)*(cff1*rv(i,j,k,nrhs)-
     &                                 cff2*rv(i,j,k,indx))+
     &                        FC(i,k)-FC(i,k-1)
              enddo
            enddo
          endif
        endif
      enddo
# ifndef DIAGNOSTIC
!
!=====================================================================
!  Apply lateral boundary conditions.
!=====================================================================
!
      do itrc=1,NT
        call t3dbc_tile (Istr,Iend,Jstr,Jend,3,itrc,grad)
#  if defined EW_PERIODIC || defined NS_PERIODIC
        call exchange_r3d_tile (Istr,Iend,Jstr,Jend,
     &                          t(START_2D_ARRAY,1,3,itrc))
#  endif
      enddo
# endif /* !DIAGNOSTIC */
#else
      subroutine pre_step3d
#endif
      return
      end
