#include "cppdefs.h"
#if defined MY25_MIXING && defined SOLVE3D
      subroutine my25_corstep (tile)
!
!======================================== Alexander F. Shchepetkin ===
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine perfoms the corrector step for turbulent kinetic     !
!  energy and length scale prognostic variables, tke and gls.        !
!                                                                    !
!  References:                                                       !
!                                                                    !
!  Mellor, G.L. and T. Yamada, 1982:  Development of turbulence      !
!    closure model for geophysical fluid problems, Rev. Geophys.     !
!    Space Phys., 20, 851-875.                                       !
!                                                                    !
!  Galperin, B., L.H. Kantha, S. Hassid, and A.Rosati, 1988:  A      !
!    quasi-equilibrium  turbulent  energy model for geophysical      !
!    flows, J. Atmos. Sci., 45, 55-62.                               !
!                                                                    !
!====================================================================!
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "scratch.h"
# include "tile.h"
!
# ifdef PROFILE
      call wclock_on (20)
# endif
      call my25_corstep_tile (Istr,Iend,Jstr,Jend,
     &                        A3d(1,1),A3d(1,2),A2d(1,1),A2d(1,2),
     &                        A2d(1,3),A2d(1,4),A2d(1,5),A2d(1,6),
     &                        A2d(1,7),A2d(1,8),A2d(1,9),A2d(1,1),
     &                        A2d(1,2),A2d(1,3),A2d(1,4),A2d(1,5),
     &                        A2d(1,6))
# ifdef PROFILE
      call wclock_off (20)
# endif
      return
      end
!
!*********************************************************************
      subroutine my25_corstep_tile (Istr,Iend,Jstr,Jend,shear2,buoy2,
     &                              gradK,gradP,curvK,curvP,FXK,FXP,
     &                              FEK,FEP,CF,dU,dV,FCK,FCP,BCK,BCP)
!*********************************************************************
!
       implicit none
# include "param.h"
# include "forces.h"
# include "grid.h"
# include "mask.h"
# include "mixing.h"
# include "ocean.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, itrc, j, k
      REAL_TYPE
     &        Gadv, Gh, Ls_unlmt, Ls_lmt, Qprod, Qdiss, Sh, Sm,
     &        Wscale, cff, cff1, cff2, cff3, eps, ql, strat2
      REAL_TYPE
     &           BCK(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &           BCP(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            CF(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            dU(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            dV(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &           FCK(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &           FCP(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &           FEK(PRIVATE_2D_SCRATCH_ARRAY),
     &           FEP(PRIVATE_2D_SCRATCH_ARRAY),
     &           FXK(PRIVATE_2D_SCRATCH_ARRAY),
     &           FXP(PRIVATE_2D_SCRATCH_ARRAY),
     &         curvK(PRIVATE_2D_SCRATCH_ARRAY),
     &         curvP(PRIVATE_2D_SCRATCH_ARRAY),
     &         gradK(PRIVATE_2D_SCRATCH_ARRAY),
     &         gradP(PRIVATE_2D_SCRATCH_ARRAY),
     &        shear2(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &         buoy2(PRIVATE_2D_SCRATCH_ARRAY,0:N)
!
      parameter (Gadv=1.0_r8/3.0_r8, eps=1.0_e8-10)
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Compute vertical velocity shear at W-points.
!---------------------------------------------------------------------
!
# ifdef SPLINES
      do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
        do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
          CF(i,0)=0.0_r8
          dU(i,0)=0.0_r8
          dV(i,0)=0.0_r8
        enddo
        do k=1,Nm
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            cff=1.0_r8/(2.0_r8*Hz(i,j,k+1)+
     &                  Hz(i,j,k)*(2.0_r8-CF(i,k-1)))
            CF(i,k)=cff*Hz(i,j,k+1)
            dU(i,k)=cff*(3.0_r8*(u(i  ,j,k+1,nstp)-u(i,  j,k,nstp)+
     &                           u(i+1,j,k+1,nstp)-u(i+1,j,k,nstp))-
     &                   Hz(i,j,k)*dU(i,k-1))
            dV(i,k)=cff*(3.0_r8*(v(i,j  ,k+1,nstp)-v(i,j  ,k,nstp)+
     &                           v(i,j+1,k+1,nstp)-v(i,j+1,k,nstp))-
     &                   Hz(i,j,k)*dV(i,k-1))
          enddo
        enddo
        do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
          dU(i,N)=0.0_r8
          dV(i,N)=0.0_r8
        enddo
        do k=Nm,1,-1
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            dU(i,k)=dU(i,k)-CF(i,k)*dU(i,k+1)
            dV(i,k)=dV(i,k)-CF(i,k)*dV(i,k+1)
          enddo
        enddo
        do k=1,Nm
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            shear2(i,j,k)=dU(i,k)*dU(i,k)+dV(i,k)*dV(i,k)
          enddo
        enddo
      enddo
# else
      do k=1,Nm
        do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            cff=0.5_r8/(z_r(i,j,k+1)-z_r(i,j,k))
            shear2(i,j,k)=(cff*(u(i  ,j,k+1,nstp)-u(i  ,j,k,nstp)+
     &                          u(i+1,j,k+1,nstp)-u(i+1,j,k,nstp)))**2+
     &                    (cff*(v(i,j  ,k+1,nstp)-v(i,j  ,k,nstp)+
     &                          v(i,j+1,k+1,nstp)-v(i,j+1,k,nstp)))**2
          enddo
        enddo
      enddo
# endif
!
! Load Brunt-Vaisala frequency.
!
      do k=1,Nm
        do j=Jstr-1,Jend+1
          do i=Istr-1,Iend+1
            buoy2(i,j,k)=bvf(i,j,k)
          enddo
        enddo
      enddo
# ifdef N2S2_HORAVG
!
!---------------------------------------------------------------------
!  Smooth horizontally buoyancy and shear.  Use buoy2(:,:,0) and
!  shear2(:,:,0) as scratch utility array.
!---------------------------------------------------------------------
!
      do k=1,Nm
        if (WESTERN_EDGE) then
          do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
            shear2(Istr-1,j,k)=shear2(Istr,j,k)
          enddo
        endif
        if (EASTERN_EDGE) then
          do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
            shear2(Iend+1,j,k)=shear2(Iend,j,k)
          enddo
        endif
        if (SOUTHERN_EDGE) then
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            shear2(i,Jstr-1,k)=shear2(i,Jstr,k)
          enddo
        endif
        if (NORTHERN_EDGE) then
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            shear2(i,Jend+1,k)=shear2(i,Jend,k)
          enddo
        endif
        if (SOUTH_WEST_CORNER) then
          shear2(Istr-1,Jstr-1,k)=shear2(Istr,Jstr,k)
        endif
        if (NORTH_WEST_CORNER) then
          shear2(Istr-1,Jend+1,k)=shear2(Istr,Jend,k)
        endif
        if (SOUTH_EAST_CORNER) then
          shear2(Iend+1,Jstr-1,k)=shear2(Iend,Jstr,k)
        endif
        if (NORTH_EAST_CORNER) then
          shear2(Iend+1,Jend+1,k)=shear2(Iend,Jend,k)
        endif
!
!  Average horizontally.
!
        do j=Jstr-1,Jend
          do i=Istr-1,Iend
            buoy2(i,j,0)=0.25_r8*(buoy2(i,j  ,k)+buoy2(i+1,j  ,k)+
     &                            buoy2(i,j+1,k)+buoy2(i+1,j+1,k))
            shear2(i,j,0)=0.25_r8*(shear2(i,j  ,k)+shear2(i+1,j  ,k)+
     &                             shear2(i,j+1,k)+shear2(i+1,j+1,k))
          enddo
        enddo
        do j=Jstr,Jend
          do i=Istr,Iend
            buoy2(i,j,k)=0.25_r8*(buoy2(i,j  ,0)+buoy2(i-1,j  ,0)+
     &                            buoy2(i,j-1,0)+buoy2(i-1,j-1,0))
            shear2(i,j,k)=0.25_r8*(shear2(i,j  ,0)+shear2(i-1,j  ,0)+
     &                             shear2(i,j-1,0)+shear2(i-1,j-1,0))
          enddo
        enddo
      enddo
# endif /* N2S2_HORAVG */
!
!---------------------------------------------------------------------
!  Time-step advective terms.
!---------------------------------------------------------------------
!
!  At entry, it is assumed that the turbulent kinetic energy fields
!  "tke" and "gls", at time level "nnew", are set to its values at
!  time level "nstp" multiplied by the grid box thicknesses Hz
!  (from old time step and at W-points).
!
      do k=1,N-1
# ifdef K_C2ADVECTION
!
!  Second-order, centered differences advection.
!
        do j=Jstr,Jend
          do i=Istr,Iend+1
            cff=0.25_r8*(Huon(i,j,k)+Huon(i,j,k+1))
            FXK(i,j)=cff*(tke(i,j,k,3)+tke(i-1,j,k,3))
            FXP(i,j)=cff*(gls(i,j,k,3)+gls(i-1,j,k,3))
          enddo
        enddo
        do j=Jstr,Jend+1
          do i=Istr,Iend
            cff=0.25_r8*(Hvom(i,j,k)+Hvom(i,j,k+1))
            FEK(i,j)=cff*(tke(i,j,k,3)+tke(i,j-1,k,3))
            FEP(i,j)=cff*(gls(i,j,k,3)+gls(i,j-1,k,3))
          enddo
        enddo
# else
#  ifdef EW_PERIODIC
#   define I_RANGE Istr-1,Iend+2
#  else
#   define I_RANGE max(Istr-1,1),min(Iend+2,Lm+1)
#  endif
        do j=Jstr,Jend
          do i=I_RANGE
            gradK(i,j)=(tke(i,j,k,3)-tke(i-1,j,k,3))
#  ifdef MASKING
     &                *umask(i,j)
#  endif
            gradP(i,j)=(gls(i,j,k,3)-gls(i-1,j,k,3))
#  ifdef MASKING
     &                *umask(i,j)
#  endif
          enddo
        enddo
#  undef I_RANGE
#  ifndef EW_PERIODIC
        if (WESTERN_EDGE) then
          do j=Jstr,Jend
            gradK(Istr-1,j)=gradK(Istr,j)
            gradP(Istr-1,j)=gradP(Istr,j)
          enddo
        endif
        if (EASTERN_EDGE) then
          do j=Jstr,Jend
            gradK(Iend+2,j)=gradK(Iend+1,j)
            gradP(Iend+2,j)=gradP(Iend+1,j)
          enddo
        endif
#  endif
#  ifdef K_C4ADVECTION
!
!  Fourth-order, centered differences advection.
!
        cff1=1.0_r8/6.0_r8
        do j=Jstr,Jend
          do i=Istr,Iend+1
            cff=0.5_r8*(Huon(i,j,k)+Huon(i,j,k+1))
            FXK(i,j)=cff*0.5_r8*(tke(i-1,j,k,3)+tke(i,j,k,3)-
     &                           cff1*(gradK(i+1,j)-gradK(i-1,j)))
            FXP(i,j)=cff*0.5_r8*(gls(i-1,j,k,3)+gls(i,j,k,3)-
     &                           cff1*(gradP(i+1,j)-gradP(i-1,j)))
          enddo
        enddo
#  else
!
!  Third-order, upstream bias advection with velocity dependent
!  hyperdiffusion.
!
        do j=Jstr,Jend
          do i=Istr-1,Iend+1
            curvK(i,j)=gradK(i+1,j)-gradK(i,j)
            curvP(i,j)=gradP(i+1,j)-gradP(i,j)
          enddo
        enddo
        do j=Jstr,Jend
          do i=Istr,Iend+1
            cff=0.5_r8*(Huon(i,j,k)+Huon(i,j,k+1))
            if (cff.gt.0.0_r8) then
              cff1=curvK(i-1,j)
              cff2=curvP(i-1,j)
            else
              cff1=curvK(i,j)
              cff2=curvP(i,j)
            endif
            FXK(i,j)=cff*0.5_r8*(tke(i-1,j,k,3)+tke(i,j,k,3)-
     &                           Gadv*cff1)
            FXP(i,j)=cff*0.5_r8*(gls(i-1,j,k,3)+gls(i,j,k,3)-
     &                           Gadv*cff2)
          enddo
        enddo
#  endif
#  ifdef NS_PERIODIC
#   define J_RANGE Jstr-1,Jend+2
#  else
#   define J_RANGE max(Jstr-1,1),min(Jend+2,Mm+1)
#  endif
        do j=J_RANGE
          do i=Istr,Iend
            gradK(i,j)=(tke(i,j,k,3)-tke(i,j-1,k,3))
#  ifdef MASKING
     &                *vmask(i,j)
#  endif
            gradP(i,j)=(gls(i,j,k,3)-gls(i,j-1,k,3))
#  ifdef MASKING
     &                *vmask(i,j)
#  endif
          enddo
        enddo
#  undef J_RANGE
#  ifndef NS_PERIODIC
        if (SOUTHERN_EDGE) then
          do i=Istr,Iend
            gradK(i,Jstr-1)=gradK(i,Jstr)
            gradP(i,Jstr-1)=gradP(i,Jstr)
          enddo
        endif
        if (NORTHERN_EDGE) then
          do i=Istr,Iend
            gradK(i,Jend+2)=gradK(i,Jend+1)
            gradP(i,Jend+2)=gradP(i,Jend+1)
          enddo
        endif
#  endif
#  ifdef K_C4ADVECTION
        cff1=1.0_r8/6.0_r8
        do j=Jstr,Jend+1
          do i=Istr,Iend
            cff=0.5_r8*(Hvom(i,j,k)+Hvom(i,j,k+1))
            FEK(i,j)=cff*0.5_r8*(tke(i,j-1,k,3)+tke(i,j,k,3)-
     &                           cff1*(gradK(i,j+1)-gradK(i,j-1)))
            FEP(i,j)=cff*0.5_r8*(gls(i,j-1,k,3)+gls(i,j,k,3)-
     &                           cff1*(gradP(i,j+1)-gradP(i,j-1)))
          enddo
        enddo
#  else
        do j=Jstr-1,Jend+1
          do i=Istr,Iend
            curvK(i,j)=gradK(i,j+1)-gradK(i,j)
            curvP(i,j)=gradP(i,j+1)-gradP(i,j)
          enddo
        enddo
        do j=Jstr,Jend+1
          do i=Istr,Iend
            cff=0.5_r8*(Hvom(i,j,k)+Hvom(i,j,k+1))
            if (cff.gt.0.0_r8) then
              cff1=curvK(i,j-1)
              cff2=curvP(i,j-1)
            else
              cff1=curvK(i,j)
              cff2=curvP(i,j)
            endif
            FEK(i,j)=cff*0.5_r8*(tke(i,j-1,k,3)+tke(i,j,k,3)-
     &                           Gadv*cff1)
            FEP(i,j)=cff*0.5_r8*(gls(i,j-1,k,3)+gls(i,j,k,3)-
     &                           Gadv*cff2)
          enddo
        enddo
#  endif
# endif
!
!  Time-step horizontal advection.
!
        do j=Jstr,Jend
          do i=Istr,Iend
            cff=dt*pm(i,j)*pn(i,j)
            tke(i,j,k,nnew)=tke(i,j,k,nnew)-
     &                      cff*(FXK(i+1,j)-FXK(i,j)+
     &                           FEK(i,j+1)-FEK(i,j))
            gls(i,j,k,nnew)=gls(i,j,k,nnew)-
     &                      cff*(FXP(i+1,j)-FXP(i,j)+
     &                           FEP(i,j+1)-FEP(i,j))
          enddo
        enddo
      enddo
!
! Compute vertical advection.
!
      do j=Jstr,Jend
# ifdef K_C2ADVECTION
        do k=1,N
          do i=Istr,Iend
            cff=0.25_r8*(W(i,j,k)+W(i,j,k-1))
            FCK(i,k)=cff*(tke(i,j,k,3)+tke(i,j,k-1,3))
            FCP(i,k)=cff*(gls(i,j,k,3)+gls(i,j,k-1,3))
          enddo
        enddo
# else
        cff1=7.0_r8/12.0_r8
        cff2=1.0_r8/12.0_r8
        do k=2,N-1
          do i=Istr,Iend
            cff=0.5*(W(i,j,k)+W(i,j,k-1))
            FCK(i,k)=cff*(cff1*(tke(i,j,k-1,3)+
     &                          tke(i,j,k  ,3))-
     &                    cff2*(tke(i,j,k-2,3)+
     &                          tke(i,j,k+1,3)))
            FCP(i,k)=cff*(cff1*(gls(i,j,k-1,3)+
     &                          gls(i,j,k  ,3))-
     &                    cff2*(gls(i,j,k-2,3)+
     &                          gls(i,j,k+1,3)))
          enddo
        enddo
        cff1=1.0_r8/3.0_r8
        cff2=5.0_r8/6.0_r8
        cff3=1.0_r8/6.0_r8
         do i=Istr,Iend
          cff=0.5_r8*(W(i,j,0)+W(i,j,1))
          FCK(i,1)=cff*(cff1*tke(i,j,0,3)+
     &                  cff2*tke(i,j,1,3)-
     &                  cff3*tke(i,j,2,3))
          FCP(i,1)=cff*(cff1*gls(i,j,0,3)+
     &                  cff2*gls(i,j,1,3)-
     &                  cff3*gls(i,j,2,3))
          cff=0.5_r8*(W(i,j,N)+W(i,j,N-1))
          FCK(i,N)=cff*(cff1*tke(i,j,N  ,3)+
     &                  cff2*tke(i,j,N-1,3)-
     &                  cff3*tke(i,j,N-2,3))
          FCP(i,N)=cff*(cff1*gls(i,j,N  ,3)+
     &                  cff2*gls(i,j,N-1,3)-
     &                  cff3*gls(i,j,N-2,3))
        enddo
# endif
!
!  Time-step vertical advection term.
!
        do k=1,N-1
          do i=Istr,Iend
            cff=dt*pm(i,j)*pn(i,j)
            tke(i,j,k,nnew)=tke(i,j,k,nnew)-
     &                      cff*(FCK(i,k+1)-FCK(i,k))
            gls(i,j,k,nnew)=gls(i,j,k,nnew)-
     &                      cff*(FCP(i,k+1)-FCP(i,k))
          enddo
        enddo
!
!--------------------------------------------------------------------
!  Compute vertical mixing, turbulent production and turbulent
!  dissipation terms.
!--------------------------------------------------------------------
!
!  Set term for vertical mixing of turbulent fields.
!
        cff=-0.5_r8*dt
        do k=1,N
          do i=Istr,Iend
            FCK(i,k)=cff*(Akk(i,j,k)+Akk(i,j,k-1))/Hz(i,j,k)
            CF(i,k)=0.0_r8
          enddo
        enddo
!
!  Compute production and dissipation terms.
!
        cff3=my_E2/(vonKar*vonKar)
        do k=1,N-1
          do i=Istr,Iend
!
!  Compute shear and bouyant production of turbulent energy (m3/s3)
!  at W-points (ignore small negative values of buoyancy).
!
            if ((buoy2(i,j,k).gt.-5.0_e8-5).and.
     &          (buoy2(i,j,k).lt.0.0_r8)) then
              strat2=0.0_r8
            else
              strat2=buoy2(i,j,k)
            endif
            Qprod=shear2(i,j,k)*(Akv(i,j,k)-Akv_bak)-
     &            strat2*(Akt(i,j,k,itemp)-Akt_bak(itemp))
!
!  Recalculate old time-step unlimited length scale.
!
            Ls_unlmt=MAX(eps,
     &                   gls(i,j,k,nstp)/(MAX(tke(i,j,k,nstp),eps)))
!
!  Time-step production term.
!
            cff1=0.5_r8*(Hz(i,j,k)+Hz(i,j,k+1))
            tke(i,j,k,nnew)=tke(i,j,k,nnew)+
     &                      dt*cff1*Qprod*2.0_r8
            gls(i,j,k,nnew)=gls(i,j,k,nnew)+
     &                      dt*cff1*Qprod*my_E1*Ls_unlmt
!
!  Compute dissipation of turbulent energy (m3/s3).  Add in vertical
!  mixing term.
!
            Qdiss=dt*SQRT(tke(i,j,k,nstp))/(my_B1*Ls_unlmt)
            cff=Ls_unlmt*(1.0_r8/(z_w(i,j,N)-z_w(i,j,k))+
     &                    1.0_r8/(z_w(i,j,k)-z_w(i,j,0)))
            Wscale=1.0_r8+cff3*cff*cff
            BCK(i,k)=cff1*(1.0_r8+2.0_r8*Qdiss)-FCK(i,k)-FCK(i,k+1)
            BCP(i,k)=cff1*(1.0_r8+Wscale*Qdiss)-FCK(i,k)-FCK(i,k+1)
          enddo
        enddo
!
!---------------------------------------------------------------------
!  Time-step dissipation and vertical diffusion terms implicitly.
!---------------------------------------------------------------------
!
!  Set surface and bottom boundary conditions.
!
        do i=Istr,Iend
          tke(i,j,N,nnew)=my_B1p2o3*
     &                    0.5_r8*SQRT((sustr(i,j)+sustr(i+1,j))**2+
     &                                (svstr(i,j)+svstr(i,j+1))**2)
          gls(i,j,N,nnew)=0.0_r8
          tke(i,j,0,nnew)=my_B1p2o3*
     &                    0.5_r8*SQRT((bustr(i,j)+bustr(i+1,j))**2+
     &                                (bvstr(i,j)+bvstr(i,j+1))**2)
          gls(i,j,0,nnew)=0.0_r8
        enddo
!
!  Solve tri-diagonal system for "tke".
!
        do i=Istr,Iend
          cff=1.0_r8/BCK(i,N-1)
          CF(i,N-1)=cff*FCK(i,N-1)
          tke(i,j,N-1,nnew)=cff*(tke(i,j,N-1,nnew)-
     &                           FCK(i,N)*tke(i,j,N,nnew))
        enddo
        do k=N-2,1,-1
          do i=Istr,Iend
            cff=1.0_r8/(BCK(i,k)-CF(i,k+1)*FCK(i,k+1))
            CF(i,k)=cff*FCK(i,k)
            tke(i,j,k,nnew)=cff*(tke(i,j,k,nnew)-
     &                           FCK(i,k+1)*tke(i,j,k+1,nnew))
          enddo
        enddo
        do k=1,N-1
          do i=Istr,Iend
            tke(i,j,k,nnew)=tke(i,j,k,nnew)-CF(i,k)*tke(i,j,k-1,nnew)
          enddo
        enddo
!
!  Solve tri-diagonal system for "gls".
!
        do i=Istr,Iend
          cff=1.0_r8/BCP(i,N-1)
          CF(i,N-1)=cff*FCK(i,N-1)
          gls(i,j,N-1,nnew)=cff*(gls(i,j,N-1,nnew)-
     &                           FCK(i,N)*gls(i,j,N,nnew))
        enddo
        do k=N-2,1,-1
          do i=Istr,Iend
            cff=1.0_r8/(BCP(i,k)-CF(i,k+1)*FCK(i,k+1))
            CF(i,k)=cff*FCK(i,k)
            gls(i,j,k,nnew)=cff*(gls(i,j,k,nnew)-
     &                           FCK(i,k+1)*gls(i,j,k+1,nnew))
          enddo
        enddo
        do k=1,N-1,+1
          do i=Istr,Iend
            gls(i,j,k,nnew)=gls(i,j,k,nnew)-CF(i,k)*gls(i,j,k-1,nnew)
          enddo
        enddo
!
!-------------------------------------------------------------------
!  Compute vertical mixing coefficients (m2/s).
!-------------------------------------------------------------------
!
        do k=1,N-1
          do i=Istr,Iend
!
!  Compute turbulent length scale (m).  The length scale is only
!  limited in the K-related calculations and not in QL production,
!  dissipation, wall-proximity, etc.
!
            tke(i,j,k,nnew)=MAX(tke(i,j,k,nnew),my_qmin)
            gls(i,j,k,nnew)=MAX(gls(i,j,k,nnew),my_qmin)
            Ls_unlmt=gls(i,j,k,nnew)/tke(i,j,k,nnew)
            Ls_lmt=MIN(Ls_unlmt,
     &                 my_lmax*SQRT(tke(i,j,k,nnew)/
     &                         (MAX(0.0_r8,buoy2(i,j,k))+eps)))
!
!  Compute Galperin et al. (1988) nondimensional stability function,
!  Gh.  Then, compute nondimensional stability functions for tracers
!  (Sh) and momentum (Sm).  The limit on length scale, sets the lower
!  limit on Gh.  Use Kantha and Clayton or Galperin et al. expression
!  for Sm.
!
            Gh=MIN(my_Gh0,-buoy2(i,j,k)*Ls_lmt*Ls_lmt/
     &                    tke(i,j,k,nnew))
            cff=1.0_r8-my_Sh2*Gh
            Sh=my_Sh1/cff
# ifdef KANTHA_CLAYSON
            Sm=(my_B1pm1o3+Sh*Gh*my_Sm4)/(1.0_r8-my_Sm2*Gh)
# else
            Sm=(my_Sm3+Sh*Gh*my_Sm4)/(1.0_r8-my_Sm2*Gh)
# endif
!
!  Compute vertical mixing (m2/s) coefficients of momentum and
!  tracers.  Average ql over the two timesteps rather than using
!  the new Lscale and just averaging tke.
!
            ql=0.5_r8*(Ls_lmt*SQRT(tke(i,j,k,nnew))+
     &                 Lscale(i,j,k)*SQRT(tke(i,j,k,nstp)))
            Akv(i,j,k)=Akv_bak+ql*Sm
	    do itrc=1,NAT
	      Akt(i,j,k,itrc)=Akt_bak(itrc)+ql*Sh
	    enddo
!
!  Compute vertical mixing (m2/s) coefficient of turbulent kinetic
!  energy.  Use original formulation (Mellor and Yamada 1982;
!  Blumberg 1991; Kantha and Clayson 1994).
!
            Akk(i,j,k)=Akk_bak+ql*my_Sq
!
!  Save limited length scale.
!
            Lscale(i,j,k)=Ls_lmt
          enddo
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Set lateral boundary conditions.
!---------------------------------------------------------------------
!
      call tkebc_tile (Istr,Iend,Jstr,Jend,nnew,gradK,gradP)
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_w3d_tile (Istr,Iend,Jstr,Jend,
     &                        tke(START_2D_ARRAY,0,nnew))
      call exchange_w3d_tile (Istr,Iend,Jstr,Jend,
     &                        gls(START_2D_ARRAY,0,nnew))
      call exchange_w3d_tile (Istr,Iend,Jstr,Jend,
     &                        Akv(START_2D_ARRAY,0))
      do itrc=1,NAT
        call exchange_w3d_tile (Istr,Iend,Jstr,Jend,
     &                          Akt(START_2D_ARRAY,0,itrc))
      enddo
# endif
#else
      subroutine my25_corstep
#endif
      return
      end
