#include "cppdefs.h"
      subroutine main3d
#ifdef SOLVE3D
!
!======================================== Alexander F. Shchepetkin ===
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine is the main driver for ROMS when configurated as     !
!  a full 3D baroclinic ocean model.   It advances the primitive     !
!  equations for a single time step.                                 !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "floats.h"
# include "iounits.h"
# include "ocean.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        my_iif, next_indx1, subs, tile, thread
# ifdef FLOATS
      INTEGER_TYPE
     &        Lend, Lstr, chunk_size
# endif
!
!=====================================================================
!  Time-step 3D primitive equations.
!=====================================================================
!
!  Set time indices and time clock.
!
      nstp=1+MOD(iic-ntstart,2)
      nnew=3-nstp
      nrhs=nstp
      time=time+dt
      tdays=time*sec2day
!
!---------------------------------------------------------------------
!  Read in forcing, climatology and assimilation data from input
!  NetCDF files, if any.
!---------------------------------------------------------------------
!
      call get_data
      if (exit_flag.ne.0) return
!
!---------------------------------------------------------------------
!  Compute density related fields and horizontal mass fluxes
!  (Hz*u/n and Hz*v/m). Process forcing, climatology and assimilation
!  input data: time interpolate between snapshots.  If applicable,
!  compute and report diagnostics.
!---------------------------------------------------------------------
!
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(numthreads)
      do thread=0,numthreads-1
        subs=NSUB_X*NSUB_E/numthreads
        do tile=subs*thread,subs*(thread+1)-1,+1
# ifndef DIAGNOSTIC
          call rho_eos (tile)
# endif
          call set_massflux (tile)
          call set_data (tile)
          call diag (tile)
        enddo
      enddo
!$OMP END PARALLEL DO
      if (exit_flag.ne.0) return
!
!---------------------------------------------------------------------
!  Set fields for vertical boundary conditions. Process tidal forcing,
!  if any.
!---------------------------------------------------------------------
!
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(numthreads)
      do thread=0,numthreads-1
        subs=NSUB_X*NSUB_E/numthreads
        do tile=subs*thread,subs*(thread+1)-1,+1
# ifdef BULK_FLUXES
          call bulk_flux (tile)
# endif
# ifdef BBL
          call bbl (tile)
# endif
          call set_vbc (tile)
# if defined SSH_TIDES || defined UV_TIDES
          call clm_tides (tile)
# endif
        enddo
      enddo
!$OMP END PARALLEL DO 
!
!---------------------------------------------------------------------
!  Compute vertical mixing coefficients for momentum and tracers.
!  Compute S-coordinate vertical velocity, diagnostically from
!  horizontal mass divergence.
!---------------------------------------------------------------------
!
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(numthreads)
      do thread=0,numthreads-1
        subs=NSUB_X*NSUB_E/numthreads
        do tile=subs*(thread+1)-1,subs*thread,-1
# if defined ANA_VMIX
          call ana_vmix (tile)
# elif defined LMD_MIXING
          call lmd_vmix (tile)
# elif defined BVF_MIXING
          call bvf_mix (tile)
# endif
          call omega (tile)
        enddo
      enddo
!$OMP END PARALLEL DO
!
!---------------------------------------------------------------------
!  Compute right-hand-side terms for 3D equations.  If applicable,
!  accumulate time-averaged output data.
!---------------------------------------------------------------------
!
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(numthreads)
      do thread=0,numthreads-1
        subs=NSUB_X*NSUB_E/numthreads
        do tile=subs*thread,subs*(thread+1)-1,+1
          call rhs3d (tile)
# ifdef MY25_MIXING
          call my25_prestep (tile)
# elif defined GLS_MIXING
          call gls_prestep (tile)
# endif
# ifdef AVERAGES
          call set_avg (tile)
# endif
        enddo
      enddo
!$OMP END PARALLEL DO
!
!---------------------------------------------------------------------
!  If appropriate, write out fields into output NetCDF files.  Notice
!  that IO data is written in delayed and serial mode.  If last time
!  step, stop all clocks.
!---------------------------------------------------------------------
!
      call output
      if (exit_flag.ne.0) return
      if (iic.eq.(ntend+1)) then
        write(stdout,10) ' Elapsed CPU time (seconds):'
 10     format(/,a,/)
!$OMP PARALLEL DO PRIVATE(thread) SHARED(numthreads)
        do thread=0,numthreads-1
          call wclock_off (0)
        enddo
!$OMP END PARALLEL DO
        return
      endif
!
!---------------------------------------------------------------------
!  Solve the vertically integrated primitive equations for the
!  free-surface and barotropic momentum components.
!---------------------------------------------------------------------
!
      do my_iif=1,nfast+1
!
!  Set time indices for predictor step. The PREDICTOR_2D_STEP switch
!  it is assumed to be false before the first time-step.
!
        next_indx1=3-indx1
        if (.not.PREDICTOR_2D_STEP) then
          PREDICTOR_2D_STEP=.TRUE.
          iif=my_iif
          if (FIRST_2D_STEP) then
            kstp=indx1
          else
            kstp=3-indx1
          endif
          knew=3
          krhs=indx1
        endif
!
!  Predictor step - Advance barotropic equations using 2D time-step
!  ==============   predictor scheme.  No actual time-stepping is
!  performed during the auxiliary (nfast+1) time-step. It is needed
!  to finalize the fast-time averaging of 2D fields, if any, and
!  compute the new time-evolving depths.
!
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(numthreads)
        do thread=0,numthreads-1
          subs=NSUB_X*NSUB_E/numthreads
          do tile=subs*(thread+1)-1,subs*thread,-1
            call step2d (tile)
          enddo
        enddo
!$OMP END PARALLEL DO
!
!  Set time indices for corrector step.
!
        if (PREDICTOR_2D_STEP) then
          PREDICTOR_2D_STEP=.FALSE.
          knew=next_indx1
          kstp=3-knew
          krhs=3
          if (my_iif.lt.(nfast+1)) indx1=next_indx1
        endif
!
!  Corrector step - Apply 2D time-step corrector scheme.  Notice that
!  ==============   there is not need for a corrector step during the
!  auxiliary (nfast+1) time-step.
!
        if (my_iif.lt.(nfast+1)) then
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(numthreads)
          do thread=0,numthreads-1
            subs=NSUB_X*NSUB_E/numthreads
            do tile=subs*thread,subs*(thread+1)-1,+1
              call step2d (tile)
            enddo
          enddo
!$OMP END PARALLEL DO
        endif
      enddo
!
!---------------------------------------------------------------------
!  Time-step 3D equations.
!---------------------------------------------------------------------
!
!  Time-step 3D momentum equations and couple with vertically
!  integrated equations.
!
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(numthreads)
      do thread=0,numthreads-1
        subs=NSUB_X*NSUB_E/numthreads
        do tile=subs*(thread+1)-1,subs*thread,-1
          call step3d_uv (tile)
        enddo
      enddo
!$OMP END PARALLEL DO
!
!---------------------------------------------------------------------
!  Time-step tracer equations.
!---------------------------------------------------------------------
!
!$OMP PARALLEL DO PRIVATE(thread,subs,tile) SHARED(numthreads)
      do thread=0,numthreads-1
        subs=NSUB_X*NSUB_E/numthreads
        do tile=subs*thread,subs*(thread+1)-1,+1
          call omega (tile)
# ifdef MY25_MIXING
          call my25_corstep (tile)
# elif defined GLS_MIXING
          call gls_corstep (tile)
# endif
# ifndef DIAGNOSTIC
          call step3d_t (tile)
# endif
# ifdef ASSIMILATION
          call oi_update (tile)
# endif
        enddo
      enddo
!$OMP END PARALLEL DO
# ifdef FLOATS
!
!---------------------------------------------------------------------
!  Compute Lagrangian drifters trajectories.
!---------------------------------------------------------------------
!
!$OMP PARALLEL DO PRIVATE(thread,chunk_size,Lstr,Lend)
!$OMP&            SHARED(numthreads,nfloats)
      do thread=0,numthreads-1
        chunk_size=(nfloats+numthreads-1)/numthreads
        Lstr=1+thread*chunk_size
        Lend=MIN(nfloats,Lstr+chunk_size-1)
        call step_floats (Lstr,Lend)
      enddo
!$OMP END PARALLEL DO
!
!  Shift floats time indices.
!
      nfp1=MOD(nfp1+1,NFT+1)
      nf  =MOD(nf  +1,NFT+1)
      nfm1=MOD(nfm1+1,NFT+1)
      nfm2=MOD(nfm2+1,NFT+1)
      nfm3=MOD(nfm3+1,NFT+1)
# endif /* FLOATS */
#endif
      return
      end
