#include "cppdefs.h"
#ifdef LMD_MIXING
      subroutine lmd_vmix (tile)
!
!======================================== Alexander F. Shchepetkin ===
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This subroutine computes the vertical mixing coefficients for     !
!  momentum and tracers  at the ocean surface boundary layer and     !
!  interior using the Large, McWilliams and Doney  (1994) mixing     !
!  scheme.                                                           !
!                                                                    !
!  Reference:                                                        !
!                                                                    !
!    Large, W.G., J.C. McWilliams, and S.C. Doney, 1994: A Review    !
!      and model with a nonlocal boundary layer parameterization,    !
!      Reviews of Geophysics, 32,363-403.                            !
!                                                                    !
!====================================================================!
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "scratch.h"
# include "tile.h"
!
# ifdef PROFILE
      call wclock_on (18)
# endif
      call  lmd_vmix_tile (Istr,Iend,Jstr,Jend,
     &                     A3d(1,1),A3d(1,2),A3d(1,3),A3d(1,4),
     &                     A2d(1,1),A2d(1,2),A2d(1,3),A2d(1,4))
# ifdef LMD_SKPP
      call lmd_skpp_tile (Istr,Iend,Jstr,Jend,
     &                    A3d(1, 2),A3d(1, 3),A3d(1, 4),A3d(1, 1),
     &                    A2d(1, 1),A2d(1, 2),A2d(1, 3),A2d(1, 4),
     &                    A2d(1, 5),A2d(1, 6),A2d(1, 7),A2d(1, 8),
     &                    A2d(1, 9),A2d(1,10),A2d(1,11),A2d(1,12),
     &                    A2d(1,13),A2d(1,10),A2d(1,11),A2d(1,12),
     &                    A2d(1,13),A2d(1,14),A2d(1,15),A2d(1,16))
# endif
# ifdef LMD_BKPP
      call lmd_bkpp_tile (Istr,Iend,Jstr,Jend,
     &                    A3d(1, 2),A3d(1, 3),A3d(1, 4),A3d(1, 1),
     &                    A2d(1, 1),A2d(1, 2),A2d(1, 3),A2d(1, 4),
     &                    A2d(1, 5),A2d(1, 6),A2d(1, 7),A2d(1, 8),
     &                    A2d(1, 9),A2d(1,10),A2d(1,11),A2d(1,12),
     &                    A2d(1,13),A2d(1,10),A2d(1,11),A2d(1,12),
     &                    A2d(1,13),A2d(1,14),A2d(1,15),A2d(1,16))
# endif
# ifdef PROFILE
      call wclock_off (18)
# endif
      return
      end
!
!*********************************************************************
      subroutine lmd_vmix_tile (Istr,Iend,Jstr,Jend,Rig,Kv,Kt,Ks,FC,
     &                          dR,dU,dV)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "ocean.h"
# include "mixing.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
      REAL_TYPE
     &        cff, lmd_iwm, lmd_iws, nu_sx, nu_sxc
# ifdef LMD_DDMIX
      REAL_TYPE
     &        Rrho, ddDS, ddDT, nu_dds, nu_ddt
# endif
# ifdef LMD_RIMIX
      REAL_TYPE
     &        eps, shear2
      parameter (eps=1.0_e8-14)
# endif
      REAL_TYPE
     &         Kt(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &         Ks(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &         Kv(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &        Rig(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &         FC(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &         dR(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &         dU(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &         dV(PRIVATE_1D_SCRATCH_ARRAY,0:N)
# ifdef LMD_RIMIX
!
!---------------------------------------------------------------------
! Compute gradient Richardson number.
!---------------------------------------------------------------------
!
!  Compute gradient Richardson number at horizontal RHO-points and
!  vertical W-points.  If zero or very small velocity shear, bound
!  computation by a large negative value.
!
#  ifdef SPLINES
      do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
        do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
          FC(i,0)=0.0_r8
          dR(i,0)=0.0_r8
          dU(i,0)=0.0_r8
          dV(i,0)=0.0_r8
        enddo
        do k=1,Nm
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            cff=1.0_r8/(2.0_r8*Hz(i,j,k+1)+
     &                  Hz(i,j,k)*(2.0_r8-FC(i,k-1)))
            FC(i,k)=cff*Hz(i,j,k+1)
            dR(i,k)=cff*(6.0_r8*(rho(i,j,k+1)-rho(i,j,k))-
     &                   Hz(i,j,k)*dR(i,k-1))
            dU(i,k)=cff*(3.0_r8*(u(i  ,j,k+1,nstp)-u(i  ,j,k,nstp)+
     &                           u(i+1,j,k+1,nstp)-u(i+1,j,k,nstp))-
     &                   Hz(i,j,k)*dU(i,k-1))
            dV(i,k)=cff*(3.0_r8*(v(i,j  ,k+1,nstp)-v(i,j  ,k,nstp)+
     &                           v(i,j+1,k+1,nstp)-v(i,j+1,k,nstp))-
     &                   Hz(i,j,k)*dV(i,k-1))
          enddo
        enddo
        do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
          dR(i,N)=0.0_r8
          dU(i,N)=0.0_r8
          dV(i,N)=0.0_r8
        enddo
        do k=Nm,1,-1
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            dR(i,k)=dR(i,k)-FC(i,k)*dR(i,k+1)
            dU(i,k)=dU(i,k)-FC(i,k)*dU(i,k+1)
            dV(i,k)=dV(i,k)-FC(i,k)*dV(i,k+1)
          enddo
        enddo
        do k=1,Nm
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            shear2=dU(i,k)*dU(i,k)+dV(i,k)*dV(i,k)
            Rig(i,j,k)=bvf(i,j,k)/(shear2+eps)
!!          Rig(i,j,k)=-gorho0*dR(i,k)/(shear2+eps)
          enddo
        enddo
      enddo
#  else
      do k=1,Nm
        do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            cff=0.5_r8/(z_r(i,j,k+1)-z_r(i,j,k))
            shear2=(cff*(u(i  ,j,k+1,nstp)-u(i  ,j,k,nstp)+
     &                   u(i+1,j,k+1,nstp)-u(i+1,j,k,nstp)))**2+
     &             (cff*(v(i,j  ,k+1,nstp)-v(i,j  ,k,nstp)+
     &                   v(i,j+1,k+1,nstp)-v(i,j+1,k,nstp)))**2
            Rig(i,j,k)=bvf(i,j,k)/(shear2+eps)
          enddo
        enddo
      enddo
#  endif /* !SPLINES */
#  ifdef RI_HORAVG
      do k=1,Nm
        if (WESTERN_EDGE) then
          do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
            Rig(Istr-1,j,k)=Rig(Istr,j,k)
          enddo
        endif
        if (EASTERN_EDGE) then
          do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
            Rig(Iend+1,j,k)=Rig(Iend,j,k)
          enddo
        endif
        if (SOUTHERN_EDGE) then
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            Rig(i,Jstr-1,k)=Rig(i,Jstr,k)
          enddo
        endif
        if (NORTHERN_EDGE) then
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            Rig(i,Jend+1,k)=Rig(i,Jend,k)
          enddo
        endif
        if (SOUTH_WEST_CORNER) then
          Rig(Istr-1,Jstr-1,k)=Rig(Istr,Jstr,k)
        endif
        if (NORTH_WEST_CORNER) then
          Rig(Istr-1,Jend+1,k)=Rig(Istr,Jend,k)
        endif
        if (SOUTH_EAST_CORNER) then
          Rig(Iend+1,Jstr-1,k)=Rig(Iend,Jstr,k)
        endif
        if (NORTH_EAST_CORNER) then
          Rig(Iend+1,Jend+1,k)=Rig(Iend,Jend,k)
        endif
!
!  Smooth gradient Richardson number horizontally.  Use Rig(:,:,0)
!  as scratch utility array.
!
        do j=Jstr-1,Jend
          do i=Istr-1,Iend
            Rig(i,j,0)=0.25_r8*(Rig(i,j  ,k)+Rig(i+1,j  ,k)+
     &                          Rig(i,j+1,k)+Rig(i+1,j+1,k))
          enddo
        enddo
        do j=Jstr,Jend
          do i=Istr,Iend
            Rig(i,j,k)=0.25_r8*(Rig(i,j  ,0)+Rig(i-1,j  ,0)+
     &                          Rig(i,j-1,0)+Rig(i-1,j-1,0))
          enddo
        enddo
      enddo
#  endif /* RI_HORAVG */
#  ifdef RI_VERAVG
!
!  Smooth gradient Richardson number vertically at the interior points.
!
      do k=Nm-1,2,-1
        do j=Jstr,Jend
          do i=Istr,Iend
            Rig(i,j,k)=0.25_r8*Rig(i,j,k-1)+
     &                 0.50_r8*Rig(i,j,k  )+
     &                 0.25_r8*Rig(i,j,k+1)
          enddo
        enddo
      enddo
#  endif /* RI_VERAVG */
# endif /* LMD_RIMIX */
!
!---------------------------------------------------------------------
!  Compute "interior" viscosities and diffusivities everywhere as
!  the superposition of three processes: local Richardson number
!  instability due to resolved vertical shear, internal wave
!  breaking, and double diffusion.
!---------------------------------------------------------------------
!
      do k=1,Nm
        do j=Jstr,Jend
          do i=Istr,Iend
!
!  Compute interior diffusivity due to shear instability mixing.
!
# ifdef LMD_RIMIX
            cff=MIN(1.0_r8,MAX(0.0_r8,Rig(i,j,k))/lmd_Ri0)
            nu_sx=1.0_r8-cff*cff
            nu_sx=nu_sx*nu_sx*nu_sx
!
!  The shear mixing should be also a function of the actual magnitude
!  of the shear, see Polzin (1996, JPO, 1409-1425).
!
            shear2=bvf(i,j,k)/(Rig(i,j,k)+eps)
            cff=shear2*shear2/(shear2*shear2+16.0_e8-10)
            nu_sx=cff*nu_sx
# else
            nu_sx=0.0_r8
# endif /* LMD_RIMIX */
!
!  Compute interior diffusivity due to wave breaking (Gargett and
!  Holloway.
!
            cff=1.0_r8/SQRT(MAX(bvf(i,j,k),1.0_e8-7))
            lmd_iwm=1.0_e8-6*cff
            lmd_iws=1.0_e8-7*cff
!           lmd_iwm=lmd_nuwm
!           lmd_iws=lmd_nuws
!
!  Compute interior convective diffusivity due to static instability
!  mixing.
!
# ifdef LMD_CONVEC
            cff=MAX(bvf(i,j,k),lmd_bvfcon)
            cff=MIN(1.0_r8,(lmd_bvfcon-cff)/lmd_bvfcon)
            nu_sxc=1.0_r8-cff*cff
            nu_sxc=nu_sxc*nu_sxc*nu_sxc
# else
            nu_sxc=0.0_r8
# endif
!
! Sum contributions due to internal wave breaking, shear instability
! and convective diffusivity due to shear instability.
!
            Kv(i,j,k)=lmd_iwm+lmd_nu0m*nu_sx+lmd_nu0c*nu_sxc
            Kt(i,j,k)=lmd_iws+lmd_nu0s*nu_sx+lmd_nu0c*nu_sxc
            Ks(i,j,k)=Kt(i,j,k)
          enddo
        enddo
# ifdef LMD_DDMIX
!
!---------------------------------------------------------------------
!  Compute double-diffusive mixing.  It can occur when vertical
!  gradient of density is stable but the vertical gradient of
!  salinity (salt figering) or temperature (diffusive convection)
!  is unstable.
!---------------------------------------------------------------------
!
!  Compute double-diffusive density ratio, Rrho.
!
        do j=Jstr,Jend
          do i=Istr,Iend
            ddDT=t(i,j,k+1,nstp,itemp)-t(i,j,k,nstp,itemp)
            ddDS=t(i,j,k+1,nstp,isalt)-t(i,j,k,nstp,isalt)
            ddDS=SIGN(1.0_r8,ddDS)*MAX(ABS(ddDS),1.0_e8-14)
            Rrho=alfaobeta(i,j,k)*ddDT/ddDS
!
!  Salt fingering case.
!
            if ((Rrho.gt.1.0_r8).and.(ddDS.gt.0.0_r8)) then
!
!  Compute interior diffusivity for double diffusive mixing of
!  salinity.  Upper bound "Rrho" by "Rrho0"; (lmd_Rrho0=1.9,
!  lmd_nuf=0.001).
!
              Rrho=MIN(Rrho,lmd_Rrho0)
              nu_dds=1.0_r8-((Rrho-1.0_r8)/(lmd_Rrho0-1.0_r8))**2
              nu_dds=lmd_nuf*nu_dds*nu_dds*nu_dds
!
!  Compute interior diffusivity for double diffusive mixing
!  of temperature (lmd_fdd=0.7).
!
              nu_ddt=lmd_fdd*nu_dds
!
!  Diffusive convection case.
!
            elseif ((0.0_r8.lt.Rrho).and.(Rrho.lt.1.0_r8).and.
     &              (ddDS.lt.0.0_r8)) then
!
!  Compute interior diffusivity for double diffusive mixing of
!  temperature (Marmorino and Caldwell, 1976); (lmd_nu=1.5e-6,
!  lmd_tdd1=0.909, lmd_tdd2=4.6, lmd_tdd3=0.54).
!
              nu_ddt=lmd_nu*lmd_tdd1*
     &               EXP(lmd_tdd2*
     &                   EXP(-lmd_tdd3*((1.0_r8/Rrho)-1.0_r8)))
!
!  Compute interior diffusivity for double diffusive mixing
!  of salinity (lmd_sdd1=0.15, lmd_sdd2=1.85, lmd_sdd3=0.85).
!
              if (Rrho.lt.0.5_r8) then
                nu_dds=nu_ddt*lmd_sdd1*Rrho
              else
                nu_dds=nu_ddt*(lmd_sdd2*Rrho-lmd_sdd3)
              endif
            else
              nu_ddt=0.0_r8
              nu_dds=0.0_r8
            endif
!
!  Add double diffusion contribution to temperature and salinity
!  mixing coefficients.
!
            Kt(i,j,k)=Kt(i,j,k)+nu_ddt
            Ks(i,j,k)=Ks(i,j,k)+nu_dds
          enddo
        enddo
# endif /* LMD_DDMIX */
      enddo
# ifndef LMD_SKPP
!
!---------------------------------------------------------------------
!  Set vertical mixing coefficients "Akv" and "Akt" to interior
!  values.
!---------------------------------------------------------------------
!
      do k=1,Nm
        do j=Jstr,Jend
          do i=Istr,Iend
            Akv(i,j,k)=Kv(i,j,k)
            Akt(i,j,k,itemp)=Kt(i,j,k)
#  ifdef SALINITY
            Akt(i,j,k,isalt)=Ks(i,j,k)
#  endif
          enddo
        enddo
      enddo
!
! Set gradient or periodic boundary conditions.
!
      call fill_w3d_tile (Istr,Iend,Jstr,Jend,
     &                    Akv(START_2D_ARRAY,0))
      call fill_w3d_tile (Istr,Iend,Jstr,Jend,
     &                    Akt(START_2D_ARRAY,0,itemp))
#  ifdef SALINITY
      call fill_w3d_tile (Istr,Iend,Jstr,Jend,
     &                    Akt(START_2D_ARRAY,0,isalt))
#  endif
# endif /* !LMD_SKPP */
#else
      subroutine lmd_vmix
#endif
      return
      end
