#include "cppdefs.h"
#undef LMD_BOUND
#define SASHA
#ifdef LMD_SKPP
      subroutine lmd_skpp_tile (Istr,Iend,Jstr,Jend,Kv,Kt,Ks,Bflux,Bo,
     &                          Bosol,Bfsfc,swdk,Ustar,wm,ws,sl_dpth,
     &                          zgrid,FC,dR,dU,dV,f1,Gm1,dGm1dS,Gt1,
     &                          dGt1dS,Gs1,dGs1dS)
!
!======================================== Alexander F. Shchepetkin ===
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine determines the depth of surface  oceanic boundary    !
!  layer, hsbl, as the shallowest depth where the bulk Richardson    !
!  number is equal to the critical value, Ric.                       !
!                                                                    !
!  Then,  it computes the vertical mixing coefficients  within the   !
!  boundary layer. They depend on surface forcing and the magnitude  !
!  and gradient of interior mixing below  the boundary layer.  The   !
!  ocean interior is allowed to force the boundary layer through a   !
!  dependence of the nondimensional vertical shape function G(sigma) !
!  and its vertical derivative at  sigma=1  on the interior  mixing  !
!  coefficients, and it vertical derivative at d=hsbl. The boundary  !
!  layer mixing coefficients are computed by matching these values.  !
!                                                                    !
! Reference:                                                         !
!                                                                    !
!  Large, W.G., J.C. McWilliams, and S.C. Doney, 1994: A Review      !
!    and model with a nonlocal boundary layer parameterization,      !
!    Reviews of Geophysics, 32,363-403.                              !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "grid.h"
# include "ocean.h"
# include "forces.h"
# include "mask.h"
# include "mixing.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
      REAL_TYPE
     &        Gm, Gt, Gs, K_bl, Ribot, Ritop, Rk, Rref, Uk, Uref,
     &        Ustarb, Vk, Vref, Vtc, a1, a2, a3, cff, cff1, cff2,
     &        cff_up, cff_dn, depth, dK_bl, eps, hekman, hmonob,
     &        sigma, zbl
      REAL_TYPE
     &         Bflux(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &            Kv(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &            Kt(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &            Ks(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &            FC(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            dR(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            dU(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            dV(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            Bo(PRIVATE_2D_SCRATCH_ARRAY),
     &         Bosol(PRIVATE_2D_SCRATCH_ARRAY),
     &         Bfsfc(PRIVATE_2D_SCRATCH_ARRAY),
     &           Gm1(PRIVATE_2D_SCRATCH_ARRAY),
     &           Gt1(PRIVATE_2D_SCRATCH_ARRAY),
     &           Gs1(PRIVATE_2D_SCRATCH_ARRAY),
     &         Ustar(PRIVATE_2D_SCRATCH_ARRAY),
     &        dGm1dS(PRIVATE_2D_SCRATCH_ARRAY),
     &        dGt1dS(PRIVATE_2D_SCRATCH_ARRAY),
     &        dGs1dS(PRIVATE_2D_SCRATCH_ARRAY),
     &            f1(PRIVATE_2D_SCRATCH_ARRAY),
     &       sl_dpth(PRIVATE_2D_SCRATCH_ARRAY),
     &          swdk(PRIVATE_2D_SCRATCH_ARRAY),
     &            wm(PRIVATE_2D_SCRATCH_ARRAY),
     &            ws(PRIVATE_2D_SCRATCH_ARRAY),
     &         zgrid(PRIVATE_2D_SCRATCH_ARRAY)
      parameter (eps=1.0_e8-20)
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Initialize relevant parameters.
!---------------------------------------------------------------------
!
      Vtc=lmd_Cv*SQRT(-lmd_betaT)/(SQRT(lmd_cs*lmd_epsilon)*
     &                             lmd_Ric*vonKar*vonKar)
!
!---------------------------------------------------------------------
!  Get approximation of surface layer depth using "lmd_eps" and
!  boundary layer depth from previous time step.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
          sl_dpth(i,j)=lmd_epsilon*(z_w(i,j,N)-hsbl(i,j))
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute turbulent friction velocity (m/s) "Ustar" from wind stress
!  at RHO-points.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
          Ustar(i,j)=SQRT(SQRT((0.5_r8*(sustr(i,j)+sustr(i+1,j)))**2+
     &                         (0.5_r8*(svstr(i,j)+svstr(i,j+1)))**2))
# ifdef MASKING
     &              *rmask(i,j)
# endif
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute surface turbulent buoyancy forcing "Bo" (m2/s3). Remove
!  incoming solar shortwave radiation because this contribution is
!  included in "Bosol".  Compute surface radiative buoyancy forcing
!  "Bosol" (m2/s3).
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
# ifdef SALINITY
          Bo(i,j)=g*(alpha(i,j)*(stflx(i,j,itemp)-srflx(i,j))-
     &               beta (i,j)*stflx(i,j,isalt))
# else
          Bo(i,j)=g*alpha(i,j)*(stflx(i,j,itemp)-srflx(i,j))
# endif /* SALINITY */
          Bosol(i,j)=g*alpha(i,j)*srflx(i,j)
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute total buoyancy flux (m2/s3) at W-points.  Notice that the
!  radiative bouyancy flux is distributed vertically using decay
!  function, swdk. Begin computation of nonlocal transport, "ghats".
!---------------------------------------------------------------------
!
      do k=0,N
        do j=Jstr,Jend
          do i=Istr,Iend
            zgrid(i,j)=z_w(i,j,N)-z_w(i,j,k)
          enddo
        enddo
        call lmd_swfrac_tile (Istr,Iend,Jstr,Jend,-1.0_r8,zgrid,swdk)
        do j=Jstr,Jend
          do i=Istr,Iend
            Bflux(i,j,k)=(Bo(i,j)+Bosol(i,j)*(1.0_r8-swdk(i,j)))
# ifdef MASKING
     &                  *rmask(i,j)
# endif
# ifdef LMD_NONLOCAL
            cff=1.0_r8-(0.5_r8+SIGN(0.5_r8,Bflux(i,j,k)))
            ghats(i,j,k,itemp)=-cff*(stflx(i,j,itemp)-srflx(i,j)+
     &                               srflx(i,j)*(1.0_r8-swdk(i,j)))
#  ifdef SALINITY
            ghats(i,j,k,isalt)=cff*stflx(i,j,isalt)
#  endif
# endif
          enddo
        enddo
      enddo
!
!=====================================================================
!  Compute bulk Richardson number "Rib" and then find depth of the
!  oceanic surface boundary layer "hsbl", such that Rib(hsbl)=Ric.
!=====================================================================
!
      do j=Jstr,Jend
# ifdef SPLINES
!
! Construct parabolic splines for vertical derivatives of potential
! density and velocity components at W-points.  FC is a scratch array.
!
        do i=Istr,Iend
          FC(i,0)=0.0_r8
          dR(i,0)=0.0_r8
          dU(i,0)=0.0_r8
          dV(i,0)=0.0_r8
        enddo
        do k=1,N-1
          do i=Istr,Iend
            cff=1.0_r8/(2.0_r8*Hz(i,j,k+1)+
     &                  Hz(i,j,k)*(2.0_r8-FC(i,k-1)))
            FC(i,k)=cff*Hz(i,j,k+1)
            dR(i,k)=cff*(6.0_r8*(pden(i,j,k+1)-pden(i,j,k))-
     &                   Hz(i,j,k)*dR(i,k-1))
            dU(i,k)=cff*(3.0_r8*(u(i  ,j,k+1,nstp)-u(i,  j,k,nstp)+
     &                           u(i+1,j,k+1,nstp)-u(i+1,j,k,nstp))-
     &                   Hz(i,j,k)*dU(i,k-1))
            dV(i,k)=cff*(3.0_r8*(v(i,j  ,k+1,nstp)-v(i,j  ,k,nstp)+
     &                           v(i,j+1,k+1,nstp)-v(i,j+1,k,nstp))-
     &                   Hz(i,j,k)*dV(i,k-1))
          enddo
        enddo
        do i=Istr,Iend
          dR(i,N)=0.0_r8
          dU(i,N)=0.0_r8
          dV(i,N)=0.0_r8
        enddo
        do k=N-1,1,-1
          do i=Istr,Iend
            dR(i,k)=dR(i,k)-FC(i,k)*dR(i,k+1)
            dU(i,k)=dU(i,k)-FC(i,k)*dU(i,k+1)
            dV(i,k)=dV(i,k)-FC(i,k)*dV(i,k+1)
          enddo
        enddo
# else
!
! Compute vertical derivatives of potential density and velocity
! components at W-points.
!
        do k=1,N-1
          do i=Istr,Iend
            cff=1.0_r8/(z_r(i,j,k+1)-z_r(i,j,k))
            dR(i,k)=cff*(pden(i,j,k+1)-pden(i,j,k))
            cff=0.5_r8*cff
            dU(i,k)=cff*(u(i  ,j,k+1,nstp)-u(i,  j,k,nstp)+
     &                   u(i+1,j,k+1,nstp)-u(i+1,j,k,nstp))
            dV(i,k)=cff*(v(i,j  ,k+1,nstp)-v(i,j  ,k,nstp)+
     &                   v(i,j+1,k+1,nstp)-v(i,j+1,k,nstp))
          enddo
        enddo
        do i=Istr,Iend
          dR(i,0)=0.0_r8
          dR(i,N)=0.0_r8
          dU(i,0)=0.0_r8
          dU(i,N)=0.0_r8
          dV(i,0)=0.0_r8
          dV(i,N)=0.0_r8
        enddo
# endif /* !SPLINES */
!
!---------------------------------------------------------------------
!  Compute bulk Richardson number "Rib" and then find depth of oceanic
!  surface boundary layer "hsbl".
!
!                  [Br - B(d)] * d
!     Rib(d) = ----------------------- ;     Rib(hsbl)=Ric     (1)
!              |Vr - V(d)|^2 + Vt(d)^2
!
!  where "Br" and "Vr" are the surface reference buoyancy and velocity
!  while "B(d)" and "V(d)" are the bouyancy and velocity at depth "d".
!
!  In the code below, the criterion "Rib(hsbl)=Ric" is reformulated
!  as follows:
!
!     Rib(d)       Ritop(d)
!     ------ = --------------- = 1                             (2)
!      Ric      Ric * Ribot(d)
!
!  where "Ritop" and "Ribot" are numerator and denominator in Eq. (1).
!  In its turn, Eq. (2) is rewritten in the following form:
!
!     FC(d) = Ritop(d) - Ric * Ribot(d) = 0                    (3)
!
!  That is, the planetary boundary layer extends to the depth where
!  the critical function "FC(d)" changes its sign.
!---------------------------------------------------------------------
!
!  Compute potential density and velocity components surface reference
!  values.
!
        cff1=1.0_r8/3.0_r8
        cff2=1.0_r8/6.0_r8
        do i=Istr,Iend
          Rref=pden(i,j,N)+
     &         Hz(i,j,N)*(cff1*dR(i,N)+cff2*dR(i,N-1))
          Uref=0.5_r8*(u(i,j,N,nstp)+u(i+1,j,N,nstp))+
     &         Hz(i,j,N)*(cff1*dU(i,N)+cff2*dU(i,N-1))
          Vref=0.5_r8*(v(i,j,N,nstp)+v(i,j+1,N,nstp))+
     &         Hz(i,j,N)*(cff1*dV(i,N)+cff2*dV(i,N-1))
!
!  Compute critical function, FC, for bulk Richardson number.
!
          FC(i,N)=0.0_r8
          do k=N,1,-1
            depth=z_w(i,j,N)-z_w(i,j,k-1)
            if (Bflux(i,j,k-1).lt.0.0_r8) then
              sigma=MIN(sl_dpth(i,j),depth)
            else
              sigma=depth
            endif
            call lmd_wscale (Bflux(i,j,k-1),Ustar(i,j),sigma,
     &                       wm(i,j),ws(i,j))
            Rk=pden(i,j,k)-
     &         Hz(i,j,k)*(cff1*dR(i,k-1)+cff2*dR(i,k))
            Uk=0.5_r8*(u(i,j,k,nstp)+u(i+1,j,k,nstp))-
     &         Hz(i,j,k)*(cff1*dU(i,k-1)+cff2*dU(i,k))
            Vk=0.5_r8*(v(i,j,k,nstp)+v(i,j+1,k,nstp))-
     &         Hz(i,j,k)*(cff1*dV(i,k-1)+cff2*dV(i,k))
!
            Ritop=-gorho0*(Rref-Rk)*depth
            Ribot=(Uref-Uk)**2+(Vref-Vk)**2+
     &            Vtc*depth*ws(i,j)*SQRT(ABS(bvf(i,j,k-1)))
# ifdef SASHA
            FC(i,k-1)=Ritop-lmd_Ric*Ribot
# else
            FC(i,k-1)=Ritop/(Ribot+eps)
# endif
          enddo
        enddo
!
! Linearly interpolate to find "hsbl" where Rib/Ric=1.
!
        do i=Istr,Iend
          ksbl(i,j)=1
          hsbl(i,j)=z_w(i,j,1)
        enddo
# ifdef SASHA
        do k=N,2,-1
          do i=Istr,Iend
            if ((ksbl(i,j).eq.1).and.(FC(i,k-1).gt.0.0_r8)) then
              hsbl(i,j)=(z_w(i,j,k)*FC(i,k-1)-z_w(i,j,k-1)*FC(i,k))/
     &                  (FC(i,k-1)-FC(i,k))
              ksbl(i,j)=k
            endif
          enddo
        enddo
# else
        do k=N,2,-1
          do i=Istr,Iend
            if ((ksbl(i,j).eq.1).and.((FC(i,k  ).lt.lmd_Ric).and.
     &                                (FC(i,k-1).ge.lmd_Ric))) then
              hsbl(i,j)=((FC(i,k-1)-lmd_Ric)*z_w(i,j,k  )+
     &                  (lmd_Ric-FC(i,k  ))*z_w(i,j,k-1))/
     &                  (FC(i,k-1)-FC(i,k))
              ksbl(i,j)=k
            endif
          enddo
        enddo
# endif
      enddo
!
!  Compute total buoyancy flux at surface boundary layer depth,
!  "Bfsfc".
!
      do j=Jstr,Jend
        do i=Istr,Iend
          zgrid(i,j)=z_w(i,j,N)-hsbl(i,j)
        enddo
      enddo
      call lmd_swfrac_tile (Istr,Iend,Jstr,Jend,-1.0_r8,zgrid,swdk)
      do j=Jstr,Jend
        do i=Istr,Iend
          Bfsfc(i,j)=(Bo(i,j)+Bosol(i,j)*(1.0_r8-swdk(i,j)))
# ifdef MASKING
     &              *rmask(i,j)
# endif
        enddo
      enddo
!
!  Under neutral and stable conditions, the depth of the surface
!  boundary layer is required to be less than Ekman and Monin-Obukov
!  depths.
!
      do j=Jstr,Jend
        do i=Istr,Iend
          if ((Ustar(i,j).gt.0.0_r8).and.(Bfsfc(i,j).gt.0.0_r8)) then
            hekman=lmd_cekman*Ustar(i,j)/MAX(ABS(f(i,j)),eps)
            hmonob=lmd_cmonob*Ustar(i,j)*Ustar(i,j)*Ustar(i,j)/
     &             MAX(vonKar*Bfsfc(i,j),eps)
            hsbl(i,j)=(z_w(i,j,N)-
     &                 MIN(hekman,hmonob,z_w(i,j,N)-hsbl(i,j)))
          endif
          hsbl(i,j)=MIN(hsbl(i,j),z_w(i,j,N))
          hsbl(i,j)=MAX(hsbl(i,j),z_w(i,j,0))
# ifdef MASKING
     &             *rmask(i,j)
# endif
        enddo
      enddo
!
!  Apply gradient or periodic boundary conditions.
!
      call fill_r2d_tile (Istr,Iend,Jstr,Jend,hsbl(START_2D_ARRAY))
!
!  Find new boundary layer index "ksbl".
!
      do j=Jstr,Jend
        do i=Istr,Iend
          ksbl(i,j)=1
          do k=N,2,-1
            if ((ksbl(i,j).eq.1).and.(z_w(i,j,k-1).lt.hsbl(i,j))) then
              ksbl(i,j)=k
            endif
          enddo
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute total buoyancy flux at final surface boundary layer depth.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
          zgrid(i,j)=z_w(i,j,N)-hsbl(i,j)
        enddo
      enddo
      call lmd_swfrac_tile (Istr,Iend,Jstr,Jend,-1.0_r8,zgrid,swdk)
      do j=Jstr,Jend
        do i=Istr,Iend
          Bfsfc(i,j)=(Bo(i,j)+Bosol(i,j)*(1.0_r8-swdk(i,j)))
# ifdef MASKING
     &              *rmask(i,j)
# endif
        enddo
      enddo
!
!=====================================================================
!  Compute vertical mixing coefficients within the planetary boundary
!  layer.
!=====================================================================
!
!  Compute tubulent velocity scales (wm,ws) at "hsbl".
!
      do j=Jstr,Jend
        do i=Istr,Iend
          sl_dpth(i,j)=lmd_epsilon*(z_w(i,j,N)-hsbl(i,j))
          if (Bfsfc(i,j).gt.0.0_r8) then
            cff=1.0_r8
          else
            cff=lmd_epsilon
          endif
          sigma=cff*(z_w(i,j,N)-hsbl(i,j))
          call lmd_wscale (Bfsfc(i,j),Ustar(i,j),sigma,
     &                     wm(i,j),ws(i,j))
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute nondimensional shape function Gx(sigma) in terms of the
!  interior diffusivities at sigma=1 (Gm1, Gs1, Gt1) and its vertical
!  derivative evaluated "hsbl" via interpolation.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
          f1(i,j)=5.0_r8*MAX(0.0_r8,Bfsfc(i,j))*vonKar/
     &            (Ustar(i,j)*Ustar(i,j)*Ustar(i,j)*Ustar(i,j)+eps)
        enddo
      enddo
!
      do j=Jstr,Jend
        do i=Istr,Iend
          zbl=z_w(i,j,N)-hsbl(i,j)
          if (hsbl(i,j).gt.z_w(i,j,1)) then
            k=ksbl(i,j)
            cff=1.0_r8/(z_w(i,j,k)-z_w(i,j,k-1))
            cff_dn=cff*(hsbl(i,j)-z_w(i,j,k-1))
            cff_up=cff*(z_w(i,j,k)-hsbl(i,j))
!
!  Compute nondimensional shape function for viscosity "Gm1" and its
!  vertical derivative "dGm1dS" evaluated at "hsbl".
!
            K_bl=cff_dn*Kv(i,j,k)+cff_up*Kv(i,j,k-1)
            dK_bl=cff*(Kv(i,j,k)-Kv(i,j,k-1))
            Gm1(i,j)=K_bl/(zbl*wm(i,j)+eps)
# ifdef MASKING
     &              *rmask(i,j)
# endif
            dGm1dS(i,j)=MIN(0.0_r8,-dK_bl/(wm(i,j)+eps)-K_bl*f1(i,j))
!
!  Compute nondimensional shape function for diffusion of temperature
!  "Gt1" and its vertical derivative "dGt1dS" evaluated at "hsbl".
!
            K_bl=cff_dn*Kt(i,j,k)+cff_up*Kt(i,j,k-1)
            dK_bl=cff*(Kt(i,j,k)-Kt(i,j,k-1))
            Gt1(i,j)=K_bl/(zbl*ws(i,j)+eps)
# ifdef MASKING
     &              *rmask(i,j)
# endif
            dGt1dS(i,j)=MIN(0.0_r8,-dK_bl/(ws(i,j)+eps)-K_bl*f1(i,j))
# ifdef SALINITY
!
!  Compute nondimensional shape function for diffusion of salinity
!  "Gs1" and its vertical derivative "dGs1dS" evaluated at "hsbl".
!
            K_bl=cff_dn*Ks(i,j,k)+cff_up*Ks(i,j,k-1)
            dK_bl=cff*(Ks(i,j,k)-Ks(i,j,k-1))
            Gs1(i,j)=K_bl/(zbl*ws(i,j)+eps)
#  ifdef MASKING
     &              *rmask(i,j)
#  endif
            dGs1dS(i,j)=MIN(0.0_r8,-dK_bl/(ws(i,j)+eps)-K_bl*f1(i,j))
# endif /* SALINITY */
          else
!
!  If the surface boundary layer extends to the bottom, assume that
!  the neutral boundary layer similarity theory holds at the bottom.
!
            ksbl(i,j)=0
!
!  Compute nondimensional bottom shape function for viscosity.
!
            Ustarb=SQRT(SQRT((0.5_r8*(bustr(i,j)+bustr(i+1,j)))**2+
     &                       (0.5_r8*(bvstr(i,j)+bvstr(i,j+1)))**2))
# ifdef MASKING
     &            *rmask(i,j)
# endif
            dK_bl=vonKar*Ustarb
            K_bl=dK_bl*(hsbl(i,j)-z_w(i,j,0))
            Gm1(i,j)=K_bl/(zbl*wm(i,j)+eps)
# ifdef MASKING
     &              *rmask(i,j)
# endif
            dGm1dS(i,j)=MIN(0.0_r8,-dK_bl/(wm(i,j)+eps)-K_bl*f1(i,j))
!
!  Compute nondimensional bottom shape function for diffusion of
!  temperature.
!
            Gt1(i,j)=K_bl/(zbl*ws(i,j)+eps)
# ifdef MASKING
     &              *rmask(i,j)
# endif
            dGt1dS(i,j)=MIN(0.0_r8,-dK_bl/(ws(i,j)+eps)-K_bl*f1(i,j))
!
!  Compute nondimensional bottom shape function for diffusion of
!  salinity.
!
# ifdef SALINITY
            Gs1(i,j)=Gt1(i,j)
            dGs1dS(i,j)=dGt1dS(i,j)
# endif /* SALINITY */
          endif
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute surface boundary layer mixing coefficients.
!---------------------------------------------------------------------
!
      do k=1,Nm
        do j=Jstr,Jend
          do i=Istr,Iend
            zbl=z_w(i,j,N)-hsbl(i,j)
            if (k.gt.ksbl(i,j)) then
!
!  Compute turbulent velocity scales at vertical W-points.
!
              depth=z_w(i,j,N)-z_w(i,j,k)
              if (Bflux(i,j,k).lt.0.0_r8) then
                sigma=MIN(sl_dpth(i,j),depth)
              else
                sigma=depth
              endif
              call lmd_wscale (Bflux(i,j,k),Ustar(i,j),sigma,
     &                         wm(i,j),ws(i,j))
!
!  Set polynomial coefficients for shape function.
!
              sigma=depth/(zbl+eps)
# ifdef MASKING
     &             *rmask(i,j)
# endif
              a1=sigma-2.0_r8
              a2=3.0_r8-2.0_r8*sigma
              a3=sigma-1.0_r8
!
!  Compute nondimesional shape functions.
!
              Gm=a1+a2*Gm1(i,j)+a3*dGm1dS(i,j)
              Gt=a1+a2*Gt1(i,j)+a3*dGt1dS(i,j)
# ifdef SALINITY
              Gs=a1+a2*Gs1(i,j)+a3*dGs1dS(i,j)
# endif /* SALINITY */
!
!  Compute boundary layer mixing coefficients, combine them
!  with interior mixing coefficients.
!
              Kv(i,j,k)=depth*wm(i,j)*(1.0_r8+sigma*Gm)
              Kt(i,j,k)=depth*ws(i,j)*(1.0_r8+sigma*Gt)
# ifdef SALINITY
              Ks(i,j,k)=depth*ws(i,j)*(1.0_r8+sigma*Gs)
# endif /* SALINITY */
# ifdef LMD_NONLOCAL
!
!  Compute boundary layer nonlocal transport (m/s2).
!
              cff=lmd_Cg*(1.0_r8-(0.5_r8+SIGN(0.5_r8,Bflux(i,j,k))))/
     &            (zbl*ws(i,j)+eps)
              ghats(i,j,k,itemp)=cff*ghats(i,j,k,itemp)
#  ifdef SALINITY
              ghats(i,j,k,isalt)=cff*ghats(i,j,k,isalt)
#  endif
# endif
!
!  Set vertical mixing coefficients to fit neutral log layer
!  similarity theory.
!
            else
# ifdef LMD_NONLOCAL
              ghats(i,j,k,itemp)=0.0_r8
#  ifdef SALINITY
              ghats(i,j,k,isalt)=0.0_r8
#  endif
# endif
            endif
          enddo
        enddo
!
!---------------------------------------------------------------------
!  Finally, set vertical mixing coefficients "Akv" and "Akt".
!---------------------------------------------------------------------
!
        do j=Jstr,Jend
          do i=Istr,Iend
# ifdef LMD_BOUND
            Akv(i,j,k)=MIN(lmd_nu0c,Kv(i,j,k))
            Akt(i,j,k,itemp)=MIN(lmd_nu0c,Kt(i,j,k))
#  ifdef SALINITY
            Akt(i,j,k,isalt)=MIN(lmd_nu0c,Ks(i,j,k))
#  endif
# else
            Akv(i,j,k)=Kv(i,j,k)
            Akt(i,j,k,itemp)=Kt(i,j,k)
#  ifdef SALINITY
            Akt(i,j,k,isalt)=Ks(i,j,k)
#  endif
# endif
          enddo
        enddo
      enddo
# ifndef LMD_BKPP
!
! Set gradient or periodic boundary conditions.
!
      call fill_w3d_tile (Istr,Iend,Jstr,Jend,
     &                    Akv(START_2D_ARRAY,0))
      call fill_w3d_tile (Istr,Iend,Jstr,Jend,
     &                    Akt(START_2D_ARRAY,0,itemp))
#  ifdef SALINITY
      call fill_w3d_tile (Istr,Iend,Jstr,Jend,
     &                    Akt(START_2D_ARRAY,0,isalt))
#  endif
# endif /* !LMD_BKPP */
#else
      subroutine lmd_skpp
#endif /* LMD_SKPP */
      return
      end
