#include "cppdefs.h"
#define SASHA
#ifdef LMD_BKPP
      subroutine lmd_bkpp_tile (Istr,Iend,Jstr,Jend,Kv,Kt,Ks,Bflux,Bo,
     &                          Bosol,Bfbot,swdk,Ustar,wm,ws,bl_dpth,
     &                          zgrid,FC,dR,dU,dV,f1,Gm1,dGm1dS,Gt1,
     &                          dGt1dS,Gs1,dGs1dS)
!
!=====================================================================
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine determines the depth of  bottom  oceanic boundary    !
!  layer,  hbbl,  as the deepest depth  where the bulk Richardson    !
!  number is equal to the critical value, Ric.                       !
!                                                                    !
!  Then,  it computes the vertical mixing coefficients  within the   !
!  boundary layer. They depend on surface forcing and the magnitude  !
!  and gradient of interior mixing below  the boundary layer.  The   !
!  ocean interior is allowed to force the boundary layer through a   !
!  dependence of the nondimensional vertical shape function G(sigma) !
!  and its vertical derivative at  sigma=1  on the interior  mixing  !
!  coefficients, and it vertical derivative at d=hsbl. The boundary  !
!  layer mixing coefficients are computed by matching these values.  !
!                                                                    !
! Reference:                                                         !
!                                                                    !
!  Large, W.G., J.C. McWilliams, and S.C. Doney, 1994: A Review      !
!    and model with a nonlocal boundary layer parameterization,      !
!    Reviews of Geophysics, 32,363-403.                              !
!                                                                    !
!  This routine was adapted from Bill Large 1995 code.               !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "grid.h"
# include "ocean.h"
# include "forces.h"
# include "mask.h"
# include "mixing.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
      REAL_TYPE
     &        Gm, Gt, Gs, K_bl, Ribot, Ritop, Rk, Rref, Uk, Uref,
     &        Vk, Vref, Vtc, a1, a2, a3, cff, cff1, cff2, cff_up,
     &        cff_dn, depth, dK_bl, eps, hekman, sigma, zbl
      REAL_TYPE
     &         Bflux(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &            Kv(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &            Kt(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &            Ks(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &            FC(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            dR(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            dU(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            dV(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            Bo(PRIVATE_2D_SCRATCH_ARRAY),
     &         Bosol(PRIVATE_2D_SCRATCH_ARRAY),
     &         Bfbot(PRIVATE_2D_SCRATCH_ARRAY),
     &           Gm1(PRIVATE_2D_SCRATCH_ARRAY),
     &           Gt1(PRIVATE_2D_SCRATCH_ARRAY),
     &           Gs1(PRIVATE_2D_SCRATCH_ARRAY),
     &         Ustar(PRIVATE_2D_SCRATCH_ARRAY),
     &       bl_dpth(PRIVATE_2D_SCRATCH_ARRAY),
     &        dGm1dS(PRIVATE_2D_SCRATCH_ARRAY),
     &        dGt1dS(PRIVATE_2D_SCRATCH_ARRAY),
     &        dGs1dS(PRIVATE_2D_SCRATCH_ARRAY),
     &            f1(PRIVATE_2D_SCRATCH_ARRAY),
     &          swdk(PRIVATE_2D_SCRATCH_ARRAY),
     &            wm(PRIVATE_2D_SCRATCH_ARRAY),
     &            ws(PRIVATE_2D_SCRATCH_ARRAY),
     &         zgrid(PRIVATE_2D_SCRATCH_ARRAY)
      parameter (eps=1.0_e8-20)
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Initialize relevant parameters.
!---------------------------------------------------------------------
!
      Vtc=lmd_Cv*SQRT(-lmd_betaT)/(SQRT(lmd_cs*lmd_epsilon)*
     &                             lmd_Ric*vonKar*vonKar)
!
!---------------------------------------------------------------------
!  Get approximation of bottom layer depth using "lmd_eps" and
!  boundary layer depth from previous time step.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
          bl_dpth(i,j)=lmd_epsilon*(hbbl(i,j)-z_w(i,j,0))
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute turbulent friction velocity (m/s) "Ustar" from bottom
!  stress at RHO-points.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
          Ustar(i,j)=SQRT(SQRT((0.5_r8*(bustr(i,j)+bustr(i+1,j)))**2+
     &                         (0.5_r8*(bvstr(i,j)+bvstr(i,j+1)))**2))
# ifdef MASKING
     &              *rmask(i,j)
# endif
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute bottom turbulent buoyancy forcing "Bo" (m2/s3). Compute
!  surface radiative buoyancy forcing "Bosol" (m2/s3) that can be
!  used in shallow areas when it can penetrate all the way to the
!  bottom.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
# ifdef SALINITY
          Bo(i,j)=g*(alpha(i,j)*(stflx(i,j,itemp)-srflx(i,j))-
     &               beta (i,j)*stflx(i,j,isalt))
# else
          Bo(i,j)=g*alpha(i,j)*(stflx(i,j,itemp)-srflx(i,j))
# endif /* SALINITY */
!!        Bosol(i,j)=g*alpha(i,j)*srflx(i,j)
          Bosol(i,j)=0.0_r8
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute total buoyancy flux (m2/s3) at W-points.
!---------------------------------------------------------------------
!
      do k=0,N
        do j=Jstr,Jend
          do i=Istr,Iend
            zgrid(i,j)=z_w(i,j,N)-z_w(i,j,k)
          enddo
        enddo
        call lmd_swfrac_tile (Istr,Iend,Jstr,Jend,-1.0_r8,zgrid,swdk)
        do j=Jstr,Jend
          do i=Istr,Iend
            Bflux(i,j,k)=(Bo(i,j)+Bosol(i,j)*(1.0_r8-swdk(i,j)))
# ifdef MASKING
     &                  *rmask(i,j)
# endif
          enddo
        enddo
      enddo
!
!=====================================================================
!  Compute bulk Richardson number "Rib" and then find depth of the
!  oceanic bottom boundary layer "hbbl", such that Rib(hbbl)=Ric.
!=====================================================================
!
      do j=Jstr,Jend
# ifdef SPLINES
!
! Construct parabolic splines for vertical derivatives of potential
! density and velocity components at W-points.  FC is a scratch array.
!
        do i=Istr,Iend
          FC(i,0)=0.0_r8
          dR(i,0)=0.0_r8
          dU(i,0)=0.0_r8
          dV(i,0)=0.0_r8
        enddo
        do k=1,N-1
          do i=Istr,Iend
            cff=1.0_r8/(2.0_r8*Hz(i,j,k+1)+
     &                  Hz(i,j,k)*(2.0_r8-FC(i,k-1)))
            FC(i,k)=cff*Hz(i,j,k+1)
            dR(i,k)=cff*(6.0_r8*(pden(i,j,k+1)-pden(i,j,k))-
     &                   Hz(i,j,k)*dR(i,k-1))
            dU(i,k)=cff*(3.0_r8*(u(i  ,j,k+1,nstp)-u(i,  j,k,nstp)+
     &                           u(i+1,j,k+1,nstp)-u(i+1,j,k,nstp))-
     &                   Hz(i,j,k)*dU(i,k-1))
            dV(i,k)=cff*(3.0_r8*(v(i,j  ,k+1,nstp)-v(i,j  ,k,nstp)+
     &                           v(i,j+1,k+1,nstp)-v(i,j+1,k,nstp))-
     &                   Hz(i,j,k)*dV(i,k-1))
          enddo
        enddo
        do i=Istr,Iend
          dR(i,N)=0.0_r8
          dU(i,N)=0.0_r8
          dV(i,N)=0.0_r8
        enddo
        do k=N-1,1,-1
          do i=Istr,Iend
            dR(i,k)=dR(i,k)-FC(i,k)*dR(i,k+1)
            dU(i,k)=dU(i,k)-FC(i,k)*dU(i,k+1)
            dV(i,k)=dV(i,k)-FC(i,k)*dV(i,k+1)
          enddo
        enddo
# else
!
! Compute vertical derivatives of potential density and velocity
! components at W-points.
!
        do k=1,N-1
          do i=Istr,Iend
            cff=1.0_r8/(z_r(i,j,k+1)-z_r(i,j,k))
            dR(i,k)=cff*(pden(i,j,k+1)-pden(i,j,k))
            cff=0.5_r8*cff
            dU(i,k)=cff*(u(i  ,j,k+1,nstp)-u(i,  j,k,nstp)+
     &                   u(i+1,j,k+1,nstp)-u(i+1,j,k,nstp))
            dV(i,k)=cff*(v(i,j  ,k+1,nstp)-v(i,j  ,k,nstp)+
     &                   v(i,j+1,k+1,nstp)-v(i,j+1,k,nstp))
          enddo
        enddo
        do i=Istr,Iend
          dR(i,0)=0.0_r8
          dR(i,N)=0.0_r8
          dU(i,0)=0.0_r8
          dU(i,N)=0.0_r8
          dV(i,0)=0.0_r8
          dV(i,N)=0.0_r8
        enddo
# endif /* !SPLINES */
!
!---------------------------------------------------------------------
!  Compute bulk Richardson number "Rib" and then find depth of oceanic
!  bottom boundary layer "hbbl".
!
!                  [Br - B(d)] * d
!     Rib(d) = ----------------------- ;     Rib(hbbl)=Ric     (1)
!              |Vr - V(d)|^2 + Vt(d)^2
!
!  where "Br" and "Vr" are the bottom reference buoyancy and velocity
!  while "B(d)" and "V(d)" are the buoyancy and velocity at depth "d".
!
!  In the code below, the criterion "Rib(hbbl)=Ric" is reformulated
!  as follows:
!
!     Rib(d)       Ritop(d)
!     ------ = --------------- = 1                             (2)
!      Ric      Ric * Ribot(d)
!
!  where "Ritop" and "Ribot" are numerator and denominator in Eq. (1).
!  In its turn, Eq. (2) is rewritten in the following form:
!
!     FC(d) = Ritop(d) - Ric * Ribot(d) = 0                    (3)
!
!  That is, the planetary boundary layer extends to the depth where
!  the critical function "FC(d)" changes its sign.
!---------------------------------------------------------------------
!
!  Compute potential density and velocity components bottom reference
!  values.
!
        cff1=1.0_r8/3.0_r8
        cff2=1.0_r8/6.0_r8
        do i=Istr,Iend
          Rref=pden(i,j,1)-
     &         Hz(i,j,1)*(cff1*dR(i,0)+cff2*dR(i,1))
          Uref=0.5_r8*(u(i,j,1,nstp)+u(i+1,j,1,nstp))-
     &         Hz(i,j,1)*(cff1*dU(i,0)+cff2*dU(i,1))
          Vref=0.5_r8*(v(i,j,1,nstp)+v(i,j+1,1,nstp))-
     &         Hz(i,j,1)*(cff1*dV(i,0)+cff2*dV(i,1))
!
!  Compute critical function, FC, for bulk Richardson number.
!
          FC(i,0)=0.0_r8
          do k=1,N
            depth=z_w(i,j,k)-z_w(i,j,0)
            if (Bflux(i,j,k).lt.0.0_r8) then
              sigma=MIN(bl_dpth(i,j),depth)
            else
              sigma=depth
            endif
            call lmd_wscale (Bflux(i,j,k),Ustar(i,j),sigma,
     &                       wm(i,j),ws(i,j))
            Rk=pden(i,j,k)+
     &         Hz(i,j,k)*(cff1*dR(i,k)+cff2*dR(i,k-1))
            Uk=0.5_r8*(u(i,j,k,nstp)+u(i+1,j,k,nstp))+
     &         Hz(i,j,k)*(cff1*dU(i,k)+cff2*dU(i,k-1))
            Vk=0.5_r8*(v(i,j,k,nstp)+v(i,j+1,k,nstp))+
     &         Hz(i,j,k)*(cff1*dV(i,k)+cff2*dV(i,k-1))
!
            Ritop=-gorho0*(Rk-Rref)*depth
            Ribot=(Uk-Uref)**2+(Vk-Vref)**2+
     &            Vtc*depth*ws(i,j)*SQRT(ABS(bvf(i,j,k)))
# ifdef SASHA
            FC(i,k)=Ritop-lmd_Ric*Ribot
# else
            FC(i,k)=Ritop/(Ribot+eps)
# endif
          enddo
        enddo
!
! Linearly interpolate to find "hbbl" where Rib/Ric=1.
!
        do i=Istr,Iend
          kbbl(i,j)=N
          hbbl(i,j)=z_w(i,j,N)
        enddo
# ifdef SASHA
        do k=1,N-1
          do i=Istr,Iend
            if ((kbbl(i,j).eq.N).and.(FC(i,k).gt.0.0_r8)) then
              hbbl(i,j)=(z_w(i,j,k)*FC(i,k-1)-z_w(i,j,k-1)*FC(i,k))/
     &                  (FC(i,k-1)-FC(i,k))
              kbbl(i,j)=k
            endif
          enddo
        enddo
# else
        do k=1,N
          do i=Istr,Iend
            if ((kbbl(i,j).eq.N).and.((FC(i,k-1).lt.lmd_Ric).and.
     &                                (FC(i,k  ).ge.lmd_Ric))) then
              hbbl(i,j)=((lmd_Ric-FC(i,k-1))*z_w(i,j,k  )+
     &                   (FC(i,k  )-lmd_Ric)*z_w(i,j,k-1))/
     &                  (FC(i,k)-FC(i,k-1))
              kbbl(i,j)=k
            endif
          enddo
        enddo
# endif
      enddo
!
!  Compute total buoyancy flux at bottom boundary layer depth,
!  "Bfbot".
!
      do j=Jstr,Jend
        do i=Istr,Iend
          zgrid(i,j)=z_w(i,j,N)-hbbl(i,j)
        enddo
      enddo
      call lmd_swfrac_tile (Istr,Iend,Jstr,Jend,-1.0_r8,zgrid,swdk)
      do j=Jstr,Jend
        do i=Istr,Iend
          Bfbot(i,j)=(Bo(i,j)+Bosol(i,j)*(1.0_r8-swdk(i,j)))
# ifdef MASKING
     &              *rmask(i,j)
# endif
        enddo
      enddo
!
!  Under neutral and stable conditions, the depth of the bottom
!  boundary layer is required to be less than Ekman and Monin-Obukov
!  depths.
!
      do j=Jstr,Jend
        do i=Istr,Iend
          if (Ustar(i,j).ge.0.0_r8) then
            hekman=lmd_cekman*Ustar(i,j)/MAX(ABS(f(i,j)),eps)-h(i,j)
            hbbl(i,j)= MIN(hekman,hbbl(i,j))
          endif
          hbbl(i,j)=MIN(hbbl(i,j),z_w(i,j,N))
          hbbl(i,j)=MAX(hbbl(i,j),z_w(i,j,0))
# ifdef MASKING
     &             *rmask(i,j)
# endif
        enddo
      enddo
!
!  Apply gradient or periodic boundary conditions.
!
      call fill_r2d_tile (Istr,Iend,Jstr,Jend,hbbl(START_2D_ARRAY))
!
!  Find new boundary layer index "kbbl".
!
      do j=Jstr,Jend
        do i=Istr,Iend
          kbbl(i,j)=N
          do k=1,N
            if ((kbbl(i,j).eq.N).and.(z_w(i,j,k).gt.hbbl(i,j))) then
              kbbl(i,j)=k
            endif
          enddo
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute total buoyancy flux at final bottom boundary layer depth.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
          zgrid(i,j)=z_w(i,j,N)-hbbl(i,j)
        enddo
      enddo
      call lmd_swfrac_tile (Istr,Iend,Jstr,Jend,-1.0_r8,zgrid,swdk)
      do j=Jstr,Jend
        do i=Istr,Iend
          Bfbot(i,j)=(Bo(i,j)+Bosol(i,j)*(1.0_r8-swdk(i,j)))
# ifdef MASKING
     &              *rmask(i,j)
# endif
        enddo
      enddo
!
!=====================================================================
!  Compute vertical mixing coefficients within the planetary boundary
!  layer.
!=====================================================================
!
!  Compute tubulent velocity scales (wm,ws) at "hbbl".
!
      do j=Jstr,Jend
        do i=Istr,Iend
          bl_dpth(i,j)=lmd_epsilon*(hbbl(i,j)-z_w(i,j,0))
          if (Bfbot(i,j).gt.0.0_r8) then
            cff=1.0_r8
          else
            cff=lmd_epsilon
          endif
          sigma=cff*(hbbl(i,j)-z_w(i,j,0))
          call lmd_wscale (Bfbot(i,j),Ustar(i,j),sigma,
     &                     wm(i,j),ws(i,j))
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute nondimensional shape function Gx(sigma) in terms of the
!  interior diffusivities at sigma=1 (Gm1, Gs1, Gt1) and its vertical
!  derivative evaluated "hbbl" via interpolation.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
          f1(i,j)=5.0_r8*MAX(0.0_r8,Bfbot(i,j))*vonKar/
     &            (Ustar(i,j)*Ustar(i,j)*Ustar(i,j)*Ustar(i,j)+eps)
        enddo
      enddo
!
      do j=Jstr,Jend
        do i=Istr,Iend
          zbl=hbbl(i,j)-z_w(i,j,0)
          k=kbbl(i,j)
          cff=1.0_r8/(z_w(i,j,k)-z_w(i,j,k-1))
          cff_dn=cff*(hbbl(i,j)-z_w(i,j,k-1))
          cff_up=cff*(z_w(i,j,k)-hbbl(i,j))
!
!  Compute nondimensional shape function for viscosity "Gm1" and its
!  vertical derivative "dGm1dS" evaluated at "hbbl".
!
          K_bl=cff_dn*Kv(i,j,k)+cff_up*Kv(i,j,k-1)
          dK_bl=-cff*(Kv(i,j,k)-Kv(i,j,k-1))
          Gm1(i,j)=K_bl/(zbl*wm(i,j)+eps)
# ifdef MASKING
     &            *rmask(i,j)
# endif
          dGm1dS(i,j)=MIN(0.0_r8, K_bl*f1(i,j)-dK_bl/(wm(i,j)+eps))
!
!  Compute nondimensional shape function for diffusion of temperature
!  "Gt1" and its vertical derivative "dGt1dS" evaluated at "hbbl".
!
          K_bl=cff_dn*Kt(i,j,k)+cff_up*Kt(i,j,k-1)
          dK_bl=-cff*(Kt(i,j,k)-Kt(i,j,k-1))
          Gt1(i,j)=K_bl/(zbl*ws(i,j)+eps)
# ifdef MASKING
     &            *rmask(i,j)
# endif
          dGt1dS(i,j)=MIN(0.0_r8, K_bl*f1(i,j)-dK_bl/(ws(i,j)+eps))
# ifdef SALINITY
!
!  Compute nondimensional shape function for diffusion of salinity
!  "Gs1" and its vertical derivative "dGs1dS" evaluated at "hbbl".
!
          K_bl=cff_dn*Ks(i,j,k)+cff_up*Ks(i,j,k-1)
          dK_bl=-cff*(Ks(i,j,k)-Ks(i,j,k-1))
          Gs1(i,j)=K_bl/(zbl*ws(i,j)+eps)
#  ifdef MASKING
     &            *rmask(i,j)
#  endif
          dGs1dS(i,j)=MIN(0.0_r8, K_bl*f1(i,j)-dK_bl/(ws(i,j)+eps))
# endif /* SALINITY */
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute bottom boundary layer mixing coefficients.
!---------------------------------------------------------------------
!
      do k=1,Nm
        do j=Jstr,Jend
          do i=Istr,Iend
            if (z_w(i,j,k).lt.hbbl(i,j)) then
!
!  Compute turbulent velocity scales at vertical W-points.
!
              depth=z_w(i,j,k)-z_w(i,j,0)
              if (Bflux(i,j,k).lt.0.0_r8) then
                sigma=MIN(bl_dpth(i,j),depth)
              else
                sigma=depth
              endif
              call lmd_wscale (Bflux(i,j,k),Ustar(i,j),sigma,
     &                         wm(i,j),ws(i,j))
!
!  Set polynomial coefficients for shape function.
!
              sigma=depth/(hbbl(i,j)-z_w(i,j,0)+eps)
# ifdef MASKING
     &             *rmask(i,j)
# endif
              a1=sigma-2.0_r8
              a2=3.0_r8-2.0_r8*sigma
              a3=sigma-1.0_r8
!
!  Compute nondimesional shape functions.
!
              Gm=a1+a2*Gm1(i,j)+a3*dGm1dS(i,j)
              Gt=a1+a2*Gt1(i,j)+a3*dGt1dS(i,j)
# ifdef SALINITY
              Gs=a1+a2*Gs1(i,j)+a3*dGs1dS(i,j)
# endif /* SALINITY */
!
!  Compute boundary layer mixing coefficients, combine them
!  with interior mixing coefficients.
!
              Kv(i,j,k)=depth*wm(i,j)*(1.0_r8+sigma*Gm)
              Kt(i,j,k)=depth*ws(i,j)*(1.0_r8+sigma*Gt)
# ifdef SALINITY
              Ks(i,j,k)=depth*ws(i,j)*(1.0_r8+sigma*Gs)
# endif /* SALINITY */
            endif
          enddo
        enddo
!
!---------------------------------------------------------------------
!  Finally, set vertical mixing coefficients "Akv" and "Akt".
!---------------------------------------------------------------------
!
!  Take the maximum estimate of vertical mixing only where the surface
!  and bottom boundary layers overlap. (Do not let the interior
!  overwrite the boundary layer estimate).
!
        do j=Jstr,Jend
          do i=Istr,Iend
            if (z_w(i,j,k).lt.hbbl(i,j)) then
               if(k.gt.ksbl(i,j)) then
                  Akv(i,j,k)=MAX(Akv(i,j,k),Kv(i,j,k))
                  Akt(i,j,k,itemp)=MAX(Akt(i,j,k,itemp),Kt(i,j,k))
# ifdef SALINITY
                  Akt(i,j,k,isalt)=MAX(Akt(i,j,k,isalt),Ks(i,j,k))
# endif
               else
                 Akv(i,j,k)=Kv(i,j,k)
                 Akt(i,j,k,itemp)=Kt(i,j,k)
# ifdef SALINITY
                 Akt(i,j,k,isalt)=Ks(i,j,k)
# endif
               endif
            endif
          enddo
        enddo
      enddo
!
! Set gradient or periodic boundary conditions.
!
      call fill_w3d_tile (Istr,Iend,Jstr,Jend,
     &                    Akv(START_2D_ARRAY,0))
      call fill_w3d_tile (Istr,Iend,Jstr,Jend,
     &                    Akt(START_2D_ARRAY,0,itemp))
# ifdef SALINITY
      call fill_w3d_tile (Istr,Iend,Jstr,Jend,
     &                    Akt(START_2D_ARRAY,0,isalt))
# endif
#else
      subroutine lmd_bkpp
#endif /* LMD_BKPP */
      return
      end
