#include "cppdefs.h"
#ifdef FLOATS
      subroutine interp_floats (A,gtype,Lstr,Lend,itim,ifld,amask,
     &                          nudg)
!
!================================================== John M. Klinck ===
!  Copyright (c) 2002 Rutgers/UCLA                   Mark Hadfield   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine interpolates requested field at the float trajectory !
!  locations.                                                        !
!                                                                    !
!  On Input:                                                         !
!                                                                    !
!     A          Field to interpolate from (global real array).      !
!     gtype      Grid type (integer).  If negative, it interpolates  !
!                  floats slopes.                                    !
!     Lstr       Starting float location index to process (integer). !
!     Lend       Ending   float location index to process (integer). !
!     itim       Floats time level to process (integer).             !
!     ifld       Field ID to compute (integer).                      !
!     amask      Should the field be masked (logical)? Ignored if    !
!                  masking is not active.                            !
!     nudg       Term to be added to field (array dimensioned        !
!                  (Mfloats))                                        !
!                                                                    !
!  On Output:                                                        !
!                                                                    !
!     track      Interpolated field: track(ifld,itim,:).             !
!                                                                    !
!=====================================================================
!
      implicit none
#include "param.h"
#include "grid.h"
#include "floats.h"
#include "mask.h"
#include "ncparam.h"
#include "scalars.h"
!
      logical Irvar, Iuvar, Jrvar, Jvvar, Krvar, Kwvar, Lmask,
     &        amask, halo
      REAL_TYPE
     &        nudg(Mfloats)
      INTEGER_TYPE
     &        Ir, Iu, Jr, Jv, Kr, Kw, Lstr, Lend, ifld, iflt,
     &        itim, i1, i2, j1, j2, k1, k2, kh1, kh2, gtype, vtype
      REAL_TYPE
     &        p1, p2, q1, q2, r1, r2,
     &        s111, s211, s121, s221, s112, s212, s122, s222
# ifdef MASKING
      INTEGER_TYPE
     &        Irn, Irnm1, Irnp1, Jrn, Jrnm1, Jrnp1
      REAL_TYPE
     &        cff1, cff2
# endif
      REAL_TYPE
     &        A(GLOBAL_2D_ARRAY,*)
!
!---------------------------------------------------------------------
!  Initialize various internal variables.
!---------------------------------------------------------------------
!
!  Determine variable type switches.
!
      vtype=ABS(gtype)
      Irvar=(vtype.eq.r2dvar).or.(vtype.eq.r3dvar).or.
     &      (vtype.eq.v2dvar).or.(vtype.eq.v3dvar).or.
     &      (vtype.eq.w3dvar)
      Jrvar=(vtype.eq.r2dvar).or.(vtype.eq.r3dvar).or.
     &      (vtype.eq.u2dvar).or.(vtype.eq.u3dvar).or.
     &      (vtype.eq.w3dvar)
      Iuvar=(vtype.eq.u2dvar).or.(vtype.eq.u3dvar)
      Jvvar=(vtype.eq.v2dvar).or.(vtype.eq.v3dvar)
      Krvar=(vtype.eq.r3dvar).or.(vtype.eq.u3dvar).or.
     &      (vtype.eq.v3dvar)
      Kwvar=(vtype.eq.w3dvar)
!
!  Determine whether to allow for masking in horizontal interpolation.
!
# ifdef MASKING
      Lmask=amask
# else
      Lmask=.false.
# endif
!
!  If not interpolating slope, set multipliers to 1.
!
      if (gtype.ge.0) then
        s111=1.0_r8
        s121=1.0_r8
        s211=1.0_r8
        s221=1.0_r8
        s112=1.0_r8
        s122=1.0_r8
        s212=1.0_r8
        s222=1.0_r8
      endif
!
!---------------------------------------------------------------------
!  Loop through floats.
!---------------------------------------------------------------------
!
      do iflt=Lstr,Lend
        if (.not.bounded(iflt)) then
          track(ifld,itim,iflt)=spval
        else
!
!  Calculate indices and weights for vertical interpolation, if any.
!
          if (Krvar) then
            Kr=INT(track(izgrd,itim,iflt)+0.5_r8)
            k1=MIN(MAX(Kr  ,1),N)
            k2=MIN(MAX(Kr+1,1),N)
            r2=FLOAT(k2-k1)*(track(izgrd,itim,iflt)+0.5_r8-FLOAT(k1))
          elseif (Kwvar) then
            Kw=INT(track(izgrd,itim,iflt))+1
            k1=MIN(MAX(Kw  ,1),N+1)
            k2=MIN(MAX(Kw+1,1),N+1)
            r2=FLOAT(k2-k1)*(track(izgrd,itim,iflt)-FLOAT(k1)+1)
          else
            k1=1
            k2=1
            r2=0.0_r8
          endif
          r1=1.0_r8-r2
!
!---------------------------------------------------------------------
!  Interpolation at RHO-points.
!---------------------------------------------------------------------
!
          if (Irvar.and.Jrvar) then
!
            Ir=INT(track(ixgrd,itim,iflt))
            Jr=INT(track(iygrd,itim,iflt))
!
            i1=MIN(MAX(Ir  ,0),L)
            i2=MIN(MAX(Ir+1,1),L)
            j1=MIN(MAX(Jr  ,0),M)
            j2=MIN(MAX(Jr+1,0),M)
!
            p2=FLOAT(i2-i1)*(track(ixgrd,itim,iflt)-FLOAT(i1))
            q2=FLOAT(j2-j1)*(track(iygrd,itim,iflt)-FLOAT(j1))
            p1=1.0_r8-p2
            q1=1.0_r8-q2
# ifdef SOLVE3D
!
            if (gtype.lt.0) then
              kh1=MAX(k1-1,1)
              kh2=MIN(k2,N)
              s111=2.0_r8*pm(i1,j1)*pn(i1,j1)/
     &             (Hz(i1,j1,kh1)+Hz(i1,j1,k1))
              s211=2.0_r8*pm(i2,j1)*pn(i2,j1)/
     &             (Hz(i2,j1,kh1)+Hz(i2,j1,k1))
              s121=2.0_r8*pm(i1,j2)*pn(i1,j2)/
     &             (Hz(i1,j2,kh1)+Hz(i1,j2,k1))
              s221=2.0_r8*pm(i2,j2)*pn(i2,j2)/
     &             (Hz(i2,j2,kh1)+Hz(i2,j2,k1))
              s112=2.0_r8*pm(i1,j1)*pn(i1,j1)/
     &             (Hz(i1,j1,k2-1)+Hz(i1,j1,kh2))
              s212=2.0_r8*pm(i2,j1)*pn(i2,j1)/
     &             (Hz(i2,j1,k2-1)+Hz(i2,j1,kh2))
              s122=2.0_r8*pm(i1,j2)*pn(i1,j2)/
     &             (Hz(i1,j2,k2-1)+Hz(i1,j2,kh2))
              s222=2.0_r8*pm(i2,j2)*pn(i2,j2)/
     &             (Hz(i2,j2,k2-1)+Hz(i2,j2,kh2))
            endif
# endif
!
            if (Lmask) then
# ifdef MASKING
              cff1=p1*q1*r1*rmask(i1,j1)*s111*A(i1,j1,k1)+
     &             p2*q1*r1*rmask(i2,j1)*s211*A(i2,j1,k1)+
     &             p1*q2*r1*rmask(i1,j2)*s121*A(i1,j2,k1)+
     &             p2*q2*r1*rmask(i2,j2)*s221*A(i2,j2,k1)+
     &             p1*q1*r2*rmask(i1,j1)*s112*A(i1,j1,k2)+
     &             p2*q1*r2*rmask(i2,j1)*s212*A(i2,j1,k2)+
     &             p1*q2*r2*rmask(i1,j2)*s122*A(i1,j2,k2)+
     &             p2*q2*r2*rmask(i2,j2)*s222*A(i2,j2,k2)
              cff2=p1*q1*r1*rmask(i1,j1)+
     &             p2*q1*r1*rmask(i2,j1)+
     &             p1*q2*r1*rmask(i1,j2)+
     &             p2*q2*r1*rmask(i2,j2)+
     &             p1*q1*r2*rmask(i1,j1)+
     &             p2*q1*r2*rmask(i2,j1)+
     &             p1*q2*r2*rmask(i1,j2)+
     &             p2*q2*r2*rmask(i2,j2)
              if (cff2.gt.0.0_r8) then
                track(ifld,itim,iflt)=cff1/cff2+nudg(iflt)
              else
                track(ifld,itim,iflt)=0.0_r8
              endif
# endif
            else
              track(ifld,itim,iflt)=p1*q1*r1*s111*A(i1,j1,k1)+
     &                              p2*q1*r1*s211*A(i2,j1,k1)+
     &                              p1*q2*r1*s121*A(i1,j2,k1)+
     &                              p2*q2*r1*s221*A(i2,j2,k1)+
     &                              p1*q1*r2*s112*A(i1,j1,k2)+
     &                              p2*q1*r2*s212*A(i2,j1,k2)+
     &                              p1*q2*r2*s122*A(i1,j2,k2)+
     &                              p2*q2*r2*s222*A(i2,j2,k2)+
     &                              nudg(iflt)
            endif
!
!---------------------------------------------------------------------
!  Interpolation at horizontal velocity points.
!---------------------------------------------------------------------
!
          else
            Ir=INT(track(ixgrd,itim,iflt))
            Jr=INT(track(iygrd,itim,iflt))
            Iu=INT(track(ixgrd,itim,iflt)+0.5_r8)
            Jv=INT(track(iygrd,itim,iflt)+0.5_r8)
!
            halo=.false.
# ifdef MASKING
!
!  Is the float inside the halo (i.e. inside a masked grid cell or
!  within 0.5*delta of one)?  Depending on which portion of the cell
!  the float is in,  we look for adjacent and diagonally adjacent
!  masked cells.
!
!  Note that the halo-checking code is evaluated every time
!  interp_floats is called for a velocity variable with masking
!  activated. This is at least twice as often as it needs to be
!  called, as the float does not move between interpolating U and V.
!
!  The special-case code for periodic boundary conditions may be
!  unnecessary
!
            if (Lmask) then
              Irn=NINT(track(ixgrd,itim,iflt))
              Jrn=NINT(track(iygrd,itim,iflt))
#  ifdef EW_PERIODIC
              if (Irn.ge.Lm) then
                Irnp1=Irn+1-Lm
              else
                Irnp1=Irn+1
              endif
              if (Irn.le.1) then
                Irnm1=Irn-1+Lm
              else
                Irnm1=Irn-1
              endif
#  else
              Irnm1=Irn-1
              Irnp1=Irn+1
#  endif
#  ifdef NS_PERIODIC
              if (Jrn.ge.Mm) then
                Jrnp1=Jrn+1-Mm
              else
                Jrnp1=Jrn+1
              endif
              if (Jrn.le.1) then
                Jrnm1=Jrn-1+Mm
              else
                Jrnm1=Jrn-1
              endif
#  else
              Jrnm1=Jrn-1
              Jrnp1=Jrn+1
#  endif
              if (rmask(Irn,Jrn).lt.0.5_r8) then
                halo=.true.
              elseif ((Ir.lt.Irn).and.
     &                (rmask(Irn-1,Jrn).lt.0.5_r8)) then
                halo=.true.
              elseif ((Ir.eq.Irn).and.
     &                (rmask(Irn+1,Jrn).lt.0.5_r8)) then
                halo=.true.
              elseif ((Jr.lt.Jrn).and.
     &                (rmask(Irn,Jrn-1).lt.0.5_r8)) then
                halo=.true.
              elseif ((Jr.eq.Jrn).and.
     &                (rmask(Irn,Jrn+1).lt.0.5_r8)) then
                halo=.true.
              elseif ((Ir.lt.Irn).and.(Jr.lt.Jrn).and.
     &                (rmask(Irn-1,Jrn-1).lt.0.5_r8)) then
                halo=.true.
              elseif ((Ir.eq.Irn).and.(Jr.lt.Jrn).and.
     &                (rmask(Irn+1,Jrn-1).lt.0.5_r8)) then
                halo=.true.
              elseif ((Ir.lt.Irn).and.(Jr.eq.Jrn).and.
     &                (rmask(Irn-1,Jrn+1).lt.0.5_r8)) then
                halo=.true.
              elseif ((Ir.eq.Irn).and.(Jr.eq.Jrn).and.
     &                (rmask(Irn+1,Jrn+1).lt.0.5_r8)) then
                halo=.true.
              endif
            endif
# endif
!
!---------------------------------------------------------------------
!  Interpolation at U-points.
!---------------------------------------------------------------------
!
            if (Iuvar) then
              if (halo) then
# ifdef MASKING
!
!  Velocity interpolation inside the halo is linear in the parallel
!  direction and nearest-neighbour in the perpendicular direction.
!  The latter ensures that the perpendicular velocity is zero
!  everywhere on the perimeter of a masked cell.
!
                i1=MIN(MAX(Iu  ,1),L)
                i2=MIN(MAX(Iu+1,1),L)
                j1=Jrn
!
                p2=FLOAT(i2-i1)*
     &             (track(ixgrd,itim,iflt)-FLOAT(i1)+0.5_r8)
                p1=1.0_r8-p2
                q1=1.0_r8
!
                if (gtype.lt.0) then
                  s111=0.5_r8*(pm(i1-1,j1)+pm(i1,j1))
                  s211=0.5_r8*(pm(i2-1,j1)+pm(i2,j1))
                  s112=s111
                  s212=s112
                endif
!
                track(ifld,itim,iflt)=p1*q1*r1*s111*A(i1,j1,k1)+
     &                                p2*q1*r1*s211*A(i2,j1,k1)+
     &                                p1*q1*r2*s112*A(i1,j1,k2)+
     &                                p2*q1*r2*s212*A(i2,j1,k2)+
     &                                nudg(iflt)
# endif
              else
!
!  Bilinear interpolation outside halo.
!
                i1=MIN(MAX(Iu  ,1),L)
                i2=MIN(MAX(Iu+1,1),L)
                j1=MIN(MAX(Jr  ,0),M)
                j2=MIN(MAX(Jr+1,0),M)
!
                p2=FLOAT(i2-i1)*
     &             (track(ixgrd,itim,iflt)-FLOAT(i1)+0.5_r8)
                q2=FLOAT(j2-j1)*
     &             (track(iygrd,itim,iflt)-FLOAT(j1))
                p1=1.0_r8-p2
                q1=1.0_r8-q2
!
                if (gtype.lt.0) then
                  s111=0.5_r8*(pm(i1-1,j1)+pm(i1,j1))
                  s211=0.5_r8*(pm(i2-1,j1)+pm(i2,j1))
                  s121=0.5_r8*(pm(i1-1,j2)+pm(i1,j2))
                  s221=0.5_r8*(pm(i2-1,j2)+pm(i2,j2))
                  s112=s111
                  s212=s112
                  s122=s121
                  s222=s221
                endif
!
                track(ifld,itim,iflt)=p1*q1*r1*s111*A(i1,j1,k1)+
     &                                p2*q1*r1*s211*A(i2,j1,k1)+
     &                                p1*q2*r1*s121*A(i1,j2,k1)+
     &                                p2*q2*r1*s221*A(i2,j2,k1)+
     &                                p1*q1*r2*s112*A(i1,j1,k2)+
     &                                p2*q1*r2*s212*A(i2,j1,k2)+
     &                                p1*q2*r2*s122*A(i1,j2,k2)+
     &                                p2*q2*r2*s222*A(i2,j2,k2)+
     &                                nudg(iflt)
              endif
!
!---------------------------------------------------------------------
!  Interpolation at V-points.
!---------------------------------------------------------------------
!
            elseif (Jvvar) then
              if (halo) then
# ifdef MASKING
!
!  Velocity interpolation inside the halo is linear in the parallel
!  direction and nearest-neighbour in the perpendicular direction.
!  The latter ensures that the perpendicular velocity is zero
!  everywhere on the perimeter of a masked cell.
!
                i1=Irn
                j1=MIN(MAX(Jv  ,1),M)
                j2=MIN(MAX(Jv+1,1),M)
!
                q2=FLOAT(j2-j1)*
     &             (track(iygrd,itim,iflt)-FLOAT(j1)+0.5_r8)
                p1=1.0_r8
                q1=1.0_r8-q2
!
                if (gtype.lt.0) then
                  s111=0.5_r8*(pn(i1,j1-1)+pn(i1,j1))
                  s121=0.5_r8*(pn(i1,j2-1)+pn(i1,j2))
                  s112=s111
                  s122=s121
                endif
!
                track(ifld,itim,iflt)=p1*q1*r1*s111*A(i1,j1,k1)+
     &                                p1*q2*r1*s121*A(i1,j2,k1)+
     &                                p1*q1*r2*s112*A(i1,j1,k2)+
     &                                p1*q2*r2*s122*A(i1,j2,k2)+
     &                                nudg(iflt)
# endif
              else
!
!  Bilinear interpolation outside halo.
!
                i1=MIN(MAX(Ir  ,0),L)
                i2=MIN(MAX(Ir+1,1),L)
                j1=MIN(MAX(Jv  ,1),M)
                j2=MIN(MAX(Jv+1,1),M)
!
                p2=FLOAT(i2-i1)*
     &             (track(ixgrd,itim,iflt)-FLOAT(i1))
                q2=FLOAT(j2-j1)*
     &             (track(iygrd,itim,iflt)-FLOAT(j1)+0.5_r8)
                p1=1.0_r8-p2
                q1=1.0_r8-q2
!
                if (gtype.lt.0) then
                  s111=0.5_r8*(pn(i1,j1-1)+pn(i1,j1))
                  s211=0.5_r8*(pn(i2,j1-1)+pn(i2,j1))
                  s121=0.5_r8*(pn(i1,j2-1)+pn(i1,j2))
                  s221=0.5_r8*(pn(i2,j2-1)+pn(i2,j2))
                  s112=s111
                  s212=s112
                  s122=s121
                  s222=s221
                endif
!
                track(ifld,itim,iflt)=p1*q1*r1*s111*A(i1,j1,k1)+
     &                                p2*q1*r1*s211*A(i2,j1,k1)+
     &                                p1*q2*r1*s121*A(i1,j2,k1)+
     &                                p2*q2*r1*s221*A(i2,j2,k1)+
     &                                p1*q1*r2*s112*A(i1,j1,k2)+
     &                                p2*q1*r2*s212*A(i2,j1,k2)+
     &                                p1*q2*r2*s122*A(i1,j2,k2)+
     &                                p2*q2*r2*s222*A(i2,j2,k2)+
     &                                nudg(iflt)
              endif
            endif
          endif
        endif
      enddo
#else
      subroutine interp_floats
#endif
      return
      end
