#include "cppdefs.h"
      subroutine inp_par
!
!=====================================================================
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine reads in input model parameters from standard input. !
!  It also writes out these parameters to standard output.           !
!                                                                    !
!=====================================================================
!
      implicit none
#include "param.h"
#include "bbl.h"
#include "grid.h"
#include "iounits.h"
#include "mixing.h"
#include "ncparam.h"
#include "scalars.h"
#include "strings.h"
!
      INTEGER_TYPE
     &        i, icard, itrc, iunit, k, lstr, wtype
      INTEGER_TYPE
     &        lenstr
      REAL_TYPE
     &        cff
#ifdef SOLVE3D
      character*17 frmt
#endif /* SOLVE3D */
      character*19 ref_att
      character*80 fname
      parameter (iunit=40)
!
!---------------------------------------------------------------------
!  Read in and report input model parameters.
!---------------------------------------------------------------------
!
      icard=0
      stdinp=5
      stdout=6
      call get_date (date_str)
      lstr=lenstr(date_str)
      write(stdout,10) version, date_str(1:lstr)
  10  format(/,' Model input parameters:  ROMS version ',a,/,26x,a,/)
!
!  Read input parameters according to their input card number.
!
      do while (icard.lt.99)
!
!  Read in and report out time-stepping parameters.
!
        if (icard.eq.1) then
          read(stdinp,*,err=620) ntimes, dt, ndtfast, ERstr, ERend
          write(stdout,20) ntimes, dt, ndtfast, ERstr, ERend
  20      format(i10,2x,'ntimes',t26,
     &             'Number of timesteps for 3-D equations.',/,
     &           f10.2,2x,'dt',t26,
     &             'Timestep size (s) for 3-D equations.',/,
     &           4x,i6,2x,'ndtfast',t26,
     &             'Number of timesteps for 2-D equations between ',
     &             /,t28,'each 3D timestep.',/,
     &           4x,i6,2x,'ERstr',t26,
     &             'Starting ensemble/perturbation run number.',/,
     &           4x,i6,2x,'ERend',t26,
     &             'Ending ensemble/perturbation run number.')
!
!  Read in and report out input/output parameters.
!
        elseif (icard.eq.2) then
          read(stdinp,*,err=620) nrrec, lcycle, nrst, nsta, nflt,
     &                           ninfo
          LastRec=.false.
          if (nrrec.lt.0) LastRec=.true.
          write(stdout,30) nrrec, lcycle, nrst, ninfo
  30      format(4x,i6,2x,'nrrec',t26,
     &             'Number of restart records to read from disk.',/,
     &           9x,l1,2x,'lcycle',t26,
     &             'Switch to recycle time-records in restart NetCDF',
     &             ' file.',/,
     &           4x,i6,2x,'nrst',t26,
     &             'Number of timesteps between storage of restart ',
     &             'fields.',/,
     &           4x,i6,2x,'ninfo',t26,
     &             'Number of timesteps between print of information',
     &             /,t28,'to standard output.')
#ifdef STATIONS
          write(stdout,40) nsta
  40      format(4x,i6,2x,'nsta',t26,
     &             'Number of timesteps between writing of data into',
     &             /,t28, 'stations file.')
#endif /* STATIONS */
#ifdef FLOATS
          write(stdout,50) nflt
  50      format(4x,i6,2x,'nflt',t26,
     &             'Number of timesteps between writing of data into',
     &             /,t28, 'floats file.')
#endif /* FLOATS */
!
!  Read in and report out history and time-average files parameters.
!
        elseif (icard.eq.3) then
          read(stdinp,*,err=620) ldefout, nhis, ndefhis, ntsavg, navg,
     &                           ndefavg
          write(stdout,60) ldefout, nhis
  60      format(9x,l1,2x,'ldefout',t26,
     &             'Switch to create a new output NetCDF file(s).',/,
     &           4x,i6,2x,'nhis',t26,
     &             'Number of timesteps between writing fields into',
     &             /,t28,'history file.')
          if (ndefhis.gt.0) then
            write(stdout,70) ndefhis
  70        format(4x,i6,2x,'ndefhis',t26,
     &               'Number of timesteps between creation of new',
     &               /,t28,'history files.')
          endif
#ifdef AVERAGES
          write(stdout,80) ntsavg, navg
  80      format(4x,i6,2x,'ntsavg',t26,
     &             'Starting timestep for the accumulation of output',
     &             /,t28,'time-averaged data.',/,
     &           4x,i6,2x,'navg',t26,
     &             'Number of timesteps between writing of ',
     &             'time-averaged',/,t28,'data into averages file.')
          if (ndefavg.gt.0) then
            write(stdout,90) ndefavg
  90        format(4x,i6,2x,'ndefavg',t26,
     &               'Number of timesteps between creation of new',
     &               /,t28,'time-averaged files.')
          endif
#endif /* AVERAGES */
!
!  Read in and report out horizontal Laplacian mixing coefficients
!  for tracer type variables.
!
        elseif (icard.eq.4) then
          read(stdinp,*,err=620) (tnu2(itrc),itrc=1,NAT)
#if defined SOLVE3D && defined TS_DIF2
          do itrc=1,NAT
            lstr=lenstr(Vname(1,idTvar(itrc)))
            write(stdout,100) tnu2(itrc), itrc, itrc,
     &                        Vname(1,idTvar(itrc))(1:lstr)
          enddo
 100      format(1p,e10.3,2x,'tnu2(',i1,')',t26,
     &             'Horizontal, Laplacian mixing coefficient (m2/s)',
     &           /,t28,'for tracer ',i1,': ',a)
#endif /* SOLVE3D && TS_DIF2 */
!
!  Read in and report out horizontal biharmonic mixing coefficients
!  for tracer type variables.
!
        elseif (icard.eq.5) then
          read(stdinp,*,err=620) (tnu4(itrc),itrc=1,NAT)
#if defined SOLVE3D && defined TS_DIF4
          do itrc=1,NAT
            lstr=lenstr(Vname(1,idTvar(itrc)))
            write(stdout,110) tnu4(itrc), itrc, itrc,
     &                        Vname(1,idTvar(itrc))(1:lstr)
          enddo
 110      format(1p,e10.3,2x,'tnu4(',i1,')',t26,
     &            'Horizontal, biharmonic mixing coefficient (m4/s)',
     &           /,t28,'for tracer ',i1,': ',a)
#endif /* SOLVE3D && TS_DIF4 */
!
!  Read in and report out isopycnal thickness diffusivity for tracer
!  type variables.
!
        elseif (icard.eq.6) then
          read(stdinp,*,err=620) (Kdiff(itrc),itrc=1,NAT)
#if defined SOLVE3D && defined GENT_McWILLIAMS
          do itrc=1,NAT
            lstr=lenstr(Vname(1,idTvar(itrc)))
            write(stdout,120) Kdiff(itrc), itrc, itrc,
     &                        Vname(1,idTvar(itrc))(1:lstr)
 120        format(1p,e10.3,2x,'Kdiff(',i1,')',t26,
     &             'Isopycnal mixing thickness diffusivity (m2/s)',
     &           /,t28,'for tracer ',i1,': ',a)
          enddo
#endif /* SOLVE3D && GENT_McWILLIANS */
!
!  Read in and report out horizontal viscosity coefficients.
!
        elseif (icard.eq.7) then
          read(stdinp,*,err=620) visc2, visc4
#ifdef UV_VIS2
          write(stdout,130) visc2
 130      format(1p,e10.3,2x,'visc2',t26,
     &             'Horizontal, Laplacian mixing coefficient (m2/s)',
     &           /,t28,'for momentum.')
#endif /* UV_VIS2 */
#ifdef UV_VIS4
          write(stdout,140) visc4
 140      format(1p,e10.3,2x,'visc4',t26,
     &            'Horizontal, biharmonic mixing coefficient (m4/s)',
     &           /,t28,'for momentum.')
#endif /* UV_VIS4 */
!
!  Read in and report out background vertical mixing coefficients
!  for tracer type variables.
!
        elseif (icard.eq.8) then
          read(stdinp,*,err=620) (Akt_bak(itrc),itrc=1,NAT)
#ifdef SOLVE3D
          do itrc=1,NAT
            lstr=lenstr(Vname(1,idTvar(itrc)))
            write(stdout,150) Akt_bak(itrc), itrc, itrc,
     &                        Vname(1,idTvar(itrc))(1:lstr)
          enddo
 150      format(1p,e10.3,2x,'Akt_bak(',i1,')',t26,
     &             'Background vertical mixing coefficient (m2/s)',
     &           /,t28,'for tracer ',i1,': ',a)
#endif /* SOLVE3D */
!
!  Read in and report out background vertical mixing coefficients
!  for momentum.
!
        elseif (icard.eq.9) then
          read(stdinp,*,err=620) Akv_bak
#ifdef SOLVE3D
          write(stdout,160) Akv_bak
 160      format(1p,e10.3,2x,'Akv_back',t26,
     &             'Background vertical mixing coefficient (m2/s)',
     &           /,t28,'for momentum.')
#endif /* SOLVE3D */
!
!  Read in and report out turbulent closure parameters.
!
        elseif (icard.eq.10) then
          read(stdinp,*,err=620) Akk_bak, Akp_bak, tkenu2, tkenu4
#if defined SOLVE3D && (defined MY25_MIXING || defined GLS_MIXING)
          write(stdout,170) Akk_bak, Akp_bak
 170      format(1p,e10.3,2x,'Akk_bak',t26,
     &             'Background vertical mixing coefficient (m2/s)',
     &           /,t28,'for turbulent energy.',/,
     &           1p,e10.3,2x,'Akp_bak',t26,
     &             'Background vertical mixing coefficient (m2/s)',
     &           /,t28,'for turbulent generic statistical field.')
# ifdef TKE_DIF2
          write(stdout,180) tkenu2
 180      format(1p,e10.3,2x,'tkenu2',t26,
     &             'Horizontal, harmonic mixing coefficient (m2/s)',
     &           /,t28,'for turbulent energy.')
# endif /* TKE_DIF2 */
# ifdef TKE_DIF4
          write(stdout,190) tkenu4
 190      format(1p,e10.3,2x,'tkenu4',t26,
     &            'Horizontal, biharmonic mixing coefficient (m2/s)',
     &           /,t28,'for turbulent energy.')
# endif /* TKE_DIF4 */
#endif /* SOLVE3D && (GLS_MIXING || MY25_MIXING) */
!
!  Read in generic length-scale turbulence closure parameters.        
!
        elseif (icard.eq.11) then
          read(stdinp,*,err=620) gls_p, gls_m, gls_n,
     &                           gls_Kmin, gls_Pmin
#if defined SOLVE3D && defined GLS_MIXING
          write(stdout,200) gls_p, gls_m, gls_n, gls_Kmin, gls_Pmin
 200      format(f10.4,2x,'gls_p',t26,
     &             'GLS stability exponent.',/,
     &           f10.4,2x,'gls_m',t26,
     &             'GLS turbulent kinetic energy exponent.',/,
     &           f10.4,2x,'gls_n',t26,
     &             'GLS turbulent length scale exponent.',/,     
     &           1p,e10.3,2x,'gls_Kmin',t26,
     &             'GLS minimum value of turbulent kinetic energy.',/,
     &           1p,e10.3,2x,'gls_Pmin',t26,
     &             'GLS minimum value of dissipation.')
#endif /* SOLVE3D && GLS_MIXING */
!
!  Read in generic length-scale turbulence closure independent
!  constraint parameters (nondimensional).
!
        elseif (icard.eq.12) then
          read(stdinp,*,err=620) gls_cmu0, gls_c1, gls_c2, gls_c3m,
     &                           gls_c3p, gls_sigk, gls_sigp
#if defined SOLVE3D && defined GLS_MIXING
          write(stdout,210) gls_cmu0, gls_c1, gls_c2, gls_c3m,
     &                      gls_c3p, gls_sigk, gls_sigp
 210      format(f10.4,2x,'gls_cmu0',t26,
     &             'GLS stability coefficient.',/,
     &           f10.4,2x,'gls_c1',t26,
     &             'GLS shear production coefficient.',/,
     &           f10.4,2x,'gls_c2',t26,
     &             'GLS dissipation coefficient.',/,
     &           f10.4,2x,'gls_c3m',t26,
     &             'GLS buoyancy production coefficient (minus).',/,
     &           f10.4,2x,'gls_c3p',t26,
     &             'GLS buoyancy production coefficient (plus).',/,
     &           f10.4,2x,'gls_sigk',t26,
     &             'GLS constant Schmidt number for TKE.',/,
     &           f10.4,2x,'gls_sigp',t26,
     &             'GLS constant Schmidt number for PSI.')
#endif /* SOLVE3D && GLS_MIXING */
!
!  Read in and report out bottom drag coefficients.
!
        elseif (icard.eq.13) then
          read(stdinp,*,err=620) rdrg, rdrg2, Zob, Zos
          if (rdrg.ne.0.0_r8) write(stdout,220) rdrg
 220      format(1p,e10.3,2x,'rdrg',t26,
     &             'Linear bottom drag coefficient (m/s).')
          if (rdrg2.ne.0.0_r8) write(stdout,230) rdrg2
 230      format(1p,e10.3,2x,'rdrg2',t26,
     &             'Quadratic bottom drag coefficient.')
          if (Zob.ne.0.0_r8) write(stdout,240) Zob
 240      format(1p,e10.3,2x,'Zob',t26,
     &             'Bottom roughness (m).')
          write(stdout,250) Zos
 250      format(1p,e10.3,2x,'Zos',t26,
     &             'Surface roughness (m).')
#ifdef BBL
          if (Zob.le.0.0_r8) then
            write(stdout,260) Zob
 260        format(' INP_PAR - Invalid bottom roughness parameter, ',
     &             'Zob = ',1p,e10.3,/,11x,
     &             'It must greater than zero.')
            stop
          endif
# ifdef SG_BBL
          sg_znotdef=Zob
# endif
#endif /* BBL */
!
!  Read in and report out various parameters.
!
        elseif (icard.eq.14) then
          read(stdinp,*,err=620) wtype, adv_ord, levsfrc, levbfrc
#ifdef SOLVE3D
# if defined LMD_SKPP || defined SOLAR_SOURCE
          lmd_Jwt=wtype
          write(stdout,270) lmd_Jwt
 270      format(4x,i6,2x,'lmd_Jwt',t26,
     &             'Jerlov water type.')
          if ((lmd_Jwt.lt.1).or.(lmd_Jwt.gt.5)) then
            write(stdout,280) lmd_Jwt
 280        format(' INP_PAR - invalid Jerlow water type, lmd_Jwt = ',
     &             i5,/,11x,'It must between one and five.')
            stop
          endif
# endif /* LMD_SKPP || SOLAR_SOURCE */
# ifdef SMOLARKIEWICZ
          write(stdout,290) adv_ord
 290      format(4x,i6,2x,'adv_ord',t26,
     &             'Order of Smolarkiewicz advection scheme.')
# endif /*  SMOLARKIEWICZ */
# ifdef BODYFORCE
          if ((levsfrc.lt.1).or.(levsfrc.gt.N)) then
            write(stdout,300) 'LEVSFRC = ',levsfrc
 300        format(' INP_PAR - Illegal bodyforce level, ',a,i4)
            stop
          endif
          if ((levbfrc.lt.1).or.(levbfrc.gt.N)) then
            write(stdout,270) 'LEVBFRC = ',levbfrc
            stop
          endif
          write(stdout,310) levsfrc, levbfrc
 310      format(4x,i6,2x,'levsfrc',t26,
     &             'Deepest level to apply surface stress as a ',
     &             'bodyfroce.',/,
     &           4x,i6,2x,'levbfrc',t26,
     &             'Shallowest level to apply bottom stress as a ',
     &             'bodyfroce.')
# endif /* BODYFORCE */
#endif /* SOLVE3D */
!
!  Read in and report out vertical S-coordinates parameters.
!
        elseif (icard.eq.15) then
          read(stdinp,*,err=620) theta_s, theta_b, Tcline
#ifdef SOLVE3D
          write(stdout,320) theta_s, theta_b, Tcline
 320      format(1p,e10.3,2x,'theta_s',t26,
     &             'S-coordinate surface control parameter.',/,
     &           1p,e10.3,2x,'theta_b',t26,
     &             'S-coordinate bottom  control parameter.',/,
     &           f10.4,2x,'Tcline',t26,
     &             'S-coordinate surface/bottom layer width (m) used',
     &           /,t28,'in vertical coordinate stretching.')
#endif /* SOLVE3D */
!
!  Read in and report out mean constant density and time stamp.
!
        elseif (icard.eq.16) then
          read(stdinp,*,err=620) rho0, dstart, time_ref
          write(stdout,330) rho0, dstart, time_ref
 330      format(f10.4,2x,'rho0',t26,
     &             'Mean density (kg/m3) used in Boussinesq ',
     &             'approximation.',/,
     &           f10.4,2x,'dstart',t26,
     &             'Time stamp assigned to model initialization ',
     &             '(days).',/,
     &           f10.1,2x,'time_ref',t26,
     &             'Reference time for units attribute (yyyymmdd.f)')
!
!  Read in and report out nudging/relaxation time scales for tracer
!  type variables.  The inverse time scales will computed somewhere
!  in this routine.
!
        elseif (icard.eq.17) then
          read(stdinp,*,err=620) (Tnudg(itrc),itrc=1,NAT)
#ifdef SOLVE3D
          do itrc=1,NAT
            lstr=lenstr(Vname(1,idTvar(itrc)))
            write(stdout,340) Tnudg(itrc), itrc, itrc,
     &                        Vname(1,idTvar(itrc))(1:lstr)
          enddo
 340      format(1p,e10.3,2x,'Tnudg(',i1,')',t26,
     &             'Nudging/relaxation time scale (days)',
     &           /,t28,'for tracer ',i1,': ',a)
# ifdef SCORRECTION
          if (Tnudg(isalt).le.0.0_r8) then
            write(stdout,350) Tnudg(isalt)
 350        format('INP_PAR - Inappropriate value for ',
     &             'Tnudg(isalt) = ',1p,e15.8)
            stop
          endif
# endif
#endif /* SOLVE3D */
!
!  Read in and report out nudging/relaxation time scales.  The inverse
!  time scales will computed somewhere in this routine.
!
        elseif (icard.eq.18) then
          read(stdinp,*,err=620) Znudg, M2nudg, M3nudg, obcfac
          write(stdout,360) Znudg, M2nudg
 360      format(1p,e10.3,2x,'Znudg',t26,
     &             'Nudging/relaxation time scale (days)',
     &           /,t28,'for free-surface.',/,
     &           1p,e10.3,2x,'M2nudg',t26,
     &             'Nudging/relaxation time scale (days)',
     &           /,t28,'for 2D momentum.')
#ifdef SOLVE3D
          write(stdout,370) M3nudg
 370      format(1p,e10.3,2x,'M3nudg',t26,
     &             'Nudging/relaxation time scale (days)',
     &           /,t28,'for 3D momentum.')
#endif /* SOLVE3D */
          write(stdout,380) obcfac
 380      format(1p,e10.3,2x,'obcfac',t26,
     &             'Factor between passive and active',
     &           /,t28,'open boundary conditions.')
!
!  Read in and report out linear equations of state parameters.
!
        elseif (icard.eq.19) then
          read(stdinp,*,err=620) R0, T0, S0, Tcoef, Scoef
#ifdef SOLVE3D
          write(stdout,390) T0, S0
 390      format(f10.4,2x,'T0',t26,
     &             'Background potential temperature (Celsius) ',
     &             'constant.',/,
     &           f10.4,2x,'S0',t26,
     &             'Background salinity (PSU) constant.')
# ifndef NONLIN_EOS
          write(stdout,400) R0, Tcoef, Scoef
 400      format(f10.4,2x,'R0',t26,
     &             'Background density (kg/m3) used in linear ',
     &             'Equation',/,t28,'of State.',/,
     &           1p,e10.3,2x,'Tcoef',t26,
     &             'Thermal expansion coefficient (1/Celsius).',/,
     &           1p,e10.3,2x,'Scoef',t26,
     &             'Saline contraction coefficient (1/PSU).')
# endif /* !NONLIN_EOS */
#endif /* SOLVE3D */
!
!  Lateral boundary parameters.
!
        elseif (icard.eq.20) then
          read(stdinp,*,err=620) gamma2
          write(stdout,410) gamma2
 410      format(f10.2,2x,'gamma2',t26,
     &             'Slipperiness variable: free-slip (1.0) or ',
     &           /,50x,'no-slip (-1.0).')
!
!  Writing switches for fields associated with momentum equations.
!
        elseif (icard.eq.21) then
          read(stdinp,*,err=620) Hout(idUvel), Hout(idVvel),
     &                           Hout(idWvel), Hout(idOvel),
     &                           Hout(idUbar), Hout(idVbar),
     &                           Hout(idFsur)
#ifdef SOLVE3D
          write(stdout,420) Hout(idUvel), Hout(idVvel), Hout(idWvel),
     &                      Hout(idOvel)
 420      format(9x,l1,2x,'Hout(idUvel)',t26,
     &             'Write out 3D U-momentum component (T/F).',/,
     &           9x,l1,2x,'Hout(idVvel)',t26,
     &             'Write out 3D V-momentum component (T/F).',/,
     &           9x,l1,2x,'Hout(idWvel)',t26,
     &             'Write out W-momentum component (T/F).',/,
     &           9x,l1,2x,'Hout(idOvel)',t26,
     &             'Write out omega vertical velocity (T/F).')
#endif /* SOLVE3D */
          write(stdout,430) Hout(idUbar), Hout(idVbar), Hout(idFsur)
 430      format(9x,l1,2x,'Hout(idUbar)',t26,
     &             'Write out 2D U-momentum component (T/F).',/,
     &           9x,l1,2x,'Hout(idVbar)',t26,
     &             'Write out 2D V-momentum component (T/F).',/,
     &           9x,l1,2x,'Hout(idFsur)',t26,
     &             'Write out free-surface (T/F).')
!
!  Writing switches for fields associated with tracers equations.
!
        elseif (icard.eq.22) then
          read(stdinp,*,err=620) (Hout(idTvar(itrc)),itrc=1,NAT)
#ifdef SOLVE3D
          do itrc=1,NAT
            lstr=lenstr(Vname(1,idTvar(itrc)))
            write(stdout,440) Hout(idTvar(itrc)), itrc,
     &                        Vname(1,idTvar(itrc))(1:lstr)
          enddo
 440      format(9x,l1,2x,'Hout(idTvar)',t26,
     &             'Write out tracer ',i1,': ',a,' (T/F).')
#endif /* SOLVE3D */
!
!  Writing switches for surface and bottom momentum stresses.
!
        elseif (icard.eq.23) then
          read(stdinp,*,err=620) Hout(idUsms), Hout(idVsms),
     &                           Hout(idUbms), Hout(idVbms),
     &                           Hout(idUbws), Hout(idVbws)
          write(stdout,450) Hout(idUsms), Hout(idVsms), Hout(idUbms),
     &                      Hout(idVbms)
 450      format(9x,l1,2x,'Hout(idUsms)',t26,
     &             'Write out surface U-momentum stress (T/F).',/,
     &           9x,l1,2x,'Hout(idVsms)',t26,
     &             'Write out surface V-momentum stress (T/F).',/,
     &           9x,l1,2x,'Hout(idUbms)',t26,
     &             'Write out bottom U-momentum stress (T/F).',/,
     &           9x,l1,2x,'Hout(idVbms)',t26,
     &             'Write out bottom U-momentum stress (T/F).')
#ifdef BBL
          write(stdout,460) Hout(idUbws), Hout(idVbws)
 460      format(9x,l1,2x,'Hout(idUbws)',t26,
     &             'Write out wind-induced, bottom U-wave stress.',/,
     &           9x,l1,2x,'Hout(idVbws)',t26,
     &             'Write out wind-induced, bottom V-wave stress.')
#endif /* BBL */
!
!  Writing switches for bed boundary layer fields.
!
        elseif (icard.eq.24) then
          read(stdinp,*,err=620) Hout(idAbed), Hout(idUbed),
     &                           Hout(idVbed), Hout(idUbot),
     &                           Hout(idVbot), Hout(idHrip),
     &                           Hout(idLrip), Hout(idZnot),
     &                           Hout(idZapp)
#ifdef BBL
          write(stdout,470) Hout(idAbed), Hout(idUbed), Hout(idVbed),
     &                      Hout(idUbot), Hout(idVbot), Hout(idHrip),
     &                      Hout(idLrip), Hout(idZnot), Hout(idZapp)
 470      format(9x,l1,2x,'Hout(idAbed)',t26,
     &             'Write out bed wave excursion amplitude (T/F).',/,
     &           9x,l1,2x,'Hout(idUbed)',t26,
     &             'Write out bed wave orbital U-velocity (T/F).',/,
     &           9x,l1,2x,'Hout(idVbed)',t26,
     &             'Write out bed wave orbital V-velocity (T/F).',/,
     &           9x,l1,2x,'Hout(idUbot)',t26,
     &             'Write out bottom U-momentum above bed (T/F).',/,
     &           9x,l1,2x,'Hout(idVbot)',t26,
     &             'Write out bottom V-momentum above bed (T/F).',/,
     &           9x,l1,2x,'Hout(idHrip)',t26,
     &             'Write out bed ripple height (T/F).',/,
     &           9x,l1,2x,'Hout(idLrip)',t26,
     &             'Write out bed ripple length (T/F).',/,
     &           9x,l1,2x,'Hout(idZnot)',t26,
     &             'Write out bottom roughness (T/F).',/,
     &           9x,l1,2x,'Hout(idVbws)',t26,
     &             'Write out apparent bottom roughness (T/F).')
#endif /* BBL */
!
!  Writing switches for surface heat flux components.
!
        elseif (icard.eq.25) then
          read(stdinp,*,err=620) Hout(idTsur(1)), Hout(idLhea),
     &                           Hout(idShea),    Hout(idLrad),
     &                           Hout(idSrad)
#ifdef SOLVE3D
          write(stdout,480) Hout(idTsur(1))
 480      format(9x,l1,2x,'Hout(idTsur)',t26,
     &             'Write out surface net heat flux (T/F).')
# ifdef SHORTWAVE
          write(stdout,490) Hout(idSrad)
 490      format(9x,l1,2x,'Hout(idSrad)',t26,
     &             'Write out shortwave radiation flux (T/F).')
# endif
# ifdef BULK_FLUXES
          write(stdout,500) Hout(idLrad), Hout(idLhea), Hout(idShea)
 500      format(9x,l1,2x,'Hout(idLrad)',t26,
     &             'Write out longwave radiation flux (T/F).',/,
     &           9x,l1,2x,'Hout(idLhea)',t26,
     &             'Write out latent heat flux (T/F).',/,
     &           9x,l1,2x,'Hout(idShea)',t26,
     &             'Write out sensible heat flux (T/F).')
# endif
#endif /* SOLVE3D */
!
!  Writing switches for other fields.
!
        elseif (icard.eq.26) then
          read(stdinp,*,err=620) Hout(idDano), Hout(idVvis),
     &                           Hout(idTdif), Hout(idSdif),
     &                           Hout(idHsbl), Hout(idHbbl),
     &                           Hout(idMtke), Hout(idMtls)
#ifdef SOLVE3D
          write(stdout,510) Hout(idDano), Hout(idVvis), Hout(idTdif),
     &                      Hout(idSdif)
 510      format(9x,l1,2x,'Hout(idDano)',t26,
     &             'Write out density anomaly (T/F).',/,
     &           9x,l1,2x,'Hout(idVvis)',t26,
     &             'Write out vertical viscosity coefficient ',
     &             '(T/F).',/,
     &           9x,l1,2x,'Hout(idTdif)',t26,
     &             'Write out vertical T-diffusion coefficient ',
     &             '(T/F).',/
     &           9x,l1,2x,'Hout(idSdif)',t26,
     &             'Write out vertical S-diffusion coefficient ',
     &             '(T/F).')
# ifdef LMD_SKPP
          write(stdout,520) Hout(idHsbl)
 520      format(9x,l1,2x,'Hout(idHsbl)',t26,
     &             'Write out depth of surface boundary layer (T/F).')
# endif /* LMD_SKPP */
# ifdef LMD_BKPP
          write(stdout,530) Hout(idHbbl)
 530      format(9x,l1,2x,'Hout(idHbbl)',t26,
     &             'Write out depth of bottom boundary layer (T/F).')
# endif /* LMD_BKPP */
# if defined GLS_MIXING || defined MY25_MIXING
          write(stdout,540) Hout(idMtke), Hout(idMtls)
 540      format(9x,l1,2x,'Hout(idMtke)',t26,
     &             'Write out turbulent kinetic energy (T/F).',/,
     &           9x,l1,2x,'Hout(idMtls)',t26,
     &             'Write out turbulent generic length-scale (T/F).')
# endif /* GLS_MIXING || MY25_MIXING */
#endif /* SOLVE3D */
!
!  Number and levels to write out. If Nlev<0, all model levels will
!  be written out. If NLEV=0, the history file will not be created.
!
        elseif (icard.eq.27) then
          read(stdinp,*,err=620) Nlev, (Lev(k),k=1,Nlev)
          if (Nlev.lt.0) then
            Nlev=N
            do k=1,Nlev
              Lev(k)=k
            enddo
          endif
#ifdef SOLVE3D
          if (Nlev.gt.N) then
            write(stdout,550)
 550        format(' INP_PAR - Number of levels requested are',
     &                        'greater than N.')
            stop
          endif
          do k=1,Nlev
            if ((Lev(k).le.0).or.(Lev(k).gt.N)) then
              write(stdout,560) Lev(k)
              stop
            endif
          enddo
 560      format(' INP_PAR - Illegal output level, Lev = ',i3)
          do i=1,NV
            if (Hout(i)) LdefHIS=.true.
          enddo
          if (Nlev.eq.0) LdefHIS=.false.
          if (LdefHIS) then
            write(stdout,570) Nlev
 570        format(4x,i6,2x,'Nlev',t26,
     &               'Number of levels to write out.',/,
     &             12x,'Lev',t26,
     &               'Levels to write out:')
            if (Nlev.eq.N) then
              write(stdout,575) Nlev
 575          format(t26,'001-',i3.3)
            else
              write(frmt,580) Nlev
 580          format('(t26,',i3,'(i3.3,1x))')
              write(stdout,frmt) (Lev(k),k=1,Nlev)
            endif
          endif
#else
          if (Hout(idFsur).or.Hout(idUbar).or.Hout(idVbar)) then
            LdefHIS=.true.
          endif
#endif /* SOLVE3D */
!
!  Read in and report generic USERs parameters.
!
        elseif (icard.eq.28) then
          read(stdinp,*,err=620) Nuser, (user(k),k=1,Nuser)
          if (Nuser.gt.0) then
            write(stdout,590)
 590        format(/,/,' Generic User Parameters:',/)
            do i=1,Nuser
              write(stdout,600) user(i), i, i
            enddo
 600        format(1p,e10.3,2x,'user(',i2.2,')',t26,
     &               'User parameter ',i2.2,'.')
          endif
!
!  Read in and report out run title.
!
        elseif (icard.eq.29) then
          read(stdinp,'(a)',err=620) title
          write(stdout,610) title
 610      format(/,a)
!
!  Read in output restart file name, if any.
!
        elseif (icard.eq.30) then
          read(stdinp,'(a)',err=620) rstname
!
!  Read in output history file name, if any.
!
        elseif (icard.eq.31) then
          read(stdinp,'(a)',err=620) hisname
          hisbase=hisname
!
!  Read in output averages file name, if any.
!
        elseif (icard.eq.32) then
          read(stdinp,'(a)',err=620) avgname
          avgbase=avgname
!
!  Read in output stations file name, if any.
!
        elseif (icard.eq.33) then
          read(stdinp,'(a)',err=620) staname
!
!  Read in output floats file name, if any.
!
        elseif (icard.eq.34) then
          read(stdinp,'(a)',err=620) fltname
!
!  Read in input grid file name, if any.
!
        elseif (icard.eq.35) then
          read(stdinp,'(a)',err=620) grdname
!
!  Read in input initial conditions file name, if any.
!
        elseif (icard.eq.36) then
          read(stdinp,'(a)',err=620) ininame
!
!  Read in input forcing file name, if any.
!
        elseif (icard.eq.37) then
          read(stdinp,'(a)',err=620) frcname
!
!  Read in input climatology file name, if any.
!
        elseif (icard.eq.38) then
          read(stdinp,'(a)',err=620) clmname
!
!  Read in input climatology file name, if any.
!
        elseif (icard.eq.39) then
          read(stdinp,'(a)',err=620) bryname
!
!  Read in input forward solution file name, if any.
!
        elseif (icard.eq.40) then
          read(stdinp,'(a)',err=620) fwdname
!
!  Read in input assimilation parameters file name, if any.
!
        elseif (icard.eq.41) then
          read(stdinp,'(a)',err=620) aparnam
!
!  Read in input station positions file name, if any.
!
        elseif (icard.eq.42) then
          read(stdinp,'(a)',err=620) sposnam
!
!  Read in input initial floats positions file name.
!
        elseif (icard.eq.43) then
          read(stdinp,'(a)',err=620) fposnam
!
!  Read in input biology parameters file name.
!
        elseif (icard.eq.44) then
          read(stdinp,'(a)',err=620) bparnam
!
!  Read in input sediment parameters file name.
!
        elseif (icard.eq.45) then
          read(stdinp,'(a)',err=620) sparnam
!
!  Read in USERs input/output generic file name.
!
        elseif (icard.eq.46) then
          read(stdinp,'(a)',err=620) usrname
        endif
!
!  Read next input card ID.
!
        read(stdinp,*,err=620) icard
      enddo
      goto 640
!
!  Error while reading input parameters.
!
 620  write(stdout,630) icard
 630  format(/,' INP_PAR - error while reading standard input line: ',
     &       i2)
      stop
 640  continue
!
!---------------------------------------------------------------------
!  Report output/input files and check availability of input files.
!---------------------------------------------------------------------
!
      write(stdout,650)
 650  format(/,/,' Output/Input Files:',/)
      lstr=lenstr(rstname)
      write(stdout,660) '          Output Restart File:  ',
     &                  rstname(1:lstr)
 660  format(2x,a,a)
      if (LdefHIS) then
        lstr=lenstr(hisname)
        if (ndefhis.eq.0) then
          write(stdout,660) '          Output History File:  ',
     &                      hisname(1:lstr)
        else
          write(stdout,660) '     Prefix for History Files:  ',
     &                      hisname(1:lstr-3)
        endif
      endif
#ifdef AVERAGES
      lstr=lenstr(avgname)
      if (ndefavg.eq.0) then
        write(stdout,660) '         Output Averages File:  ',
     &                    avgname(1:lstr)
      else
        write(stdout,660) '    Prefix for Averages Files:  ',
     &                    avgname(1:lstr-3)
      endif
#endif /* AVERAGES */
#ifdef STATIONS
      lstr=lenstr(staname)
      write(stdout,660) '         Output Stations File:  ',
     &                  staname(1:lstr)
#endif /* STATIONS */
#ifdef FLOATS
      lstr=lenstr(fltname)
      write(stdout,660) '           Output Floats File:  ',
     &                  fltname(1:lstr)
#endif /* FLOATS */
#ifndef ANA_GRID
      fname=grdname
      lstr=lenstr(grdname)
      open(iunit,file=grdname(1:lstr),status='old',err=670)
      write(stdout,660) '              Input Grid File:  ',
     &                  grdname(1:lstr)
      close(iunit)
#endif /* ANA_GRID */
#ifndef ANA_INITIAL
      fname=ininame
      lstr=lenstr(ininame)
      open(iunit,file=ininame(1:lstr),status='old',err=670)
      write(stdout,660) '           Input Initial File:  ',
     &                  ininame(1:lstr)
      close(iunit)
#endif /* ANA_INITIAL */
#ifdef FRC_FILE
      fname=frcname
      lstr=lenstr(frcname)
      open(iunit,file=frcname(1:lstr),status='old',err=670)
      write(stdout,660) '           Input Forcing File:  ',
     &                  frcname(1:lstr)
      close(iunit)
#endif /* FRC_FILE */
#if (defined  ZCLIMATOLOGY && !defined ANA_SSH)     || \
    (defined M2CLIMATOLOGY && !defined ANA_M2CLIMA) || \
    (defined  TCLIMATOLOGY && !defined ANA_TCLIMA)  || \
    (defined M3CLIMATOLOGY && !defined ANA_M3CLIMA)
      fname=clmname
      lstr=lenstr(clmname)
      open(iunit,file=clmname(1:lstr),status='old',err=670)
      write(stdout,660) '       Input Climatology File:  ',
     &                  clmname(1:lstr)
      close(iunit)
#endif
#ifdef OBC_DATA
      fname=bryname
      lstr=lenstr(bryname)
      open(iunit,file=bryname(1:lstr),status='old',err=670)
      write(stdout,660) '          Input Boundary File:  ',
     &                  bryname(1:lstr)
      close(iunit)
#endif /* OBC_DATA */
#ifdef STATIONS
      fname=sposnam
      lstr=lenstr(sposnam)
      open(iunit,file=sposnam(1:lstr),status='old',err=670)
      write(stdout,660) '       Station positions File:  ',
     &                  sposnam(1:lstr)
      close(iunit)
#endif /* STATIONS */
#if defined ASSIMILATION || defined ESSE || defined NUDGING
      fname=aparnam
      lstr=lenstr(aparnam)
      open(iunit,file=aparnam(1:lstr),status='old',err=670)
      close(iunit)
#endif /* ASSIMILATION || ESSE || NUDGING */
#ifdef FLOATS
      fname=fposnam
      lstr=lenstr(fposnam)
      open(iunit,file=fposnam(1:lstr),status='old',err=670)
      write(stdout,660) 'Initial Floats Positions File:  ',
     &                  fposnam(1:lstr)
      close(iunit)
#endif /* FLOATS */
#ifdef BIOLOGY
      fname=bparnam
      lstr=lenstr(bparnam)
      open(iunit,file=bparnam(1:lstr),status='old',err=670)
      write(stdout,660) '       Biology Parameter File:  ',
     &                  bparnam(1:lstr)
      close(iunit)
#endif /* FLOATS */
      fname=usrname
      lstr=lenstr(usrname)
      write(stdout,660) '       Input/Output USER File:  ',
     &                  usrname(1:lstr)
      close(iunit)
      goto 690
 670  write(stdout,680) fname(1:lstr)
 680  format(/,' INP_PAR - could not find input file:  ',a)
      stop
 690  continue
#if defined ASSIMILATION || defined ESSE || defined NUDGING
!
!---------------------------------------------------------------------
!  Read in input assimilation parameters.
!---------------------------------------------------------------------
!
      call inp_Apar
#endif /* ASSIMILATION || ESSE || NUDGING */
#ifdef BIOLOGY
!
!---------------------------------------------------------------------
!  Read input stations parameters.
!---------------------------------------------------------------------
!
      call inp_Bpar
#endif /* BIOLOGY */
#ifdef SEDIMENT
!
!---------------------------------------------------------------------
!  Read input sediment parameters.
!---------------------------------------------------------------------
!
      call inp_sed
#endif /* SEDIMENT */
#ifdef STATIONS
!
!---------------------------------------------------------------------
!  Read input stations parameters.
!---------------------------------------------------------------------
!
      call inp_Spar
#endif /* STATIONS */
!
!---------------------------------------------------------------------
!  Check C-preprocessing options and definitions.
!---------------------------------------------------------------------
!
      call checkdefs
!
!---------------------------------------------------------------------
!  Compute various constants.
!---------------------------------------------------------------------
!
      gorho0=g/rho0
      dtfast=dt/FLOAT(ndtfast)
!
!  Take the square root of the biharmonic coefficients so it can
!  be applied to each harmonic operator.
!
      visc4=SQRT(ABS(visc4))
      tkenu4=SQRT(ABS(tkenu4))
      do itrc=1,NAT
        tnu4(itrc)=SQRT(ABS(tnu4(itrc)))
      enddo
!
!  Compute inverse nudging coefficients (1/s) used in various tasks.
!
      if (Znudg.gt.0.0_r8) then
        Znudg=1.0_r8/(Znudg*86400.0_r8)
      else
        Znudg=0.0_r8
      endif
      if (M2nudg.gt.0.0_r8) then
        M2nudg=1.0_r8/(M2nudg*86400.0_r8)
      else
        M2nudg=0.0_r8
      endif
#ifdef SOLVE3D
      if (M3nudg.gt.0.0_r8) then
        M3nudg=1.0_r8/(M3nudg*86400.0_r8)
      else
        M3nudg=0.0_r8
      endif
      do itrc=1,NAT
        if (Tnudg(itrc).gt.0.0_r8) then
          Tnudg(itrc)=1.0_r8/(Tnudg(itrc)*86400.0_r8)
        else
          Tnudg(itrc)=0.0_r8
        endif
      enddo
#endif /* SOLVE3D */
!
!  Set scales for input forcing fields.  Notice that for consitency
!  with Boussinesq approximation, these scales are overwritten here
!  to use the provided value for "rho0".
!
      cff=1.0_r8/rho0
      Fscale(idUsms)=cff
      Fscale(idVsms)=cff
      Fscale(idUbms)=cff
      Fscale(idVbms)=cff
      Fscale(idUbws)=cff
      Fscale(idVbws)=cff
      cff=1.0_r8/(rho0*Cp)
      Fscale(idTsur(itemp))=cff
      Fscale(idTbot(itemp))=cff
      Fscale(idSrad)=cff
      Fscale(idLrad)=cff
      Fscale(idLhea)=cff
      Fscale(idShea)=cff
      Fscale(iddQdT)=cff
!
!  Build time reference attribute string.
!
      r_text=ref_att(time_ref,r_date)
      return
      end
#if defined ASSIMILATION || defined ESSE || defined NUDGING
      subroutine inp_Apar
!
!=====================================================================
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This subroutine reads in input model assimilation parameters from !
!  input file.  It also reports these parameters to standard output. !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "ncparam.h"
# include "obs.h"
# include "iounits.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        icard, itrc, iunit, k, lfstr, lstr
      INTEGER_TYPE
     &        lenstr
      character*80 fname
      parameter (iunit=40)
!
!---------------------------------------------------------------------
!  Read in assimilation parameters.
!---------------------------------------------------------------------
!
      lfstr=0
      lstr=lenstr(aparnam)
      open(iunit,file=aparnam(1:lstr),form='formatted',status='old')
      write(stdout,10)
  10  format(/,' Data assimilation parameters:',/)
# if defined ASSIMILATION || defined NUDGING
!
!  Read input parameters according to their input card number.
!
      icard=0
      do while (icard.lt.99)
!
!  Read in number of stations.
!
        if (icard.eq.1) then
          read(iunit,*,err=140) Emod0, Tgrowth, cor
#  ifdef ASSIMILATION
          write(stdout,20) Emod0, Tgrowth, cor
  20      format(f10.2,2x,'Emod0',t26,
     &             'Initial model error (percentage).',/,
     &           f10.2,2x,'Tgrowth',t26,
     &             'Empirical model error growth scale (days).',/,
     &           1p,e10.3,2x,'cor',t26,
     &             'Correlation between model and observations.')
#  endif
          Emod0=1.0_r8-0.01_r8*Emod0
!
!  Read in assimilation activation switches for tracers.
!
        elseif (icard.eq.2) then
          read(iunit,*,err=140) (assi_T(itrc),itrc=1,NAT)
#  if defined ASSIMILATION && defined SOLVE3D
          do itrc=1,NAT
            write(stdout,30) assi_T(itrc), itrc, itrc
          enddo
  30      format(9x,l1,2x,'assi_T(',i1,')',t26,
     &             'Activate assimilation of tracer ',i1,' (T/F).')
#  endif
!
!  Read in assimilation activation switches for other variables.
!
        elseif (icard.eq.3) then
          read(iunit,*,err=140) assi_SSH, assi_SST, assi_UVsur,
     &                          assi_UV
#  ifdef ASSIMILATION
          write(stdout,40) assi_SSH
  40      format(9x,l1,2x,'assi_SSH',t26,
     &             'Activate assimilation of SSH (T/F).')
#   ifdef SOLVE3D
          write(stdout,50) assi_SST, assi_UVsur, assi_UV
  50      format(9x,l1,2x,'assi_SST',t26,
     &             'Activate assimilation of SST (T/F).',/,
     &           9x,l1,2x,'assi_UVsur',t26,
     &             'Activate assimilation of surface currents ',
     &             '(T/F).',/,
     &           9x,l1,2x,'assi_UV',t26,
     &             'Activate assimilation of currents (T/F).')
#   endif
#  endif
!
!  Read in and report out nudging assimilation inverse time scales.
!
        elseif (icard.eq.4) then
          read(iunit,*,err=140) (Tnudass(itrc),itrc=1,NAT)
#  ifdef SOLVE3D
#   if defined NUDGING_T || defined NUDGING_SST
          do itrc=1,NAT
            if (assi_T(itrc)) then
              lfstr=lenstr(Vname(1,idTvar(itrc)))
              write(stdout,60) Tnudass(itrc), itrc, itrc,
     &                         Vname(1,idTvar(itrc))(1:lfstr)
            endif
          enddo
  60      format(1p,e10.3,2x,'Tnudass(',i1,')',t26,
     &             'Nudging assimilation time scale (days)',
     &           /,t28,'for tracer ',i1,': ',a)
#   endif
#  endif /* SOLVE3D */
!
!  Read in and report out nudging assimilation inverse time scales.
!
        elseif (icard.eq.5) then
          read(iunit,*,err=140) Znudass, M2nudass, M3nudass
#  ifdef NUDGING_SSH
          write(stdout,70) Znudass
  70      format(1p,e10.3,2x,'Znudass',t26,
     &             'Nudging assimilation time scale (days)',
     &           /,t28,'for free-surface.')
#  endif
#  ifdef SOLVE3D
#   if defined NUDGING_UV || defined NUDGING_UVsur
          write(stdout,80) M3nudass
  80      format(1p,e10.3,2x,'M3nudass',t26,
     &             'Nudging assimilation time scale (days)',
     &           /,t28,'for 3D momentum.')
#   endif
#  endif
!
!  Read in extension parameters for SST.
!
        elseif (icard.eq.6) then
          read(iunit,*,err=140) extend_SST, npSST, ZmSST, ZoSST
          ZmSST=-ABS(ZmSST)
          ZoSST=-ABS(ZoSST)
#  ifdef SOLVE3D
#   if defined NUDGING_SST || defined ASSIMILATION_SST
          if (extend_SST) then
            write(stdout,90) extend_SST, npSST, ZmSST, ZoSST
  90        format(9x,l1,2x,'extend_SST',t26,
     &               'Extend SST vertically (T/F).',/,
     &             7x,i3,2x,'npSST',t26,
     &               'Order of polynomial for SST extension.',/,
     &             f10.2,2x,'ZmSST',t26,
     &               'SST, valid maximum extension depth (m).',/,
     &             f10.2,2x,'ZoSST',t26,
     &               'E-folding depth (m) to extend SST error ',
     &               'variance.',/)
            if (KP.lt.npSST) then
              write(stdout,100) KP, npSST
 100          format(/,' INP_APAR - too small dimension parameter, ',
     &               'KP: ',2i4,/,12x,
     &               'change file  obs.h  and recompile.')
              stop
            endif
          endif
#   endif
#  endif
!
!  Read in polynomials for extending SST and error variance
!  vertically.
!
        elseif (icard.eq.7) then
          do itrc=0,npSST
            read(iunit,*,err=140) k, pcoef_SST(k), perr_SST(k)
#  ifdef SOLVE3D
#   if defined NUDGING_SST || defined ASSIMILATION_SST
            if (npSST.gt.0) then
              write(stdout,110) k, pcoef_SST(k), perr_SST(k)
 110          format(1x,'SST,Err polynomial order ',
     &               i2.2,1x,1p,e15.8,1x,1p,e15.8)
            endif
#   endif
#  endif
          enddo
!
!  Read in extension parameters for surface currents.
!
        elseif (icard.eq.8) then
          read(iunit,*,err=140) extend_UV, npUV, ZmUV, ZoUV
          ZmUV=-ABS(ZmUV)
          ZoUV=-ABS(ZoUV)
#  ifdef SOLVE3D
#   if defined NUDGING_UVsur || defined ASSIMILATION_UVsur
          if (extend_UV) then
            write(stdout,120) extend_UV, npUV, ZmUV, ZoUV
 120        format(9x,l1,2x,'extend_UV',t26,
     &               'Extend surface currents vertically (T/F).',/,
     &             7x,i3,2x,'npUV',t26,
     &               'Order of polynomial for surface currents',
     &               ' extension.',/,
     &             f10.2,2x,'ZmUV',t26,
     &               'Surface currents, valid maximum extension',
     &               ' depth (m).',/,
     &             f10.2,2x,'ZoUV',t26,
     &               'E-folding depth (m) to extend surface currents',
     &               ' error',/,t28,'variance.',/)
            if (KP.lt.npSST) then
              write(stdout,125) KP, npUV
 125          format(/,' INP_APAR - too small dimension parameter, ',
     &               'KP: ',2i4,/,12x,
     &               'change file  obs.h  and recompile.')
              stop
            endif
          endif
#   endif
#  endif
!
!  Read in polynomial for extending surface currents vertically.
!
        elseif (icard.eq.9) then
          do itrc=0,npUV
            read(iunit,*,err=140) k, pcoef_U(k), pcoef_V(k),
     &                            perr_V(k)
#  ifdef SOLVE3D
#   if defined NUDGING_UVsur || defined ASSIMILATION_UVsur
            if (npUV.gt.0) then
              write(stdout,130) k, pcoef_U(k), pcoef_V(k), perr_V(k)
 130          format(1x,'U,V,Err polynomial order: ',
     &               i2.2,1x,1p,e15.8,1x,1p,e15.8,1x,1p,e15.8)
            endif
#   endif
#  endif
          enddo
!
!  Read in input SSH observations file name, if any.
!
        elseif (icard.eq.10) then
          read(iunit,'(a)',err=140) SSHname
!
!  Read in input SST observations file name, if any.
!
        elseif (icard.eq.11) then
          read(iunit,'(a)',err=140) SSTname
!
!  Read in input surface currents observations file name, if any.
!
        elseif (icard.eq.12) then
          read(iunit,'(a)',err=140) VSURname
!
!  Read in input current observations file name, if any.
!
        elseif (icard.eq.13) then
          read(iunit,'(a)',err=140) VOBSname
!
!  Read in input tracers observations file name, if any.
!
        elseif (icard.eq.14) then
          read(iunit,'(a)',err=140) TOBSname
        endif
!
!  Read next input card ID.
!
        read(iunit,*,err=140) icard
      enddo
      goto 160
!
!  Error while reading input parameters.
!
 140  write(stdout,150) icard, aparnam(1:lstr)
 150  format(/,' INP_APAR - error while reading input line: ',i2,
     &       /,12x,'from assimilation file: ',a)
      stop
 160  close(iunit)
!
!  Report input assimilation files.
!
      write(stdout,170)
 170  format(/,' Input Assimilation Files:',/)
      lstr=lenstr(aparnam)
      write(stdout,'(2x,a,a)') ' Assimilation parameters File:  ',
     &                  aparnam(1:lstr)
#  if defined ASSIMILATION_SSH || defined NUDGING_SSH
      fname=SSHname
      lstr=lenstr(SSHname)
      open(iunit,file=SSHname(1:lstr),status='old',err=180)
      write(stdout,'(2x,a,a)') '      Sea Surface Height File:  ',
     &                         SSHname(1:lstr)
      close(iunit)
#  endif
#  ifdef SOLVE3D
#   if defined ASSIMILATION_SST || defined NUDGING_SST
      fname=SSTname
      lstr=lenstr(SSTname)
      open(iunit,file=SSTname(1:lstr),status='old',err=180)
      write(stdout,'(2x,a,a)') ' Sea Surface Temperature File:  ',
     &                         SSTname(1:lstr)
      close(iunit)
#   endif
#   if defined ASSIMILATION_T || defined NUDGING_T
      fname=TOBSname
      lstr=lenstr(TOBSname)
      open(iunit,file=TOBSname(1:lstr),status='old',err=180)
      write(stdout,'(2x,a,a)') '                 Tracers File:  ',
     &                         TOBSname(1:lstr)
      close(iunit)
#   endif
#   if defined ASSIMILATION_UVsur || defined NUDGING_UVsur
      fname=VSURname
      lstr=lenstr(VSURname)
      open(iunit,file=VSURname(1:lstr),status='old',err=180)
      write(stdout,'(2x,a,a)') '        Surface Currents File:  ',
     &                         VSURname(1:lstr)
      close(iunit)
#   endif
#   if defined ASSIMILATION_UV || defined NUDGING_UV
      fname=VOBSname
      lstr=lenstr(VOBSname)
      open(iunit,file=VOBSname(1:lstr),status='old',err=180)
      write(stdout,'(2x,a,a)') '     Horizontal Currents File:  ',
     &                         VOBSname(1:lstr)
      close(iunit)
#   endif
#  endif /* SOLVE3D */
      goto 200
 180  write(stdout,190) fname(1:lstr)
 190  format(/,' INP_APAR - could not find input file:  ',a)
      stop
 200  continue
#  ifdef NUDGING
!
!---------------------------------------------------------------------
!  Compute inverse nudging coefficients (1/s) used in data
!  assimilation.
!---------------------------------------------------------------------
!
      if (Znudass.gt.0.0_r8) then
        Znudass=1.0_r8/(Znudass*86400.0_r8)
      else
        Znudass=0.0_r8
      endif
      if (M2nudass.gt.0.0_r8) then
        M2nudass=1.0_r8/(M2nudass*86400.0_r8)
      else
        M2nudass=0.0_r8
      endif
#   ifdef SOLVE3D
      if (M3nudass.gt.0.0_r8) then
        M3nudass=1.0_r8/(M3nudass*86400.0_r8)
      else
        M3nudass=0.0_r8
      endif
      do itrc=1,NAT
        if (Tnudass(itrc).gt.0.0_r8) then
          Tnudass(itrc)=1.0_r8/(Tnudass(itrc)*86400.0_r8)
        else
          Tnudass(itrc)=0.0_r8
        endif
      enddo
#   endif /* SOLVE3D */
#  endif /* NUDGING */
# endif /* ASSIMILATION || NUDGING */
# ifdef ESSE
!
!  Read input parameters according to their input card number.
!
      icard=0
      do while (icard.lt.99)
!
!  Read in ensemble run parameters.
!
        if (icard.eq.1) then
          read(iunit,*,err=160) Nsev
#  ifdef PERTURBATION
          write(stdout,20) Nsev
  20      format(4x,i6,2x,'Nsev',t26,
     &             'Number of state error vectors to use.')
#  endif
!
!  Read in various paramenters.
!
        elseif (icard.eq.2) then
          read(iunit,*,err=160) err_fct
          write(stdout,30) err_fct
  30      format(1p,e10.3,2x,'err_fct',t26,
     &             'Model RMS adjustment factor.')
!
!  Read in perturbation switches for state variables.
!
        elseif (icard.eq.3) then
          read(iunit,*,err=160) pert(isFsur), pert(isUbar),
     &                          pert(isVbar), pert(isUvel),
     &                          pert(isVvel)
#  ifdef PERTURBATION
          write(stdout,40) pert(isFsur), isFsur,
     &                     pert(isUbar), isUbar,
     &                     pert(isVbar), isVbar
  40      format(9x,l1,2x,'pert(',i1')',t26,
     &             'Activate free-surface perturbation (T/F).',/,
     &           9x,l1,2x,'pert(',i1')',t26,
     &             'Activate 2D U-momentum perturbation (T/F).',/,
     &           9x,l1,2x,'pert(',i1')',t26,
     &             'Activate 2D V-momentum perturbation (T/F).')
#   ifdef SOLVE3D
          write(stdout,50) pert(isUvel), isUvel,
     &                     pert(isVvel), isVvel
  50      format(9x,l1,2x,'pert(',i1')',t26,
     &             'Activate 3D U-momentum perturbation (T/F).',/,
     &           9x,l1,2x,'pert(',i1')',t26,
     &             'Activate 3D V-momentum perturbation (T/F).')
#   endif
#  endif
!
!  Read in perturbation switches for tracer state variables.
!
        elseif (icard.eq.4) then
          read(iunit,*,err=160) (pert(isTvar(itrc)),itrc=1,NAT)
#  if defined PERTURBATION && defined SOLVE3D
          do itrc=1,NAT
             write(stdout,60) pert(isTvar(itrc)), isTvar(itrc), itrc
          enddo
  60      format(9x,l1,2x,'pert(',i1,')',t26,
     &             'Activate perturbation of tracer ',i1,' (T/F).')
#  endif
!
!  Read in nondimensional perturbation strenght for state variables.
!
        elseif (icard.eq.5) then
          read(iunit,*,err=160) Prms(isFsur), Prms(isUbar),
     &                          Prms(isVbar), Prms(isUvel),
     &                          Prms(isVvel)
#  ifdef PERTURBATION
          if (pert(isFsur)) then
            write(stdout,70) Prms(isFsur), isFsur
  70        format(1p,e10.3,2x,'Prms(',i1')',t26,
     &               'Perturbation strenght for free-surface.')
          endif
          if (pert(isUbar)) then
            write(stdout,80) Prms(isUbar), isUbar
  80        format(1p,e10.3,2x,'Prms(',i1')',t26,
     &               'Perturbation strenght for 2D U-momentum.')
          endif
          if (pert(isVbar)) then
            write(stdout,90) Prms(isVbar), isVbar
  90        format(1p,e10.3,2x,'Prms(',i1')',t26,
     &               'Perturbation strenght for 2D V-momentum.')
          endif
#   ifdef SOLVE3D
          if (pert(isUvel)) then
            write(stdout,100) Prms(isUvel), isUvel
 100        format(1p,e10.3,2x,'Prms(',i1')',t26,
     &               'Perturbation strenght for 3D U-momentum.')
          endif
          if (pert(isVvel)) then
            write(stdout,110) Prms(isVvel), isVvel
 110        format(1p,e10.3,2x,'Prms(',i1')',t26,
     &               'Perturbation strenght for 3D V-momentum.')
          endif
#   endif
#  endif
!
!  Read in nondimensional perturbation strenght for tracer variables.
!
        elseif (icard.eq.6) then
          read(iunit,*,err=160) (Prms(isTvar(itrc)),itrc=1,NAT)
#  if defined PERTURBATION && defined SOLVE3D
          do itrc=1,NAT
            if (pert(isTvar(itrc))) then
              write(stdout,120) Prms(isTvar(itrc)), isTvar(itrc), itrc
            endif
          enddo
 120      format(1p,e10.3,2x,'Prms(',i1,')',t26,
     &             'Perturbation strenght for tracer ',i1)
#  endif
!
!  Read in assimilation switches for state variables.
!
        elseif (icard.eq.7) then
          read(iunit,*,err=160) assi(isFsur), assi(isUbar),
     &                          assi(isVbar), assi(isUvel),
     &                          assi(isVvel)
#  ifdef ASSIMILATION
          write(stdout,130) assi(isFsur), isFsur,
     &                     assi(isUbar), isUbar,
     &                     assi(isVbar), isVbar
 130      format(9x,l1,2x,'pert(',i1')',t26,
     &             'Activate free-surface assimilation (T/F).',/,
     &           9x,l1,2x,'pert(',i1')',t26,
     &             'Activate 2D U-momentum assimilation (T/F).',/,
     &           9x,l1,2x,'pert(',i1')',t26,
     &             'Activate 2D V-momentum assimilation (T/F).')
#   ifdef SOLVE3D
          write(stdout,140) assi(isUvel), isUvel,
     &                      assi(isVvel), isVvel
 140      format(9x,l1,2x,'assi(',i1')',t26,
     &             'Activate 3D U-momentum perturbation (T/F).',/,
     &           9x,l1,2x,'assi(',i1')',t26,
     &             'Activate 3D V-momentum perturbation (T/F).')
#   endif
#  endif
!
!  Read in assimilation switches for tracer state variables.
!
        elseif (icard.eq.8) then
          read(iunit,*,err=160) (assi(isTvar(itrc)),itrc=1,NAT)
#  if defined ASSIMILATION && defined SOLVE3D
          do itrc=1,NAT
             write(stdout,150) assi(isTvar(itrc)), isTvar(itrc), itrc
          enddo
 150      format(9x,l1,2x,'pert(',i1,')',t26,
     &             'Activate assimilation of tracer ',i1,' (T/F).')
#  endif
!
!  Read in input error subspace file name, if any.
!
        elseif (icard.eq.9) then
          read(iunit,'(a)',err=160) ESname
        endif
!
!  Read next input card ID.
!
        read(iunit,*,err=160) icard
      enddo
      goto 180
!
!  Error while reading input parameters.
!
 160  write(stdout,170) icard, aparnam(1:lstr)
 170  format(/,' INP_APAR - error while reading input card: ',i2,
     &       /,12x,'from assimilation file: ',a)
      stop
 180  close(iunit)
!
!  Report input assimilation files.
!
      write(stdout,190)
 190  format(/,' Input Assimilation Files:',/)
      lstr=lenstr(aparnam)
      write(stdout,'(2x,a,a)') ' Assimilation parameters File:  ',
     &                         aparnam(1:lstr)
#  ifdef PERTURBATION
      fname=ESname
      lstr=lenstr(ESname)
      open(iunit,file=ESname(1:lstr),status='old',err=200)
      write(stdout,'(2x,a,a)') '          Error Subspace File:  ',
     &                         ESname(1:lstr)
      close(iunit)
#  endif
      goto 220
 200  write(stdout,210) fname(1:lstr)
 210  format(/,' INP_APAR - cannot not find input file:  ',a)
      stop
 220  continue
# endif /* ESSE */
      return
      end
#endif /* ASSIMILATION || ESSE || NUDGING */
#ifdef BIOLOGY
      subroutine inp_Bpar
!
!=====================================================================
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine reads in biology parameters from input file. It also !
!  reports these parameters to standard output.                      !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "biology.h"
# include "iounits.h"
!
      INTEGER_TYPE
     &        icard, iunit, lstr
      INTEGER_TYPE
     &        lenstr
      parameter (iunit=40)
!
!---------------------------------------------------------------------
!  Read in biology parameters.
!---------------------------------------------------------------------
!
      lstr=lenstr(bparnam)
      open(iunit,file=bparnam(1:lstr),form='formatted',status='old')
      write(stdout,10)
  10  format(/,' Biology parameters:',/)
# ifdef BIO_FASHAM
!
!  Read input parameters according to their input card number.
!
      icard=0
      do while (icard.lt.99)
!
!  Read in numerical parmeters.
!
        if (icard.eq.1) then
          read(iunit,*,err=100) BioIter
          write(stdout,20) BioIter
  20      format(i10,2x,'BioIter',t26,
     &             'Number of iterations for nonlinear convergence.')
!
!  Read in light parameters.
!
        elseif (icard.eq.2) then
          read(iunit,*,err=100) AttSW, AttChl, PARfrac
          write(stdout,30) AttSW, AttChl, PARfrac
  30      format(1p,e10.3,2x,'AttSW',t26,
     &             'Light attenuation of seawater (m-1).',/,
     &           1p,e10.3,2x,'AttChl',t26,
     &             'Light attenuation by chlorophyll (mg-1 m-2).',/,
     &           f10.6,2x,'PARfrac',t26,
     &             'Fraction of shortwave radiation that is',/,t28,
     &             'photosynthetically active (nondimensional).')
!
!  Read in phytoplankton saturation parameters.
!
        elseif (icard.eq.3) then
          read(iunit,*,err=100) K_NO3, K_NH4, K_Phy
          write(stdout,40) K_NO3, K_NH4, K_Phy
  40      format(f10.6,2x,'K_NO3',t26,
     &             'Inverse half-saturation for phytoplankton NO3',
     &             /,t28,'uptake (m2 mMol-1).',/,
     &           f10.6,2x,'K_NH4',t26,
     &             'Inverse half-saturation for phytoplankton NH4',
     &             /,t28,'uptake (m2 mMol-1).',/,
     &           f10.6,2x,'K_Phy',t26,
     &             'Zooplankton half-saturation constant for '
     &             'ingestion',/,t28,'(day-1).')
!
!  Read in phytoplankton parameters.
!
        elseif (icard.eq.4) then
          read(iunit,*,err=100) PhyIS, PhyCN, PhyIP, PhyMR
          write(stdout,50) PhyIS, PhyCN, PhyIP, PhyMR
  50      format(f10.6,2x,'PhyIS',t26,
     &             'Phytoplankton growth, initial slope of P-I curve',
     &             /,t28,'(mg C/mg Chl Watts m-2 day).',/,
     &           f10.6,2x,'PhyCN',t26,
     &             'Phytoplankton C:N ratio (mMol C/mMol N).',/,
     &           f10.6,2x,'PhyIP',t26,
     &             'Phytoplankton NH4 inhibition parameter ',
     &             ' (1/mMol N).',/,
     &           f10.6,2x,'PhyMR',t26,
     &             'Phytoplankton mortality rate (day-1).')
!
!  Read in zooplankton parameters.
!
        elseif (icard.eq.5) then
          read(iunit,*,err=100) ZooAE, ZooGR, ZooER, ZooMR
          write(stdout,60) ZooAE, ZooGR, ZooER, ZooMR
  60      format(f10.6,2x,'ZooAE',t26,
     &             'Zooplankton assimilation efficiency fraction',
     &             /,t28,'(nondimensional).',/,
     &           f10.6,2x,'ZooGR',t26,
     &             'Zooplankton maximum growth rate (day-1).',/,
     &           f10.6,2x,'ZooER',t26,
     &             'Zooplankton specific excretion rate (day-1).',/,
     &           f10.6,2x,'ZooMR',t26,
     &             'Zooplankton mortality rate (day-1).')
!
!  Read in chlorophyll parameters.
!
        elseif (icard.eq.6) then
          read(iunit,*,err=100) ChlMB, CPrMax
          write(stdout,70) ChlMB, CPrMax
  70      format(f10.6,2x,'ChlMB',t26,
     &             'Mass balance for chlorophyll molecule',
     &             /,t28,'(mg Chl/mg C).',/,
     &           f10.6,2x,'CPrMax',t26,
     &             'Maximum cellular chlorphyll to carbon ratio'
     &             /,t28,'(mg Chl/mg C).')
!
!  Read in detritus parameters.
!
        elseif (icard.eq.7) then
          read(iunit,*,err=100) SDeBR, SDeAR, LDeRR
          write(stdout,80) SDeBR, SDeAR, LDeRR
  80      format(f10.6,2x,'SDeBR',t26,
     &             'Small detritus beakdown rate (day-1).',/,
     &           f10.6,2x,'SDeAR',t26,
     &             'Small detritus aggreagation rate (day-1).',/,
     &           f10.6,2x,'LDeRR',t26,
     &             'Large detritus re-mineralization rate (day-1).')
!
!  Read in vertical sinking parameters.
!
        elseif (icard.eq.8) then
          read(iunit,*,err=100) wSDe, wLDe
          write(stdout,90) wSDe, wLDe
  90      format(f10.6,2x,'wSDe',t26,
     &             'Small detritus sinking velocity (m/day).',/,
     &           f10.6,2x,'wLDe',t26,
     &             'Large detritus sinking velocity (m/day).')
        endif
!
!  Read next input card ID.
!
        read(iunit,*,err=100) icard
      enddo
      goto 120
!
!  Error while reading input parameters.
!
 100  write(stdout,110) icard, bparnam(1:lstr)
 110  format(/,' INP_BPAR - error while reading input line: ',i2,
     &       /,12x,'from biology file: ',a)
      stop
 120  close(iunit)
# endif /* BIO_FASHAM */
      return
      end
#endif /* BIOLOGY */
#ifdef SEDIMENT
      subroutine inp_sed
!
!=====================================================================
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine reads in sediment parameters from input file.  It    !
!  also reports these parameters to standard output.                 !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "iounits.h"
# include "mixing.h"
# include "ncparam.h"
# include "scalars.h"
# include "sediment.h"
!
      INTEGER_TYPE
     &        i, icard, iunit, lstr
      INTEGER_TYPE
     &        lenstr
      parameter (iunit=40)
!
!---------------------------------------------------------------------
!  Read in sediment parameters.
!---------------------------------------------------------------------
!
      lstr=lenstr(sparnam)
      open(iunit,file=sparnam(1:lstr),form='formatted',status='old')
      write(stdout,10) version
  10  format(/,/,' Sediment parameters:  ROMS version ',a,/)
!
!  Read input parameters according to their input card number.
!
      icard=0
      do while (icard.lt.99)
!
!  Cohesive sediments: median sediment grain diameter (mm).
!
        if (icard.eq.1) then
          read(iunit,*,err=20) (Sd50(i),i=1,NCS)
!
!  Cohesive sediments: concentration (mg/l).
!
        elseif (icard.eq.2) then
          read(iunit,*,err=20) (Csed(i),i=1,NCS)
!
!  Cohesive sediments: grain density (kg/m3).
!
        elseif (icard.eq.3) then
          read(iunit,*,err=20) (Srho(i),i=1,NCS)
!
!  Cohesive sediments: particle settling velocity (mm/s).
!
        elseif (icard.eq.4) then
          read(iunit,*,err=20) (Wsed(i),i=1,NCS)
!
!  Cohesive sediments: surface erosion rate (kg/m2/s).
!
        elseif (icard.eq.5) then
          read(iunit,*,err=20) (Erate(i),i=1,NCS)
!
!  Cohesive sediments: critical shear for erosion (N/m2).
!
        elseif (icard.eq.6) then
          read(iunit,*,err=20) (tau_ce(i),i=1,NCS)
!
!  Cohesive sediments: critical shear for deposition (N/m2).
!
        elseif (icard.eq.7) then
          read(iunit,*,err=20) (tau_cd(i),i=1,NCS)
!
!  Cohesive sediments: porosity (nondimensional: 0 - 1).
!
        elseif (icard.eq.8) then
          read(iunit,*,err=20) (poros(i),i=1,NCS)
!
!  Cohesive sediments: harmonic horizontal mixing (m2/s).
!
        elseif (icard.eq.9) then
          read(iunit,*,err=20) (tnu2(idsed(i)),i=1,NCS)
!
!  Cohesive sediments: biharmonic horizontal mixing (m2/s).
!
        elseif (icard.eq.10) then
          read(iunit,*,err=20) (tnu4(idsed(i)),i=1,NCS)
!
!  Cohesive sediments: background vertical mixing coefficient (m2/s).
!
        elseif (icard.eq.11) then
          read(iunit,*,err=20) (Akt_bak(idsed(i)),i=1,NCS)
!
!  Cohesive sediments: nudging time scale (days).
!
        elseif (icard.eq.12) then
          read(iunit,*,err=20) (Tnudg(idsed(i)),i=1,NCS)
!
!  Cohesive sediments: writing switches for suspended concentration.
!
        elseif (icard.eq.13) then
          read(iunit,*,err=20) (Hout(idTvar(idsed(i))),i=1,NCS)
!
!  Cohesive sediments: writing switches for bed layer fraction.
!
        elseif (icard.eq.14) then
          read(iunit,*,err=20) (Hout(idfrac(i)),i=1,NCS)
!
!  Noncohesive sediments: median sediment grain diameter (mm).
!
        elseif (icard.eq.15) then
          read(iunit,*,err=20) (Sd50(i),i=NCS+1,NST)
!
!  Noncohesive sediments: concentration (mg/l).
!
        elseif (icard.eq.16) then
          read(iunit,*,err=20) (Csed(i),i=NCS+1,NST)
!
!  Noncohesive sediments: grain density (kg/m3).
!
        elseif (icard.eq.17) then
          read(iunit,*,err=20) (Srho(i),i=NCS+1,NST)
!
!  Noncohesive sediments: particle settling velocity (mm/s).
!
        elseif (icard.eq.18) then
          read(iunit,*,err=20) (Wsed(i),i=NCS+1,NST)
!
!  Noncohesive sediments: surface erosion rate (kg/m2/s).
!
        elseif (icard.eq.19) then
          read(iunit,*,err=20) (Erate(i),i=NCS+1,NST)
!
!  Noncohesive sediments: critical shear for erosion (N/m2).
!
        elseif (icard.eq.20) then
          read(iunit,*,err=20) (tau_ce(i),i=NCS+1,NST)
!
!  Noncohesive sediments: critical shear for deposition (N/m2).
!
        elseif (icard.eq.21) then
          read(iunit,*,err=20) (tau_cd(i),i=NCS+1,NST)
!
!  Noncohesive sediments: porosity (nondimensional: 0 - 1).
!
        elseif (icard.eq.22) then
          read(iunit,*,err=20) (poros(i),i=NCS+1,NST)
!
!  Noncohesive sediments: harmonic horizontal mixing (m2/s).
!
        elseif (icard.eq.23) then
          read(iunit,*,err=20) (tnu2(idsed(i)),i=NCS+1,NST)
!
!  Noncohesive sediments: biharmonic horizontal mixing (m2/s).
!
        elseif (icard.eq.24) then
          read(iunit,*,err=20) (tnu4(idsed(i)),i=NCS+1,NST)
!
!  Noncohesive sediments: background vertical mixing coefficient (m2/s).
!
        elseif (icard.eq.25) then
          read(iunit,*,err=20) (Akt_bak(idsed(i)),i=NCS+1,NST)
!
!  Noncohesive sediments: nudging time scale (days).
!
        elseif (icard.eq.26) then
          read(iunit,*,err=20) (Tnudg(idsed(i)),i=NCS+1,NST)
!
!  Noncohesive sediments: writing switches for suspended concentration.
!
        elseif (icard.eq.27) then
          read(iunit,*,err=20) (Hout(idTvar(idsed(i))),i=NCS+1,NST)
!
!  Noncohesive sediments: writing switches for bed layer fraction.
!
        elseif (icard.eq.28) then
          read(iunit,*,err=20) (Hout(idfrac(i)),i=NCS+1,NST)
!
!  Time-stepping parameters.
!
        elseif (icard.eq.29) then
          read(iunit,*,err=20) SedIter
!
!  Writing switches for bed layer sediment properties.
!
        elseif (icard.eq.30) then
          read(iunit,*,err=20) (Hout(idSbed(i)),i=1,MBEDP)
!
!  Writing switches for bottom sediment properties.
!
        elseif (icard.eq.31) then
          read(iunit,*,err=20) (Hout(idBott(i)),i=1,MBOTP)
        endif
!
!  Read next input card ID.
!
        read(iunit,*,err=20) icard
      enddo
      goto 40
!
!  Error while reading input parameters.
!
  20  write(stdout,30) icard, sparnam(1:lstr)
  30  format(/,' INP_SED - error while reading input line: ',i2,
     &       /,12x,'from sediment file: ',a)
      stop
  40  close(iunit)
!
!  Report sediment paramenters.
!
      write(stdout,50)
  50  format(/,2x,'Size',5x,'Sd50',7x,'Csed',7x,'Srho',7x,'Wsed',
     &       7x,'Erate',6x,'poros',/,2x,'Class',4x,'(mm)',6x,
     &       '(mg/l)',5x,'(kg/m3)',4x,'(mm/s)',4x,'(kg/m2/s)',3x,
     &       '(nondim)',/)
      do i=1,NST
        write(stdout,60) i, Sd50(i), Csed(i), Srho(i), Wsed(i),
     &                   Erate(i), poros(i)
      enddo
  60  format(3x,i2,2x,6(1x,1p,e10.3))
      write(stdout,70)
  70  format(/,10x,'tau_ce',5x,'tau_cd',6x,'tnu2',7x,'tnu4',6x,
     &       'Akt_bak',5x,'Tnud',/,10x,'(N/m2)',5x,'(N/m2)',5x,
     &       '(m2/s)',5x,'(m4/s)',6x,'(m2/s)',5x,'(day)',/)
      do i=1,NST
        write(stdout,60) i, tau_ce(i), tau_cd(i), tnu2(idsed(i)),
     &                   tnu4(idsed(i)), Akt_bak(idsed(i)),
     &                   Tnudg(idsed(i))
      enddo
!
!  Report IO switches.
!
      write(stdout,'(/)')
      write(stdout,80) SedIter
  80  format(i10,2x,'SedIter',t26,
     &       'number of time-steps iterations.')
      do i=1,NST
        lstr=lenstr(Vname(1,idTvar(idsed(i))))
        write(stdout,90) Hout(idTvar(idsed(i))), i,
     &                   Vname(1,idTvar(idsed(i)))(1:lstr)
      enddo
  90  format(9x,l1,2x,'Hout(idTvar)',t26,
     &       'Write out sediment ',i2.2,': ',a,' (T/F).')
      do i=1,NST
        lstr=lenstr(Vname(1,idfrac(i)))
        write(stdout,100) Hout(idfrac(i)), i,
     &                    Vname(1,idfrac(i))(1:lstr)
      enddo
 100  format(9x,l1,2x,'Hout(idfrac)',t26,
     &       'Write out bed fraction, sediment ',i2.2,': ',a,
     &       ' (T/F).')
      do i=1,MBEDP
        lstr=lenstr(Vname(1,idSbed(i)))
        write(stdout,110) Hout(idSbed(i)), i,
     &                   Vname(1,idSbed(i))(1:lstr)
      enddo
 110  format(9x,l1,2x,'Hout(idSbed)',t26,
     &       'Write out BED property ',i2.2,': ',a,' (T/F).')
      do i=1,MBOTP
        lstr=lenstr(Vname(1,idBott(i)))
        write(stdout,120) Hout(idBott(i)), i,
     &                   Vname(1,idBott(i))(1:lstr)
      enddo
 120  format(9x,l1,2x,'Hout(idBott)',t26,
     &       'Write out BOT property ',i2.2,': ',a,' (T/F).')
!
!  Scale input parameters.
!
      do i=1,NST
        Sd50(i)=Sd50(i)*0.001_r8
        Wsed(i)=Wsed(i)*0.001_r8
        tau_ce(i)=tau_ce(i)/rho0
        tau_cd(i)=tau_cd(i)/rho0
        tnu4(idsed(i))=SQRT(ABS(tnu4(idsed(i))))
        if (Tnudg(idsed(i)).gt.0.0_r8) then
          Tnudg(idsed(i))=1.0_r8/(Tnudg(idsed(i))*86400.0_r8)
        else
          Tnudg(idsed(i))=0.0_r8
        endif
      enddo
      return
      end
#endif /* SEDIMENT */
#ifdef STATIONS
      subroutine inp_Spar
!
!=====================================================================
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine reads in input station parameters from station file. !
!  It also reports these parameters to standard output.              !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "bbl.h"
# include "iounits.h"
# include "ncparam.h"
# include "scalars.h"
# include "strings.h"
!
      INTEGER_TYPE
     &        icard, itrc, iunit, k, lstr
      INTEGER_TYPE
     &        lenstr
      parameter (iunit=40)
!
!---------------------------------------------------------------------
!  Read in station positions.
!---------------------------------------------------------------------
!
      lstr=lenstr(sposnam)
      open(iunit,file=sposnam(1:lstr),form='formatted',status='old')
      write(stdout,10)
  10  format(/,' STATION processing parameters:',/)
!
!  Read input parameters according to their input card number.
!
      icard=0
      do while (icard.lt.99)
!
!  Read in number of stations.
!
        if (icard.eq.1) then
          read(iunit,*,err=190) nstation
          if (NS.lt.nstation) then
            write(stdout,20) NS, nstation
  20        format(/,' INP_SPAR - too small dimension parameter, NS:',
     &             1x,2i4,/,12x,'change file  param.h and recompile.')
            stop
          endif
          write(stdout,30) nstation
  30      format(4x,i6,2x,'nstation',t26,
     &          'Number of stations to write out into stations file.')
!
!  Writing switches for fields associated with momentum equations.
!
        elseif (icard.eq.2) then
          read(iunit,*,err=190) Sout(idUvel), Sout(idVvel),
     &                          Sout(idWvel), Sout(idOvel),
     &                          Sout(idUbar), Sout(idVbar),
     &                          Sout(idFsur)
# ifdef SOLVE3D
          write(stdout,40) Sout(idUvel), Sout(idVvel), Sout(idWvel),
     &                     Sout(idOvel)
  40      format(9x,l1,2x,'Sout(idUvel)',t26,
     &             'Write out 3D U-momentum component (T/F).',/,
     &           9x,l1,2x,'Sout(idVvel)',t26,
     &             'Write out 3D V-momentum component (T/F).',/,
     &           9x,l1,2x,'Sout(idWvel)',t26,
     &             'Write out W-momentum component (T/F).',/,
     &           9x,l1,2x,'Sout(idOvel)',t26,
     &             'Write out omega vertical velocity (T/F).')
# endif /* SOLVE3D */
          write(stdout,50) Sout(idUbar), Sout(idVbar), Sout(idFsur)
  50      format(9x,l1,2x,'Sout(idUbar)',t26,
     &             'Write out 2D U-momentum component (T/F).',/,
     &           9x,l1,2x,'Sout(idVbar)',t26,
     &             'Write out 2D V-momentum component (T/F).',/,
     &           9x,l1,2x,'Sout(idFsur)',t26,
     &             'Write out free-surface (T/F).')
!
!  Writing switches for fields associated with tracers equations.
!
        elseif (icard.eq.3) then
          read(iunit,*,err=190) (Sout(idTvar(itrc)),itrc=1,NT)
# ifdef SOLVE3D
          do itrc=1,NT
            write(stdout,60) Sout(idTvar(itrc)), itrc
          enddo
  60      format(9x,l1,2x,'Sout(idTvar)',t26,
     &             'Write out tracer ',i1,' (T/F).')
# endif /* SOLVE3D */
!
!  Writing switches for surface and bottom momentum stresses.
!
        elseif (icard.eq.4) then
          read(iunit,*,err=190) Sout(idUsms), Sout(idVsms),
     &                          Sout(idUbms), Sout(idVbms),
     &                          Sout(idUbws), Sout(idVbws)
          write(stdout,70) Sout(idUsms), Sout(idVsms), Sout(idUbms),
     &                     Sout(idVbms)
  70      format(9x,l1,2x,'Sout(idUsms)',t26,
     &             'Write out surface U-momentum stress (T/F).',/,
     &           9x,l1,2x,'Sout(idVsms)',t26,
     &             'Write out surface V-momentum stress (T/F).',/,
     &           9x,l1,2x,'Sout(idUbms)',t26,
     &             'Write out bottom U-momentum stress (T/F).',/,
     &           9x,l1,2x,'Sout(idVbms)',t26,
     &             'Write out bottom U-momentum stress (T/F).')
# ifdef BBL
          write(stdout,80) Sout(idUbws), Sout(idVbws)
  80      format(9x,l1,2x,'Sout(idUbws)',t26,
     &             'Write out wind-induced, bottom U-wave stress.',/,
     &           9x,l1,2x,'Sout(idVbws)',t26,
     &             'Write out wind-induced, bottom V-wave stress.')
# endif /* BBL */
!
!  Writing switches for bed boundary layer fields.
!
        elseif (icard.eq.5) then
          read(iunit,*,err=190) Sout(idAbed), Sout(idUbed),
     &                          Sout(idVbed), Sout(idUbot),
     &                          Sout(idVbot), Sout(idHrip),
     &                          Sout(idLrip), Sout(idZnot),
     &                          Sout(idZapp)
# ifdef BBL
          write(stdout,90) Sout(idAbed), Sout(idUbed), Sout(idVbed),
     &                     Sout(idUbot), Sout(idVbot), Sout(idHrip),
     &                     Sout(idLrip), Sout(idZnot), Sout(idZapp)
  90      format(9x,l1,2x,'Sout(idAbed)',t26,
     &             'Write out bed wave excursion amplitude (T/F).',/,
     &           9x,l1,2x,'Sout(idUbed)',t26,
     &             'Write out bed wave orbital U-velocity (T/F).',/,
     &           9x,l1,2x,'Sout(idVbed)',t26,
     &             'Write out bed wave orbital V-velocity (T/F).',/,
     &           9x,l1,2x,'Sout(idUbot)',t26,
     &             'Write out bottom U-momentum above bed (T/F).',/,
     &           9x,l1,2x,'Sout(idVbot)',t26,
     &             'Write out bottom V-momentum above bed (T/F).',/,
     &           9x,l1,2x,'Sout(idHrip)',t26,
     &             'Write out bed ripple height (T/F).',/,
     &           9x,l1,2x,'Sout(idLrip)',t26,
     &             'Write out bed ripple length (T/F).',/,
     &           9x,l1,2x,'Sout(idZnot)',t26,
     &             'Write out bottom roughness (T/F).',/,
     &           9x,l1,2x,'Sout(idVbws)',t26,
     &             'Write out apparent bottom roughness (T/F).')
# endif /* BBL */
!
!  Writing switches for surface heat flux components.
!
        elseif (icard.eq.6) then
          read(iunit,*,err=190) Sout(idTsur(1)), Sout(idLhea),
     &                          Sout(idShea),    Sout(idLrad),
     &                          Sout(idSrad)
# ifdef SOLVE3D
          write(stdout,100) Sout(idTsur(1))
 100      format(9x,l1,2x,'Sout(idTsur)',t26,
     &             'Write out surface net heat flux (T/F).')
# ifdef SHORTWAVE
          write(stdout,110) Sout(idSrad)
 110      format(9x,l1,2x,'Sout(idSrad)',t26,
     &             'Write out shortwave radiation flux (T/F).')
# endif
# ifdef BULK_FLUXES
          write(stdout,120) Sout(idLrad), Sout(idLhea), Sout(idShea)
 120      format(9x,l1,2x,'Sout(idLrad)',t26,
     &             'Write out longwave radiation flux (T/F).',/,
     &           9x,l1,2x,'Sout(idLhea)',t26,
     &             'Write out latent heat flux (T/F).',/,
     &           9x,l1,2x,'Sout(idShea)',t26,
     &             'Write out sensible heat flux (T/F).')
# endif
# endif /* SOLVE3D */
!
!  Writing switches for other fields.
!
        elseif (icard.eq.7) then
          read(iunit,*,err=190) Sout(idDano), Sout(idVvis),
     &                          Sout(idTdif), Sout(idSdif),
     &                          Sout(idHsbl), Sout(idHbbl),
     &                          Sout(idMtke), Sout(idMtls)
# ifdef SOLVE3D
          write(stdout,130) Sout(idDano), Sout(idVvis), Sout(idTdif),
     &                      Sout(idSdif)
 130      format(9x,l1,2x,'Sout(idDano)',t26,
     &             'Write out density anomaly (T/F).',/,
     &           9x,l1,2x,'Sout(idVvis)',t26,
     &             'Write out vertical viscosity coefficient ',
     &             '(T/F).',/,
     &           9x,l1,2x,'Sout(idTdif)',t26,
     &             'Write out vertical T-diffusion coefficient ',
     &             '(T/F).',/
     &           9x,l1,2x,'Sout(idSdif)',t26,
     &             'Write out vertical S-diffusion coefficient ',
     &             '(T/F).')
#  ifdef LMD_SKPP
          write(stdout,140) Sout(idHsbl)
 140      format(9x,l1,2x,'Sout(idHsbl)',t26,
     &             'Write out depth of surface boundary layer (T/F).')
#  endif /* LMD_SKPP */
#  ifdef LMD_BKPP
          write(stdout,150) Sout(idHbbl)
 150      format(9x,l1,2x,'Sout(idHbbl)',t26,
     &             'Write out depth of bottom boundary layer (T/F).')
#  endif /* LMD_BKPP */
#  ifdef MY25_MIXING
          write(stdout,160) Sout(idMtke), Sout(idMtls)
 160      format(9x,l1,2x,'Sout(idMtke)',t26,
     &             'Write out turbulent kinetic energy (T/F).',/,
     &           9x,l1,2x,'Sout(idMtls)',t26,
     &             'Write out turbulent KE times length scale (T/F).')
#  endif /* MY25_MIXING */
# endif /* SOLVE3D */
!
!  Station I-J positions.
!
        elseif (icard.eq.8) then
          write(stdout,*) ' '
          do k=1,nstation
            read(iunit,*,err=190) ispos(k), jspos(k)
            if ((ispos(k).lt.1).or.(ispos(k).gt.Lm)) then
              write(stdout,170) ' ISPOS = ', ispos(k)
              stop
            endif
            if ((jspos(k).lt.1).or.(jspos(k).gt.Mm)) then
              write(stdout,170) ' JSPOS = ', jspos(k)
              stop
            endif
            write(stdout,180) k, ispos(k), jspos(k)
          enddo
 170      format(/,' INP_SPAR - out of range station index position',
     &           a,i4)
 180      format(12x,'(I,J) for station ',i4.4,':',2i6)
        endif
!
!  Read next input card ID.
!
        read(iunit,*,err=190) icard
      enddo
      goto 210
!
!  Error while reading input parameters.
!
 190  write(stdout,200) icard, sposnam(1:lstr)
 200  format(/,' INP_SPAR - error while reading input line: ',i2,
     &       /,12x,'from stations file: ',a)
      stop
 210  close(iunit)
      return
      end
#endif /* STATIONS */
