#include "cppdefs.h"
#ifdef GLS_MIXING && defined SOLVE3D
      subroutine gls_corstep (tile)
!
!======================================== Alexander F. Shchepetkin ===
!  Copyright (c) 2002 Rutgers/UCLA                  John C. Warner   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine perfoms the corrector step for turbulent kinetic     !
!  energy and generic length scale prognostic variables, tke and     !
!  gls.                                                              !
!                                                                    !
!  References:                                                       !
!                                                                    !
!  Umlauf, L. and H. Burchard, 2001:  A generic length-scale         !
!    Equation for geophysical turbulence models.                     !
!                                                                    !
!====================================================================!
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "scratch.h"
# include "tile.h"
!
# ifdef PROFILE
      call wclock_on (19)
# endif
      call gls_corstep_tile (Istr,Iend,Jstr,Jend,
     &                       A3d(1,1),A3d(1,2),A2d(1,1),A2d(1,2),
     &                       A2d(1,3),A2d(1,4),A2d(1,5),A2d(1,6),
     &                       A2d(1,7),A2d(1,8),A2d(1,9),A2d(1,1),
     &                       A2d(1,2),A2d(1,3),A2d(1,4),A2d(1,5),
     &                       A2d(1,6))
# ifdef PROFILE
      call wclock_off (19)
# endif
      return
      end
!
!*********************************************************************
      subroutine gls_corstep_tile (Istr,Iend,Jstr,Jend,shear2,buoy2,
     &                             gradK,gradP,curvK,curvP,FXK,FXP,
     &                             FEK,FEP,CF,dU,dV,FCK,FCP,BCK,BCP)
!*********************************************************************
!
       implicit none
# include "param.h"
# include "forces.h"
# include "grid.h"
# include "mask.h"
# include "mixing.h"
# include "ocean.h"
# include "scalars.h"
!
      Logical
     &        Lmy25, Lkw98
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, itrc, j, k
      REAL_TYPE
     &        Gadv, Gh, Gm, Kprod, Ls_unlmt, Ls_lmt, Pprod, Sh, Sm,
     &        cff, cff1, cff2, cff3, cmu_fac1, cmu_fac2, cmu_fac3,
     &        gls_c3, gls_exp1, gls_fac1, gls_fac2, gls_fac3,
     &        gls_fac4, gls_fac5, gls_fac6, eps, ql, sqrt2, strat2,
     &        tke_exp1, tke_exp2, tke_exp3, tke_exp4, wall_fac
      REAL_TYPE
     &           BCK(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &           BCP(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            CF(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            dU(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &            dV(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &           FCK(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &           FCP(PRIVATE_1D_SCRATCH_ARRAY,0:N),
     &           FEK(PRIVATE_2D_SCRATCH_ARRAY),
     &           FEP(PRIVATE_2D_SCRATCH_ARRAY),
     &           FXK(PRIVATE_2D_SCRATCH_ARRAY),
     &           FXP(PRIVATE_2D_SCRATCH_ARRAY),
     &         curvK(PRIVATE_2D_SCRATCH_ARRAY),
     &         curvP(PRIVATE_2D_SCRATCH_ARRAY),
     &         gradK(PRIVATE_2D_SCRATCH_ARRAY),
     &         gradP(PRIVATE_2D_SCRATCH_ARRAY),
     &        shear2(PRIVATE_2D_SCRATCH_ARRAY,0:N),
     &         buoy2(PRIVATE_2D_SCRATCH_ARRAY,0:N)
!
      parameter (Gadv=1.0_r8/3.0_r8, eps=1.0_e8-10)
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Compute several constants.
!---------------------------------------------------------------------
!
      Lmy25=.false.
      Lkw98=.false.
      if ((gls_p.eq.0.0_r8).and.
     &    (gls_n.eq.1.0_r8).and.
     &    (gls_m.eq.1.0_r8)) then
        Lmy25=.true.
      endif
      if ((gls_p.eq.-1.0_r8).and.
     &    (gls_n.eq. 0.5_r8).and.
     &    (gls_m.eq.-1.0_r8)) then
        Lkw98=.true.
      endif
!
      sqrt2=SQRT(2.0_r8)
      cmu_fac1=gls_cmu0**(-gls_p/gls_n)
      cmu_fac2=gls_cmu0**(3.0_r8+gls_p/gls_n)
      cmu_fac3=1.0_r8/gls_cmu0**2.0_r8
!
      gls_fac1=gls_cmu0**(1.0_r8+gls_p)*gls_n*vonKar**(1.0_r8+gls_n)
      gls_fac2=gls_cmu0**(gls_p)*gls_n*vonKar**(gls_n)
      gls_fac3=gls_cmu0**(gls_p)*(vonKar*Zos)**(gls_n)
      gls_fac4=gls_cmu0**(gls_p)*(vonKar*Zob)**(gls_n)
      gls_fac5=0.045_r8**(-0.5_r8*gls_n)*
     &         gls_cmu0**(gls_p+3.0_r8*gls_n)
!
      gls_exp1=1.0_r8/gls_n
      tke_exp1=gls_m/gls_n
      tke_exp2=0.5_r8+gls_m/gls_n
      tke_exp3=0.5_r8+gls_m
      tke_exp4=gls_m+0.5_r8*gls_n
!
!---------------------------------------------------------------------
!  Compute vertical velocity shear at W-points.
!---------------------------------------------------------------------
!
# ifdef SPLINES
      do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
        do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
          CF(i,0)=0.0_r8
          dU(i,0)=0.0_r8
          dV(i,0)=0.0_r8
        enddo
        do k=1,Nm
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            cff=1.0_r8/(2.0_r8*Hz(i,j,k+1)+
     &                  Hz(i,j,k)*(2.0_r8-CF(i,k-1)))
            CF(i,k)=cff*Hz(i,j,k+1)
            dU(i,k)=cff*(3.0_r8*(u(i  ,j,k+1,nstp)-u(i,  j,k,nstp)+
     &                           u(i+1,j,k+1,nstp)-u(i+1,j,k,nstp))-
     &                   Hz(i,j,k)*dU(i,k-1))
            dV(i,k)=cff*(3.0_r8*(v(i,j  ,k+1,nstp)-v(i,j  ,k,nstp)+
     &                           v(i,j+1,k+1,nstp)-v(i,j+1,k,nstp))-
     &                   Hz(i,j,k)*dV(i,k-1))
          enddo
        enddo
        do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
          dU(i,N)=0.0_r8
          dV(i,N)=0.0_r8
        enddo
        do k=Nm,1,-1
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            dU(i,k)=dU(i,k)-CF(i,k)*dU(i,k+1)
            dV(i,k)=dV(i,k)-CF(i,k)*dV(i,k+1)
          enddo
        enddo
        do k=1,Nm
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            shear2(i,j,k)=dU(i,k)*dU(i,k)+dV(i,k)*dV(i,k)
          enddo
        enddo
      enddo
# else
      do k=1,Nm
        do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            cff=0.5_r8/(z_r(i,j,k+1)-z_r(i,j,k))
            shear2(i,j,k)=(cff*(u(i  ,j,k+1,nstp)-u(i  ,j,k,nstp)+
     &                          u(i+1,j,k+1,nstp)-u(i+1,j,k,nstp)))**2+
     &                    (cff*(v(i,j  ,k+1,nstp)-v(i,j  ,k,nstp)+
     &                          v(i,j+1,k+1,nstp)-v(i,j+1,k,nstp)))**2
          enddo
        enddo
      enddo
# endif
!
! Load Brunt-Vaisala frequency.
!
      do k=1,Nm
        do j=Jstr-1,Jend+1
          do i=Istr-1,Iend+1
            buoy2(i,j,k)=bvf(i,j,k)
          enddo
        enddo
      enddo
# ifdef N2S2_HORAVG
!
!---------------------------------------------------------------------
!  Smooth horizontally buoyancy and shear.  Use buoy2(:,:,0) and
!  shear2(:,:,0) as scratch utility array.
!---------------------------------------------------------------------
!
      do k=1,Nm
        if (WESTERN_EDGE) then
          do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
            shear2(Istr-1,j,k)=shear2(Istr,j,k)
          enddo
        endif
        if (EASTERN_EDGE) then
          do j=MAX(1,Jstr-1),MIN(Jend+1,Mm)
            shear2(Iend+1,j,k)=shear2(Iend,j,k)
          enddo
        endif
        if (SOUTHERN_EDGE) then
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            shear2(i,Jstr-1,k)=shear2(i,Jstr,k)
          enddo
        endif
        if (NORTHERN_EDGE) then
          do i=MAX(1,Istr-1),MIN(Iend+1,Lm)
            shear2(i,Jend+1,k)=shear2(i,Jend,k)
          enddo
        endif
        if (SOUTH_WEST_CORNER) then
          shear2(Istr-1,Jstr-1,k)=shear2(Istr,Jstr,k)
        endif
        if (NORTH_WEST_CORNER) then
          shear2(Istr-1,Jend+1,k)=shear2(Istr,Jend,k)
        endif
        if (SOUTH_EAST_CORNER) then
          shear2(Iend+1,Jstr-1,k)=shear2(Iend,Jstr,k)
        endif
        if (NORTH_EAST_CORNER) then
          shear2(Iend+1,Jend+1,k)=shear2(Iend,Jend,k)
        endif
!
!  Average horizontally.
!
        do j=Jstr-1,Jend
          do i=Istr-1,Iend
            buoy2(i,j,0)=0.25_r8*(buoy2(i,j  ,k)+buoy2(i+1,j  ,k)+
     &                            buoy2(i,j+1,k)+buoy2(i+1,j+1,k))
            shear2(i,j,0)=0.25_r8*(shear2(i,j  ,k)+shear2(i+1,j  ,k)+
     &                             shear2(i,j+1,k)+shear2(i+1,j+1,k))
          enddo
        enddo
        do j=Jstr,Jend
          do i=Istr,Iend
            buoy2(i,j,k)=0.25_r8*(buoy2(i,j  ,0)+buoy2(i-1,j  ,0)+
     &                            buoy2(i,j-1,0)+buoy2(i-1,j-1,0))
            shear2(i,j,k)=0.25_r8*(shear2(i,j  ,0)+shear2(i-1,j  ,0)+
     &                             shear2(i,j-1,0)+shear2(i-1,j-1,0))
          enddo
        enddo
      enddo
# endif /* N2S2_HORAVG */
!
!---------------------------------------------------------------------
!  Time-step advective terms.
!---------------------------------------------------------------------
!
!  At entry, it is assumed that the turbulent kinetic energy fields
!  "tke" and "gls", at time level "nnew", are set to its values at
!  time level "nstp" multiplied by the grid box thicknesses Hz
!  (from old time step and at W-points).
!
      do k=1,N-1
# ifdef K_C2ADVECTION
!
!  Second-order, centered differences advection.
!
        do j=Jstr,Jend
          do i=Istr,Iend+1
            cff=0.25_r8*(Huon(i,j,k)+Huon(i,j,k+1))
            FXK(i,j)=cff*(tke(i,j,k,3)+tke(i-1,j,k,3))
            FXP(i,j)=cff*(gls(i,j,k,3)+gls(i-1,j,k,3))
          enddo
        enddo
        do j=Jstr,Jend+1
          do i=Istr,Iend
            cff=0.25_r8*(Hvom(i,j,k)+Hvom(i,j,k+1))
            FEK(i,j)=cff*(tke(i,j,k,3)+tke(i,j-1,k,3))
            FEP(i,j)=cff*(gls(i,j,k,3)+gls(i,j-1,k,3))
          enddo
        enddo
# else
#  ifdef EW_PERIODIC
#   define I_RANGE Istr-1,Iend+2
#  else
#   define I_RANGE max(Istr-1,1),min(Iend+2,Lm+1)
#  endif
        do j=Jstr,Jend
          do i=I_RANGE
            gradK(i,j)=(tke(i,j,k,3)-tke(i-1,j,k,3))
#  ifdef MASKING
     &                *umask(i,j)
#  endif
            gradP(i,j)=(gls(i,j,k,3)-gls(i-1,j,k,3))
#  ifdef MASKING
     &                *umask(i,j)
#  endif
          enddo
        enddo
#  undef I_RANGE
#  ifndef EW_PERIODIC
        if (WESTERN_EDGE) then
          do j=Jstr,Jend
            gradK(Istr-1,j)=gradK(Istr,j)
            gradP(Istr-1,j)=gradP(Istr,j)
          enddo
        endif
        if (EASTERN_EDGE) then
          do j=Jstr,Jend
            gradK(Iend+2,j)=gradK(Iend+1,j)
            gradP(Iend+2,j)=gradP(Iend+1,j)
          enddo
        endif
#  endif
#  ifdef K_C4ADVECTION
!
!  Fourth-order, centered differences advection.
!
        cff1=1.0_r8/6.0_r8
        do j=Jstr,Jend
          do i=Istr,Iend+1
            cff=0.5_r8*(Huon(i,j,k)+Huon(i,j,k+1))
            FXK(i,j)=cff*0.5_r8*(tke(i-1,j,k,3)+tke(i,j,k,3)-
     &                           cff1*(gradK(i+1,j)-gradK(i-1,j)))
            FXP(i,j)=cff*0.5_r8*(gls(i-1,j,k,3)+gls(i,j,k,3)-
     &                           cff1*(gradP(i+1,j)-gradP(i-1,j)))
          enddo
        enddo
#  else
!
!  Third-order, upstream bias advection with velocity dependent
!  hyperdiffusion.
!
        do j=Jstr,Jend
          do i=Istr-1,Iend+1
            curvK(i,j)=gradK(i+1,j)-gradK(i,j)
            curvP(i,j)=gradP(i+1,j)-gradP(i,j)
          enddo
        enddo
        do j=Jstr,Jend
          do i=Istr,Iend+1
            cff=0.5_r8*(Huon(i,j,k)+Huon(i,j,k+1))
            if (cff.gt.0.0_r8) then
              cff1=curvK(i-1,j)
              cff2=curvP(i-1,j)
            else
              cff1=curvK(i,j)
              cff2=curvP(i,j)
            endif
            FXK(i,j)=cff*0.5_r8*(tke(i-1,j,k,3)+tke(i,j,k,3)-
     &                           Gadv*cff1)
            FXP(i,j)=cff*0.5_r8*(gls(i-1,j,k,3)+gls(i,j,k,3)-
     &                           Gadv*cff2)
          enddo
        enddo
#  endif
#  ifdef NS_PERIODIC
#   define J_RANGE Jstr-1,Jend+2
#  else
#   define J_RANGE max(Jstr-1,1),min(Jend+2,Mm+1)
#  endif
        do j=J_RANGE
          do i=Istr,Iend
            gradK(i,j)=(tke(i,j,k,3)-tke(i,j-1,k,3))
#  ifdef MASKING
     &                *vmask(i,j)
#  endif
            gradP(i,j)=(gls(i,j,k,3)-gls(i,j-1,k,3))
#  ifdef MASKING
     &                *vmask(i,j)
#  endif
          enddo
        enddo
#  undef J_RANGE
#  ifndef NS_PERIODIC
        if (SOUTHERN_EDGE) then
          do i=Istr,Iend
            gradK(i,Jstr-1)=gradK(i,Jstr)
            gradP(i,Jstr-1)=gradP(i,Jstr)
          enddo
        endif
        if (NORTHERN_EDGE) then
          do i=Istr,Iend
            gradK(i,Jend+2)=gradK(i,Jend+1)
            gradP(i,Jend+2)=gradP(i,Jend+1)
          enddo
        endif
#  endif
#  ifdef K_C4ADVECTION
        cff1=1.0_r8/6.0_r8
        do j=Jstr,Jend+1
          do i=Istr,Iend
            cff=0.5_r8*(Hvom(i,j,k)+Hvom(i,j,k+1))
            FEK(i,j)=cff*0.5_r8*(tke(i,j-1,k,3)+tke(i,j,k,3)-
     &                           cff1*(gradK(i,j+1)-gradK(i,j-1)))
            FEP(i,j)=cff*0.5_r8*(gls(i,j-1,k,3)+gls(i,j,k,3)-
     &                           cff1*(gradP(i,j+1)-gradP(i,j-1)))
          enddo
        enddo
#  else
        do j=Jstr-1,Jend+1
          do i=Istr,Iend
            curvK(i,j)=gradK(i,j+1)-gradK(i,j)
            curvP(i,j)=gradP(i,j+1)-gradP(i,j)
          enddo
        enddo
        do j=Jstr,Jend+1
          do i=Istr,Iend
            cff=0.5_r8*(Hvom(i,j,k)+Hvom(i,j,k+1))
            if (cff.gt.0.0_r8) then
              cff1=curvK(i,j-1)
              cff2=curvP(i,j-1)
            else
              cff1=curvK(i,j)
              cff2=curvP(i,j)
            endif
            FEK(i,j)=cff*0.5_r8*(tke(i,j-1,k,3)+tke(i,j,k,3)-
     &                           Gadv*cff1)
            FEP(i,j)=cff*0.5_r8*(gls(i,j-1,k,3)+gls(i,j,k,3)-
     &                           Gadv*cff2)
          enddo
        enddo
#  endif
# endif
!
!  Time-step horizontal advection.
!
        do j=Jstr,Jend
          do i=Istr,Iend
            cff=dt*pm(i,j)*pn(i,j)
            tke(i,j,k,nnew)=tke(i,j,k,nnew)-
     &                      cff*(FXK(i+1,j)-FXK(i,j)+
     &                           FEK(i,j+1)-FEK(i,j))
            gls(i,j,k,nnew)=gls(i,j,k,nnew)-
     &                      cff*(FXP(i+1,j)-FXP(i,j)+
     &                           FEP(i,j+1)-FEP(i,j))
            if (tke(i,j,k,nnew).lt.0.0_r8) then
              tke(i,j,k,nnew)=MAX(tke(i,j,k,nnew),gls_Kmin)
            endif
            if (gls(i,j,k,nnew).lt.0.0_r8) then
              gls(i,j,k,nnew)=MAX(gls(i,j,k,nnew),gls_Pmin)
            endif
          enddo
        enddo
      enddo
!
! Compute vertical advection.
!
      do j=Jstr,Jend
# ifdef K_C2ADVECTION
        do k=1,N
          do i=Istr,Iend
            cff=0.25_r8*(W(i,j,k)+W(i,j,k-1))
            FCK(i,k)=cff*(tke(i,j,k,3)+tke(i,j,k-1,3))
            FCP(i,k)=cff*(gls(i,j,k,3)+gls(i,j,k-1,3))
          enddo
        enddo
# else
        cff1=7.0_r8/12.0_r8
        cff2=1.0_r8/12.0_r8
        do k=2,N-1
          do i=Istr,Iend
            cff=0.5*(W(i,j,k)+W(i,j,k-1))
            FCK(i,k)=cff*(cff1*(tke(i,j,k-1,3)+
     &                          tke(i,j,k  ,3))-
     &                    cff2*(tke(i,j,k-2,3)+
     &                          tke(i,j,k+1,3)))
            FCP(i,k)=cff*(cff1*(gls(i,j,k-1,3)+
     &                          gls(i,j,k  ,3))-
     &                    cff2*(gls(i,j,k-2,3)+
     &                          gls(i,j,k+1,3)))
          enddo
        enddo
        cff1=1.0_r8/3.0_r8
        cff2=5.0_r8/6.0_r8
        cff3=1.0_r8/6.0_r8
         do i=Istr,Iend
          cff=0.5_r8*(W(i,j,0)+W(i,j,1))
          FCK(i,1)=cff*(cff1*tke(i,j,0,3)+
     &                  cff2*tke(i,j,1,3)-
     &                  cff3*tke(i,j,2,3))
          FCP(i,1)=cff*(cff1*gls(i,j,0,3)+
     &                  cff2*gls(i,j,1,3)-
     &                  cff3*gls(i,j,2,3))
          cff=0.5_r8*(W(i,j,N)+W(i,j,N-1))
          FCK(i,N)=cff*(cff1*tke(i,j,N  ,3)+
     &                  cff2*tke(i,j,N-1,3)-
     &                  cff3*tke(i,j,N-2,3))
          FCP(i,N)=cff*(cff1*gls(i,j,N  ,3)+
     &                  cff2*gls(i,j,N-1,3)-
     &                  cff3*gls(i,j,N-2,3))
        enddo
# endif
!
!  Time-step vertical advection term.
!
        do k=1,N-1
          do i=Istr,Iend
            cff=dt*pm(i,j)*pn(i,j)
            tke(i,j,k,nnew)=tke(i,j,k,nnew)-
     &                      cff*(FCK(i,k+1)-FCK(i,k))
            gls(i,j,k,nnew)=gls(i,j,k,nnew)-
     &                      cff*(FCP(i,k+1)-FCP(i,k))
            if (tke(i,j,k,nnew).lt.0.0_r8) then
              tke(i,j,k,nnew)=MAX(tke(i,j,k,nnew),gls_Kmin)
            endif
            if (gls(i,j,k,nnew).lt.0.0_r8) then
              gls(i,j,k,nnew)=MAX(gls(i,j,k,nnew),gls_Pmin)
            endif
          enddo
        enddo
!
!--------------------------------------------------------------------
!  Compute vertical mixing, turbulent production and turbulent
!  dissipation terms.
!--------------------------------------------------------------------
!
!  Set term for vertical mixing of turbulent fields.
!
        cff=-0.5_r8*dt
        do i=Istr,Iend
          do k=2,N-1
            FCK(i,k)=cff*(Akk(i,j,k)+Akk(i,j,k-1))/Hz(i,j,k)
            FCP(i,k)=cff*(Akp(i,j,k)+Akp(i,j,k-1))/Hz(i,j,k)
            CF(i,k)=0.0_r8
          enddo
          FCP(i,1)=0.0_r8
          FCP(i,N)=0.0_r8
          FCK(i,1)=0.0_r8
          FCK(i,N)=0.0_r8
        enddo
!
!  Compute production and dissipation terms.
!
        do i=Istr,Iend
          do k=1,N-1
!
!  Compute shear and bouyant production of turbulent energy (m3/s3)
!  at W-points (ignore small negative values of buoyancy).
!
            if ((buoy2(i,j,k).gt.-5.0_e8-5).and.
     &          (buoy2(i,j,k).lt.0.0_r8)) then
              strat2=0.0_r8
            else
              strat2=buoy2(i,j,k)
            endif
!
            if (strat2.gt.0.0_r8) then
              gls_c3=gls_c3m
            else
              gls_c3=gls_c3p
            endif
            Kprod=shear2(i,j,k)*(Akv(i,j,k)-Akv_bak)-
     &            strat2*(Akt(i,j,k,itemp)-Akt_bak(itemp))
            Pprod=gls_c1*shear2(i,j,k)*(Akv(i,j,k)-Akv_bak)-
     &            gls_c3*strat2*(Akt(i,j,k,itemp)-Akt_bak(itemp))
!
!  If negative production terms, then add buoyancy to dissipation terms
!  (BCK and BCP) below, using "cff1" and "cff2" as the on/off switch.
!
            cff1=1.0_r8
            if (Kprod.lt.0.0_r8) then
              Kprod=Kprod+strat2*(Akt(i,j,k,itemp)-Akt_bak(itemp))
              cff1=0.0_r8
            endif
            cff2=1.0_r8
            if (Pprod.lt.0.0_r8) then
              Pprod=Pprod+gls_c3*strat2*(Akt(i,j,k,itemp)-
     &                                   Akt_bak(itemp))
              cff2=0.0_r8
            endif
!
!  Time-step shear and buoyancy production terms.
!
            cff=0.5_r8*(Hz(i,j,k)+Hz(i,j,k+1))
            tke(i,j,k,nnew)=tke(i,j,k,nnew)+
     &                      dt*cff*Kprod
            gls(i,j,k,nnew)=gls(i,j,k,nnew)+
     &                      dt*cff*Pprod*gls(i,j,k,nstp)/
     &                      MAX(tke(i,j,k,nstp),gls_Kmin)
!
!  Compute dissipation of turbulent energy (m3/s3).
!
            wall_fac=1.0_r8
            if (Lmy25) then
!
!  Parabolic wall function,  L = ds db / (ds + db).
!
              wall_fac=1.0_r8+gls_E2/(vonKar*vonKar)*
     &                 (gls(i,j,k,nstp)**( gls_exp1)*cmu_fac1*
     &                  tke(i,j,k,nstp)**(-tke_exp1)*
     &                  (1.0_r8/(z_w(i,j,N)-z_w(i,j,k))+
     &                   1.0_r8/(z_w(i,j,k)-z_w(i,j,0))))**2
!!
!! linear wall function for , L = 
!!            wall_fac=1.0_r8+gls_E2/(vonKar*vonKar)*
!!    &                (gls(i,j,k,nstp)**( gls_exp1)*cmu_fac1*
!!    &                 tke(i,j,k,nstp)**(-tke_exp1)*
!!    &                 (1.0_r8/MIN((z_w(i,j,N)-z_w(i,j,k)),
!!    &                             (z_w(i,j,k)-z_w(i,j,0)))))**2
!!
!! Linear wall function for , L = dist to surface
!!
!!            wall_fac=1.0_r8+gls_E2/(vonKar*vonKar)*
!!    &                (gls(i,j,k,nstp)**( gls_exp1)*cmu_fac1*
!!    &                 tke(i,j,k,nstp)**(-tke_exp1)*
!!    &                 (1.0_r8/ (z_w(i,j,N)-z_w(i,j,k))))**2
            elseif (Lkw98) then
              wall_fac=1.0_r8         ! Will be added later.
!              gls_W=0.5*(du/dx-dv/dy)+fomn(i,j)
!
!       !!    gls_S(1,1)= 1/2*(du/dx+du/dx)  --->
!            gls_S(1,1)=(u(i,j,k,nstp)-u(i+1,j  ,k  ,nstp))*om_r(i,j)
!       !!    gls_S(2,2)= 1/2*(dv/dy+dv/dy)  --->
!            gls_S(2,2)=(v(i,j,k,nstp)-v(i  ,j+1,k  ,nstp))*on_r(i,j)
!       !!    gls_S(3,3)= 1/2*(dw/dz+dw/dz)  --->
!            gls_S(3,3)=0.5_r8*((W(i,j,k  )-W(i,j,k-1))*Hz(i,j,k  )+
!      &                        (W(i,j,k+1)-W(i,j,k  ))*Hz(i,j,k+1))
!       !!    gls_S(1,2)= 1/2*(du/dy+dv/dx)  --->
!            gls_S(1,2)= 0.5_r8*
!      &     (0.5_r8*((0.5_r8*(u(i,j,k,nstp)-u(i,j-1,k,nstp))*on_p(i,j)+
!      &     (u(i,j+1,k,nstp)-u(i,j,k,nstp))*on_p(i,j+1))+
!      &     (0.5_r8*(u(i+1,j,k,nstp)-u(i+1,j-1,k  ,nstp))*on_p(i+1,j)+
!      &     (u(i+1,j+1,k,nstp)-u(i+1,j,k  ,nstp))*on_p(i+1,j+1)))+
!      &     0.5_r8*((0.5_r8*(v(i,j,k,nstp)-v(i-1,j,k  ,nstp))*om_p(i,j)+
!      &     (v(i+1,j,k,nstp)-v(i,j,k  ,nstp))*om_p(i+1,j))+
!      &     (0.5_r8*(v(i,j+1,k,nstp)-v(i-1,j+1,k  ,nstp))*om_p(i,j+1)+
!      &     (v(i+1,j+1,k,nstp)-v(i,j+1,k  ,nstp))*om_p(i+1,j+1))))
!           gls_S(1,3)= 0.5_r8*(du/dz+dw/dx)  --->
!
!              Xw=ABS(gls_W(i,j)*gls_W(j,k)*gls_S(k,i)/
!     &             ((gls_cmu0**4*gls(i,j,k,nstp))**3))
!              Xk=dk/dx dw/dx / w**3
!              if (Xk.le.0.) then 
!                fbstar=1
!              else
!                fbstar=(1+680Xk**2)/(1+400Xk**2)
!              endif
!              fb=(1+70*Xw)/(1+80*Xw)
!              wall_fac=gls_fb/gls_fbstar
            endif
!
            BCK(i,k)=cff*(1.0_r8+dt*   
     &                    gls(i,j,k,nstp)**(-gls_exp1)*cmu_fac2* 
     &                    tke(i,j,k,nstp)**( tke_exp2)+
     &                    dt*(1.0_r8-cff1)*strat2*
     &                    (Akt(i,j,k,itemp)-Akt_bak(itemp))/
     &                    tke(i,j,k,nstp))-
     &                    FCK(i,k)-FCK(i,k+1)
     &
            BCP(i,k)=cff*(1.0_r8+dt*gls_c2*wall_fac*
     &                    gls(i,j,k,nstp)**(-gls_exp1)*cmu_fac2*
     &                    tke(i,j,k,nstp)**( tke_exp2)+
     &                    dt*(1.0_r8-cff2)*gls_c3*strat2*
     &                    (Akt(i,j,k,itemp)-Akt_bak(itemp))/
     &                    tke(i,j,k,nstp))-
     &                    FCP(i,k)-FCP(i,k+1)
          enddo
        enddo
!
!---------------------------------------------------------------------
!  Time-step dissipation and vertical diffusion terms implicitly.
!---------------------------------------------------------------------
!
!  Set Dirichlet surface and bottom boundary conditions.
!
        do i=Istr,Iend
          tke(i,j,N,nnew)=0.5_r8*SQRT((sustr(i,j)+sustr(i+1,j))**2+
     &                                (svstr(i,j)+svstr(i,j+1))**2)*
     &                    cmu_fac3
          tke(i,j,0,nnew)=0.5_r8*SQRT((bustr(i,j)+bustr(i+1,j))**2+
     &                                (bvstr(i,j)+bvstr(i,j+1))**2)*
     &                    cmu_fac3
          gls(i,j,N,nnew)=gls_fac3*tke(i,j,N,nnew)**(gls_m)
          gls(i,j,0,nnew)=gls_fac4*tke(i,j,0,nnew)**(gls_m)
        enddo
!
!  Solve tri-diagonal system for tubulent kinetic energy.
!
        do i=Istr,Iend
          cff=1.0_r8/BCK(i,N-1)
          CF(i,N-1)=cff*FCK(i,N-1)
          tke(i,j,N-1,nnew)=cff*tke(i,j,N-1,nnew)
        enddo
        do i=Istr,Iend
          do k=N-2,1,-1
            cff=1.0_r8/(BCK(i,k)-CF(i,k+1)*FCK(i,k+1))
            CF(i,k)=cff*FCK(i,k)
            tke(i,j,k,nnew)=cff*(tke(i,j,k,nnew)-
     &                           FCK(i,k+1)*tke(i,j,k+1,nnew))
          enddo
        enddo
        do k=2,N-1
          do i=Istr,Iend
            tke(i,j,k,nnew)=tke(i,j,k,nnew)-CF(i,k)*tke(i,j,k-1,nnew)
          enddo
        enddo
!
!  Solve tri-diagonal system for generic statistical field.
!
        do i=Istr,Iend
          cff=1.0_r8/BCP(i,N-1)
          CF(i,N-1)=cff*FCP(i,N-1)
          gls(i,j,N-1,nnew)=cff*(gls(i,j,N-1,nnew)-
     &                      dt*gls_fac2*tke(i,j,N-1,nnew)**(gls_m)*
     &                      (0.5_r8*Hz(i,j,N))**(gls_n-1.0_r8)*
     &                      0.5_r8*(Akp(i,j,N)+Akp(i,j,N-1))) 
        enddo               
        do i=Istr,Iend      
          do k=N-2,1,-1     
            cff=1.0_r8/(BCP(i,k)-CF(i,k+1)*FCP(i,k+1))
            CF(i,k)=cff*FCP(i,k)
            gls(i,j,k,nnew)=cff*(gls(i,j,k,nnew)-
     &                           FCP(i,k+1)*gls(i,j,k+1,nnew))
          enddo                  
          gls(i,j,1,nnew)=gls(i,j,1,nnew)-
     &                    cff*dt*gls_fac2*tke(i,j,1,nnew)**(gls_m)*
     &                    (0.5_r8*Hz(i,j,1))**(gls_n-1.0_r8)*
     &                    0.5_r8*(Akp(i,j,0)+Akp(i,j,1))
        enddo             
        do k=2,N-1        
          do i=Istr,Iend    
            gls(i,j,k,nnew)=gls(i,j,k,nnew)-CF(i,k)*gls(i,j,k-1,nnew)
          enddo
        enddo
!
!-------------------------------------------------------------------
!  Compute vertical mixing coefficients (m2/s).
!-------------------------------------------------------------------
!
        do i=Istr,Iend
          do k=1,N-1
!
!  Compute turbulent length scale (m).  The length scale is only
!  limited in the K-related calculations and not in QL production,
!  dissipation, wall-proximity, etc.
!
            tke(i,j,k,nnew)=MAX(tke(i,j,k,nnew),gls_Kmin)
            gls(i,j,k,nnew)=MAX(gls(i,j,k,nnew),gls_Pmin)
            if (gls_n.ge.0.0_r8) then
              gls(i,j,k,nnew)=MIN(gls(i,j,k,nnew),gls_fac5*
     &                            tke(i,j,k,nnew)**(tke_exp4)*
     &                            (SQRT(MAX(0.0_r8,
     &                                  buoy2(i,j,k)))+eps)**(-gls_n))
       else
              gls(i,j,k,nnew)=MAX(gls(i,j,k,nnew),gls_fac5*
     &                            tke(i,j,k,nnew)**(tke_exp4)*
     &                            (SQRT(MAX(0.0_r8,
     &                                  buoy2(i,j,k)))+eps)**(-gls_n))
      endif
            Ls_unlmt=MAX(eps,
     &                   gls(i,j,k,nnew)**( gls_exp1)*cmu_fac1*
     &                   tke(i,j,k,nnew)**(-tke_exp1))
            Ls_lmt=MIN(Ls_unlmt,
     &                 SQRT(0.56_r8*tke(i,j,k,nnew)/
     &                      (MAX(0.0_r8,buoy2(i,j,k))+eps)))
!
!  Compute nondimensional stability functions for tracers (Sh) and
!  momentum (Sm).  Apply smoothing of Gh between critical and max,
!  Burchard (1999).
!
            Gh=MIN(gls_Gh0,-buoy2(i,j,k)*Ls_lmt*Ls_lmt/
     &                    (2.0_r8*tke(i,j,k,nnew)))
            Gh=MIN(Gh,Gh-(Gh-gls_Ghcri)**2/
     &                    (Gh+gls_Gh0-2.0_r8*gls_Ghcri))
            Gh=MAX(Gh,gls_Ghmin)
# if defined CANUTO_A || defined CANUTO_B
!
!  Compute shear number.
!
            Gm=(gls_b0/gls_fac6-gls_b1*Gh+gls_b3*gls_fac6*(Gh**2))/
     &         (gls_b2-gls_b4*gls_fac6*Gh)
            Gm=MIN(Gm,shear2(i,j,k)*Ls_lmt*Ls_lmt/
     &                    (2.0_r8*tke(i,j,k,nnew)))
!!          Gm=MIN(Gm,(gls_s1*gls_fac6*Gh-gls_s0)/(gls_s2*gls_fac6))
!
!  Compute stability functions
!
            cff=gls_b0-gls_b1*gls_fac6*Gh+gls_b2*gls_fac6*Gm+
     &          gls_b3*gls_fac6**2*Gh**2-gls_b4*gls_fac6**2*Gh*Gm+
     &          gls_b5*gls_fac6**2*Gm*Gm
            Sm=(gls_s0-gls_s1*gls_fac6*Gh+gls_s2*gls_fac6*Gm)/cff
            Sh=(gls_s4-gls_s5*gls_fac6*Gh+gls_s6*gls_fac6*Gm)/cff
            Sm=MAX(Sm,0.0_r8)
            Sh=MAX(Sh,0.0_r8)
!
!  Relate Canuto stability to ROMS notation
!
            Sm=Sm*sqrt2/gls_cmu0**3
            Sh=Sh*sqrt2/gls_cmu0**3 
# elif defined KANTHA_CLAYSON
            cff=1.0_r8-my_Sh2*Gh
            Sh=my_Sh1/cff
            Sm=(my_B1pm1o3+my_Sm4*Sh*Gh)/(1.0_r8-my_Sm2*Gh)
# else ! Galperin
            cff=1.0_r8-my_Sh2*Gh
            Sh=my_Sh1/cff
            Sm=(my_Sm3+Sh*Gh*my_Sm4)/(1.0_r8-my_Sm2*Gh)
# endif
!
!  Compute vertical mixing (m2/s) coefficients of momentum and
!  tracers.  Average ql over the two timesteps rather than using
!  the new Lscale and just averaging tke.
!
            ql=sqrt2*0.5_r8*(Ls_lmt*SQRT(tke(i,j,k,nnew))+
     &                       Lscale(i,j,k)*SQRT(tke(i,j,k,nstp)))
            Akv(i,j,k)=Akv_bak+Sm*ql
            do itrc=1,NAT
              Akt(i,j,k,itrc)=Akt_bak(itrc)+Sh*ql
            enddo
!
!  Compute vertical mixing (m2/s) coefficents of turbulent kinetic
!  energy and generic statistical field.
!
            if (Lmy25) then
              Akk(i,j,k)=Akk_bak+gls_cmu0*ql/(gls_sigk*sqrt2)
              Akp(i,j,k)=Akp_bak+gls_cmu0*ql/(gls_sigp*sqrt2)
            else
              Akk(i,j,k)=Akk_bak+Akv(i,j,k)/gls_sigk
              Akp(i,j,k)=Akp_bak+Akv(i,j,k)/gls_sigp
            endif
!
!  Save limited length scale.
!
            Lscale(i,j,k)=Ls_lmt
          enddo
!
!  Compute vertical mixing coefficients at the surface and bottom.
!
          Akv(i,j,N)=Akv_bak+vonKar*Zos*
     &               SQRT(0.5_r8*SQRT((sustr(i,j)+sustr(i+1,j))**2+
     &                                (svstr(i,j)+svstr(i,j+1))**2))
          Akv(i,j,0)=Akv_bak+vonKar*Zob*
     &               SQRT(0.5_r8*SQRT((bustr(i,j)+bustr(i+1,j))**2+
     &                                (bvstr(i,j)+bvstr(i,j+1))**2))
!
!!        if (Lmy25) then
!!          Akk(i,j,N)=Akk_bak
!!          Akk(i,j,0)=Akk_bak
!!          Akp(i,j,N)=Akp_bak
!!          Akp(i,j,0)=Akp_bak
!!        else
            Akk(i,j,N)=Akk_bak+Akv(i,j,N)/gls_sigk
            Akk(i,j,0)=Akk_bak+Akv(i,j,0)/gls_sigk
            Akp(i,j,N)=Akp_bak+Akv(i,j,N)/gls_sigp
            Akp(i,j,0)=Akp_bak+Akv(i,j,0)/gls_sigp
!!        endif
          do itrc=1,NAT
            Akt(i,j,N,itrc)=Akt_bak(itrc)
            Akt(i,j,0,itrc)=Akt_bak(itrc)
          enddo
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Set lateral boundary conditions.
!---------------------------------------------------------------------
!
      call tkebc_tile (Istr,Iend,Jstr,Jend,nnew,gradK,gradP)
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_w3d_tile (Istr,Iend,Jstr,Jend,
     &                        tke(START_2D_ARRAY,0,nnew))
      call exchange_w3d_tile (Istr,Iend,Jstr,Jend,
     &                        gls(START_2D_ARRAY,0,nnew))
      call exchange_w3d_tile (Istr,Iend,Jstr,Jend,
     &                        Akv(START_2D_ARRAY,0))
      do itrc=1,NAT
        call exchange_w3d_tile (Istr,Iend,Jstr,Jend,
     &                          Akt(START_2D_ARRAY,0,itrc))
      enddo
# endif
#else
      subroutine gls_corstep
#endif /* GLS_MIXING && SOLVE3D */
      return
      end
