#include "cppdefs.h"
      subroutine get_3dfld (Fout,Iout,Irec,ifield,ncid,ncfile,update)
!
!=====================================================================
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine reads in requested 3D field (point or grided) from   !
!  specified NetCDF file.                                            !
!                                                                    !
!  On Input:                                                         !
!                                                                    !
!     Iout       Size of the outer dimension,  if any.  Otherwise,   !
!                  Iout must be set to one by the calling program.   !
!     Irec       Number of 3D field records to read (integer).       !
!     ifield     Field ID (integer).                                 !
!     ncid       NetCDF file ID (integer).                           !
!     ncfile     NetCDF file name (string).                          !
!                                                                    !
!  On Output:                                                        !
!                                                                    !
!     Fout       Read field (real).                                  !
!     update     Switch indicating reading of the requested field    !
!                  the current time step.                            !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "iounits.h"
# include "ncparam.h"
# include "netcdf.inc"
# include "scalars.h"
!
      logical Liocycle, Lgrided, Lonerec, got_var, got_time, update
      INTEGER_TYPE
     &        Iout, Irec, Tid, Tindex, Trec, Vid, Vtype, i, ifield,
     &        latt, lstr, lfvar, ltvar, lvar, ncid, nrec, nvatts,
     &        status, vartype
      INTEGER_TYPE
     &        lenstr, nf_fread
      REAL_TYPE
     &        Clength, Fmax, Fmin, Fval, Tdelta, Tend, Tscale, Tstart,
     &        Tstr, Tval, scale
      REAL_TYPE
     &        Fout(GLOBAL_2D_ARRAY,N,Iout)
      character*45  attnam
      character*(*) ncfile
!
!---------------------------------------------------------------------
!  On first call, inquire about the contents of input NetCDF file.
!---------------------------------------------------------------------
!
      if (exit_flag.ne.0) return
!
      if (iic.eq.0) then
!
!  Intialize local variables.
!
        Liocycle=.false.
        Lgrided=.false.
        Lonerec=.false.
        got_var=.false.
        got_time=.false.
        Vid=-1
        Tid=-1
        Vtype=Iinfo(1,ifield)
!
!  Inquire about the dimensions and variables. Check for consistency.
!
        lstr=lenstr(ncfile)
        call opencdf (ncfile,N,ifield,nrec)
        if (exit_flag.ne.0) return
!
!  Scan variable list from input NetCDF and check for requested
!  variables.
!
        lfvar=lenstr(Vname(1,ifield))
        ltvar=lenstr(Tname(ifield))
        do i=1,nvars
          lvar=lenstr(varnam(i))
          if (lfvar.gt.0) then
            if (varnam(i)(1:lvar).eq.Vname(1,ifield)(1:lfvar)) then
              Vid=i
              got_var=.true.
              if (nvdims(i).gt.1) Lgrided=.true.
            endif
          endif
          if (ltvar.gt.0) then
            if (varnam(i)(1:lvar).eq.Tname(ifield)(1:ltvar)) then
              Tid=i
              got_time=.true.
            endif
          endif
        enddo
        Linfo(1,ifield)=Lgrided
        Iinfo(2,ifield)=Vid
        Iinfo(3,ifield)=Tid
        Iinfo(4,ifield)=nrec
!
!  Terminate execution requested variables are not found.
!
        if (.not.got_var.and.(lfvar.gt.0)) then
          write(stdout,10) Vname(1,ifield)(1:lfvar), ncfile(1:lstr)
          exit_flag=2
          return
        endif
        if (.not.got_time.and.(ltvar.gt.0)) then
          write(stdout,10) Tname(ifield)(1:ltvar), ncfile(1:lstr)
          exit_flag=2
          return
        endif
!
!  If appropriate, open input NetCDF file for reading.
!
        if (ncid.eq.-1) then
          status=nf_open(ncfile(1:lstr),nf_nowrite,ncid)
          if (status.ne.nf_noerr) then
            write(stdout,20) ncfile(1:lstr)
            exit_flag=2
            return
          endif
        endif
!
!  If "scale_factor" attribute is present for a variable, the data are
!  to be multiplied by this factor.  Check if only water points are
!  available.
!
        if (got_var) then
          status=nf_inq_var(ncid,Vid,varnam(Vid),vartype,nvdims(Vid),
     &                      vdims(1,Vid),nvatts)
          lvar=lenstr(varnam(Vid))
          if (status.eq.nf_noerr) then
            do i=1,nvatts
              status=nf_inq_attname(ncid,Vid,i,attnam)
              if (status.eq.nf_noerr) then
                latt=lenstr(attnam)
                if (attnam(1:latt).eq.'scale_factor') then
                  status=nf_get_att_FTYPE(ncid,Vid,attnam(1:latt),
     &                                    scale)
                  if (status.eq.nf_noerr) then
                    Fscale(ifield)=Fscale(ifield)*scale
                  else
                    write (stdout,30) attnam(1:latt)
                    exit_flag=2
                    return
                  endif
                elseif (attnam(1:latt).eq.'water_points') then
                  Iinfo(1,ifield)=-ABS(Iinfo(1,ifield))
                  Vtype=Iinfo(1,ifield)
                endif
              else
                write(stdout,40) varnam(Vid)(1:lvar)
                exit_flag=2
                return
              endif
            enddo
          else
            write(stdout,50)
            exit_flag=2
            return
          endif
        endif
!
!  Determine initial time record to read and cycling switch.
!
        call get_cycle (ncid,Tid,nrec,tdays,Liocycle,Clength,Trec,
     &                  Tstart,Tstr,Tend,Tscale)
        if (exit_flag.ne.0) return
        Linfo(2,ifield)=Liocycle
        Finfo(1,ifield)=Tstart
        Finfo(2,ifield)=Tend
        Finfo(3,ifield)=Clength
        Finfo(6,ifield)=Tscale
!
!  The strategy here is to create a local, monotonically increasing
!  time variable so the interpolation between snapshots is trivial
!  when cycling forcing fields. Subtract one to time record counter
!  "Trec" to avoid doing special case at initialization.
!
        if (Irec.eq.1) then
          Tindex=Iout
        else
          Tindex=1
        endif
        Vtime(Tindex,ifield)=Tstart
        if (Liocycle) then
          if (Trec.eq.nrec) then
            if (tdays.lt.Tend) then
              Tmono(ifield)=Tstart-Clength
            else
              Tmono(ifield)=tdays+(Tstart-Clength)
              if (Tstart.eq.Tend) then
                Tmono(ifield)=Tmono(ifield)+
     &                        (Tstr-MOD(tdays+Tstr,Clength))
              else
                Tmono(ifield)=Tmono(ifield)+
     &                        (Tstart-MOD(tdays+Tstart,Clength))
              endif
            endif
            Tmono(ifield)=Tmono(ifield)*day2sec
          else
            if (tdays.gt.Clength) then
              Tmono(ifield)=time-MOD(tdays-Tstart,Clength)*day2sec
            else
              Tmono(ifield)=Tstart*day2sec
            endif
          endif
        else
          Tmono(ifield)=Tstart*day2sec
        endif
        Trec=Trec-1
!
!  Set switch for one time record dataset. In this case, the input
!  data is always the same and time interpolation is not performed.
!
        if (nrec.eq.1) Lonerec=.true.
        Linfo(3,ifield)=Lonerec
!
!---------------------------------------------------------------------
!  Else, get requested field information from global storage.
!---------------------------------------------------------------------
!
      else
        Lgrided =Linfo(1,ifield)
        Liocycle=Linfo(2,ifield)
        Lonerec =Linfo(3,ifield)
        Vtype   =Iinfo(1,ifield)
        Vid     =Iinfo(2,ifield)
        Tid     =Iinfo(3,ifield)
        nrec    =Iinfo(4,ifield)
        Tindex  =Iinfo(5,ifield)
        Trec    =Iinfo(6,ifield)
        Clength =Finfo(3,ifield)
        Tscale  =Finfo(6,ifield)
      endif
!
!---------------------------------------------------------------------
!  If appropriate, read in new data.
!---------------------------------------------------------------------
!
      update=.false.
      if ((Tmono(ifield).lt.time).or.(iic.eq.0).or.
     &    (iic.eq.ntstart)) then
        if (Liocycle) then
          Trec=MOD(Trec,nrec)+1
        else
          Trec=Trec+1
        endif
        Iinfo(6,ifield)=Trec
        if (Trec.le.nrec) then
!
!  Set rolling index for two-time record storage of input data.  If
!  "Iout" is unity, input data is stored in timeless array by the
!  calling program.  If Irec > 1, this routine is used to read a 3D
!  field varying in another non-time dimension.
!
          if (Irec.eq.1) then
            if (Iout.eq.1) then
              Tindex=1
            else
              Tindex=3-Tindex
            endif
            Iinfo(5,ifield)=Tindex
          endif
!
!  Read in time coordinate and scale it to day units.
!
          if (Tid.ge.0) then
            status=nf_get_var1_FTYPE(ncid,Tid,Trec,Tval)
            Vtime(Tindex,ifield)=Tval*Tscale
            if (status.ne.nf_noerr) then
              ltvar=lenstr(Tname(ifield))
              write(stdout,60) Tname(ifield)(1:ltvar), Trec
              exit_flag=2
              return
            endif
          endif
!
!  Read in 3D-grided or point data.
!
          if (Vid.ge.0) then
            Fmin=0.0_r8
            Fmax=0.0_r8
            if (Lgrided) then
              if (Irec.gt.1) then
                Finfo(4,ifield)=1.0_e8+35
                Finfo(5,ifield)=-1.0_e8+35
                do i=1,Irec
                  status=nf_fread(Fout(START_2D_ARRAY,1,i),Fmin,Fmax,
     &                            Fscale(ifield),ncid,Vid,i,Vtype)
                  Finfo(4,ifield)=MIN(Fmin,Finfo(4,ifield))
                  Finfo(5,ifield)=MAX(Fmax,Finfo(5,ifield))
                enddo
              else
                status=nf_fread(Fout(START_2D_ARRAY,1,Tindex),Fmin,Fmax,
     &                          Fscale(ifield),ncid,Vid,Trec,Vtype)
                Finfo(4,ifield)=Fmin
                Finfo(5,ifield)=Fmax
              endif
            else
              status=nf_get_var1_FTYPE(ncid,Vid,Trec,Fval)
              Fpoint(Tindex,ifield)=Fval*Fscale(ifield)
              Fmin=Fval
              Fmax=Fval
            endif
            if (status.ne.nf_noerr) then
              lfvar=lenstr(Vname(1,ifield))
              write(stdout,60) Vname(1,ifield)(1:lfvar), Trec
              exit_flag=2
              return
            endif
            lfvar=lenstr(Vname(2,ifield))
            if (Irec.gt.1) then
              write(stdout,70) Vname(2,ifield)(1:lfvar), Fmin, Fmax
            else
              write(stdout,80) Vname(2,ifield)(1:lfvar), Tval*Tscale,
     &                         Fmin, Fmax
            endif
            update=.true.
          endif
        endif
!
!  Increment the local time variable "Tmono" by the interval between
!  snapshots. If the interval is negative, indicating cycling, add in
!  a cycle length.  Load time value (sec) into "Tintrp" which used
!  during interpolation between snapshots.
!
        if (.not.Lonerec) then
          Tdelta=Vtime(Tindex,ifield)-Vtime(3-Tindex,ifield)
          if (Liocycle.and.(Tdelta.lt.0.0_r8)) then
            Tdelta=Tdelta+Clength
          endif
          Tmono(ifield)=Tmono(ifield)+Tdelta*day2sec
          Tintrp(Tindex,ifield)=Tmono(ifield)
        endif
      endif
!
  10  format(/,' GET_3DFLD   - unable to find requested variable: ',a,
     &       /,15x,'in input NetCDF file: ',a)
  20  format(/,' GET_3DFLD   - unable to open input NetCDF file: ',a)
  30  format(/,' GET_2DFLD   - error while reading attribute: ',a)
  40  format(/,' GET_2DFLD   - error while inquiring attributes for',
     &        ' variable: ',a)
  50  format(/,' GET_2DFLD   - cannot inquire about time variable from',
     &        ' input NetCDF file.')
  60  format(/,' GET_3DFLD   - error while reading variable: ',a,2x,
     &       ' at TIME index = ',i4)
  70  format(' GET_3DFLD   - ',a,/,17x,'(Min = ',1pe15.8,' Max = ',
     &       1pe15.8,')')
  80  format(' GET_3DFLD   - ',a,',',t64,'t = ',f12.4,/,17x,'(Min = ',
     &       1pe15.8,' Max = ',1pe15.8,')')
      return
      end
