#include "cppdefs.h"
      subroutine biology_tile (Istr,Iend,Jstr,Jend)
!
!======================================== Alexander F. Shchepetkin ===
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine computes the biological sources and sinks and adds   !
!  then the global biological fields.                                !
!                                                                    !
!  Reference:                                                        !
!                                                                    !
!    Fasham, M.J.R., H.W. Ducklow, and S.M. McKelvie,  1990:  A      !
!      nitrogen-based model of plankton dynamics in the oceanic      !
!      mixed layer, J. of Marine Res., 48, 591-639.                  !
!                                                                    !
!  Adapted from code written originally by John Moisan and modified  !
!  by Emanule Di Lorenzo.                                            !
!                                                                    !
!=====================================================================
!
      implicit none
#include "param.h"
#include "biology.h"
#include "forces.h"
#include "grid.h"
#include "ocean.h"
#include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Iter, Jend, Jstr, i, j, k
      REAL_TYPE
     &        Att, Epp, PAR, Qval, Vp, aL, aR, cff, cff1, cff2, cu,
     &        eps, dtdays
      REAL_TYPE
     &        aJ(N), dSDe(0:N), dLDe(0:N), FC(0:N)
      REAL_TYPE
     &        CPr(N), CPr_bak(N), LDe(N), LDe_bak(N),
     &        NH4(N), NH4_bak(N), NO3(N), NO3_bak(N),
     &        Phy(N), Phy_bak(N), SDe(N), SDe_bak(N),
     &        Zoo(N), Zoo_bak(N)
!
      parameter (eps=1.0_e8-20)
!
#include "set_bounds.h"
#ifdef EW_PERIODIC
# define I_RANGE Istr,Iend
#else
# define I_RANGE IstrR,IendR
#endif
#ifdef NS_PERIODIC
# define J_RANGE Jstr,Jend
#else
# define J_RANGE JstrR,JendR
#endif
#ifdef PROFILE
!
!---------------------------------------------------------------------
!  Turn on biological module time wall clock.
!---------------------------------------------------------------------
!
      call wclock_on (15)
#endif
!
!---------------------------------------------------------------------
!  Add biological Source/Sink terms.
!---------------------------------------------------------------------
!
!  Since the following solver is iteractive to achieve implicit
!  discretization of the biological interaction, two time levels are
!  required, BIO and BIO_bak, where BIO is understood as a vector of
!  biological state variables: BIO=[NO3,NH4,Phy,Zoo,SDe,Lde]. Assuming
!  that the iterations converge, the newly obtained state variables
!  satisfy
!
!           BIO = BIO_bak + dtdays * rhs(BIO)
!
!  where rhs(BIO) is the vector of biological right-hand-side
!  computed at the new time step. During the iterative procedure
!  a series of fractional time steps are performed in a chained
!  mode (splitting  by different biological conversion processes)
!  in sequence NO3 =>  NH4 => Phy => Zoo => SDe => Lde, that is the
!  main food chain.  In all stages the concentration of the component
!  being consumed is treated in fully implicit manner, so that the
!  algorithm guarantees non-negative values, no matter how strong
!  is the concentration of active consuming component (Phy or Zoo).
!  The overall algorithm, as well as any stage of it, is formulated
!  in conservative form (except explicit sinking) in sense that the
!  sum of concentration of all components is conserved.
!
      dtdays=dt*sec2day/FLOAT(BioIter)
!
      do j=J_RANGE
        do i=I_RANGE
!
!  Extract biological variables from tracer arrays, place them into
!  scratch arrays, and restrict their values to be positive definite.
!
          do k=1,N
            NO3_bak(k)=MAX(t(i,j,k,nnew,iNO3_),0.0_r8)
            NH4_bak(k)=MAX(t(i,j,k,nnew,iNH4_),0.0_r8)
            Phy_bak(k)=MAX(t(i,j,k,nnew,iPhyt),0.0_r8)
            Zoo_bak(k)=MAX(t(i,j,k,nnew,iZoop),0.0_r8)
            SDe_bak(k)=MAX(t(i,j,k,nnew,iSDet),0.0_r8)
            LDe_bak(k)=MAX(t(i,j,k,nnew,iLDet),0.0_r8)
            CPr_bak(k)=MAX(t(i,j,k,nnew,iChlo),0.0_r8)/
     &                 (Phy_bak(k)*PhyCN*12.0_r8+eps)
            NO3(k)=NO3_bak(k)                         ! Nitrate
            NH4(k)=NH4_bak(k)                         ! Ammonium
            Phy(k)=Phy_bak(k)                         ! Phytoplankton
            Zoo(k)=Zoo_bak(k)                         ! Zooplankton
            SDe(k)=SDe_bak(k)                         ! Small Detritus
            LDe(k)=LDe_bak(k)                         ! Large Detritus
            CPr(k)=CPr_bak(k)                         ! Chl/Phy ratio
          enddo
!
!  Calulate Photosynthetically Available Radiation (PAR).  The net
!  shortwave radiation is scaled back to Watts/m2 and multiplied by
!  the fraction that is photosynthetically available, PARfrac.
!
          PAR=PARfrac*srflx(i,j)*rho0*Cp
!
!  Start internal iterations to achieve convergence of the nonlinear
!  backward-implicit solution.
!
          do Iter=1,BioIter
!
!  Compute sunlight dependent quantities.
!
            if (PAR.gt.0.0_r8) then
              do k=N,1,-1
                cff=AttSW+AttChl*PhyCN*12.0_r8*CPr(k)*Phy(k)
                Att=EXP(-0.5_r8*cff*(z_w(i,j,k)-z_w(i,j,k-1)))
                PAR=PAR*Att
                Vp=0.59_r8*(1.066_r8**t(i,j,k,nnew,itemp))
!
                cff1=PAR*PhyIS*CPr(k)
                Epp=Vp/SQRT(Vp*Vp+cff1*cff1)
                aJ(k)=Epp*cff1
!
                cff=K_NO3*NO3(k)+K_NH4*NH4(k)
                Qval=cff/(1.0_r8+cff)
                cff2=dtdays*aJ(k)*Qval
                CPr(k)=(CPr_bak(k)+CPrMax*ChlMB*Epp*Qval*cff2)/
     &                 (1.0_r8+cff2)
                PAR=PAR*Att
              enddo
!
!  Nitrate uptake by Phytoplankton.
!
              do k=1,N
                cff=dtdays*Phy(k)*aJ(k)*K_NO3*EXP(-PhyIP*NH4(k))/
     &              (1.0_r8+K_NO3*NO3(k)+K_NH4*NH4(k))
                NO3(k)=NO3_bak(k)/(1.0_r8+cff)
                Phy(k)=Phy_bak(k)+cff*NO3(k)
              enddo
!
!  Ammonium uptake by Phytoplankton.
!
              do k=1,N
                cff=dtdays*Phy(k)*aJ(k)*K_NH4/
     &              (1.0_r8+K_NO3*NO3(k)+K_NH4*NH4(k))
                NH4(k)=NH4_bak(k)/(1.0_r8+cff)
                Phy(k)=Phy(k)+cff*NH4(k)
              enddo
!
!  Processes that require sunlight are absent during the night.
!  (Set quatities old step values).

            else
              do k=1,N
                NO3(k)=NO3_bak(k)
                NH4(k)=NH4_bak(k)
                Phy(k)=Phy_bak(k)
                CPr(k)=CPr_bak(k)
              enddo
            endif
!
!  Phytoplankton grazing by Zooplankton (rate: ZooGR) and mortality
!  to Small Detritus (rate: PhyMR).
!
            do k=1,N
              cff=dtdays*ZooGR*Zoo(k)/(K_Phy+Phy(k))
              Phy(k)=Phy(k)/(1.0_r8+cff+dtdays*PhyMR)
              Zoo(k)=Zoo_bak(k)+Phy(k)*cff*ZooAE
              SDe(k)=SDe_bak(k)+Phy(k)*(cff*(1.0_r8-ZooAE)+
     &                                  dtdays*PhyMR)
            enddo
!
!  Zooplankton excretion to NH4 (rate: ZooER) and mortality to
!  Small Detritus (rate: ZooMR).
!
            do k=1,N
              Zoo(k)=Zoo(k)/(1.0_r8+dtdays*(ZooER+ZooMR))
              NH4(k)=NH4(k)+dtdays*ZooER*Zoo(k)
              SDe(k)=SDe(k)+dtdays*ZooMR*Zoo(k)
            enddo
!
!  Small Detritus breakdown to NH4 (rate: SDeBR) and conversion into
!  Large Detritus (coagulation rate: SDeAR), also loss, if any.
!
            do k=1,N
              cff=SDeAR*SDe(k)
              SDe(k)=SDe(k)/(1.0_r8+dtdays*(SDeBR+cff))
              NH4(k)=NH4(k)+dtdays*SDeBR*SDe(k)
              LDe(k)=LDe_bak(k)+dtdays*cff*SDe(k)
            enddo
!
!  Large Detritus conversion to NO3 (re-mineralization rate: LDeRR).
!
            do k=1,N
              LDe(k)=LDe(k)/(1.0_r8+dtdays*LDeRR)
              NO3(k)=NO3(k)+dtdays*LDeRR*LDe(k)
            enddo
!
!  Vertical sinking: Vertical advection algorithm based on monotonic,
!  continuous conservative parabolic splines.
!
!  Construct parabolic splines:  Compute vertical derivatives for
!  "SDe" and "LDe" at W-points.  Neumann boundary conditions are
!  assumed on top and bottom.
!
            FC(0)=0.0_r8
            dSDe(0)=0.0_r8
            dLDe(0)=0.0_r8
            do k=1,N-1
              cff=1.0_r8/(2.0_r8*Hz(i,j,k+1)+
     &                    Hz(i,j,k)*(2.0_r8-FC(k-1)))
              FC(k)=cff*Hz(i,j,k+1)
              dSDe(k)=cff*(6.0_r8*(SDe(k+1)-SDe(k))-
     &                     Hz(i,j,k)*dSDe(k-1))
              dLDe(k)=cff*(6.0_r8*(LDe(k+1)-LDe(k))-
     &                     Hz(i,j,k)*dLDe(k-1))
            enddo
            dSDe(N)=0.0_r8
            dLDe(N)=0.0_r8
            do k=N-1,1,-1
              dSDe(k)=dSDe(k)-FC(k)*dSDe(k+1)
              dLDe(k)=dLDe(k)-FC(k)*dLDe(k+1)
            enddo
!
!  Convert vertical derivatives "dSDe" and "dLDe" into field values
!  at grid box interfaces assuming parabolic profiles within each
!  grid box.  Restrict these values to lie between bounds determined
!  from box-averaged values of grid boxes adjscent from above and
!  below. This restriction is part of the PPM-like monotonization
!  procedure.
!
            cff=1.0_r8/3.0_r8
            dSDe(0)=SDe(1) ! -cff*Hz(i,j,1)*(dSDe(0)+0.5_r8*dSDe(1))
            dLDe(0)=LDe(1) ! -cff*Hz(i,j,1)*(dLDe(0)+0.5_r8*dLDe(1))
            dSDe(N)=SDe(N) ! +cff*Hz(i,j,N)*(dSDe(N)+0.5_r8*dSDe(N-1))
            dLDe(N)=LDe(N) ! +cff*Hz(i,j,N)*(dLDe(N)+0.5_r8*dLDe(N-1))
            do k=2,N
              dSDe(k-1)=SDe(k)-cff*Hz(i,j,k)*(0.5_r8*dSDe(k)+dSDe(k-1))
              dSDe(k-1)=MAX(dSDe(k-1),MIN(SDe(k-1),SDe(k)))
              dSDe(k-1)=MIN(dSDe(k-1),MAX(SDe(k-1),SDe(k)))
              dLDe(k-1)=LDe(k)-cff*Hz(i,j,k)*(0.5_r8*dLDe(k)+dLDe(k-1))
              dLDe(k-1)=MAX(dLDe(k-1),MIN(LDe(k-1),LDe(k)))
              dLDe(k-1)=MIN(dLDe(k-1),MAX(LDe(k-1),LDe(k)))
            enddo
!
!  Convert "dSDe" and "dLDe" into flux-integrated values;  complete
!  PPM flux limiting.  This procedure starts from assigning left and
!  right (aR,aL) values of the interpolating parabolae, then the
!  monotonicity conditions are checked and aL, aR are modified to fit.
!
            do k=1,N
              FC(k)=dtdays/Hz(i,j,k)
              aR=dSDe(k)
              aL=dSDe(k-1)
              cff1=(aR-aL)*6.0_r8*(SDe(k)-0.5_r8*(aR+aL))
              cff2=(aR-aL)**2
              if ((aR-SDe(k))*(SDe(k)-aL).lt.0.0_r8) then
                aL=SDe(k)
                aR=SDe(k)
              elseif (cff1.gt.cff2) then
                aL=3.0_r8*SDe(k)-2.0_r8*aR
              elseif (cff1.lt.-cff2) then
                aR=3.0_r8*SDe(k)-2.0_r8*aL
              endif
              cu=wSDe*FC(k)
              dSDe(k-1)=SDe(k)-(1.0_r8-cu)*(0.5_r8*(aR-aL)-
     &                  (0.5_r8*(aR+aL)-SDe(k))*(1.0_r8-2.0_r8*cu))
              aR=dLDe(k)
              aL=dLDe(k-1)
              cff1=(aR-aL)*6.0_r8*(LDe(k)-0.5_r8*(aR+aL))
              cff2=(aR-aL)**2
              if ((aR-LDe(k))*(LDe(k)-aL).lt.0.0_r8) then
                aL=LDe(k)
                aR=LDe(k)
              elseif (cff1.gt.cff2) then
                aL=3.0_r8*LDe(k)-2.0_r8*aR
              elseif (cff1.lt.-cff2) then
                aR=3.0_r8*LDe(k)-2.0_r8*aL
              endif
              cu=wLDe*FC(k)
              dLDe(k-1)=LDe(k)-(1.0_r8-cu)*(0.5_r8*(aR-aL)-
     &                  (0.5_r8*(aR+aL)-LDe(k))*(1.0_r8-2.0_r8*cu))
            enddo
            dSDe(N)=0.0_r8
            dLDe(N)=0.0_r8
!
!  Apply vertical fluxes.
!
            do k=1,N
              SDe(k)=SDe(k)+wSDe*FC(k)*(dSDe(k)-dSDe(k-1))
              LDe(k)=LDe(k)+wLDe*FC(k)*(dLDe(k)-dLDe(k-1))
            enddo
          enddo
!
!  Update global tracer variables.
!
          do k=1,N
            t(i,j,k,nnew,iNO3_)=MAX(0.0_r8,
     &                          t(i,j,k,nnew,iNO3_)+NO3(k)-NO3_bak(k))
            t(i,j,k,nnew,iNH4_)=MAX(0.0_r8,
     &                          t(i,j,k,nnew,iNH4_)+NH4(k)-NH4_bak(k))
            t(i,j,k,nnew,iPhyt)=MAX(0.0_r8,
     &                          t(i,j,k,nnew,iPhyt)+Phy(k)-Phy_bak(k))
            t(i,j,k,nnew,iZoop)=MAX(0.0_r8,
     &                          t(i,j,k,nnew,iZoop)+Zoo(k)-Zoo_bak(k))
            t(i,j,k,nnew,iSDet)=MAX(0.0_r8,
     &                          t(i,j,k,nnew,iSDet)+SDe(k)-SDe_bak(k))
            t(i,j,k,nnew,iLDet)=MAX(0.0_r8,
     &                          t(i,j,k,nnew,iLDet)+LDe(k)-LDe_bak(k))
            t(i,j,k,nnew,iChlo)=MAX(0.0_r8,
     &                          t(i,j,k,nnew,iChlo)+PhyCN*12.0_r8*
     &                          (Phy(k)*CPr(k)-Phy_bak(k)*CPr_bak(k)))
          enddo
        enddo
      enddo
#ifdef PROFILE
!
!---------------------------------------------------------------------
!  Turn off biological module time wall clock.
!---------------------------------------------------------------------
!
      call wclock_off (15)
#endif
      return
      end
