#include "cppdefs.h"
       subroutine biology_tile (Istr,Iend,Jstr,Jend)
!
!================================================= W. Paul Bissett ===
!  Copyright (c) 2002 TOMS Group/FERI                                !
!  Copyright (c) 1997 W. Paul Bissett                                !
!                                                                    !
!  The EcoSim code has been developed for research purposes only. It !
!  consists of unpublished, proprietary formulations protected under !
!  U.S. copyright law. It is freely available on request from the    !
!  Florida Environmental Research Institute (FERI). Commercial usage !
!  of these formulations is forbidden without express written        !
!  permission from FERI. All rights reserved.                        !
!                                                                    !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine computes the EcoSim sources and sinks and adds them  !
!  to the global biological fields.                                  !
!                                                                    !
!  Reference:                                                        !
!                                                                    !
!    Bissett, W.P., J.J. Walsh, D.A. Dieterle, K.L. Carder, 1999:    !
!      Carbon cycling in the upper waters of the Sargasso Sea: I.    !
!      Numerical  simulation of  differential carbon and nitrogen    !
!      fluxes,  Deep-Sea Res., 46, 205-269.                          !
!                                                                    !
!    Bissett, W.P., K.L. Carder, J.J. Walsh, D.A. Dieterle, 1999:    !
!      Carbon cycling in the upper waters of the Sargasso Sea: II.   !
!      Numerical  simulation  of  apparent  and  inherent optical    !
!      properties, Deep-Sea Res., 46, 271-317                        !
!                                                                    !
!  NOTES to EcoSim                                                   !
!  This version uses a descending index for depth that is different  !
!  than the original coding.                                         !
!                                                                    !
!                                                                    !
!=====================================================================
!
!
      implicit none
#include "globaldefs.h"
#include "param.h"
#include "ecosim.h"
#include "forces.h"
#include "grid.h"
#include "mask.h"
#include "ocean.h"
#include "scalars.h"
!
      INTEGER_TYPE
     &          Iend, Istr, Jend, Jstr, Tindex, i, id, j, k
!
!  Local variable declarations.
!
      INTEGER_TYPE
     &          ibac, ilab, iphy, itemp, isalt, ibnd, ichl,
     &          ipig, irct, isfc, iffc, ifec, idom,
     &          Iter, BioIter, bot_NZ
!
      INTEGER_TYPE
     &          RtUVR_flag, RtNIT_flag, Regen_flag, CDOC_flag,
     &          NFIX_flag, restart_flag
      parameter (RtUVR_flag=1, RtNIT_flag=1, Regen_flag=0,
     &           CDOC_flag=1, NFIX_flag=0, restart_flag=0)
!
      REAL_TYPE
     &          FV1, FV2, FV3, FV4, FV5, FV6, FV7, MinVal
      REAL_TYPE
     &          dtbio
!
!  Start Light Arrays Declarations.
!
      REAL_TYPE
     &          Ed_tot, aph442, tot_ab, tot_b, tot_bb, par_b, par_bb,
     &          avgcos_min, slope_AC, aPHYN_wa, tChl, theta_m,
     &          RtUVR_DIC, RtUVR_DOC, photo_decay, total_photo,
     &          photo_DIC, photo_DOC
      REAL_TYPE
     &          dATT_sum(Nbnd), wavedp(Nbnd)
      REAL_TYPE
     &          Ed_nz(N), E0_nz(N)
      REAL_TYPE
     &          specir_d(N, Nbnd), specir_scal(N, Nbnd),
     &          avgcos(N, Nbnd), dATT(N, Nbnd)
      REAL_TYPE
     &          aPHYN_al(N, Nphy, Nbnd), aPHYN_at(N, Nphy, Nbnd)
!
      REAL_TYPE
     &          DOC_lab, Bac_Ceff, I_Bac_Ceff, Nup_max,
     &          NH4toNO3, NtoNBAC, NtoPBAC, NtoFeBAC, RtNIT,
     &          HsNIT, Het_BAC, ExBAC_c, ExBAC_n, Frac_ExBAC_n, N_quota,
     &          R_ExBAC_c, RelDOC1, RelDON1, RelDOP1, RelFe,
     &          RelDOC1, RelDON1, RelDOP1, RelFe, GtBAC_max,
     &          BacCYC, BacDOC, BacPEL
!
      REAL_TYPE
     &          Bac_G(4)
!
      REAL_TYPE
     &          NH4(N), NO3(N), PO4(N), SiO(N), FeO(N), DIC(N), DZ(N),
     &          NH4P1(N), NO3P1(N), PO4P1(N), SiOP1(N), FeOP1(N),
     &          DICP1(N), DOC_frac(N), NitrBAC(N),
     &          totNH4_d(N), totNO3_d(N), totPO4_d(N), totSiO_d(N),
     &          totFe_d(N), totDOC_d(N), totDON_d(N), totDOP_d(N),
     &          temp(N), salt(N), Zr(N), Zw(0:N)
!
      REAL_TYPE
     &          BacC(N, Nbac), BacF(N, Nbac), BacN(N, Nbac),
     &          BacP(N, Nbac),
     &          GtBAC(N, Nbac), BacCP1(N, Nbac), BacFP1(N, Nbac),
     &          BacNP1(N, Nbac), BacPP1(N, Nbac),
     &          NupDOC_ba(N, Nbac), NupDON_ba(N, Nbac),
     &          NupDOP_ba(N, Nbac), NupFe_ba(N, Nbac),
     &          NupNH4_ba(N, Nbac), NupPO4_ba(N, Nbac)
!
      REAL_TYPE
     &          PhyC(N, Nphy), PhyN(N, Nphy), PhyS(N, Nphy),
     &          PhyP(N, Nphy), PhyF(N, Nphy), GtALG(N, Nphy),
     &          PhyCP1(N, Nphy), PhyNP1(N, Nphy), PhySP1(N, Nphy),
     &          PhyPP1(N, Nphy), PhyFP1(N, Nphy),
     &          NupNO3(N, Nphy) ,NupNH4(N, Nphy), NupSiO(N, Nphy),
     &          NupPO4(N, Nphy), NupFe(N, Nphy),
     &          NupDOP(N, Nphy), NupDON(N, Nphy),
     &          C2nALG(N, Nphy), C2pALG(N, Nphy), C2CHL(N, Nphy),
     &          mu_bar_n(N, Nphy), mu_bar_s(N, Nphy), mu_bar_p(N, Nphy),
     &          mu_bar_f(N, Nphy), alph_(N, Nphy), pac_eff(N, Nphy),
     &          Gt_ll(N, Nphy), Gt_nl(N, Nphy), Gt_sl(N, Nphy),
     &          Gt_pl(N, Nphy),
     &          Gt_fl(N, Nphy), GtALG_r(N, Nphy),
     &          refuge(N, Nphy), graz_act(N, Nphy),
     &          C2CHL_w(N, Nphy)
!
      REAL_TYPE
     &         DOMC(N, Ndom), DOMN(N, Ndom), DOMP(N, Ndom),
     &         CDMC(N, Ndom), DOMCP1(N, Ndom), DOMNP1(N, Ndom),
     &         DOMPP1(N, Ndom), CDMCP1(N, Ndom)
!
      REAL_TYPE
     &         FecC(N, Nfec), FecN(N, Nfec), FecS(N, Nfec),
     &         FecP(N, Nfec), FecF(N, Nfec),
     &         FecCP1(N, Nfec), FecNP1(N, Nfec), FecSP1(N, Nfec),
     &         FecPP1(N, Nfec), FecFP1(N, Nfec),
     &         Regen_C(N, Nfec), Regen_N(N, Nfec), Regen_S(N, Nfec),
     &         Regen_P(N, Nfec), Regen_F(N, Nfec)
!
      REAL_TYPE
     &         Pigs(N, Nphy, Npig), PigsP1(N, Nphy, Npig),
     &         Pigs_w(N, Nphy, Npig)

      REAL_TYPE
     &        DIC_bak(N), FeO_bak(N), NH4_bak(N), NO3_bak(N),
     &        PO4_bak(N), SiO_bak(N)
      REAL_TYPE
     &        BacC_bak(N,Nbac), BacF_bak(N,Nbac), BacN_bak(N,Nbac),
     &        BacP_bak(N,Nbac)
      REAL_TYPE
     &        CDMC_bak(N,Ndom), DOMC_bak(N,Ndom), DOMN_bak(N,Ndom),
     &        DOMP_bak(N,Ndom)
      REAL_TYPE
     &        FecC_bak(N,Nfec), FecF_bak(N,Nfec), FecN_bak(N,Nfec),
     &        FecP_bak(N,Nfec), FecS_bak(N,Nfec)
      REAL_TYPE
     &        PhyC_bak(N,Nphy), PhyF_bak(N,Nphy), PhyN_bak(N,Nphy),
     &        PhyP_bak(N,Nphy), PhyS_bak(N,Nphy)
      REAL_TYPE
     &        Pigs_bak(N,Nphy,Npig)
!
#include "set_bounds.h"
#ifdef PROFILE
!
!---------------------------------------------------------------------
!  Turn on EcoSim module time wall clock.
!---------------------------------------------------------------------
!
      call wclock_on (15)
#endif
!
!---------------------------------------------------------------------
!  Compute relevant local quantities. some of these could be put into
!  parameters.
!---------------------------------------------------------------------
!
!  Load light parameters.
!
      Tindex=INT((time-dstart*day2sec)/dt)+1
      theta_m=Irr_angle(Tindex)
      do ibnd=1,Nbnd
        specper_m(ibnd)=D_irr(ibnd,Tindex)
        specir_m (ibnd)=T_irr(ibnd,Tindex)
      enddo
!
!  Bacteria Parameters
!
      BioIter=1
!!    BioIter=3
      GtBAC_max=2.0_r8
      BacCYC=4.583_e8-1
      BacPEL=8.340_e8-2
      BacDOC=4.583_e8-1
      HsDOC_ba(1)=1.300_e8+2
!
!  Calculated fecal parameters values.
!
      WF(1)=0.0_r8
      WF(2)=100.0_r8*day2sec
!
!  Calculated bacteria parameters values.
!
      HsNH4_ba(1) =HsDOC_ba(1)/C2nBAC
      HsPO4_ba(1) =HsDOC_ba(1)/C2pBAC
      HsFe_ba(1)  =HsFe_ba(1)/50.0_r8      ! v8r5
      ExBAC_c     =0.04_r8                 ! bacterial DOC excretion
      R_ExBAC_c   =1.0_r8/(1.0_r8-ExBAC_c) ! Recipical ExBAC_c
      ExBAC_n     =ExBAC_c*C2nBAC/15.0_r8
      Frac_ExBAC_n=1.0_r8-ExBAC_n          ! Fraction of ExBAC_n
      Bac_Ceff    =0.30_r8                 ! bacterial carbon eff.
      I_Bac_Ceff  =1.0_r8/Bac_Ceff         ! inverse of Bac_Ceff
      RtNIT       =0.04_r8*sec2day         ! MAX Nitrification rate
!
!  Calculated CDMC parameter values.
!

      RtUVR_DIC=0.0193_r8/3600.0_r8   ! uv degradation of DOC into DIC
      RtUVR_DOC=0.0034_r8/3600.0_r8   ! uv degradation of DOC into DOC
!
!  Spectral dependency for scattering and backscattering.
!
      do ibnd=1,Nbnd
        wavedp(ibnd)=(550.0_r8/(397.0_r8+FLOAT(ibnd)*DLAM))
      enddo
!
      ilab=1                     ! labile index for DOC.
      irct=2                     ! relict index for DOC.
      ichl=1                     ! pigment index for chlorophyll a
      isfc=1                     ! slow fecal pellet index
      iffc=2                     ! fast fecal pellet index
!
#ifdef EW_PERIODIC
# define I_RANGE Istr,Iend
#else
# define I_RANGE IstrR,IendR
#endif
#ifdef NS_PERIODIC
# define J_RANGE Jstr,Jend
#else
# define J_RANGE JstrR,JendR
#endif
!
!=====================================================================
!  Add EcoSim Source/Sink terms.
!=====================================================================
!
      do j=Jstr,Jend
        do i=Istr,Iend
#ifdef MASKING
          if (rmask(i,j).gt.0.5) then
#endif
!
!---------------------------------------------------------------------
!  Extract biological variables from tracer arrays, place them into
!  scratch arrays, and restrict their values to be positive definite.
!---------------------------------------------------------------------
!
          MinVal=0.0_r8
          do k=1,N
            temp(k)=t(i,j,k,nnew,itemp)
            salt(k)=t(i,j,k,nnew,isalt)
            DIC(k)=MAX(t(i,j,k,nnew,iDIC_),MinVal)
            FeO(k)=MAX(t(i,j,k,nnew,iFeO_),MinVal)
            NH4(k)=MAX(t(i,j,k,nnew,iNH4_),MinVal)
            NO3(k)=MAX(t(i,j,k,nnew,iNO3_),MinVal)
            PO4(k)=MAX(t(i,j,k,nnew,iPO4_),MinVal)
            SiO(k)=MAX(t(i,j,k,nnew,iSiO_),MinVal)
            DIC_bak(k)=DIC(k)
            FeO_bak(k)=FeO(k)
            NH4_bak(k)=NH4(k)
            NO3_bak(k)=NO3(k)
            PO4_bak(k)=PO4(k)
            SiO_bak(k)=SiO(k)
!
            do ibac=1,Nbac
              BacC(k,ibac)=MAX(t(i,j,k,nnew,iBacC(ibac)),MinVal)
              BacF(k,ibac)=MAX(t(i,j,k,nnew,iBacF(ibac)),MinVal)
              BacN(k,ibac)=MAX(t(i,j,k,nnew,iBacN(ibac)),MinVal)
              BacP(k,ibac)=MAX(t(i,j,k,nnew,iBacP(ibac)),MinVal)
              BacC_bak(k,ibac)=BacC(k,ibac)
              BacF_bak(k,ibac)=BacF(k,ibac)
              BacN_bak(k,ibac)=BacN(k,ibac)
              BacP_bak(k,ibac)=BacP(k,ibac)
            enddo
!
            do idom=1,Ndom
              CDMC(k,idom)=MAX(t(i,j,k,nnew,iCDMC(idom)),MinVal)
              DOMC(k,idom)=MAX(t(i,j,k,nnew,iDOMC(idom)),MinVal)
              DOMN(k,idom)=MAX(t(i,j,k,nnew,iDOMN(idom)),MinVal)
              DOMP(k,idom)=MAX(t(i,j,k,nnew,iDOMP(idom)),MinVal)
              CDMC_bak(k,idom)=CDMC(k,idom)
              DOMC_bak(k,idom)=DOMC(k,idom)
              DOMN_bak(k,idom)=DOMN(k,idom)
              DOMP_bak(k,idom)=DOMP(k,idom)
            enddo
!
            do ifec=1,Nfec
              FecC(k,ifec)=MAX(t(i,j,k,nnew,iFecC(ifec)),MinVal)
              FecF(k,ifec)=MAX(t(i,j,k,nnew,iFecF(ifec)),MinVal)
              FecN(k,ifec)=MAX(t(i,j,k,nnew,iFecN(ifec)),MinVal)
              FecP(k,ifec)=MAX(t(i,j,k,nnew,iFecP(ifec)),MinVal)
              FecS(k,ifec)=MAX(t(i,j,k,nnew,iFecS(ifec)),MinVal)
              FecC_bak(k,ifec)=FecC(k,ifec)
              FecF_bak(k,ifec)=FecF(k,ifec)
              FecN_bak(k,ifec)=FecN(k,ifec)
              FecP_bak(k,ifec)=FecP(k,ifec)
              FecS_bak(k,ifec)=FecS(k,ifec)
            enddo
!
            do iphy=1,Nphy
              PhyC(k,iphy)=MAX(t(i,j,k,nnew,iPhyC(iphy)),MinVal)
              PhyF(k,iphy)=MAX(t(i,j,k,nnew,iPhyF(iphy)),MinVal)
              PhyN(k,iphy)=MAX(t(i,j,k,nnew,iPhyN(iphy)),MinVal)
              PhyP(k,iphy)=MAX(t(i,j,k,nnew,iPhyP(iphy)),MinVal)
              if (iPhyS(iphy).gt.0) then
                PhyS(k,iphy)=MAX(t(i,j,k,nnew,iPhyS(iphy)),MinVal)
              else
                PhyS(k,iphy)=0.0_r8
              endif
              PhyC_bak(k,iphy)=PhyC(k,iphy)
              PhyF_bak(k,iphy)=PhyF(k,iphy)
              PhyN_bak(k,iphy)=PhyN(k,iphy)
              PhyP_bak(k,iphy)=PhyP(k,iphy)
              PhyS_bak(k,iphy)=PhyS(k,iphy)
              do ipig=1,Npig
                id=iPigs(iphy,ipig)
                Pigs(k,iphy,ipig)=0.0_r8
                if (id.gt.0) then
                  Pigs(k,iphy,ipig)=MAX(t(i,j,k,nnew,id),MinVal)
                  PigsP1(k,iphy,ipig)=Pigs(k,iphy,ipig)
                endif
                Pigs_bak(k,iphy,ipig)=Pigs(k,iphy,ipig)
              enddo
            enddo
          enddo
!
!---------------------------------------------------------------------
!  Compute temperature and salinity dependent variables.
!---------------------------------------------------------------------
!
!  Get level thicknesses.
!
          do k=1,N
            DZ(k)=Hz(i,j,k)
            Zr(k)=z_r(i,j,k)
          enddo
          do k=1,N
            Zw(k)=z_w(i,j,k)
          enddo
!
!  Initialize regenreation arrays.
!
          do ifec=1,Nfec
            do k=1,N
              Regen_C(k,ifec)=0.0_r8
              Regen_N(k,ifec)=0.0_r8
              Regen_P(k,ifec)=0.0_r8
              Regen_F(k,ifec)=0.0_r8
              Regen_S(k,ifec)=0.0_r8
            enddo
          enddo
!
!  Refuge depth calculation.
!
          do iphy = 1, Nphy
            do k = 1, N
              refuge(k,iphy)= MinRefuge(iphy)
            enddo
          enddo
!!
!!  Fecal remineralization rates. Depth and time dependent.  Calculated
!!  from VERTEX data in Martin et al., 1987 DSR. As there is no
!!  information on POP or POFe, assume that the regeneration is the
!!  same as nitrogen. Silica is given by values from Bidle & Azam.
!!
!!        if (Regen_flag.eq.1) then
!!
!!  Calculate values beneath 100 m. If total depth > 100 m, k loop 8054
!!  does not execute.
!!
!!          FV1=0.0_r8
!!          k=1
!!  NOTE: k2 is used to avoid EXIT call in IF test below.
!!            do k = 1, N                      !k loop, find 100 m
!!            do while ( ( FV1 .ge. 100.0_r8 )  !k loop, find 100 m
!!     &                 .and. ( k .le. N ) )
!!              FV1 = FV1 + DZ(k)
!!  NOTE: EXIT is F90 function.
!!              if ( FV1 .ge. 100.0_r8) EXIT
!!              k = k + 1
!!            enddo                            !k loop
!!            IV1 = k
!!            do ifec = 1, Nfec                !ifec loop 8053
!!
!!  If statement used because silica is the only slow sinking fecal
!!  material (ifec = 1). Trying to save cycles.
!!
!!              if (ifec .eq. 1) then          !if ifec = slow fecal
!!                do k = IV1, N                !k loop 8054
!!                  Regen_C(k,ifec)  = 0.0_r8
!!                  Regen_N(k,ifec)  = 0.0_r8
!!                  Regen_P(k,ifec)  = 0.0_r8
!!                  Regen_F(k,ifec) = 0.0_r8
!!                  Regen_S(k,ifec) = 0.13_r8 * sec2day
!!                enddo
!!
!!  REMEMBER - IV1-1 and N may evaluate to the same value. Will the MIN
!!             function fail?
!!                IV1 = MIN( IV1-1, N )
!!
!!                do k = 1, IV1              !k loop 8057
!!                  FV4 = EXP( 0.092_r8 * ( temp(k) - 20.0_r8 ) )
!!                  Regen_S(k,ifec) = 0.13d0 * sec2day * FV4
!!                enddo                      !k loop 8057
!!              else
!!                FV2 = 0.0_r8
!!                do k = IV1, N                !k loop 8054
!!                  FV2 = FV1 * 0.01_r8              !/ 100.d0
!!  REMEMBER - minus DZ(k) may yield value < 100.0 m. Slight difference
!!             to original coding.
!!                  FV3 = ( FV1 - DZ(k) ) * 0.01_r8  !/ 100.d0
!!                  FV5 = WF(ifec) / DZ(k)           ! inverse of DZ/WF
!!
!!                 Regen_C(k,ifec)  = ( 1.0_r8 - ( FV2 ** ( -0.858_r8 ) )
!!     &                              / ( FV3 ** ( -0.858_r8 ) ) )
!!                 Regen_N(k,ifec)  = ( 1.0_r8 - ( FV2 ** ( -0.988_r8 ) )
!!     &                              / ( FV3 ** ( -0.988_r8 ) ) )
!!                  Regen_P(k,ifec)  = Regen_N(k,ifec)
!!                  Regen_F(k,ifec) = Regen_N(k,ifec)
!!
!!                  Regen_C(k,ifec)  = Regen_C(k,ifec) * FV5
!!                  Regen_N(k,ifec)  = Regen_N(k,ifec) * FV5
!!                  Regen_P(k,ifec)  = Regen_P(k,ifec) * FV5
!!                  Regen_F(k,ifec) = Regen_F(k,ifec) * FV5
!!                  Regen_S(k,ifec) = 0.13_r8 * sec2day
!!
!!                  FV1              = FV1 + DZ(k)
!!                enddo                      !k loop 8054
!!
!! NOTE: If the total depth is < 100 m the above loop is not expected to
!!       to execute. Thus, we need to set the temperature dependent
!!       calculation for these areas to the hypothetical 100 m depth
!!       % reduction of a similar DZ(k) value, and assume a 20 deg C
!!       temperature at this hypothetical depth to calculate the
!!       regeneration in these shallower waters.
!!
!!       There will be a slight change from ES1 which used the
!!       temperature from the 100 m level, rather than assume 20 deg C.
!!
!!  REMEMBER - IV1-1 and N may evaluate to the same value. Will the MIN
!!             function fail?
!!                IV1 = MIN( IV1-1, N )
!!
!!                do k = 1, IV1              !k loop 8057
!!
!!                  FV2 = ( 1.0_r8 -
!!     &                 ( ( ( 100._r8 + DZ(k) ) * 0.01_r8 )
!!     &                 ** (-0.858_r8) ) / ( 1.0_r8 ** (-0.858_r8) ) )
!!                  FV3 = ( 1.0_r8 -
!!     &                 ( ( ( 100._r8 + DZ(k) ) * 0.01_r8 )
!!     &                 ** (-0.988_r8) ) / ( 1.0_r8 ** (-0.988_r8) ) )
!!
!!                  FV5 = WF(ifec) / DZ(k)          ! inverse of DZ/WF
!!
!! Surface factor increase. Temperature at 100 m is set to 20.0 degrees.
!!                  FV4 = EXP( 0.092_r8 * ( temp(k) - 20.0_r8 ) )
!!                  Regen_C(k,ifec)  = FV2 * FV4 * FV5
!!                  Regen_N(k,ifec)  = FV3 * FV4 * FV5
!!                  Regen_P(k,ifec)  = Regen_N(k,ifec)
!!                  Regen_F(k,ifec) = Regen_N(k,ifec)
!!                  Regen_S(k,ifec) = 0.13d0 * sec2day * FV4
!!                enddo                      !k loop 8057
!!              endif                        !ifec = slow fecal
!!            enddo                          !ifec loop 8053
!!          endif                            ! Regen if flag
!
!  Minimum intra-cellular nutrient stock calculations.
!
          FV1=dt*sec2day
          do iphy=1,Nphy
            C2nALGminABS (iphy)=minC2nALG(iphy)/
     &                          (1.0_r8+
     &                           GtALG_max(iphy)*Norm_Vol(iphy)*FV1)
            C2SiALGminABS(iphy)=minC2SiALG(iphy)/
     &                          (1.0_r8+
     &                           GtALG_max(iphy)*Norm_Vol(iphy)*FV1)
            C2pALGminABS (iphy)=minC2pALG(iphy)/
     &                          (1.0_r8+
     &                           GtALG_max(iphy)*Norm_Vol(iphy)*FV1)
            C2FeALGminABS(iphy)=minC2FeALG(iphy)/
     &                          (1.0_r8+
     &                           GtALG_max(iphy)*Norm_Vol(iphy)*FV1)
          enddo
!
!  Calculate temperature dependent growth rate.
!
          do iphy=1,Nphy
            do k=1,N
              GtALG(k,iphy)=GtALG_max(iphy)*sec2day*
     &                      EXP(0.0633_r8*(temp(k)-27.0_r8))
!
!  Calculate mu_bar for droop equation.
!
              mu_bar_n(k,iphy)=GtALG(k,iphy)*
     &                         ((maxC2nALG(iphy)*
     &                           (1.0_r8+GtALG(k,iphy)))/
     &                          (maxC2nALG(iphy)*
     &                           (1.0_r8+GtALG(k,iphy))-
     &                            minC2nALG(iphy)))
              if (HsSiO(iphy).lt.LARGE) then
                mu_bar_s(k,iphy)=GtALG(k,iphy)*
     &                           ((maxC2SiALG(iphy)*
     &                             (1.0_r8+GtALG(k,iphy)))/
     &                            (maxC2SiALG(iphy)*
     &                             (1.0_r8+GtALG(k,iphy))-
     &                              minC2SiALG(iphy)))
              else
                mu_bar_s(k,iphy)=LARGE
              endif
              if (HsPO4(iphy).lt.LARGE) then
                mu_bar_p(k,iphy)=GtALG(k,iphy)*
     &                           ((maxC2pALG(iphy)*
     &                             (1.0_r8+GtALG(k,iphy)))/
     &                            (maxC2pALG(iphy)*
     &                             (1.0_r8+GtALG(k,iphy))-
     &                              minC2pALG(iphy)))
              else
                mu_bar_p(k,iphy)=LARGE
              endif
              if (HsFe(iphy).lt.LARGE) then
                mu_bar_f(k,iphy)=GtALG(k,iphy)*
     &                           ((maxC2FeALG(iphy)*
     &                             (1.0_r8+GtALG(k,iphy)))/
     &                            (maxC2FeALG(iphy)*
     &                             (1.0_r8+GtALG(k,iphy))-
     &                              minC2FeALG(iphy)))
              else
                mu_bar_f(k,iphy)=LARGE
              endif
            enddo
          enddo
!
!  Bacterial growth rate from Fasham et al., 1990.
!
          do ibac=1,Nbac
            do k=1,N
              GtBAC(k,ibac)=GtBAC_max*sec2day*
     &                      EXP(0.092_r8*(temp(k)-27.0_r8))
            enddo
          enddo
!
!  Grazing rate calculation.
!  NOTE: ES1 included separation calculations for grazing beneath the
!        zone of refuge (250 m). This has been removed and may
!        result in differences in deeper waters.
!
          do iphy=1,Nphy
            do k=1,N
              graz_act(k,iphy)=GtALG(k,iphy)*
     &                         PhyC(k,iphy)/(HsGRZ(iphy)+PhyC(k,iphy))
            enddo
          enddo
!
!---------------------------------------------------------------------
!  Iterate biology source and sink terms.
!---------------------------------------------------------------------
!
          dtbio=dt/FLOAT(BioIter)
          do Iter=1,BioIter
            do k=1,N
              totNH4_d(k)=0.0_r8
              totNO3_d(k)=0.0_r8
              totPO4_d(k)=0.0_r8
              totSiO_d(k)=0.0_r8
              totFe_d (k)=0.0_r8
              totDOC_d(k)=0.0_r8
              totDON_d(k)=0.0_r8
              totDOP_d(k)=0.0_r8
            enddo
            do iphy=1,Nphy
              do k=1,N
                NupNH4(k,iphy)=0.0_r8
                NupNO3(k,iphy)=0.0_r8
                NupPO4(k,iphy)=0.0_r8
                NupSiO(k,iphy)=0.0_r8
                NupFe(k,iphy)=0.0_r8
                NupDON(k,iphy)=0.0_r8
                NupDOP(k,iphy)=0.0_r8
              enddo
            enddo
!
!  Compute Ratio Arrays.
!  (Calculating only those that are accessed more than once.)
!
            do iphy=1,Nphy
              do k=1,N
                if (PhyN(k,iphy).gt.0.0_r8) then
                  C2nALG(k,iphy)=PhyC(k,iphy)/PhyN(k,iphy)
                else
                  C2nALG(k,iphy)=0.0_r8
                endif 
                if (PhyP(k,iphy).gt.0.0_r8) then
                  C2pALG(k,iphy)=PhyC(k,iphy)/PhyP(k,iphy)
                else
                  C2pALG(k,iphy)=0.0_r8
                endif 
              enddo
            enddo
!
!---------------------------------------------------------------------
!  Daylight Computations.
!---------------------------------------------------------------------
!
! Initialize.
!
            Ed_nz(N)=0.0_r8
            E0_nz(N)=0.0_r8
            bot_NZ=N+1
            if (specir_m(21).gt.VSMALL) then
              do k=1,N-1
                Ed_nz(k)=0.0_r8
                E0_nz(k)=0.0_r8
              enddo
              do iphy=1,Nphy
                do k=1,N
                  alph_(k,iphy)=0.0_r8
                enddo
              enddo
              do ibnd=1,Nbnd
                dATT_sum(ibnd)=0.0_r8
                do k=1,N
                  avgcos(k,ibnd)=0.0_r8
                  dATT(k,ibnd)=0.0_r8
                enddo
              enddo
              do ibnd=1,Nbnd
                do iphy=1,Nphy
                  do k=1,N
                    aPHYN_at(k,iphy,ibnd)=0.0_r8
                    aPHYN_al(k,iphy,ibnd)=0.0_r8
                  enddo
                enddo
              enddo
!
!  Calculate average cosine at surface.
!  (See equation 14 Morel, 1991 Prog. Ocean.)
!
              Ed_tot=0.0_r8
              do ibnd=1,Nbnd
                Ed_tot=Ed_tot+specir_m(ibnd)*DLAM
                avgcos(N,ibnd)=theta_m*(1.0_r8-specper_m(ibnd))+
     &                         0.86_r8*specper_m(ibnd)
              enddo
!
!  Total aph(442). adp(442) is set to 50% of aph(442).
!  NOTE: choosing sbands=9 which is band 442 using v8r16
!        sbands formulation. If spectral resolution changes, this
!        value must change!
!
              do k=N,1,-1
                if (Ed_tot.ge.1.0_r8) then
                  aph442=0.0_r8
                  tChl=0.0_r8
                  do iphy=1,Nphy
                    if (PhyC(k,iphy).gt.0.0_r8) then
                      tChl=tChl+Pigs(k,iphy,ichl)
                      pac_eff(k,iphy)=1.0_r8
                      if (b_PacEff(iphy).gt.SMALL) then
                        FV2=Pigs(k,iphy,ichl)/(PhyC(k,iphy)*12.0_r8)
                        pac_eff(k,iphy)=MAX(0.5_r8,
     &                                      (MIN(1.0_r8,b_PacEff(iphy)+
     &                                           mxPacEff(iphy)*
     &                                           (FV2-b_C2Cl(iphy)))))
                      endif
                      ibnd=9
                      do ipig=1,Npig
                        if (iPigs(iphy,ipig).gt.0) then
                          aph442=aph442+Pigs(k,iphy,ipig)*
     &                                  apigs(ipig,ibnd)*pac_eff(k,iphy)
                        endif
                      enddo
                    endif
                  enddo
!
!  Calculate absorption.
!  Calculating phytoplankton absorption for attentuation calculation.
!  NOTE: 12 factor to convert to ugrams (mg m-3)
!
                  aph442=0.5_r8*aph442
                  do ibnd=1,Nbnd
                    tot_ab=0.0_r8
                    do iphy=1,Nphy
                      do ipig=1,Npig
                        if (iPigs(iphy,ipig).gt.0) then
                          aPHYN_at(k,iphy,ibnd)=aPHYN_at(k,iphy,ibnd)+
     &                                          Pigs(k,iphy,ipig)*
     &                                          apigs(ipig,ibnd)*
     &                                          pac_eff(k,iphy)
                        endif
                      enddo
                      tot_ab=tot_ab+aPHYN_at(k,iphy,ibnd)
!
!  Removing absorption due to PPC for alph_ calculation
!
                      ipig=5
                      aPHYN_al(k,iphy,ibnd)=aPHYN_at(k,iphy,ibnd)-
     &                                      Pigs(k,iphy,ipig)*
     &                                      apigs(ipig,ibnd)*
     &                                      pac_eff(k,iphy)
                    enddo
!
!  Adding detrital absorption.
!
                    tot_ab=tot_ab+
     &                     aph442*EXP(0.011_r8*(442.0_r8-
     &                                (397.0_r8+FLOAT(ibnd)*DLAM)))
!
!  Calculate CDOC absorption.
!  NOTE: 12 factor is to convert ugrams per liter, and 0.001 converts
!        to mg/liter.  Specific absorption
!        coefficients were calculated as m-1 / (mg DOC/liters sw).
!        net factor = (12*0.001) = 0.012
!
                    tot_ab=tot_ab+
     &                     0.012_r8*(CDMC(k,ilab)*aDOC(ilab,ibnd)+
     &                               CDMC(k,irct)*aDOC(irct,ibnd))+
     &                     awater(ibnd)
!
!  Calculate scattering and backscattering
!  See equation 19 Morel, 1991 Prog. Ocean.
!  Morel, 1988 puts spectral dependency in backscattering. Since
!  Morel, 1991 does not have a backscattering equation, use '88
!  paper.
!
                    par_b =0.3_r8*(tChl**0.62_r8)
                    par_bb=0.0_r8
                    if (tChl.gt.0.0_r8) then
                      par_bb=par_b*(0.002_r8+0.02_r8*
     &                            (0.5_r8-0.25_r8*LOG10(tChl)*
     &                             wavedp(ibnd)))
                    endif
                    par_bb=MAX(par_bb,0.0_r8)
!
!  However, for omega0 calculation, par_b must be spectral, so use
!  dependency from Sathy and Platt 1988
!
                    tot_b=bwater(ibnd)+par_b*wavedp(ibnd)
!
!  Morel, '88 instead of '91. see methods
!
                    tot_bb=0.5_r8*bwater(ibnd)+par_bb
!
!  Sathy. and Platt JGR 1988.  This is set with the average cosine of
!  the box above, and used to calculate a new avgcos for this level.
!  This new average cosine is then used to recalculate the attenuation
!  coefficient
!
                    dATT(k,ibnd)=(tot_ab+tot_bb)/avgcos(k,ibnd)
!
!  See Mobley, 1995 for graphical depiction of this equation.
!  JJolliff 2001 working on correction of this equation.
!
                    avgcos_min=avgcos(k,ibnd)+
     &                         (0.5_r8-avgcos(k,ibnd))*
     &                         (tot_b/(tot_ab+tot_b))
!
!  Calculate average cosine.
!  Linear fit to average cosine versus optical depth relationship.
!  The FV1 calculation keeps the denominator of the slope calculation
!  from going negative and above 1.
!
                    FV1=MAX(1.0_r8,
     &                      7.0_r8-dATT(k,ibnd)*ABS(Zr(k)))
                    slope_AC =MIN(0.0_r8,
     &                            (avgcos_min-avgcos(k,ibnd))/FV1)
                    avgcos(k,ibnd)=avgcos(k,ibnd)+
     &                             slope_AC*dATT(k,ibnd)*DZ(k)
                    dATT(k,ibnd)=(tot_ab+tot_bb)/avgcos(k,ibnd)
!
!  Set avgcos for next level.
!
                    if (k.ne.1) then
                      avgcos(k-1,ibnd)=avgcos(k,ibnd)
                    endif
!
!  Calculate spectral irradiance with depth.
!
                    FV1=dATT(k,ibnd)*DZ(k)
                    FV2=dATT_sum(ibnd)+0.5_r8*FV1
                    dATT_sum(ibnd)=dATT_sum(ibnd)+FV1
                    specir_d(k,ibnd)=specir_m(ibnd)*
     &                               EXP(-FV2)*DLAM
!
!  Calculate spectral scalar irradiance.  Morel, 1991 Prog. Ocean.
!
                    specir_scal(k,ibnd)=specir_d(k,ibnd)*
     &                                  (dATT(k,ibnd)/tot_ab)
                    E0_nz(k)=E0_nz(k)+specir_scal(k,ibnd)
!
!  Calculate Ed_nz.
!
                    Ed_nz(k)=Ed_nz(k)+specir_d(k,ibnd)
                  enddo
                  Ed_tot=E0_nz(k)
!
!  Set bottom of the euphotic zone.
!
                  bot_NZ=k
                endif
              enddo
            endif
!
!---------------------------------------------------------------------
!  Bacterial nutrient uptake.
!---------------------------------------------------------------------
!
            do ibac=1,Nbac
              do k=1,N
!
!  DOM uptake.
!
                if ((DOMC(k,ilab).gt.0.0_r8).and.
     &              (DOMN(k,ilab).gt.0.0_r8).and.
     &              (DOMP(k,ilab).gt.0.0_r8)) then
                  DOC_lab=DOMC(k,ilab)
                  DOC_frac(k)=DOMC(k,ilab)/DOC_lab
                  NupDOC_ba(k,ibac)=GtBAC(k,ibac)*
     &                              BacC(k,ibac)*I_Bac_Ceff*
     &                              (DOC_lab/(HsDOC_ba(ibac)+DOC_lab))
                  NupDON_ba(k,ibac)=NupDOC_ba(k,ibac)*
     &                              DOC_frac(k)*DOMN(k,ilab)/
     &                              DOMC(k,ilab)
                  NupDOP_ba(k,ibac)=NupDOC_ba(k,ibac)*
     &                              DOC_frac(k)*DOMP(k,ilab)/
     &                              DOMC(k,ilab)
                else
                  NupDOC_ba(k,ibac)=0.0_r8
                  NupDON_ba(k,ibac)=0.0_r8
                  NupDOP_ba(k,ibac)=0.0_r8
                endif
                totDOC_d(k)=totDOC_d(k)+NupDOC_ba(k,ibac)
                totDON_d(k)=totDON_d(k)+NupDON_ba(k,ibac)
                totDOP_d(k)=totDOP_d(k)+NupDOP_ba(k,ibac)
!
!  NH4 uptake.
!
                NupNH4_ba(k,ibac)=GtBAC(k,ibac)*BacN(k,ibac)*
     &                            NH4(k)/(HsNH4_ba(ibac)+NH4(k))
                totNH4_d(k)=totNH4_d(k)+NupNH4_ba(k,ibac)
!
!  PO4 uptake.
!
                NupPO4_ba(k,ibac)=GtBAC(k,ibac)*BacP(k,ibac)*
     &                            PO4(k)/(HsPO4_ba(ibac)+PO4(k))
                totPO4_d(k)=totPO4_d(k)+NupPO4_ba(k,ibac)
!
!  Fe uptake.
!
                NupFe_ba(k,ibac)=GtBAC(k,ibac)*BacF(k,ibac)*
     &                           FeO(k)/(HsFe_ba(ibac)+FeO(k))
                totFe_d(k)=totFe_d(k)+NupFe_ba(k,ibac)
!
              enddo
            enddo
!
!---------------------------------------------------------------------
!  Phytoplankton dark nutrient uptake.
!---------------------------------------------------------------------
!
            do k=1,N
              do iphy=1,Nphy
                if (C2nALG(k,iphy).gt.C2nALGminABS(iphy)) then
!
!  NOTE: these are being saved to test for total nutrient uptake.
!        If nutrient uptake is greater than maximum nutrient, then
!        each of the uptakes are reduced by their fractional contri-
!        bution to the total.
!
                  N_quota=1.0_r8/MAX(6.625_r8,C2nALG(k,iphy))
                  Nup_max=GtALG(k,iphy)
                  NupNO3(k,iphy)=(NO3(k)/(HsNO3(iphy)+NO3(k))*
     &                           EXP(-BET_(iphy)*NH4(k)))
                  NupNH4(k,iphy)=NH4(k)/(HsNH4(iphy)+NH4(k))
!
!  Test that Wroblewski's equation does not exceed 1.0.
!
                  FV1=NupNO3(k,iphy)+NupNH4(k,iphy)
                  if (FV1.gt.1.0_r8) then
                    FV1=1.0_r8/FV1
                    NupNO3(k,iphy)=NupNO3(k,iphy)*FV1
                    NupNH4(k,iphy)=NupNH4(k,iphy)*FV1
                  endif
!
!  Change from percentage of maximum to mass per second.
!
                  FV1=(Nup_max*PhyN(k,iphy))
                  NupNO3(k,iphy)=NupNO3(k,iphy)*FV1
                  NupNH4(k,iphy)=NupNH4(k,iphy)*FV1
!
!  Test for DON uptake.
!
                  if (C2nALG(k,iphy).gt.C2nNupDON(iphy)) then
                    NupDON(k,iphy)=FV1*(DOMN(k,ilab)/
     &                                  (HsDON(iphy)+DOMN(k,ilab)))
                  endif
!
!  Accumulate total demand for nutrients.
!
                  totNO3_d(k)=totNO3_d(k)+NupNO3(k,iphy)
                  totNH4_d(k)=totNH4_d(k)+NupNH4(k,iphy)
                  totDON_d(k)=totDON_d(k)+NupDON(k,iphy)
                endif
!
!  Dark silica uptake, min C2Si test.
!  The LARGE test can be removed after testing phase.
!
                if (HsSiO(iphy).lt.LARGE) then
!
!  FV2 added to eliminate C2SiALG array. However, if PhyS array is
!    zero because the functional group does not use silica this
!    calculation will cause a divide by zero. Use LARGE test to avoid
!    this situation.
!
                  if (PhyS(k,iphy).gt.0.0_r8) then
                    FV2=PhyC(k,iphy)/PhyS(k,iphy)
                    if (FV2.gt.C2SiALGminABS(iphy)) then
                      Nup_max=GtALG(k,iphy)
                      NupSiO(k,iphy)=SiO(k)/(HsSiO(iphy)+SiO(k))
!
!  Change from percentage of maximum to mass per second.
!
                      FV1=Nup_max*PhyS(k,iphy)
                      NupSiO(k,iphy)=NupSiO(k,iphy)*FV1
!
!  Accumulate total demand for nutrients.
!
                      totSiO_d(k)=totSiO_d(k)+NupSiO(k,iphy)
                    endif
                  endif
                endif
!
!  Dark phophorus uptake, min C2P test.
!  The LARGE test can be removed after testing phase.
!
                if (HsPO4(iphy).lt.LARGE) then
                  if (C2pALG(k,iphy).gt.C2pALGminABS(iphy)) then
                    Nup_max=GtALG(k,iphy)
                    NupPO4(k,iphy)=PO4(k)/(HsPO4(iphy)+PO4(k))
!
!  Change from percentage of maximum to mass per second.
!
                    FV1=(Nup_max*PhyP(k,iphy))
                    NupPO4(k,iphy)=NupPO4(k,iphy)*FV1
!
!  Test for alk. phosphatase
!
                    if (C2pALG(k,iphy).gt.C2pALKPHOS(iphy)) then
                      NupDOP(k,iphy)=FV1*(DOMP(k,ilab)/
     &                                    (HsDOP(iphy)+DOMP(k,ilab)))
                    endif
!
!  Accumulate total demand for nutrients.
!
                    totPO4_d(k)=totPO4_d(k)+NupPO4(k,iphy)
                    totDOP_d(k)=totDOP_d(k)+NupDOP(k,iphy)
                  endif
                endif
!
!  Dark iron uptake, min C2Fe test.
!  The LARGE test can be removed after testing phase.
!
                if (HsFe(iphy).lt.LARGE) then
!
!  FV2 added to eliminate C2FeALG array. However, if PhyF array is
!  zero because the functional group does not use silica this
!  calculation will cause a divide by zero. Use LARGE test to avoid
!  this situation.
!
                  if (PhyF(k,iphy).gt.0.0_r8) then
                    FV2=PhyC(k,iphy)/PhyF(k,iphy)
                    if (FV2.gt.C2FeALGminABS(iphy)) then
                      Nup_max=GtALG(k,iphy)
                      NupFe(k,iphy)=FeO(k)/(HsFe(iphy)+FeO(k))
!
!  Change from percentage of maximum to mass per second.
!
                      FV1=(Nup_max*PhyF(k,iphy))
                      NupFe(k,iphy)=NupFe(k,iphy)*FV1
!
!  Accumulate total demand for nutrients.
!
                      totFe_d(k)=totFe_d(k)+NupFe(k,iphy)
                    endif
                  endif
                endif
              enddo
!
!  Calculate bacterial nitrification as a Michaelis-Menton function
!  of ambient NH4 concentration, beneath the euphotic zone (light
!  inhibits nitrification).
!
              NH4toNO3=0.0_r8
              NtoNBAC=0.0_r8
              NtoPBAC=0.0_r8
              NtoFeBAC=0.0_r8
              if (k.lt.bot_NZ) then
                NH4toNO3=RtNIT*(NH4(k)/(HsNIT+NH4(k)))
!
!  Nitrification fixes DIC into POC.
!  Conversion factor of 7.0 from Kaplan 1983 "Nitrogen in the Sea"
!  factor equals (1.0 / (7.0 * C2nBAC)). Adds NH4 uptake as biomass.
!!              NitrBAC(k) = NH4toNO3 * ( 1.0d0 / (7.0d0) )
!
                NitrBAC(k)=NH4toNO3*0.142857_r8
                NtoNBAC =NitrBAC(k)*N2cBAC
                NtoPBAC =NitrBAC(k)*P2cBAC
                NtoFeBAC=NitrBAC(k)*Fe2cBAC
                totNH4_d(k)=totNH4_d(k)+NH4toNO3+NtoNBAC
                totPO4_d(k)=totPO4_d(k)+NtoPBAC
                totFe_d (k)=totFe_d (k)+NtoFeBAC
              endif
!
!---------------------------------------------------------------------
!  Test that total nutrient demand does not exceed supply.  If it does
!  total demand is normalized to the total supply. Each species demand
!  is reduced to its weighted average percentage of the supply.
!---------------------------------------------------------------------
!
              FV2=totNO3_d(k)*dtbio
              if (FV2.gt.NO3(k)) then
                FV1=NO3(k)/FV2
!!              FV1=MAX(0.0_r8,NO3(k)/FV2)
                do iphy=1,Nphy
                  NupNO3(k,iphy)=NupNO3(k,iphy)*FV1
                enddo
              endif
!
              FV2=totNH4_d(k)*dtbio
              if (FV2.gt.NH4(k)) then
                FV1=NH4(k)/FV2
!!              FV1=MAX(0.0_r8,NH4(k)/FV2)
                do iphy=1,Nphy
                  NupNH4(k,iphy)=NupNH4(k,iphy)*FV1
                enddo
                do ibac=1,Nbac
                  NupNH4_ba(k,ibac)=NupNH4_ba(k,ibac)*FV1
                enddo
                NH4toNO3=NH4toNO3*FV1
                NtoNBAC=NtoNBAC*FV1
              endif
!
              FV2=totSiO_d(k)*dtbio
              if (FV2.gt.SiO(k)) then
                FV1=SiO(k)/FV2
!!              FV1=MAX(0.0_r8,SiO(k)/FV2)
                do iphy=1,Nphy
                  NupSiO(k,iphy)=NupSiO(k,iphy)*FV1
                enddo
              endif
!
              FV2=totPO4_d(k)*dtbio
              if (FV2.gt.PO4(k)) then
                FV1=PO4(k)/FV2
!!              FV1=MAX(0.0_r8,PO4(k)/FV2)
                do iphy=1,Nphy
                  NupPO4(k,iphy)=NupPO4(k,iphy)*FV1
                enddo
                NtoPBAC=NtoPBAC*FV1
                NupPO4_ba(k,ibac)=NupPO4_ba(k,ibac)*FV1
              endif
!
              FV2=totFe_d(k)*dtbio
              if (FV2.gt.FeO(k)) then
                FV1=FeO(k)/FV2
!!              FV1=MAX(0.0_r8,FeO(k)/FV2)
                do iphy=1,Nphy
                  NupFe(k,iphy)=NupFe(k,iphy)*FV1
                enddo
                do ibac=1,Nbac
                  NupFe_ba(k,ibac)=NupFe_ba(k,ibac)*FV1
                enddo
                NtoFeBAC=NtoFeBAC*FV1
              endif
!
!  Bacteria are the only group to take up DOC.  Remove BAC DON and
!  BAC DOP uptake from total uptake; adjust uptake and add back.
!
              FV2=totDOC_d(k)*dtbio
              if (FV2.gt.DOMC(k,ilab)) then
                FV1=DOMC(k,ilab)/FV2
!!              FV1=MAX(0.0_r8,DOMC(k,ilab)/FV2)
                totDOC_d(k)=totDOC_d(k)*FV1
                do ibac=1,Nbac
                  NupDOC_ba(k,ibac)=NupDOC_ba(k,ibac)*FV1
                  totDON_d(k)=totDON_d(k)-NupDON_ba(k,ibac)
                  NupDON_ba(k,ibac)=NupDON_ba(k,ibac)*FV1
                  totDON_d(k)=totDON_d(k)+NupDON_ba(k,ibac)
                  totDOP_d(k)=totDOP_d(k)-NupDOP_ba(k,ibac)
                  NupDOP_ba(k,ibac)=NupDOP_ba(k,ibac)*FV1
                  totDOP_d(k)=totDOP_d(k)+NupDOP_ba(k,ibac)
                enddo
              endif
!
!  Bacteria are only group to take up DOC.  Remove BAC DOP uptake
!  from total uptake; adjust uptake and add back.
!
              FV2=totDON_d(k)*dtbio
              if (FV2.gt.DOMN(k,ilab)) then
                FV1=DOMN(k,ilab)/FV2
!!              FV1=MAX(0.0_r8,DOMN(k,ilab)/FV2)
                totDON_d(k)=totDON_d(k)*FV1
                totDOC_d(k)=totDOC_d(k)*FV1
                do iphy=1,Nphy
                  NupDON(k,iphy)=NupDON(k,iphy)*FV1
                enddo
                do ibac=1,Nbac
                  NupDON_ba(k,ibac)=NupDON_ba(k,ibac)*FV1
                  NupDOC_ba(k,ibac)=NupDOC_ba(k,ibac)*FV1
                  totDOP_d(k)=totDOP_d(k)-NupDOP_ba(k,ibac)
                  NupDOP_ba(k,ibac)=NupDOP_ba(k,ibac)*FV1
                  totDOP_d(k)=totDOP_d(k)+NupDOP_ba(k,ibac)
                enddo
              endif
!
!  Remove BAC DON uptake from total uptake; adjust uptake and add back.
!  Bacteria are only group to take up DOC.
!
              FV2=totDOP_d(k)*dtbio
              if (FV2.gt.DOMP(k,ilab)) then
                FV1=DOMP(k,ilab)/FV2
!!              FV1=MAX(0.0_r8,DOMP(k,ilab)/FV2)
                totDOP_d(k)=totDOP_d(k)*FV1
                totDOC_d(k)=totDOC_d(k)*FV1
                do iphy=1,Nphy
                  NupDOP(k,iphy)=NupDOP(k,iphy)*FV1
                enddo
                do ibac=1,Nbac
                  NupDOP_ba(k,ibac)=NupDOP_ba(k,ibac)*FV1
                  totDON_d(k)=totDON_d(k)-NupDON_ba(k,ibac)
                  NupDON_ba(k,ibac)=NupDON_ba(k,ibac)*FV1
                  totDON_d(k)=totDON_d(k)+NupDON_ba(k,ibac)
                  NupDOC_ba(k,ibac)=NupDOC_ba(k,ibac)*FV1
                enddo
              endif
!
!  Increase particulate nutrients by the amount of the uptake.
!
              do iphy=1,Nphy
                PhyNP1(k,iphy)=PhyNP1(k,iphy)+
     &                         NupNO3(k,iphy)+NupNH4(k,iphy)+
     &                         NupDON(k,iphy)
                PhySP1(k,iphy)=PhySP1(k,iphy)+
     &                         NupSiO(k,iphy)
                PhyPP1(k,iphy)=PhyPP1(k,iphy)+
     &                         NupPO4(k,iphy)+NupDOP(k,iphy)
                PhyFP1(k,iphy)=PhyFP1(k,iphy)+
     &                         NupFe(k,iphy)
!
!  Update nutrient arrays for growth and budgets. Bacterial uptake
!  included below.
!
                NO3P1(k)=NO3P1(k)-NupNO3(k,iphy)
                NH4P1(k)=NH4P1(k)-NupNH4(k,iphy)
                SiOP1(k)=SiOP1(k)-NupSiO(k,iphy)
                PO4P1(k)=PO4P1(k)-NupPO4(k,iphy)
                FeOP1(k)=FeOP1(k)-NupFe (k,iphy)
                DOMNP1(k,ilab)=DOMNP1(k,ilab)-NupDON(k,iphy)
                DOMPP1(k,ilab)=DOMPP1(k,ilab)-NupDOP(k,iphy)
              enddo
!!
!!  Budgeting to be addressed later.
!!
!!            do iphy=1,Nphy
!!              BGrwthNUT(1)=BGrwthNUT(1)+DZ(k)*NupNO3(k,iphy)
!!              BGrwthNUT(2)=BGrwthNUT(2)+DZ(k)*NupNH4(k,iphy)
!!              BGrwthNUT(3)=BGrwthNUT(3)+DZ(k)*NupSiO(k,iphy)
!!              BGrwthNUT(4)=BGrwthNUT(4)+DZ(k)*NupPO4(k,iphy)
!!              BGrwthNUT(5)=BGrwthNUT(5)+DZ(k)*NupFe (k,iphy)
!!              BGrwthORG(3)=BGrwthORG(3)+DZ(k)*NupDON(k,iphy)
!!              BGrwthORG(4)=BGrwthORG(4)+DZ(k)*NupDOP(k,iphy)
!!              BGrwthPS(iphy)=BGrwthPS(iphy)+DZ(k)*NupSiO(k,iphy)
!!              BGrwthPF(iphy)=BGrwthPF(iphy)+DZ(k)*NupFe (k,iphy)
!!              BGrwthPN(iphy)=BGrwthPN(iphy)+
!!   &                         DZ(k)*(NupNO3(k,iphy)+
!!   &                                NupNH4(k,iphy)+
!!   &                                NupDON(k,iphy))
!!              BGrwthPP(iphy)=BGrwthPP(iphy)+
!!                             DZ(k)*(NupPO4(k,iphy)+
!!                                    NupDOP(k,iphy))
!!            enddo
!
!  Nitrification fixes DIC into DOC.
!
              DICP1(k)=DICP1(k)-NitrBAC(k)
!
!  Add nitrifying bacteria biomass to heterotrophic bacteria biomass.
!  Adding PON, POP, POFe to BacC arrays at current C2_BAC ratios.
!
              do ibac=1,Nbac
                BacCP1(k,ibac)=BacCP1(k,ibac)+NitrBAC(k)
                BacNP1(k,ibac)=BacNP1(k,ibac)+NtoNBAC
                BacPP1(k,ibac)=BacPP1(k,ibac)+NtoPBAC
                BacFP1(k,ibac)=BacFP1(k,ibac)+NtoFeBAC
              enddo
!
!  Update nutrient arrays for nitrification and budgets.
!
              NO3P1(k)=NO3P1(k)+NH4toNO3
              NH4P1(k)=NH4P1(k)-(NH4toNO3+NtoNBAC)
              PO4P1(k)=PO4P1(k)-NtoPBAC
              FeOP1(k)=FeOP1(k)-NtoFeBAC
!!
!! Budgeting to be addressed later.
!!
!!            BNupBAC(1)=BNupBAC(1)+DZ(k)*NitrBAC(k)
!!            BNitrfNUT(1)=BNitrfNUT(1)+DZ(k)*NH4toNO3
!!            BNitrfNUT(2)=BNitrfNUT(2)+DZ(k)*(NH4toNO3+NtoNBAC)
!!            BNitrfNUT(3)=BNitrfNUT(3)+DZ(k)*NtoPBAC
!!            BNitrfNUT(4)=BNitrfNUT(4)+DZ(k)*NtoFeBAC
            enddo
!
!---------------------------------------------------------------------
!  Light mediated carbon growth.
!---------------------------------------------------------------------
!
            do k=N,bot_NZ,-1
              do iphy=1,Nphy
                if (PhyC(k,iphy).gt.0.0_r8) then
!
!  Calculate weighted average spectral absorption.
!
                  aPHYN_wa=0.0_r8
                  do ibnd=1,Nbnd
                    aPHYN_wa=aPHYN_wa+
     &                       (aPHYN_al(k,iphy,ibnd)*specir_scal(k,ibnd))
                  enddo
!
!  If bot_NZ < N+1, and E0_nz(k) =0.0, this will cause pigments to 
!  blow up. This should never happen, unless bot_NZ is not calcuated 
!  properly. WPB
!
                  aPHYN_wa=aPHYN_wa/E0_nz(k)
!
!  Calculate alph_ for HTAN function of P vs. I.
!  (conversion:  Ein/microEin * 10e3)
!
                  alph_(k,iphy)=(aPHYN_wa/PhyC(k,iphy))*qu_yld(iphy)*
     &                          1.0_e8-03
!
!  Light limited growth rate.
!
                  FV1=MAX(0.0_r8,E0_nz(k)-E0_comp(iphy))
                  FV2=E0_nz(k)-E0_inhib(iphy)
                  if (FV2.gt.0.0_r8) then
                    Gt_ll(k,iphy)=GtALG(k,iphy)*
     &                            TANH(alph_(k,iphy)*FV1/
     &                            GtALG(k,iphy))*
     &                            EXP(-inhib_fac(iphy)*FV2)
                  else
                    Gt_ll(k,iphy)=GtALG(k,iphy)*
     &                            TANH(alph_(k,iphy)*FV1/
     &                            GtALG(k,iphy))
                  endif
!
!  Nutrient limited growth rates.
!
!  REMEMBER that sinking speed to be set by gradient of limiting
!       nutrient, allowing for negative sinking. Try storing growth
!       rate terms in an array and using MAXLOC for if test.
!
!  Nitrogen limited growth rate.
!
                  if (PhyN(k,iphy).gt.0.0_r8) then
                    FV1=PhyC(k,iphy)/(PhyN(k,iphy)+PhyNP1(k,iphy))
                    Gt_nl(k,iphy)=mu_bar_n(k,iphy)*
     &                            (1.0_r8-ImaxC2nALG(iphy)*FV1)
                    Gt_nl(k,iphy)=MAX(0.0_r8,MIN(Gt_nl(k,iphy),
     &                                           GtALG(k,iphy)))
                  endif
!
!  Silica limited growth rate.
!  Testing for silica incorporation.
!
                  if ((HsSiO(iphy).lt.LARGE).and.
     &                (PhyS(k,iphy).gt.0.0_r8)) then
                    FV1=PhyC(k,iphy)/(PhyS(k,iphy)+PhySP1(k,iphy))
                    Gt_sl(k,iphy)=mu_bar_s(k,iphy)*
     &                            (1.0_r8-ImaxC2sALG(iphy)*FV1)
                    Gt_sl(k,iphy)=MAX(0.0_r8,
     &                                MIN(Gt_sl(k,iphy),GtALG(k,iphy)))
                  else
                    Gt_sl(k,iphy)=LARGE
                  endif
!
!  Phosphorus limited growth rate.
!
                  if ((HsPO4(iphy).lt.LARGE).and.
     &                (PhyP(k,iphy).gt.0.0_r8)) then
                    FV1=PhyC(k,iphy)/(PhyP(k,iphy)+PhyPP1(k,iphy))
                    Gt_pl(k,iphy)=mu_bar_p(k,iphy)*
     &                            (1.0_r8-ImaxC2pALG(iphy)*FV1)
                    Gt_pl(k,iphy)=MAX(0.0_r8,
     &                                MIN(Gt_pl(k,iphy),GtALG(k,iphy)))
                  else
                    Gt_pl(k,iphy)=LARGE
                  endif
!
!  Iron limited growth rate
!
                  if ((HsFe(iphy).lt.LARGE).and.
     &                (PhyF(k,iphy).gt.0.0_r8)) then
                    FV1=PhyC(k,iphy)/(PhyF(k,iphy)+PhyFP1(k,iphy))
                    Gt_fl(k,iphy)=mu_bar_f(k,iphy)*
     &                            (1.0_r8-ImaxC2fALG(iphy)*FV1)
                    Gt_fl(k,iphy)=MAX(0.0_r8,
     &                                MIN(Gt_fl(k,iphy),GtALG(k,iphy)))
                  else
                    Gt_fl(k,iphy)=LARGE
                  endif
!
!  Realized growth rate is minimum of light or nutrient limited rate.
!
                  GtALG_r(k,iphy)=MIN(Gt_ll(k,iphy),Gt_nl(k,iphy),
     &                                Gt_sl(k,iphy),Gt_pl(k,iphy),
     &                                Gt_fl(k,iphy))
                  if (GtALG_r(k,iphy).ge.LARGE) GtALG_r(k,iphy)=0.0_r8
!
!  Carbon growth calculations.
!
                  FV1=PhyC(k,iphy)*GtALG_r(k,iphy)
                  PhyCP1(k,iphy)=PhyCP1(k,iphy)+FV1
                  DICP1(k)=DICP1(k)-FV1
!!
!! Budgeting to be addressed later.
!!
!!                BGrwthPHY(iphy)=BGrwthPHY(iphy)+DZ(k)*FV1
!!                GPHY(k,iphy)=GPHY(k,iphy)*
!!   &                         (1.0_r8+GtALG_r(k,iphy)*dtbio)
!!
                endif
              enddo
            enddo
!
!---------------------------------------------------------------------
!  Bacterioplankton carbon growth terms.
!---------------------------------------------------------------------
!
            do k=1,N
              Het_BAC=0.0_r8
              RelDOC1=0.0_r8
              RelDON1=0.0_r8
              RelDOP1=0.0_r8
              RelFe  =0.0_r8
!
!  NOTE: Only DOC2/DON2 formation is in this section.
!        Take colored excretion off the top. 03/18/00
!        also, not excreting any DOP or Fe
!  REMEMBER, if excreting DOP and Fe, must address changes in growth if
!        tests. (see DON equations). 03/21/00.
!
              do ibac=1,Nbac
                FV1=NupDOC_ba(k,ibac)*ExBAC_c*
     &              (1.0_r8-cDOCfrac_c(irct))
                FV2=NupDOC_ba(k,ibac)*ExBAC_c*cDOCfrac_c(irct)
                FV3=NupDON_ba(k,ibac)*ExBAC_n
!
                DOMCP1(k,irct)=DOMCP1(k,irct)+FV1
                CDMCP1(k,irct)=CDMCP1(k,irct)+FV2
                DOMNP1(k,irct)=DOMNP1(k,irct)+FV3
!
!  As we are taking it off the top, must remove from DOMN1 now. No other
!  organisms use DOMC1, so net term (totDOC_d) can be used in budgeting
!  below. This saves cycles, but makes code difficult to read. WPB
!
                DOMNP1(k,ilab)=DOMNP1(k,ilab)-FV3
!
!  Remove from uptake.
!
                NupDOC_ba(k,ibac)=NupDOC_ba(k,ibac)-(FV1+FV2)
                NupDON_ba(k,ibac)=NupDON_ba(k,ibac)-FV3
!!
!!  Budgeting to be addressed later.
!!
!!              BExc_CBAC (2)=BExc_CBAC (2)+DZ(k)*FV1
!!              BExc_cCBAC(2)=BExc_cCBAC(2)+DZ(k)*FV2
!!              BExc_NBAC (2)=BExc_NBAC (2)+DZ(k)*FV3
!
!  Determine growth limitation. Assuming 100% efficiency for N, P, Fe.
!  If DOMC=0, or DOMN=0, or DOMP=0, then NupDOC_ba = NupDON_ba = 
!  NupDOP_ba = 0 and none of the divisions below are accessed. WPB
!
                Bac_G(1)=NupDOC_ba(k,ibac)*Bac_Ceff
                Bac_G(2)=(NupDON_ba(k,ibac)+NupNH4_ba(k,ibac))*
     &                   C2nBAC
                Bac_G(3)=(NupDOP_ba(k,ibac)+NupPO4_ba(k,ibac))*
     &                   C2pBAC
                Bac_G(4)=NupFe_ba(k,ibac)*C2FeBAC
!
!  Energy limited case. All excess nutrients returned in inorganic form.
!
                if ((Bac_G(1).le.Bac_G(2)).and.
     &              (Bac_G(1).le.Bac_G(3)).and.
     &              (Bac_G(1) .le. Bac_G(4))) then
                  Het_BAC=Bac_G(1)
                  FV1=Bac_G(1)*N2cBAC
                  FV2=Bac_G(1)*P2cBAC
                  FV3=Bac_G(1)*Fe2cBAC
                  BacNP1(k,ibac)=BacNP1(k,ibac)+FV1
                  BacPP1(k,ibac)=BacPP1(k,ibac)+FV2
                  BacFP1(k,ibac)=BacFP1(k,ibac)+FV3
!
!  Uptake arrays should probably now be negative. If NH4 or PO4 is
!  positive, then there is some uptake of inorganic forms, but this
!  value will be less than the original Nup value because of IF test.
!
                  NupNH4_ba(k,ibac)=FV1-NupDON_ba(k,ibac)
                  NupPO4_ba(k,ibac)=FV2-NupDOP_ba(k,ibac)
!
!  Because Fe is considered to be all inorganic, only net uptake of Fe
!  is needed.
!
                  RelFe=NupFe_ba(k,ibac)-FV3
                  NupFe_ba(k,ibac)=FV3
!
!  Nitrogen limited case. Excess nutrients returned in organic form
!  first, inorganic second.
!!            else if (Bac_G_T(1) .eq. 2) then
!
                elseif ((Bac_G(2).le.Bac_G(3)).and.
     &                  (Bac_G(2).le.Bac_G(4))) then
                  Het_BAC=Bac_G(2)
                  FV2=Bac_G(2)*P2cBAC
                  FV3=Bac_G(2)*Fe2cBAC
                  BacNP1(k,ibac)=BacNP1(k,ibac)+
     &                           (NupDON_ba(k,ibac)+NupNH4_ba(k,ibac))
                  BacPP1(k,ibac)=BacPP1(k,ibac)+FV2
                  BacFP1(k,ibac)=BacFP1(k,ibac)+FV3
!
!  Uptake arrays will now reflect release of inorganic and organic
!  revision of uptake.
!
                  FV1=(Bac_G(1)-Bac_G(2))*I_Bac_Ceff
                  NupDOC_ba(k,ibac)=NupDOC_ba(k,ibac)-FV1
                  RelDOC1=FV1
!
!  To get accurate DOP from C2pDOC, must add back excreted DOC.
!
                  FV4=(FV1*R_ExBAC_c)*DOC_frac(k)
!!   &               /C2pDOC(k,ibac)
     &               * DOMP(k,ilab)/DOMC(k,ilab)
!!                FV5=FV2-(NupDOP_ba(k,ibac)-FV4)
                  FV5=FV2-(NupDOP_ba(k,ibac)-FV4+NupPO4_ba(k,ibac))
!
!  If FV5 is positive then released DOP is required for bacteria growth.
!
!!                RelDOP1=FV4-MAX(0.0_r8,FV5)
!!                NupDOP_ba(k,ibac)=NupDOP_ba(k,ibac)-
!!   &                              MAX(0.0_r8,RelDOP1)
!!                if (RelDOP1.lt.0.0_r8) then 
!!                  NupPO4_ba(k,ibac)=-RelDOP1
!!                else
!!                  NupPO4_ba(k,ibac) = FV5
!!                endif 
!!
                  if (FV5.lt.0.0_r8) then
                    RelDOP1=FV4
                    NupPO4_ba(k,ibac)=NupPO4_ba(k,ibac)+FV5
                  else
                    RelDOP1=FV4-FV5
                  endif
                  NupDOP_ba(k,ibac)=NupDOP_ba(k,ibac)-RelDOP1
!
!  Release Fe.
!
                  RelFe=NupFe_ba(k,ibac)-FV3
                  NupFe_ba(k,ibac)=FV3
!
!  Phosphorous limited case. Excess nutrients returned in organic form
!  first, inorganic second.
!!            else if (Bac_G_T(1) .eq. 3) then
!
                elseif (Bac_G(3).le.Bac_G(4)) then
                  Het_BAC=Bac_G(3)
                  FV2=Bac_G(3)*N2cBAC
                  FV3=Bac_G(3)*Fe2cBAC
                  BacNP1(k,ibac)=BacNP1(k,ibac)+FV2
                  BacPP1(k,ibac)=BacPP1(k,ibac)+
     &                           (NupDOP_ba(k,ibac)+NupPO4_ba(k,ibac))
                  BacFP1(k,ibac)=BacFP1(k,ibac)+FV3
!
!  Uptake arrays will now reflect release of inorganic and organic
!  revision of uptake.
!
                  FV1=(Bac_G(1)-Bac_G(3))*I_Bac_Ceff
                  NupDOC_ba(k,ibac)=NupDOC_ba(k,ibac)-FV1
                  RelDOC1=FV1
!
!  To get accurate DON from C2nDOC, must add back excreted DOC.
!
                  FV4=(FV1*R_ExBAC_c)*DOC_frac(k)
!!   &               /C2nDOC(k,1)*Frac_ExBAC_n
     &               *(DOMN(k,ilab)/DOMC(k,ilab))*Frac_ExBAC_n
!!                FV5=FV2-(NupDON_ba(k,ibac)-FV4)
                  FV5=FV2-(NupDON_ba(k,ibac)-FV4+NupNH4_ba(k,ibac))
!
!  If FV5 is positive then released DON is required for bacteria growth.
!
!!                RelDON1=FV4-MAX(0.0_r8,FV5)
!!                NupDON_ba(k,ibac)=NupDON_ba(k,ibac)-
!!   &                              MAX(0.0_r8,RelDON1)
!!                if (RelDON1.lt.0.0_r8) then
!!                  NupNH4_ba(k,ibac)=-RelDON1
!!                else
!!                  NupNH4_ba(k,ibac)=FV5
!!                endif
!!
                  if(FV5.lt.0.0_r8) then
                    RelDON1=FV4
                    NupNH4_ba(k,ibac)=NupNH4_ba(k,ibac)+FV5
                  else
                    RelDON1=FV4-FV5
                  endif
                  NupDON_ba(k,ibac)=NupDON_ba(k,ibac)-RelDON1
!
!  Release Fe.
!
                  RelFe=NupFe_ba(k,ibac)-FV3
                  NupFe_ba(k,ibac)=FV3
!
!  Fe limited case. Excess nutrients returned in organic form
!  first, inorganic second.
!!            else if (Bac_G_T(1) .eq. 4) then
!
                else
                  Het_BAC=Bac_G(4)
                  FV2=Bac_G(4)*N2cBAC
                  FV3=Bac_G(4)*P2cBAC
                  BacNP1(k,ibac)=BacNP1(k,ibac)+FV2
                  BacPP1(k,ibac)=BacPP1(k,ibac)+FV3
                  BacFP1(k,ibac)=BacFP1(k,ibac)+NupFe_ba(k,ibac)
!
!  Uptake arrays will now reflect release of inorganic and organic
!  revision of uptake.
!
                  FV1=(Bac_G(1)-Bac_G(4))*I_Bac_Ceff
                  NupDOC_ba(k,ibac)=NupDOC_ba(k,ibac)-FV1
                  RelDOC1=FV1
!
!  To get accurate DON from C2nDOC, must add back excreted DOC.
!
                  FV4=(FV1*R_ExBAC_c)*DOC_frac(k)
!!   &               /C2nDOC(k,1)*Frac_ExBAC_n
     &               *DOMN(k,ilab)/DOMC(k,ilab)*Frac_ExBAC_n
!!                FV5=FV2-(NupDON_ba(k,ibac)-FV4)
                  FV5=FV2-(NupDON_ba(k,ibac)-FV4+NupNH4_ba(k,ibac))
!
!  If FV5 is positive then released DON is required for bacteria growth.
!
!!                RelDON1=FV4-MAX(0.d0,FV5)
!!                NupDON_ba(k,ibac)=NupDON_ba(k,ibac)-
!!   &                              MAX(0.0_r8,RelDON1)
!!                if (RelDON1.lt.0.0_r8) then
!!                  NupNH4_ba(k,ibac)=-RelDON1
!!                else
!!                  NupNH4_ba(k,ibac)=FV5
!!                endif
!!
                  if(FV5.lt.0.0_r8) then
                    RelDON1=FV4
                    NupNH4_ba(k,ibac)=NupNH4_ba(k,ibac)+FV5
                  else
                    RelDON1=FV4-FV5
                  endif
                  NupDON_ba(k,ibac)=NupDON_ba(k,ibac)-RelDON1
!
!  To get accurate DOP from C2pDOC, must add back excreted DOC.
!
                  FV4=(FV1*R_ExBAC_c)*DOC_frac(k)
!!   &               /C2pDOC(k,1)
     &               *DOMP(k,ilab)/DOMC(k,ilab)
!!                FV5=FV3-(NupDOP_ba(k,ibac)-FV4)
                  FV5=FV2-(NupDOP_ba(k,ibac)-FV4+NupPO4_ba(k,ibac))
!
!  If FV5 is positive then released DOP is required for bacteria growth.
!
!!                 RelDOP1=FV4-MAX(0.0_r8,FV5)
!!                 NupDOP_ba(k,ibac)=NupDOP_ba(k,ibac)-
!!   &                               MAX(0.0_r8,RelDOP1)
!!                 if (RelDOP1.lt.0.0_r8) then
!!                   NupPO4_ba(k,ibac)=-RelDOP1
!!                 else
!!                   NupPO4_ba(k,ibac)=FV5
!!                endif
!!
                  if (FV5.lt.0.0_r8) then
                    RelDOP1=FV4
                    NupPO4_ba(k,ibac)=NupPO4_ba(k,ibac)+FV5
                  else
                    RelDOP1=FV4-FV5
                  endif
                  NupDOP_ba(k,ibac)=NupDOP_ba(k,ibac)-RelDOP1
                endif
!
!  Increment nutrient arrays.
!
                BacCP1(k,ibac)=BacCP1(k,ibac)+Het_BAC
                FV1=NupDOC_ba(k,ibac)-Het_BAC
                DICP1(k)=DICP1(k)+FV1
!
!  NOTE: to be strictly accurate we should remove RelDOC1 from DOCNP1,
!       and then add it back, since NupDOC_ba is a net term. This should
!       wash out in the budgeting.
!
                DOMCP1(k,ilab)=DOMCP1(k,ilab)-
     &                         (totDOC_d(k)-RelDOC1)*DOC_frac(k)
                CDMCP1(k,ilab)=CDMCP1(k,ilab)-
     &                         (totDOC_d(k)-RelDOC1)*
     &                         (1.0_r8-DOC_frac(k))
!
!  This is inclusive of RelDOX1, excretion of DON1 removed above.
!
                DOMNP1(k,ilab)=DOMNP1(k,ilab)-NupDON_ba(k,ibac)
                DOMPP1(k,ilab)=DOMPP1(k,ilab)-NupDOP_ba(k,ibac)
                NH4P1(k)=NH4P1(k)-NupNH4_ba(k,ibac)
                PO4P1(k)=PO4P1(k)-NupPO4_ba(k,ibac)
                FeOP1(k)=FeOP1(k)-NupFe_ba (k,ibac)
!!
!!  Budgeting to be addressed later.
!!
!!              BExc_CBAC (3)=BExc_CBAC (3)+DZ(k)*RelDOC1*DOC_frac(k)
!!              BExc_cCBAC(3)=BExc_cCBAC(3)+
!!   &                        DZ(k)*RelDOC1*(1.0_r8-DOC_frac(k))
!!              BExc_NBAC(3)=BExc_NBAC(3)+DZ(k)*MAX(0.0_r8,RelDON1)
!!              BExc_PBAC(3)=BExc_PBAC(3)+DZ(k)*MAX(0.0_r8,RelDOP1)
!!              BResprBAC(1)=BResprBAC(1)+DZ(k)*FV1
!!              BGrwthBAC(1)=BgrwthBAC(1)+DZ(k)*totDOC_d(k)
!!              BGrwthDOC(1)=BGrwthDOC(1)+DZ(k)*totDOC_d(k)*DOC_frac(k)
!!              BGrwthDOC(2)=BGrwthDOC(2)+
!!   &                       DZ(k)*totDOC_d(k)*(1.0_r8-DOC_frac(k))
!!              BGrwthDOC(3)=BGrwthDOC(3)+
!!                           DZ(k)*(totDON_d(k)-SUM(NupDON(k,:)))
!!
!! This only works if DOP is not excreted as DOP2. Problems arise
!! because totDOP_d includes phtyoplankton uptake, whereas totDON_d and
!! totDOC_d are only bacterial uptake.
!!
!!              BGrwthDOC(4)=BGrwthDOC(4)+
!!                           DZ(k)*(NupDOP_ba(k,1)+MAX(0.0_r8,RelDOP1))
!!
!!  MAX statement is used to select appropriate budgeting variable for
!!  inorganic uptake. If positive, net uptake. If negative, net
!!  respiration. This could be simplified, but is carried to conform
!!  to es2.r4b (and prior) budgeting protocols.
!!
!!              BNupBAC(3)=BNupBAC(3)+DZ(k)*MAX(0.0_r8,NupNH4_ba(k,1))
!!              BNupBAC(4)=BNupBAC(4)+DZ(k)*MAX(0.0_r8,NupPO4_ba(k,1))
!!              BNupBAC(5)=BNupBAC(5)+DZ(k)*NupFe_ba(k,1)+RelFe
!!
!!  MIN statement is used to select appropriate budgeting variable for
!!  inorganic respiration. If positive, net uptake. If negative, net
!!  respiration. This could be simplified, but is carried to conform
!!  to es2.r4b (and prior) budgeting protocols.
!!
!!              BResprBAC(3)=BResprBAC(3)+
!!                           DZ(k)*MIN(0.0_r8,NupNH4_ba(k,1))
!!              BResprBAC(4)=BResprBAC(4)+
!!                           DZ(k)*MIN(0.0_r8,NupPO4_ba(k,1))
!!              BResprBAC(5)=BResprBAC(5)+DZ(k)*(-RelFe)
!!
!!              GBAC(k,1)=GBAC(k,1)*
!!                        (1.0_r8+(BACNP1(k,1)/BACN(k,1))*dtbio)
!!
              enddo
            enddo
!
!---------------------------------------------------------------------
!  Phytoplankton Losses.
!---------------------------------------------------------------------
!
            do iphy=1,Nphy
              do k=1,N
!
!  Excretion.
!
                if (C2nALG(k,iphy).ge.C2nALGminABS(iphy)) then
                  FV1=PhyC(k,iphy)*ExALG(iphy)
                  PhyCP1(k,iphy)=PhyCP1(k,iphy)-FV1
!
!  No excretion of CDOC.
!
                  DOMCP1(k,ilab)=DOMCP1(k,ilab)+FV1
!!
!! Budgeting to be addressed later.
!!
!!                BExcrtPHY(iphy)=BExcrtPHY(iphy)+DZ(k)*FV1
                endif
!
!  Grazing.
!
                if (PhyC(k,iphy).gt.refuge(k,iphy)) then
!
!  Carbon calculations.
!
                  FV1=graz_act(k,iphy)*PhyC(k,iphy)
                  PhyCP1(k,iphy)=PhyCP1(k,iphy)-FV1
                  FecCP1(k,isfc)=FecCP1(k,isfc)+FecPEL(iphy,isfc)*FV1
                  FecCP1(k,iffc)=FecCP1(k,iffc)+FecPEL(iphy,iffc)*FV1
                  FV3=FecDOC(iphy)*FV1
                  DOMCP1(k,ilab)=DOMCP1(k,ilab)+
     &                           FV3*(1.0_r8-cDOCfrac_c(ilab))
                  CDMCP1(k,ilab)=CDMCP1(k,ilab)+
     &                           FV3*cDOCfrac_c(ilab)
                  DICP1(k)=DICP1(k)+FecCYC(iphy)*FV1
!!
!!  Budgeting to be addressed later.
!!
!!                BFecalPHY(iphy)=BFecalPHY(iphy)+DZ(k)*FV1
!!                BGrazeDOC(iphy)=BGrazeDOC(iphy)+
!!                                DZ(k)*FV3*(1.0_r8-cDOCfrac_c(1))
!!                BGrazeCDOC(iphy)=BGrazeCDOC(iphy)+
!!                                 DZ(k)*FV3*cDOCfrac_c(1)
!
!  Nitrogen calculations.
!
                  FV2=graz_act(k,iphy)*PhyN(k,iphy)
                  PhyNP1(k,iphy)=PhyNP1(k,iphy)-FV2
                  FecNP1(k,isfc)=FecNP1(k,isfc)+FecPEL(iphy,isfc)*FV2
                  FecNP1(k,iffc)=FecNP1(k,iffc)+FecPEL(iphy,iffc)*FV2
                  DOMNP1(k,ilab)=DOMNP1(k,ilab)+FecDOC(iphy)*FV2
                  NH4P1(k)=NH4P1(k)+FecCYC(iphy)*FV2
!!
!!  Budgeting to be addressed later.
!!
!!                BFecalPN(iphy)=BFecalPN(iphy)+DZ(k)*FV2
!!                BGrazeDON(iphy)=BGrazeDON(iphy)+
!!                                DZ(k)*FecDOC(iphy)*FV2
!!                BFecalNH4(iphy)=BFecalNH4(iphy)+
!!                                DZ(k)*FecCYC(iphy)*FV2
!
!  Silica calculations.
!
                  if (iPhyS(iphy).gt.0.0_r8) then
                    FV2=graz_act(k,iphy)*PhyS(k,iphy)
                    PhySP1(k,iphy)=PhySP1(k,iphy)-FV2
!
!  Assuming that the fraction of material lost via sloppy feeding/cell
!  lysis also results in silica tests being put into FecS pool.
!
                    FecSP1(k,isfc)=FecSP1(k,isfc)+
     &                             FecDOC(iphy)*FV2
                    FecSP1(k,iffc)=FecSP1(k,iffc)+
     &                             (1.0_r8-FecDOC(iphy))*FV2
!!
!!  Budgeting to be addressed later.
!!
!!                  BFecalPS(phy)=BFecalPS(iphy)+DZ(k)*FV2
!!
                  endif
!
!  Phosphorus calculations.
!
                  FV2=graz_act(k,iphy)*PhyP(k,iphy)
                  PhyPP1(k,iphy)=PhyPP1(k,iphy)-FV2
                  FecPP1(k,isfc)=FecPP1(k,isfc)+FecPEL(iphy,isfc)*FV2
                  FecPP1(k,iffc)=FecPP1(k,iffc)+FecPEL(iphy,iffc)*FV2
                  DOMPP1(k,ilab)=DOMPP1(k,ilab)+FecDOC(iphy)*FV2
                  PO4P1(k)=PO4P1(k)+FecCYC(iphy)*FV2
!!
!!  Budgeting to be addressed later.
!!
!!                BFecalPP(iphy)=BFecalPP(iphy)+DZ(k)*FV2
!!                BGrazeDOP(iphy)=BGrazeDOP(iphy)+
!!                                DZ(k)*FecDOC(iphy)*FV2
!!                BFecalPO4(iphy)=BFecalPO4(iphy)+
!!                                DZ(k)*FecCYC(iphy)*FV2
!
!  Iron calculations. Assuming no DOMF.
!
                  FV2=graz_act(k,iphy)*PhyF(k,iphy)
                  PhyFP1(k,iphy)=PhyFP1(k,iphy)-FV2
                  FecFP1(k,isfc)=FecFP1(k,isfc)+FecPEL(iphy,isfc)*FV2
                  FecFP1(k,iffc)=FecFP1(k,iffc)+FecPEL(iphy,iffc)*FV2
                  FeOP1(k)=FeOP1(k)+
     &                     (FecCYC(iphy)+FecDOC(iphy))*FV2
!!
!!  Budgeting to be addressed later.
!!
!!                BFecalPF(iphy)=BFecalPF(iphy)+DZ(k)*FV2
!!                BFecalFe(iphy)=BFecalFe(iphy)+
!!                               DZ(k)*(FecCYC(iphy)+FecDOC(iphy)*FV2)
!!
                endif
              enddo
            enddo
!
!---------------------------------------------------------------------
!  Bacterial losses.
!---------------------------------------------------------------------
!
!  NOTE: Bacterial growth is completely reminerialized
!
            do ibac=1,Nbac
              do k=1,N
!
!  Grazing calculation.
!
                FV1=BacCYC*BacCP1(k,ibac)
                FV2=BacPEL*BacCP1(k,ibac)
                FV3=BacDOC*BacCP1(k,ibac)
                FV4=FV1+FV2+FV3
!
!  Carbon calculations.
!
                BacCP1(k,ibac)=BacCP1(k,ibac)-FV4
                FecCP1(k,iffc)=FecCP1(k,iffc)+FV2
                DOMCP1(k,ilab)=DOMCP1(k,ilab)+
     &                         FV3*(1.0_r8-cDOCfrac_c(ilab))
                CDMCP1(k,ilab)=CDMCP1(k,ilab)+FV3*cDOCfrac_c(ilab)
                DICP1(k)=DICP1(k)+FV1
!!
!!  Budgeting to be addressed later.
!!
!!              BCycBAC(1)=BCycBAC(1)+DZ(k)*FV1
!!              BFecalBAC(1)=BFecalBAC(1)+DZ(k)*FV2
!!              BExc_CBAC(1)=BExc_CBAC(1)+
!!                           DZ(k)*FV3*(1.0_r8-cDOCfrac_c(1))
!!              BExc_cCBAC(1)=BExc_cCBAC(1)+
!!                            DZ(k)*FV3*cDOCfrac_c(1)
!
!  Nitrogen calculations.
!
                BacNP1(k,ibac)=BacNP1(k,ibac)-FV4*N2cBAC
                FecNP1(k,iffc)=FecNP1(k,iffc)+FV2*N2cBAC
                DOMNP1(k,ilab)=DOMNP1(k,ilab)+FV3*N2cBAC
                NH4P1(k)=NH4P1(k)+FV1*N2cBAC
!!
!!  Budgeting to be addressed later.
!!
!!              BCycBAC(2)=BCycBAC(2)+DZ(k)*FV1*N2cBAC
!!              BExc_NBAC(1)=BExc_NBAC(1)+DZ(k)*FV3*N2cBAC
!!              BFecalBAC(2)=BFecalBAC(2)+DZ(k)*FV2*N2cBAC
!
!  Phosphorous calculations.
!
                BacPP1(k,ibac)=BacPP1(k,ibac)-FV4*P2cBAC
                FecPP1(k,iffc)=FecPP1(k,iffc)+FV2*P2cBAC
                DOMPP1(k,ilab)=DOMPP1(k,ilab)+FV3*P2cBAC
                PO4P1(k)=PO4P1(k)+FV1*P2cBAC
!!
!!  Budgeting to be addressed later.
!!
!!              BCycBAC(3)=BCycBAC(3)+DZ(k)*FV1*P2cBAC
!!              BExc_PBAC(1)=BExc_PBAC(1)+DZ(k)*FV3*P2cBAC
!!              BFecalBAC(3)=BFecalBAC(3)+DZ(k)*FV2*P2cBAC
!
!  Iron calculations.
!
                BacFP1(k,ibac)=BacFP1(k,ibac)-FV4*Fe2cBAC
                FecFP1(k,iffc)=FecFP1(k,iffc)+FV2*Fe2cBAC
                FeOP1(k)=FeOP1(k)+(FV1+FV3)*Fe2cBAC
!!
!!  Budgeting to be addressed later.
!!
!!              BCycBAC(4)=BCycBAC(4)+DZ(k)*(FV1+FV3)*Fe2cBAC
!!              BFecalBAC(4)=BFecalBAC(4)+DZ(k)*FV2*Fe2cBAC
!!
              enddo
            enddo
!
!---------------------------------------------------------------------
!  Fecal pellet remineralization.
!---------------------------------------------------------------------
!
            do ifec=1,Nfec
              do k=1,N
!
!  Carbon calculations.  All carbon goes to CO2.
!
                FV3=Regen_C(k,ifec)*FecC(k,ifec)
                FecCP1(k,ifec)=FecCP1(k,ifec)-FV3
                DICP1(k)=DICP1(k)+FV3
!!
!!  Budgeting to be addressed later.
!!
!!              BFecREM_c(ifec)=BFecREM_c(ifec)+DZ(k)*FV3
!
!  Nitrogen calculations.  Nitrogen goes to NH4.
!
                FV2=Regen_N(k,ifec)*FecN(k,ifec)
                FecNP1(k,ifec)=FecNP1(k,ifec)-FV2
                NH4P1(k)=NH4P1(k)+FV2
!!
!!  Budgeting to be addressed later.
!!
!!              BFecREM_n(ifec)=BFecREM_n(ifec)+DZ(k)*FV2
!
!  Silica calculations.
!
                FV2=Regen_S(k,ifec)*FecS(k,ifec)
                FecSP1(k,ifec)=FecSP1(k,ifec)-FV2
                SiOP1(k)=SiOP1(k)+FV2
!!
!!  Budgeting to be addressed later.
!!
!!              BFecREM_s(ifec)=BFecREM_s(ifec)+DZ(k)*FV2
!
!  Phosphorous calculations.
!
                FV2=Regen_P(k,ifec)*FecP(k,ifec)
                FecPP1(k,ifec)=FecPP1(k,ifec)-FV2
                PO4P1(k)=PO4P1(k)+FV2
!!
!!  Budgeting to be addressed later.
!!
!!              BFecREM_p(ifec)=BFecREM_p(ifec)+DZ(k)*FV2
!
!  Iron calculations.
!
                FV2=Regen_F(k,ifec)*FecF(k,ifec)
                FecFP1(k,ifec)=FecFP1(k,ifec)-FV2
                FeOP1(k)=FeOP1(k)+FV2
!!
!!  Budgeting to be addressed later.
!!
!!              BFecREM_f(ifec)=BFecREM_f(ifec)+DZ(k)*FV2
!!
              enddo
            enddo
!
!---------------------------------------------------------------------
!  CDMC photolysis calculations.
!---------------------------------------------------------------------
!
!  If Ed_nz(N) > zero, then there is sunlight.
!
            FV1=RtUVR_DIC*Ed_nz(N)/1500.0_r8
            FV2=RtUVR_DOC*Ed_nz(N)/1500.0_r8

            if (FV1.gt.0.0_r8) then
!
!  FV4 equals the CDMC1 absorption at 410 nm. 0.012 converts to g m-3.
!  FV5 equals the CDMC2 absorption at 410 nm.
!  Weighted average attenuation of UVB at 300 nm = 0.2 m-1.
!  REMEMBER - (DZ + 0.5)?
!
              FV4=CDMC(N,ilab)*0.012*aDOC410(ilab)
              FV5=CDMC(N,irct)*0.012*aDOC410(irct)
              photo_decay=(DZ(N)*0.5_r8)*
     &                    (0.2_r8+(FV4+FV5)*aDOC300(ilab))
              FV3=EXP(-photo_decay)
              photo_decay=2.0_r8*photo_decay
!
!  Do not photolyze below the euphotic zone.
!
              do k=N,bot_NZ,-1
                if (FV3.gt.0.01_r8) then
                  FV6=FV5+FV4
                  if (FV6.gt.0.0_r8) then
                    FV7=FV4/FV6
                    photo_DIC=FV3*FV1*FV6
                    photo_DOC=FV3*FV2*FV6
                    total_photo=photo_DIC+photo_DOC
!
!  NOTE: not testing for excess photolysis (CDOC going negative).
!
                    FV4=(1.0_r8-FV7)*total_photo
                    CDMCP1(k,irct)=CDMCP1(k,irct)-FV4
                    DOMCP1(k,ilab)=DOMCP1(k,ilab)+photo_DOC
                    CDMCP1(k,ilab)=CDMCP1(k,ilab)-FV7*total_photo
                    DICP1(k)=DICP1(k)+photo_DIC
!!
!!  Budgeting to be addressed later.
!!
!!                  BDecayDOC(1)=BDecayDOC(1)+DZ(k)*FV7*photo_DIC
!!                  BDecayDOC(2)=BDecayDOC(2)+DZ(k)*FV7*photo_DOC
!!                  BDecayDOC(3)=BDecayDOC(3)+DZ(k)*(1.0_r8-FV7)*
!!   &                           photo_DIC
!!                  BDecayDOC(4)=BDecayDOC(4)+DZ(k)*(1.0_r8-FV7)*
!!   &                           photo_DOC
!!
                  endif
!
!  FV4 equals the CDMC1 absorption at 410 nm. 0.012 converts to g m-3.
!  FV5 equals the CDMC2 absorption at 410 nm.
!  Weighted average attenuation of UVB at 300 nm = 0.2 m-1.
!
                  FV4=CDMC(k,ilab)*0.012*aDOC410(ilab)
                  FV5=CDMC(k,irct)*0.012*aDOC410(irct)
                  FV1=photo_decay+
     &                0.5_r8*DZ(k)*(0.2_r8+(FV4+FV5)*aDOC300(ilab))
!
!  If k is greater than the bottom of the euphotic zone (and by
!  by extension the bottom boundary) or the decay constant is
!  greater than 4.61 (or < 1% photolysis zone) then exit do loop.
!
                  FV3=EXP(-FV1)
!
!  Store value for passage through entire DZ(k).
!
                  photo_decay=photo_decay+2.0_r8*FV1
                endif
              enddo
            endif
!
!  Update pigment ratios due to light and nutrients.
!  NOTE: 12 factor to convert to ugrams (mg m-3)
!
            do iphy=1,Nphy
              do k=1,N
                if (PhyC(k,iphy).gt.0.0_r8) then
                  C2CHL(k,iphy)=Pigs(k,iphy,ichl)/
     &                          (PhyC(k,iphy)*12.0_r8)
                else
                  C2CHL(k,iphy)=0.0_r8
                endif
              enddo
            enddo
!
!  Carbon to chlorophyll a ratio
!  This statement says that nutrient limitation of C2CHL ratio overides
!  light adaptation. Minimum of two functions may be more ecologically
!  accurate?
!
            if (bot_NZ.le.N) then
              do iphy=1,Nphy
                do k=N,bot_NZ,-1
                  if (b_C2Cn(iphy).lt.0.0_r8+SMALL) then
                    C2CHL_w(k,iphy)=MIN((b_C2Cl(iphy)+
     &                                   mxC2Cl(iphy)*E0_nz(k)),
     &                                  C2CHL_max(iphy))
                  elseif (C2nALG(k,iphy).gt.
     &                    minC2nALG(iphy)+SMALL) then
                    C2CHL_w(k,iphy)=b_C2Cn(iphy)+mxC2Cn(iphy)*
     &                              (C2nALG(k,iphy)-minC2nALG(iphy))
                  else
                    C2CHL_w(k,iphy)=MIN((b_C2Cl(iphy)+
     &                                   mxC2Cl(iphy)*E0_nz(k)),
     &                                  C2CHL_max(iphy))
                  endif
                enddo
              enddo
!
!---------------------------------------------------------------------
!  Create optimal pigment ratios.
!---------------------------------------------------------------------
!
              do iphy=1,Nphy
!
!  Chlorophyll a concentation per species. form g CHL a / g C
!
                do k=N,bot_NZ,-1
                  Pigs_w(k,iphy,ichl)=1.0_r8/C2CHL_w(k,iphy)
                enddo
!
!  Chlorophyll b concentration per species. form g CHL b / g C
!
                if (iPigs(iphy,2).gt.0) then
                  do k=N,bot_NZ,-1
                    Pigs_w(k,iphy,2)=b_ChlB(iphy)+
     &                               mxChlB(iphy)*(C2CHL_w(k,iphy)-
     &                                             b_C2Cl(iphy))
                    Pigs_w(k,iphy,2)=Pigs_w(k,iphy,2)*
     &                               Pigs_w(k,iphy,ichl)
                  enddo
                endif
!
!  Chlorophyll c concentration per species. form g CHL c / g C
!
                if (iPigs(iphy,3).gt.0) then
                  do k=N,bot_NZ,-1
                    Pigs_w(k,iphy,3)=b_ChlC(iphy)+
     &                               mxChlC(iphy)*(C2CHL_w(k,iphy)-
     &                                             b_C2Cl(iphy))
                    Pigs_w(k,iphy,3)=Pigs_w(k,iphy,3)*
     &                               Pigs_w(k,iphy,ichl)
                  enddo
                endif
!
!  Photosynthetic caroteniods per species. form g PSC / g C
!
                if (iPigs(iphy,4).gt.0) then
                  do k=N,bot_NZ,-1
                    Pigs_w(k,iphy,4)=b_PSC(iphy)+
     &                               mxPSC(iphy)*(C2CHL_w(k,iphy)-
     &                                            b_C2Cl(iphy))
                    Pigs_w(k,iphy,4)=Pigs_w(k,iphy,4)*
     &                               Pigs_w(k,iphy,ichl)
                  enddo
                endif
!
!  Photoprotective caroteniods per species. form g PPC / g C
!
                if (iPigs(iphy,5).gt.0) then
                  do k=N,bot_NZ,-1
                    Pigs_w(k,iphy,5)=b_PPC(iphy)+
     &                               mxPPC(iphy)*(C2CHL_w(k,iphy)-
     &                                            b_C2Cl(iphy))
                    Pigs_w(k,iphy,5)=Pigs_w(k,iphy,5)*
     &                               Pigs_w(k,iphy,ichl)
                  enddo
                endif
!
!  Low Urobilin Phycoeurythin concentration per species. g LPUB / g C
!
                if (iPigs(iphy,6).gt.0) then
                  do k=N,bot_NZ,-1
                    Pigs_w(k,iphy,6)=b_LPUb(iphy)+
     &                               mxLPUb(iphy)*(C2CHL_w(k,iphy)-
     &                                             b_C2Cl(iphy))
                    Pigs_w(k,iphy,6)=Pigs_w(k,iphy,6)*
     &                               Pigs_w(k,iphy,ichl)
                  enddo
                endif
!
!  High Urobilin Phycoeurythin concentration per species. g HPUB / g C
!
                if (iPigs(iphy,7).gt.0) then
                  do k=N,bot_NZ,-1
                    Pigs_w(k,iphy,7)=b_HPUb(iphy)+
     &                               mxHPUb(iphy)*(C2CHL_w(k,iphy)-
     &                                             b_C2Cl(iphy))
                    Pigs_w(k,iphy,7)=Pigs_w(k,iphy,7)*
     &                               Pigs_w(k,iphy,ichl)
                  enddo
                endif
              enddo
!
!  Calculate pigment ratio changes.
!  NOTE: 12 factor to convert to ugrams (mg m-3)
!  NOTE: dtbio is included in PigsP1 calculation and is not required
!        in the update of the tendency array section below.
!
              do ipig=1,Npig
                do iphy=1,Nphy
                  if (iPigs(iphy,ipig).gt.0) then
                    do k=N,bot_NZ,-1
                      if ((PhyC(k,iphy).gt.0.0_r8).and.
     &                    (Pigs(k,iphy,ipig).gt.0.0_r8)) then
                        FV1=PhyC(k,iphy)*12.0_r8
!!                      Pigs_w(k,iphy,ipig)=1.0_r8/Pigs_w(k,iphy,ipig)
!!                      PigsP1(k,iphy,ipig)=FV1/Pigs(k,iphy,ichl)
!!                      PigsP1(k,iphy,ipig)=GtALG_r(k,iphy)*
!!   &                                      dtbio*Pigs_w(k,iphy,ipig)+
!!   &                                      PigsP1(k,iphy,ipig)*
!!   &                                      (1.0_r8-
!!   &                                       GtALG_r(k,iphy)*dtbio)
!!                      PigsP1(k,iphy,ipig)=FV1*(1.0_r8/
!!   &                                           PigsP1(k,iphy,ipig))
                        PigsP1(k,iphy,ipig)=FV1/
     &                                     (GtALG_r(k,iphy)*dtbio/
     &                                      Pigs_w(k,iphy,ipig)+
     &                                      FV1*(1.0_r8-
     &                                           GtALG_r(k,iphy)*dtbio)/
     &                                      Pigs(k,iphy,ipig))
                      else
                        PigsP1(k,iphy,ipig)=0.0_r8
                      endif
                    enddo
                  endif
                enddo
              enddo
            endif
!
!---------------------------------------------------------------------
!  Update the tendency arrays
!---------------------------------------------------------------------
!
            do k=1,N
             DIC(k)=DIC(k)+dtbio*DICP1(k)
             FeO(k)=FeO(k)+dtbio*FeOP1(k)
             NH4(k)=NH4(k)+dtbio*NH4P1(k)
             NO3(k)=NO3(k)+dtbio*NO3P1(k)
             PO4(k)=PO4(k)+dtbio*PO4P1(k)
             SiO(k)=SiO(k)+dtbio*SiOP1(k)
             DICP1(k)=0.0_r8
             FeOP1(k)=0.0_r8
             NH4P1(k)=0.0_r8
             NO3P1(k)=0.0_r8
             PO4P1(k)=0.0_r8
             SiOP1(k)=0.0_r8
            enddo
!
            do ibac=1,Nbac
              do k=1,N
                BacC(k,ibac)=BacC(k,ibac)+dtbio*BacCP1(k,ibac)
                BacN(k,ibac)=BacN(k,ibac)+dtbio*BacNP1(k,ibac)
                BacP(k,ibac)=BacP(k,ibac)+dtbio*BacPP1(k,ibac)
                BacF(k,ibac)=BacF(k,ibac)+dtbio*BacFP1(k,ibac)
                BacCP1(k,ibac)=0.0_r8
                BacNP1(k,ibac)=0.0_r8
                BacPP1(k,ibac)=0.0_r8
                BacFP1(k,ibac)=0.0_r8
              enddo
            enddo
!
            do idom=1,Ndom
              do k=1,N
                CDMC(k,idom)=CDMC(k,idom)+dtbio*CDMCP1(k,idom)
                DOMC(k,idom)=DOMC(k,idom)+dtbio*DOMCP1(k,idom)
                DOMN(k,idom)=DOMN(k,idom)+dtbio*DOMNP1(k,idom)
                DOMP(k,idom)=DOMP(k,idom)+dtbio*DOMPP1(k,idom)
                CDMCP1(k,idom)=0.0_r8
                DOMCP1(k,idom)=0.0_r8
                DOMNP1(k,idom)=0.0_r8
                DOMPP1(k,idom)=0.0_r8
              enddo
            enddo
!
            do ifec=1,Nfec
              do k=1,N
                FecC(k,ifec)=FecC(k,ifec)+dtbio*FecCP1(k,ifec)
                FecN(k,ifec)=FecN(k,ifec)+dtbio*FecNP1(k,ifec)
                FecP(k,ifec)=FecP(k,ifec)+dtbio*FecPP1(k,ifec)
                FecS(k,ifec)=FecS(k,ifec)+dtbio*FecSP1(k,ifec)
                FecF(k,ifec)=FecF(k,ifec)+dtbio*FecFP1(k,ifec)
                FecCP1(k,ifec)=0.0_r8
                FecNP1(k,ifec)=0.0_r8
                FecPP1(k,ifec)=0.0_r8
                FecSP1(k,ifec)=0.0_r8
                FecFP1(k,ifec)=0.0_r8
              enddo
            enddo
!
!  NOTE: dtbio is excluded in PigsP1 update of the tendency arrays.
!        It is include in the PigsP1 calculation above.
!
            do iphy=1,Nphy
              do k=1,N
                PhyC(k,iphy)=PhyC(k,iphy)+dtbio*PhyCP1(k,iphy)
                PhyF(k,iphy)=PhyF(k,iphy)+dtbio*PhyFP1(k,iphy)
                PhyN(k,iphy)=PhyN(k,iphy)+dtbio*PhyNP1(k,iphy)
                PhyP(k,iphy)=PhyP(k,iphy)+dtbio*PhyPP1(k,iphy)
                PhyCP1(k,iphy)=0.0_r8
                PhyFP1(k,iphy)=0.0_r8
                PhyNP1(k,iphy)=0.0_r8
                PhyPP1(k,iphy)=0.0_r8
                if (iPhyS(iphy).gt.0) then
                  PhyS(k,iphy)=PhyS(k,iphy)+dtbio*PhySP1(k,iphy)
                  PhySP1(k,iphy)=0.0_r8
                endif
                do ipig=1,Npig
                  if (iPigs(iphy,ipig).gt.0) then
                    Pigs(k,iphy,ipig)=PigsP1(k,iphy,ipig)
!!                  Pigs(k,iphy,ipig)=Pigs(k,iphy,ipig)+
!!   &                                PigsP1(k,iphy,ipig)
                    PigsP1(k,iphy,ipig)=0.0_r8
                  endif
                enddo
              enddo
            enddo
!
!  End iterations.
!
          enddo
!
!---------------------------------------------------------------------
!  Update global tracer variables.
!---------------------------------------------------------------------
!
          do k=1,N
            t(i,j,k,nnew,iDIC_)=MAX((t(i,j,k,nnew,iDIC_)+
     &                          DIC(k)-DIC_bak(k)),0.0_r8)
            t(i,j,k,nnew,iFeO_)=MAX((t(i,j,k,nnew,iFeO_)+
     &                          FeO(k)-FeO_bak(k)),0.0_r8)
            t(i,j,k,nnew,iNH4_)=MAX((t(i,j,k,nnew,iNH4_)+
     &                          NH4(k)-NH4_bak(k)),0.0_r8)
            t(i,j,k,nnew,iNO3_)=MAX((t(i,j,k,nnew,iNO3_)+
     &                          NO3(k)-NO3_bak(k)),0.0_r8)
            t(i,j,k,nnew,iPO4_)=MAX((t(i,j,k,nnew,iPO4_)+
     &                          PO4(k)-PO4_bak(k)),0.0_r8)
            t(i,j,k,nnew,iSiO_)=MAX((t(i,j,k,nnew,iSiO_)+
     &                          SiO(k)-SiO_bak(k)),0.0_r8)
            do ibac=1,Nbac
              t(i,j,k,nnew,iBacC(ibac))=MAX((t(i,j,k,nnew,iBacC(ibac))+
     &                        BacC(k,ibac)-BacC_bak(k,ibac)),0.0_r8)
              t(i,j,k,nnew,iBacN(ibac))=MAX((t(i,j,k,nnew,iBacN(ibac))+
     &                        BacN(k,ibac)-BacN_bak(k,ibac)),0.0_r8)
              t(i,j,k,nnew,iBacP(ibac))=MAX((t(i,j,k,nnew,iBacP(ibac))+
     &                        BacP(k,ibac)-BacP_bak(k,ibac)),0.0_r8)
              t(i,j,k,nnew,iBacF(ibac))=MAX((t(i,j,k,nnew,iBacF(ibac))+
     &                        BacF(k,ibac)-BacF_bak(k,ibac)),0.0_r8)
            enddo
            do idom=1,Ndom
              t(i,j,k,nnew,iCDMC(idom))=MAX((t(i,j,k,nnew,iCDMC(idom))+
     &                        CDMC(k,idom)-CDMC_bak(k,idom)),0.0_r8)
              t(i,j,k,nnew,iDOMC(idom))=MAX((t(i,j,k,nnew,iDOMC(idom))+
     &                        DOMC(k,idom)-DOMC_bak(k,idom)),0.0_r8)
              t(i,j,k,nnew,iDOMN(idom))=MAX((t(i,j,k,nnew,iDOMN(idom))+
     &                        DOMN(k,idom)-DOMN_bak(k,idom)),0.0_r8)
              t(i,j,k,nnew,iDOMP(idom))=MAX((t(i,j,k,nnew,iDOMP(idom))+
     &                        DOMP(k,idom)-DOMP_bak(k,idom)),0.0_r8)
            enddo
            do ifec=1,Nfec
              t(i,j,k,nnew,iFecC(ifec))=MAX((t(i,j,k,nnew,iFecC(ifec))+
     &                        FecC(k,ifec)-FecC_bak(k,ifec)),0.0_r8)
              t(i,j,k,nnew,iFecF(ifec))=MAX((t(i,j,k,nnew,iFecF(ifec))+
     &                        FecF(k,ifec)-FecF_bak(k,ifec)),0.0_r8)
              t(i,j,k,nnew,iFecN(ifec))=MAX((t(i,j,k,nnew,iFecN(ifec))+
     &                        FecN(k,ifec)-FecN_bak(k,ifec)),0.0_r8)
              t(i,j,k,nnew,iFecP(ifec))=MAX((t(i,j,k,nnew,iFecP(ifec))+
     &                        FecP(k,ifec)-FecP_bak(k,ifec)),0.0_r8)
              t(i,j,k,nnew,iFecS(ifec))=MAX((t(i,j,k,nnew,iFecS(ifec))+
     &                        FecS(k,ifec)-FecS_bak(k,ifec)),0.0_r8)
            enddo
            do iphy=1,Nphy
              t(i,j,k,nnew,iPhyC(iphy))=MAX((t(i,j,k,nnew,iPhyC(iphy))+
     &                        PhyC(k,iphy)-PhyC_bak(k,iphy)),0.0_r8)
              t(i,j,k,nnew,iPhyN(iphy))=MAX((t(i,j,k,nnew,iPhyN(iphy))+
     &                        PhyN(k,iphy)-PhyN_bak(k,iphy)),0.0_r8)
              t(i,j,k,nnew,iPhyP(iphy))=MAX((t(i,j,k,nnew,iPhyP(iphy))+
     &                        PhyP(k,iphy)-PhyP_bak(k,iphy)),0.0_r8)
              t(i,j,k,nnew,iPhyF(iphy))=MAX((t(i,j,k,nnew,iPhyF(iphy))+
     &                        PhyF(k,iphy)-PhyF_bak(k,iphy)),0.0_r8)
              if (iPhyS(iphy).gt.0) then
                t(i,j,k,nnew,iPhyS(iphy))=
     &                        MAX((t(i,j,k,nnew,iPhyS(iphy))+
     &                          PhyS(k,iphy)-PhyS_bak(k,iphy)),0.0_r8)
              endif
              do ipig=1,Npig
                id=iPigs(iphy,ipig)
                if (id.gt.0) then
                  t(i,j,k,nnew,id)=MAX((t(i,j,k,nnew,id)+
     &                 Pigs(k,iphy,ipig)-Pigs_bak(k,iphy,ipig)),0.0_r8)
                endif
              enddo
            enddo
          enddo
#ifdef MASKING
          endif
#endif
        enddo
      enddo
#ifdef PROFILE
!
!---------------------------------------------------------------------
!  Turn on EcoSim module time wall clock.
!---------------------------------------------------------------------
!
      call wclock_off (15)
#endif
      return
      end
