#include "cppdefs.h"
#ifdef BULK_FLUXES
      subroutine bulk_flux (tile)
!
!=====================================================================
!  Copyright (c) 2002 Rutgers/UCLA                                   !
!================================================ Hernan G. Arango ===
!                                                                    !
!  This routine computes the bulk parameterization of surface wind   !
!  stress and surface net heat fluxes.                               !
!                                                                    !
!  References:                                                       !
!                                                                    !
!    Fairall, C.W., E.F. Bradley, D.P. Rogers, J.B. Edson and G.S.   !
!      Young, 1996:  Bulk parameterization of air-sea fluxes for     !
!      tropical ocean-global atmosphere Coupled-Ocean Atmosphere     !
!      Response Experiment, JGR, 101, 3747-3764.                     !
!                                                                    !
!    Fairall, C.W., E.F. Bradley, J.S. Godfrey, G.A. Wick, J.B.      !
!      Edson, and G.S. Young, 1996:  Cool-skin and warm-layer        !
!      effects on sea surface temperature, JGR, 101, 1295-1308.      !
!                                                                    !
!    Liu, W.T., K.B. Katsaros, and J.A. Businger, 1979:  Bulk        !
!        parameterization of the air-sea exchange of heat and        !
!        water vapor including the molecular constraints at          !
!        the interface, J. Atmos. Sci, 36, 1722-1735.                !
!                                                                    !
!  Adapted from COARE code written originally by David Rutgers and   !
!  Frank Bradley.                                                    !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "scratch.h"
# include "tile.h"
!
# ifdef PROFILE
      call wclock_on (17)
# endif
      call bulk_flux_tile (Istr,Iend,Jstr,Jend,
     &                     A2d(1,1),A2d(1,2),A2d(1,3),A2d(1,4),
     &                     A2d(1,5),A2d(1,6))
# ifdef PROFILE
      call wclock_off (17)
# endif
      return
      end
!
!********************************************************************
      subroutine bulk_flux_tile (Istr,Iend,Jstr,Jend,LRad,LHeat,
     &                           SRad,SHeat,Taux,Tauy)
!********************************************************************
!
      implicit none
# include "param.h"
# include "forces.h"
# include "grid.h"
# include "mask.h"
# include "mixing.h"
# include "ocean.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Iter, IterMax, Jend, Jstr, i, j, k
      REAL_TYPE
     &        Bf, Cd, Ce, Ch, Hl, Hlv, Hlw, Hscale, Hs, Hsr, IER,
     &        PairM, Q, Qair, Qpsi, Qsea, Qstar, RH, Ri, Rq, Rr, Rt,
     &        Scff, TairC, TairK, Taur, Tcff, Tpsi, TseaC, TseaK,
     &        Tstar, TVstar, VisAir, Wgus, Wmag, Wpsi, Wspeed, Wstar,
     &        ZQ, ZQoL, ZT, ZToL, ZW, ZWoL, ZoQ, ZoT, ZoW
      REAL_TYPE
     &        cff, cff1, cff2, delQ, delQc, delT, delTc, delW, diffh,
     &        diffw, oL, rainfall, rhoAir, rhoSea, r3, upvel, wet_bulb
# ifdef LONGWAVE
      REAL_TYPE
     &        e_sat, vap_p
# endif
# ifdef COOL_SKIN
      REAL_TYPE
     &        Clam, Cwet, Fc, Hcool, Hsb, Hlb, Qbouy, Qcool, lambd
# endif
      REAL_TYPE
     &        bulk_psi
      REAL_TYPE
     &        LHeat(PRIVATE_2D_SCRATCH_ARRAY),
     &         LRad(PRIVATE_2D_SCRATCH_ARRAY),
     &        SHeat(PRIVATE_2D_SCRATCH_ARRAY),
     &         SRad(PRIVATE_2D_SCRATCH_ARRAY),
     &         Taux(PRIVATE_2D_SCRATCH_ARRAY),
     &         Tauy(PRIVATE_2D_SCRATCH_ARRAY)
      parameter (IterMax=20, r3=1.0_r8/3.0_r8)
!
# include "set_bounds.h"
!
!=====================================================================
!  Atmosphere-Ocean bulk fluxes parameterization.
!=====================================================================
!
      Hscale=rho0*Cp
      do j=Jstr-1,JendR
        do i=Istr-1,IendR
!
!  Input bulk parameterization fields.
!
          Wmag=SQRT(Uwind(i,j)*Uwind(i,j)+Vwind(i,j)*Vwind(i,j))
          PairM=Pair(i,j)
          TairC=Tair(i,j)
          TairK=TairC+273.16_r8
          TseaC=t(i,j,N,nrhs,itemp)
          TseaK=TseaC+273.16_r8
          rhoSea=rho(i,j,N)+1000.0_r8
          RH=Hair(i,j)
          SRad(i,j)=srflx(i,j)*Hscale
          rainfall=rain(i,j)
          Tcff=alpha(i,j)
          Scff=beta(i,j)
          ZT=blk_ZT
          ZQ=blk_ZQ
          ZW=blk_ZW
!
!  Initialize.
!
          IER=0
          delTc=0.0_r8
          delQc=0.0_r8
          LHeat(i,j)=lhflx(i,j)*Hscale
          SHeat(i,j)=shflx(i,j)*Hscale
          Taur=0.0_r8
          Taux(i,j)=0.0_r8
          Tauy(i,j)=0.0_r8
!
!---------------------------------------------------------------------
!  Compute net longwave radiation (W/m2), LRad.
!---------------------------------------------------------------------
!
# ifdef LONGWAVE
!  Use Berliand (1952) formula.
!
          cff=(0.7859_r8+0.03477_r8*TairC)/(1.0_r8+0.00412_r8*TairC)
          e_sat=10.0_r8**cff
          vap_p=e_sat*RH
          cff2=TairK*TairK*TairK
          cff1=cff2*TairK
          LRad(i,j)=-emmiss*StefBo*
     &              (cff1*(0.39_r8-0.5_r8*SQRT(0.01_r8*vap_p))*
     &                    (1.0_r8-0.6823_r8*cloud(i,j)*cloud(i,j))+
     &               cff2*4.0_r8*(TseaK-TairK))
# else
          LRad(i,j)=lrflx(i,j)*Hscale
# endif
# ifdef MASKING
     &             *rmask(i,j)
# endif
!
!---------------------------------------------------------------------
!  Compute specific humidities (kg/kg).
!---------------------------------------------------------------------
!
!  Compute air saturation vapor pressure (mb), using Teten formula.
!
          Qair=(1.0007_r8+3.46_e8-6*PairM)*6.1121_r8*
     &         EXP(17.502_r8*TairC/(240.97_r8+TairC))
!
!  Compute specific humdity, Q (kg/kg).  Check whether input relative
!  humidity is in kg/kg or g/kg.
!
          if (RH.lt.2.0_r8) then
            cff=Qair*RH
            Q=0.62197_r8*(cff/(PairM-0.378_r8*cff))
          else
            Q=RH/1000.0_r8
          endif
          Qair=0.62197_r8*(Qair/(PairM-0.378_r8*Qair))
!
!  Compute water saturation vapor pressure (mb), using Teten formula.
!
          Qsea=(1.0007_r8+3.46_e8-6*PairM)*6.1121_r8*
     &         EXP(17.502_r8*TseaC/(240.97_r8+TseaC))
!
!  Qsea reduced for salinity (Kraus, 1972, pp 46).
!
          Qsea=Qsea*0.98_r8
!
!  Convert Qsea from mb to specific humidity (kg/kg).
!
          Qsea=0.62197_r8*(Qsea/(PairM-0.378_r8*Qsea))
!
!---------------------------------------------------------------------
!  Compute Monin-Obukhov similarity parameters for wind (Wstar),
!  heat (Tstar), and moisture (Qstar), Liu et al. (1979).
!---------------------------------------------------------------------
!
!  Moist air density (kg/m3).
!
          rhoAir=PairM*100.0_r8/(blk_Rgas*TairK*(1.0_r8+0.61_r8*Q))
!
!  Kinematic viscosity of dry air (m2/s), Andreas (1989).
!
          VisAir=1.326_e8-5*(1.0_r8+TairC*(6.542_e8-3+TairC*
     &                       (8.301_e8-6-4.84_e8-9*TairC)))
!
!  Compute latent heat of vaporization (J/kg) at sea surface, Hlv.
!
          Hlv=(2.501_r8-0.00237_r8*TseaC)*1.0_e8+6
# ifdef COOL_SKIN
!
!  Cool skin correction constants. Clam: part of Saunders constant
!  lambda; Cwet: slope of saturation vapor.
!
          Clam=16.0_r8*g*blk_Cpw*(rhoSea*blk_visw)**3/
     &         (blk_tcw*blk_tcw*rhoAir*rhoAir)
          Cwet=0.622_r8*Hlv*Qsea/(blk_Rgas*TseaK*TseaK)
!
!  Set initial guesses for cool-skin layer thickness (Hcool).
!
          Hcool=0.001_r8
# endif /* COOL_SKIN */
!
!  Assume that wind is measured relative to sea surface and include
!  gustiness.
!
          Wgus=0.5_r8
          delW=SQRT(Wmag*Wmag+Wgus*Wgus)
          delQ=Q-Qsea
          delT=TairC-TseaC
!
!  Initial guesses for Monon-Obukhov similarity scales.
!
          ZWoL=0.0_r8
          ZoW=0.0005_r8
          Wstar=0.04_r8*delW
          Tstar=0.04_r8*delT
          Qstar=0.04_r8*delQ
          TVstar=Tstar*(1.0_r8+0.61_r8*Q)+0.61_r8*TairK*Qstar
!
!  Compute Richardson number.
!
          if (delW.ne.0.0_r8) then
            Ri=g*ZW*(delT+0.61_r8*TairK*delQ)/(TairK*delW*delW)
!           if (Ri.gt.0.25_r8) IER=-1
          else
            IER=-1
          endif
!
!  Iterate until convergence when IER is negative.  It usually
!  converges within four iterations.
!
          do Iter=1,IterMax
            if (IER.ge.0) then
!
!  Compute Monin-Obukhov stability parameter, Z/L.
!
              oL=g*vonKar*TVstar/
     &           (TairK*(1.0_r8+0.61_r8*Q)*Wstar*Wstar)
              ZWoL=ZW*oL
              ZToL=ZT*oL
              ZQoL=ZQ*oL
!
!  Evaluate stability functions at Z/L.
!
              Wpsi=bulk_psi(1,ZWoL)
              Tpsi=bulk_psi(2,ZToL)
              Qpsi=bulk_psi(2,ZQoL)
!
!  Compute wind scaling parameters, Wstar.
!
              ZoW=0.011_r8*Wstar*Wstar/g+0.11_r8*VisAir/Wstar
              Wstar=delW*vonKar/(LOG(ZW/ZoW)-Wpsi)
!
!  Computes roughness Reynolds number for wind (Rr), heat (Rt),
!  and moisture (Rq). Use Liu et al. (1976) look-up table to
!  compute "Rt" and "Rq" as function of "Rr".
!
              Rr=ZoW*Wstar/VisAir
              if ((Rr.ge.0.0_r8).and.(Rr.lt.1000.0_r8)) then
                do k=1,8
                  if ((Liu_Rr(k).le.Rr).and.(Rr.lt.Liu_Rr(k+1))) then
                    Rt=Liu_a(k,1)*Rr**Liu_b(k,1)
                    Rq=Liu_a(k,2)*Rr**Liu_b(k,2)
                  endif
                enddo
!
!  Compute heat and moisture scaling parameters, Tstar and Qstar.
!
                cff=VisAir/Wstar
                ZoT=Rt*cff
                ZoQ=Rq*cff
                cff=vonKar*blk_fdg
                Tstar=(delT+delTc)*cff/(LOG(ZT/ZoT)-Tpsi)
                Qstar=(delQ+delQc)*cff/(LOG(ZQ/ZoQ)-Qpsi)
!
!  Compute gustiness in wind speed.
!
                TVstar=Tstar*(1.0_r8+0.61_r8*Q)+0.61_r8*TairK*Qstar
                Bf=-g/TairK*Wstar*TVstar
                if (Bf.gt.0) then
                  Wgus=blk_beta*(Bf*blk_Zabl)**r3
                else
                  Wgus=0.0_r8
                endif
                delW=SQRT(Wmag*Wmag+Wgus*Wgus)
# ifdef COOL_SKIN
!
!---------------------------------------------------------------------
!  Cool Skin correction.
!---------------------------------------------------------------------
!
!  Backgound sensible and latent heat.
!
                Hsb=-rhoAir*blk_Cpa*Wstar*Tstar
                Hlb=-rhoAir*Hlv*Wstar*Qstar
!
!  Mean absoption in cool-skin layer.
!
                Fc=0.137_r8+11.0_r8*Hcool-
     &             (1.0_r8-EXP(-Hcool/8.0e-4))*6.6_e8-5/Hcool
!
!  Total cooling at the interface.
!
                Qcool=LRad(i,j)+Hsb+Hlb-SRad(i,j)*Fc
                Qbouy=Tcff*Qcool+Scff*Hlb*blk_Cpw/Hlv
!
!  Compute temperature and moisture change.
!
                if ((Qcool.gt.0.0_r8).and.(Qbouy.gt.0.0_r8)) then
                  lambd=6.0_r8/
     &                  (1.0_r8+(Clam*Qbouy/Wstar**4)**0.75_r8)**r3
                  Hcool=lambd*blk_visw/(SQRT(rhoAir/rhoSea)*Wstar)
                  delTc=Qcool*Hcool/blk_tcw
                else
                  delTc=0.0_r8
                endif
                delQc=Cwet*delTc
# endif /* COOL_SKIN */
              else
                IER=-2
              endif
            endif
          enddo
!
!---------------------------------------------------------------------
!  Compute Atmosphere/Ocean fluxes.
!---------------------------------------------------------------------
!
          if (IER.ge.0) then
!
!  Compute transfer coefficients for momentun (Cd), heat (Ch),
!  and moisture (Ce).
!
            Wspeed=SQRT(Wmag*Wmag+Wgus*Wgus)
            Cd=Wstar*Wstar/(Wspeed*Wspeed)
            Ch=Wstar*Tstar/(Wspeed*(TairC-TseaC+delTc))
            Ce=Wstar*Qstar/(Wspeed*(Q-Qsea+delQc))
!
!  Compute turbulent sensible heat flux (W/m2), Hs.
!
            Hs=-blk_Cpa*rhoAir*Wstar*Tstar
!
!  Compute sensible heat flux (W/m2) due to rainfall (kg/m2/s), Hsr.
!
            diffw=2.11_e8-5*(TairK/273.16_r8)**1.94_r8
            diffh=0.02411_r8*(1.0_r8+TairC*(3.309_e8-3-1.44e-6*TairC))/
     &            (rhoAir*blk_Cpa)
            cff=Qair*Hlv/(blk_Rgas*TairK*TairK)
            wet_bulb=1.0_r8/(1.0_r8+0.622_r8*(cff*Hlv*diffw)/
     &                                       (blk_Cpa*diffh))
            Hsr=rainfall*wet_bulb*blk_Cpw*
     &          ((TseaC-TairC)+(Qsea-Q)*Hlv/blk_Cpa)
            SHeat(i,j)=(Hs+Hsr)
# ifdef MASKING
     &                *rmask(i,j)
# endif
!
!  Compute turbulent latent heat flux (W/m2), Hl.
!
            Hl=-Hlv*rhoAir*Wstar*Qstar
!
!  Compute Webb correction (Webb effect) to latent heat flux, Hlw.
!
            upvel=-1.61_r8*Wstar*Qstar-
     &            (1.0_r8+1.61_r8*Q)*Wstar*Tstar/TairK
            Hlw=rhoAir*Hlv*upvel*Q
            LHeat(i,j)=(Hl+Hlw)
# ifdef MASKING
     &                *rmask(i,j)
# endif
!
!  Compute momentum flux (N/m2) due to rainfall (kg/m2/s).
!
            Taur=0.85_r8*rainfall*Wmag
!
!  Compute wind stress components (N/m2), Tau.
!
            cff=rhoAir*Cd*Wspeed
            Taux(i,j)=(cff*Uwind(i,j)+Taur*SIGN(1.0_r8,Uwind(i,j)))
# ifdef MASKING
     &               *rmask(i,j)
# endif
            Tauy(i,j)=(cff*Vwind(i,j)+Taur*SIGN(1.0_r8,Vwind(i,j)))
# ifdef MASKING
     &               *rmask(i,j)
# endif
          endif
        enddo
      enddo
!
!=====================================================================
!  Compute surface net heat flux and surface wind stress.
!=====================================================================
!
!  Compute kinematic, surface, net heat flux (degC m/s).  Notice that
!  the signs of latent and sensible fluxes are reversed because fluxes
!  from the bulk formulation are positive out of the ocean.
!
      Hscale=1.0_r8/(rho0*Cp)
      do j=JstrR,JendR
        do i=IstrR,IendR
          lrflx(i,j)=LRad(i,j)*Hscale
          lhflx(i,j)=-LHeat(i,j)*Hscale
          shflx(i,j)=-SHeat(i,j)*Hscale
          stflx(i,j,itemp)=(srflx(i,j)+lrflx(i,j)+
     &                      lhflx(i,j)+shflx(i,j))
# ifdef MASKING
     &                    *rmask(i,j)
# endif
        enddo
      enddo
!
!  Compute kinematic, surface wind stress (m2/s2).
!
      cff=0.5_r8/rho0
      do j=JstrR,JendR
        do i=Istr,IendR
          sustr(i,j)=cff*(Taux(i-1,j)+Taux(i,j))
# ifdef MASKING
     &              *umask(i,j)
# endif
        enddo
      enddo
      do j=Jstr,JendR
        do i=IstrR,IendR
          svstr(i,j)=cff*(Tauy(i,j-1)+Tauy(i,j))
# ifdef MASKING
     &              *vmask(i,j)
# endif
        enddo
      enddo
# if defined EW_PERIODIC || defined NS_PERIODIC
! 
!-----------------------------------------------------------------------
!  Apply periodic boundary conditions.
!-----------------------------------------------------------------------
!
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        lrflx(START_2D_ARRAY))
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        lhflx(START_2D_ARRAY))
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        shflx(START_2D_ARRAY))
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        stflx(START_2D_ARRAY,itemp))
      call exchange_u2d_tile (Istr,Iend,Jstr,Jend,
     &                        sustr(START_2D_ARRAY))
      call exchange_v2d_tile (Istr,Iend,Jstr,Jend,
     &                        svstr(START_2D_ARRAY))
# endif
      return
      end
      function bulk_psi (iflag,ZoL)
!
!=====================================================================
!                                                                    !
!  This function evaluates the stability function, PSI, for wind     !
!  speed (iflag=1) or for air temperature and moisture (iflag=2)     !
!  profiles as function of the stability parameter, ZoL (z/L).       !
!                                                                    !
!  Reference:                                                        !
!                                                                    !
!    Liu, W.T., K.B. Katsaros, and J.A. Businger, 1979:  Bulk        !
!        parameterization of the air-sea exchange of heat and        !
!        water vapor including the molecular constraints at          !
!        the interface, J. Atmos. Sci, 36, 1722-1735.                !
!                                                                    !
!=====================================================================
!
      implicit none
!
      INTEGER_TYPE
     &        iflag
      REAL_TYPE
     &        Fw, chic, chik, cff, pi, psic, psik, r3, ZoL
      REAL_TYPE
     &        bulk_psi
      parameter (r3=1.0_r8/3.0_r8)
!
!---------------------------------------------------------------------
!  Compute stability function, PSI.
!---------------------------------------------------------------------
!
!  Initialize for the zero "ZoL" case.
!
      bulk_psi=0.0_r8
!
!  Unstable conditions.
!
      if (ZoL.lt.0.0_r8) then
        pi=4.0_r8*ATAN(1.0_r8)
        chik=(1.0_r8-16.0_r8*ZoL)**0.25_r8
        if (iflag.eq.1) then
          psik=2.0_r8*LOG(0.5_r8*(1.0_r8+chik))+
     &                LOG(0.5_r8*(1.0_r8+chik*chik))-
     &         2.0_r8*ATAN(chik)+
     &         0.5_r8*pi
        elseif (iflag.eq.2) then
          psik=2.0_r8*LOG(0.5_r8*(1.0_r8+chik*chik))
        endif
!
!  For very unstable conditions, use free-convection (Fairall).
!
        cff=SQRT(3.0_r8)
        chic=(1.0_r8-12.87_r8*ZoL)**r3
        psic=1.5_r8*LOG(r3*(1.0_r8+chic+chic*chic))-
     &       cff*ATAN((1.0_r8+2.0_r8*chic)/cff)+
     &       pi/cff
!
!  Match Kansas and free-convection forms with weighting Fw.
!
        Fw=1.0_r8/(1.0_r8+ZoL*ZoL)
        bulk_psi=Fw*psik+(1.0_r8-Fw)*psic
!
!  Stable conditions.
!
      elseif (ZoL.gt.0.0_r8) then
        bulk_psi=-4.7_r8*ZoL
      endif
#else
      subroutine bulk_flux
#endif
      return
      end
