#include "cppdefs.h"
/*
**********************************************************************
** Copyright (c) 2002 Rutgers/UCLA                                  **
************************************************* Hernan G. Arango ***
**                                                                  **
** ANALYTICAL PACKAGE:                                              **
**                                                                  **
** This package is used to provide various analytical fields to the **
** model when appropriate.                                          **
**                                                                  **
** Routines:                                                        **
**                                                                  **
** ana_biology         Analytical initial conditions for biological **
**                       tracers.                                   **
** ana_bsedim          Analytical bottom sediment grain size and    **
**                       density.                                   **
** ana_btflux          Analytical kinematic bottom flux of tracer   **
**                       type variables.                            **
** ana_cloud           Analytical cloud fraction.                   **
** ana_diag            Customized diagnostics.                      **
** ana_fsobc           Analytical free-surface boundary conditions. **
** ana_grid            Analytical model grid set-up.                **
** ana_humid           Analytical surface air humidity.             **
** ana_initial         Analytical initial conditions for momentum,  **
**                       free surface and tracers.                  **
** ana_m2clima         Analytical 2D momentum climatology.          **
** ana_m2obc           Analytical 2D momentum boundary conditions.  **
** ana_m3clima         Analytical 3D momentum climatology.          **
** ana_m3obc           Analytical 3D momentum boundary conditions.  **
** ana_mask            Analytical Land/Sea masking.                 **
** ana_pair            Analytical surface air pressure.             **
** ana_psource         Analytical mass/tracer point sources/sinks.  **
** ana_rain            Analytical rain fall rate.                   **
** ana_sediment        Analytical initial conditions for sediment   **
**                       tracers and variables.                     **
** ana_smflux          Analytical kinematic surface momentum flux   **
**                       (wind stress).                             **
** ana_spinning        Analytical time-varying rotation force.      **
** ana_srflux          Analytical kinematic surface shortwave       **
**                       radiation flux.                            **
** ana_ssh             Analytical sea surface height climatology.   **
** ana_sst             Analytical SST and dQdSST which are used     **
**                       for heat flux correction.                  **
** ana_sss             Analytical sea surface salinity.             **
** ana_stflux          Analytical kinematic surface flux of tracer  **
**                       type variables.                            **
** ana_tair            Analytical surface air temperature.          **
** ana_tclima          Analytical tracer climatology fields.        **
** ana_tobc            Analytical tracer boundary conditions.       **
** ana_vmix            Analytical vertical mixing coefficients for  **
**                       momentum and tracers.                      **
** ana_winds           Analytical surface winds.                    **
** ana_wwave           Analytical wind induced wave amplitude,      **
**                       direction and period.                      **
**                                                                  **
**********************************************************************
*/

#if defined ANA_BIOLOGY && defined BIOLOGY
      subroutine ana_biology (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets initial conditions for biological tracer fields !
!  using analytical expressions.                                     !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!    
      call ana_biology_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_biology_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "biology.h"
# include "grid.h"
# include "ocean.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, itrc, j, k
      REAL_TYPE
     &        SiO4, cff1, cff2, temp
!
# include "set_bounds.h"
!
      itrc=0
!
# ifdef BIO_FASHAM
!---------------------------------------------------------------------
!  Fasham type, nitrogen-based biology model.
!---------------------------------------------------------------------
!
      cff1=20.0_r8/3.0_r8
      cff2= 2.0_r8/3.0_r8
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            temp=t(i,j,k,1,itemp)
            if (temp.lt.8.0_r8) then
              SiO4=30.0_r8
            elseif ((temp.ge.8.0_r8).and.(temp.le.11.0_r8)) then
              SiO4=30.0_r8-((temp-8.0_r8)*cff1)
            elseif ((temp.gt.11.0_r8).and.(temp.le.13.0_r8)) then
              SiO4=10.0_r8-((temp-11.0_r8)*4.0_r8)
            elseif ((temp.gt.13.0_r8).and.(temp.le.16.0_r8)) then
              SiO4=2.0_r8-((temp-13.0_r8)*cff2)
            elseif (temp.gt.16.0_r8) then
              SiO4=0.0_r8
            endif
            t(i,j,k,1,iNO3_)=1.67_r8+0.5873_r8*SiO4+
     &                               0.0144_r8*SiO4**2+
     &                               0.0003099_r8*SiO4**3
            t(i,j,k,1,iPhyt)=0.08_r8
            t(i,j,k,1,iZoop)=0.06_r8
            t(i,j,k,1,iNH4_)=0.1_r8
            t(i,j,k,1,iLDet)=0.02_r8
            t(i,j,k,1,iSDet)=0.04_r8
            t(i,j,k,1,iChlo)=0.02_r8
!
            t(i,j,k,2,iNO3_)=t(i,j,k,1,iNO3_)
            t(i,j,k,2,iPhyt)=t(i,j,k,1,iPhyt)
            t(i,j,k,2,iNH4_)=t(i,j,k,1,iNH4_)
            t(i,j,k,2,iLDet)=t(i,j,k,1,iLDet)
            t(i,j,k,2,iSDet)=t(i,j,k,1,iSDet)
            t(i,j,k,2,iChlo)=t(i,j,k,1,iChlo)
          enddo
        enddo
      enddo
#  if defined EW_PERIODIC || defined NS_PERIODIC
      do itrc=NAT+1,NT
        call exchange_r3d_tile (Istr,Iend,Jstr,Jend,
     &                          t(START_2D_ARRAY,1,1,itrc))
        call exchange_r3d_tile (Istr,Iend,Jstr,Jend,
     &                          t(START_2D_ARRAY,1,2,itrc))
      enddo
#  endif
# endif /* BIO_FASHAM */
      return
      end
#endif /* ANA_BIOLOGY && BIOLOGY */
#if defined ANA_BSEDIM && defined BBL
      subroutine ana_bsedim (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets initial bottom sediment grain diameter size (m) !
!  and density used in the bottom boundary formulation (kg/m3).      !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_bsedim_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_bsedim_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "bbl.h"
# include "grid.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set bottom sediment grain diameter (m) and density (kg/m3).
!---------------------------------------------------------------------
!
# if defined BL_TEST || defined NJ_BIGHT
      do j=JstrR,JendR
        do i=IstrR,IendR
          Ssize(i,j)=0.0005_r8
          Sdens(i,j)=2650.0_r8
        enddo
      enddo
# else
      ANA_BSEDIM: no values provided for SSIZE and SDENS.
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        Ssize(START_2D_ARRAY))
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        Sdens(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_BSEDIM && BBL */
#if defined ANA_BTFLUX || defined ANA_BSFLUX || defined ANA_BPFLUX
      subroutine ana_btflux (tile,itrc)
!
!=====================================================================
!                                                                    !
!  This routine sets kinematic bottom flux of tracer type variables  !
!  (tracer units m/s).                                               !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        itrc, tile
# include "param.h"
# include "tile.h"
!
      call ana_btflux_tile (Istr,Iend,Jstr,Jend,itrc)
      return
      end
!
!*********************************************************************
      subroutine ana_btflux_tile (Istr,Iend,Jstr,Jend,itrc)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "forces.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, itrc, j
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set kinematic bottom heat flux (degC m/s) at horizontal RHO-points.
!---------------------------------------------------------------------
!
      if (itrc.eq.itemp) then
        do j=JstrR,JendR
          do i=IstrR,IendR
            btflx(i,j,itrc)=0.0_r8
          enddo
        enddo
!
!---------------------------------------------------------------------
!  Set kinematic bottom salt flux (m/s) at horizontal RHO-points,
!  scaling by bottom salinity is done elsewhere.
!---------------------------------------------------------------------
!
      elseif (itrc.eq.isalt) then
        do j=JstrR,JendR
          do i=IstrR,IendR
            btflx(i,j,itrc)=0.0_r8
          enddo
        enddo
!
!---------------------------------------------------------------------
!  Set kinematic bottom flux (T m/s) of passive tracers, if any.
!---------------------------------------------------------------------
!
      else
        do j=JstrR,JendR
          do i=IstrR,IendR
            btflx(i,j,itrc)=0.0_r8
          enddo
        enddo
      endif
      return
      end
#endif /* ANA_BTFLUX || ANA_BSFLUX || ANA_BPFLUX */
#if defined ANA_CLOUD && defined CLOUDS
      subroutine ana_cloud (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets cloud fraction using an analytical expression.  !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_cloud_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_cloud_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "forces.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
      REAL_TYPE
     &        Cval
# ifdef PAPA_CLM
      INTEGER_TYPE
     &        day, month, year
      REAL_TYPE
     &        Coktas(14), Cyday(14), hour, yday
      parameter
     &    (Coktas( 1)=6.29_r8, Coktas( 2)=6.26_r8, Coktas( 3)=6.31_r8,
     &     Coktas( 4)=6.31_r8, Coktas( 5)=6.32_r8, Coktas( 6)=6.70_r8,
     &     Coktas( 7)=7.12_r8, Coktas( 8)=7.26_r8, Coktas( 9)=6.93_r8,
     &     Coktas(10)=6.25_r8, Coktas(11)=6.19_r8, Coktas(12)=6.23_r8,
     &     Coktas(13)=6.31_r8, Coktas(14)=6.29_r8)
      parameter
     &    (Cyday( 1)=  0.0_r8, Cyday( 2)= 16.0_r8, Cyday( 3)= 46.0_r8,
     &     Cyday( 4)= 75.0_r8, Cyday( 5)=105.0_r8, Cyday( 6)=136.0_r8,
     &     Cyday( 7)=166.0_r8, Cyday( 8)=197.0_r8, Cyday( 9)=228.0_r8,
     &     Cyday(10)=258.0_r8, Cyday(11)=289.0_r8, Cyday(12)=319.0_r8,
     &     Cyday(13)=350.0_r8, Cyday(14)=365.0_r8)
# endif
!
# include "set_bounds.h"
!
      Cval=0.0_r8
!
!---------------------------------------------------------------------
!  Set analytical cloud fraction (%/100): 0=clear sky, 1:overcast sky.
!---------------------------------------------------------------------

# if defined PAPA_CLM
!
!  OWS Papa cloud climatology.
!
      call caldate (r_date,tdays,year,yday,month,iday,hour)
      do i=1,13
        if ((yday.ge.Cyday(i)).and.(yday.le.Cday(i+1))) then
          Cval=0.125_r8*(Coktas(i  )*(Cyday(i+1)-yday)+
     &                   Coktas(i+1)*(yday-Cyday(i)))/
     &                  (Cyday(i+1)-Cyday(i))
        endif
      enddo
# elif defined NJ_BIGHT
      Cval=0.3_r8
# elif defined ADRIATIC1 || defined ADRIATIC2
      Cval=0.4_r8
# endif
!
      do j=JstrR,JendR
        do i=IstrR,IendR
          cloud(i,j)=Cval
        enddo
      enddo
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        cloud(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_CLOUD && CLOUDS */
#ifdef ANA_DIAG
      subroutine ana_diag
!
!=====================================================================
!                                                                    !
!  This routine is provided so the USER can compute any specialized  !
!  diagnostics.  If activated, this routine is call at end of every  !
!  3D-equations timestep.                                            !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "iounits.h"
# include "ocean.h"
# include "scalars.h"
!
# ifdef SEAMOUNT
      INTEGER_TYPE
     &        i, j, k
      REAL_TYPE
     &        umax, ubarmax, vmax, vbarmax
# endif /* SEAMOUNT */
# ifdef SEAMOUNT
!
!  Open USER file.
!
      if (iic.eq.ntstart) then
        open (usrout,file=usrname,form='formatted',status='unknown',
     &        err=40)
        goto 60
  40    write(stdout,50) usrname
  50    format(' ANA_DIAG - unable to open output file: ',a)
        exit_flag=1
  60    continue
      endif
!
!  Write out maximum values of velocity.
!
      umax=0.0_r8
      vmax=0.0_r8
      ubarmax=0.0_r8
      vbarmax=0.0_r8
      do k=1,N
        do j=0,M
          do i=1,L
            umax=MAX(umax,u(i,j,k,nnew))
          enddo
        enddo
        do j=1,M
          do i=0,L
            vmax=MAX(vmax,v(i,j,k,nnew))
          enddo
        enddo
      enddo
      do j=0,M
        do i=1,L
          ubarmax=MAX(ubarmax,ubar(i,j,knew))
        enddo
      enddo
      do j=1,M
        do i=0,L
          vbarmax=MAX(vbarmax,vbar(i,j,knew))
        enddo
      enddo
!
!  Write out maximum values on velocity.
!
      write(usrout,70) tdays, ubarmax, vbarmax, umax, vmax
  70  format(2x,f13.6,2x,1pe13.6,2x,1pe13.6,2x,1pe13.6,2x,1pe13.6)
# endif /* SEAMOUNT */
      return
      end
#endif /* ANA_DIAG */
#if defined ANA_FSOBC && !defines ISWAKE
      subroutine ana_fsobc (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets free-surface open boundary conditions using     !
!  analytical expressions.                                           !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_fsobc_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_fsobc_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "boundary.h"
# include "grid.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
      REAL_TYPE
     &        fac, omega, phase, val
!
# include "set_bounds.h"
!
      i=0
      j=0
      fac=0.0_r8
      omega=0.0_r8
      phase=0.0_r8
      val=0.0_r8
!
!---------------------------------------------------------------------
!  Free-surface open boundary conditions.
!---------------------------------------------------------------------
!
# if defined NJ_BIGHT
      if (EASTERN_EDGE) then
        do j=JstrR,JendR
          zeta_east(j)=0.0_r8
        enddo
      endif
      if (SOUTHERN_EDGE) then
        do i=IstrR,IendR
          zeta_south(i)=0.0_r8
        enddo
      endif
# elif defined SED_TEST1
      if (WESTERN_EDGE) then
        fac=100.0_r8
        do j=JstrR,JendR
          zeta_west(j)=9.0_e8-6*fac
        enddo
      endif
      if (EASTERN_EDGE) then
        fac=100.0_r8
        do j=JstrR,JendR
          zeta_east(j)=9.0_e8-6*FLOAT(L)*fac
        enddo
      endif
# elif defined WEDDELL
      if (WESTERN_EDGE) then
        fac=TANH((tdays-dstart)/1.0_r8)
        omega=2.0_r8*pi*time/(12.42_r8*3600.0_r8)    !  M2 Tide period
        val=0.53_r8+(0.53_r8-0.48_r8)/FLOAT(L)
        phase=(277.0_r8+(277.0_r8-240.0_r8)/FLOAT(L))*deg2rad
        do j=JstrR,JendR
          zeta_west(j)=fac*val*COS(omega-phase)
        enddo
      endif
      if (EASTERN_EDGE) then
        fac=TANH((tdays-dstart)/1.0_r8)
        omega=2.0_r8*pi*time/(12.42_r8*3600.0_r8)    !  M2 Tide period
        val=0.53_r8+(0.53_r8-0.48_r8)
        phase=(277.0_r8+(277.0_r8-240.0_r8))*deg2rad
        do j=JstrR,JendR
          zeta_east(j)=fac*val*COS(omega-phase)
        enddo
      endif
# elif defined ISWAKE
! nothing, because it is defined by ana_initial, and kept constant.
# else
      ANA_FSOBC: no values provided for free-surface.
# endif
      return
      end
#endif /* ANA_FSOBC */
#ifdef ANA_GRID
      subroutine ana_grid (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets model grid using an analytical expressions.     !
!                                                                    !
!  On Output:  stored in common blocks:                              !
!                                                                    !
!                           "grid"    (file grid.h)                  !
!                           "scalars" (file scalar.h)                !
!                                                                    !
!     el       Length (m) of domain box in the ETA-direction.        !
!     f        Coriolis parameter (1/seconds) at RHO-points.         !
!     h        Bathymetry (meters; positive) at RHO-points.          !
!     hmin     Minimum depth of bathymetry (m).                      !
!     hmax     Maximum depth of bathymetry (m).                      !
!     pm       Coordinate transformation metric "m" (1/meters)       !
!              associated with the differential distances in XI      !
!              at RHO-points.                                        !
!     pn       Coordinate transformation metric "n" (1/meters)       !
!              associated with the differential distances in ETA.    !
!              at RHO-points.                                        !
!     xl       Length (m) of domain box in the XI-direction.         !
!     xp       XI-coordinates (m) at PSI-points.                     !
!     xr       XI-coordinates (m) at RHO-points.                     !
!     yp       ETA-coordinates (m) at PSI-points.                    !
!     yr       ETA-coordinates (m) at RHO-points.                    !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_grid_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_grid_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "scalars.h"
# include "wclock.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
      REAL_TYPE
     &        Esize, NSUB, Xsize, beta, cff, depth, dth, dx, dy, f0,
     &        r, theta, twopi, val1, val2, x0,y0,rd_inner
# ifdef WEDDELL
      REAL_TYPE
     &        hwrk(-1:235), xwrk(-1:235), zwrk
# endif
!
# include "set_bounds.h"
!
      k=0
      dth=0.0_r8
      r=0.0_r8
      theta=0.0_r8
      twopi=2.0_r8*pi
      val1=0.0_r8
      val2=0.0_r8
!
!---------------------------------------------------------------------
!  Set grid parameters:
!
!     Xsize    Length (m) of domain box in the XI-direction.
!     Esize    Length (m) of domain box in the ETA-direction.
!     depth    Maximum depth of bathymetry (m).
!     f0       Coriolis parameter, f-plane constant (1/s).
!     beta     Coriolis parameter, beta-plane constant (1/s/m).
!---------------------------------------------------------------------
!
# if defined BASIN
      Xsize=3600.0_e8+3
      Esize=2800.0_e8+3
      depth=5000.0_r8
      f0=1.0_e8-4
      beta=2.0_e8-11
# elif defined BL_TEST
      Xsize=100.0_e8+3
      Esize=5.0_e8+3
      depth=47.5_r8
      f0=9.25_e8-4
      beta=0.0_r8
# elif defined CANYON_A || defined CANYON_B
      Xsize=128.0_e8+3
      Esize=96.0_e8+3
      depth=4000.0_r8
      f0=1.0_e8-4
      beta=0.0_r8
# elif defined DOUBLE_GYRE
      Xsize=1000.0_e8+3
      Esize=2000.0_e8+3
      depth=500.0_r8
      f0=7.3_e8-5
      beta=2.0_e8-11
# elif defined FLT_TEST
      Xsize=1.0_e8+3*FLOAT(Lm)
      Esize=1.0_e8+3*FLOAT(Mm)
      depth=10.0_r8
      f0=0.0_r8
      beta=0.0_r8
# elif defined GRAV_ADJ
      Xsize=64.0_e8+3
      Esize=2.0_e8+3
      depth=20.0_r8
      f0=0.0_r8
      beta=0.0_r8
# elif defined LAB_CANYON
      Xsize=0.55_r8                  ! width of annulus
      Esize=2.0_r8*pi                ! azimuthal length (radians)
      f0=4.0_r8*pi/25.0_r8
      beta=0.0_r8
# elif defined LMD_TEST
      Xsize=100.0_e8+3
      Esize=100.0_e8+3
      depth=50.0_r8
      f0=1.09_e8-4
      beta=0.0_r8
# elif defined OVERFLOW
      Xsize=4.0_e8+3
      Esize=200.0_e8+3
      depth=4000.0_r8
      f0=0.0_r8
      beta=0.0_r8
# elif defined RIVERPLUME
      Xsize=58.5_e8+3
      Esize=201.0_e8+3
      depth=150.0_r8
      f0=1.0_e8-4
      beta=0.0_r8
# elif defined SEAMOUNT
      Xsize=320.0_e8+3
      Esize=320.0_e8+3
      depth=5000.0_r8
      f0=1.0_e8-4
      beta=0.0_r8
# elif defined SHELFRONT
      Xsize=20.0_e8+3
      Esize=200.0_e8+3
      depth=1660.0_r8
      f0=1.0_e8-4
      beta=0.0_r8
# elif defined ISWAKE
      Esize=80.0e+3                             ! width of channel
      Xsize=Esize*dble(Lm)/dble(Mm)             ! length of channel
      depth=500.                                ! depth of channel
      beta=0.                  ! f-plane
      f0=1.0E-4                ! Coriolis
      y0=Esize/2.              ! y location of island center
      x0=y0
      rd_inner=10.e+3
# elif defined SOLITON
      Xsize=48.0_r8
      Esize=16.0_r8
      depth=1.0_r8
      f0=0.0_r8
      beta=1.0_r8
      g=1.0_r8
# elif defined SED_TEST1
      Xsize=10000.0_r8
      Esize=1000.0_r8
      depth=10.0_r8
      f0=0.0_r8
      beta=0.0_r8
# elif defined UPWELLING
      Xsize=1000.0_r8*FLOAT(Lm)
      Esize=1000.0_r8*FLOAT(Mm)
      depth=150.0_r8
      f0=-8.26_e8-5
      beta=0.0_r8
# elif defined WEDDELL
      Xsize=4000.0_r8*FLOAT(Lm)
      Esize=4000.0_r8*FLOAT(Mm)
      depth=4500.0_r8
      f0=0.0_r8
      beta=0.0_r8
# else
      ANA_GRID: no values provided for Xsize, Esize, depth, f0, and beta.
# endif
!
!  Load grid parameters to global storage.
!
      if (SOUTH_WEST_CORNER) then
        xl=Xsize
        el=Esize
        hmax=depth
      endif
!
!---------------------------------------------------------------------
!  Compute the (XI,ETA) coordinates at PSI- and RHO-points.
!  Set grid spacing (m).
!---------------------------------------------------------------------
!
      dx=Xsize/FLOAT(Lm)
      dy=Esize/FLOAT(Mm)
!
!  Compute (XI,ETA) coordinate at PSI-points and RHO-points.
!
# if defined LAB_CANYON
!!    dth=twopi/FLOAT(Mm)                        ! equal azimultal spacing
      dth=0.01_r8                                ! azimultal spacing
      cff=(4.0_r8*pi/(dth*FLOAT(Mm)))-1.0_r8     ! F
      do j=JstrR,JendR
        do i=IstrR,IendR
          r=0.35_r8+dx*FLOAT(i-1)
          theta=-pi+
     &          0.5_r8*dth*((cff+1.0_r8)*FLOAT(j-1)+
     &                      (cff-1.0_r8)*(FLOAT(Mm)/twopi)*
     &                      SIN(twopi*FLOAT(j-1)/FLOAT(Mm)))
          xp(i,j)=r*COS(theta)
          yp(i,j)=r*SIN(theta)
          r=0.35_r8+dx*(FLOAT(i-1)+0.5_r8)
          theta=-pi+
     &          0.5_r8*dth*((cff+1.0_r8)*(FLOAT(j-1)+0.5_r8)+
     &                      (cff-1.0_r8)*(FLOAT(Mm)/twopi)*
     &                      SIN(twopi*(FLOAT(j-1)+0.5_r8)/FLOAT(Mm)))
          xr(i,j)=r*COS(theta)
          yr(i,j)=r*SIN(theta)
        enddo
      enddo
# else
      do j=JstrR,JendR
        do i=IstrR,IendR
#  ifdef BL_TEST
          dx=0.5_r8*(4000.0_r8/FLOAT(L))*FLOAT(i)+675.0_r8
#  endif
          xp(i,j)=dx*FLOAT(i-1)
          xr(i,j)=dx*(FLOAT(i-1)+0.5_r8)
          yp(i,j)=dy*FLOAT(j-1)
          yr(i,j)=dy*(FLOAT(j-1)+0.5_r8)
        enddo
      enddo
# endif
!
!---------------------------------------------------------------------
! Compute coordinate transformation metrics at RHO-points "pm" and
! "pn"  (1/m) associated with the differential distances in XI and
! ETA, respectively.
!---------------------------------------------------------------------
!
# if defined LAB_CANYON
      do j=JstrR,JendR
        do i=IstrR,IendR
          r=0.35_r8+dx*(FLOAT(i-1)+0.5_r8)
          theta=0.5_r8*dth*((cff+1.0_r8)+
     &                      (cff-1.0_r8)*
     &                      COS(twopi*FLOAT(j-1)/FLOAT(Mm)))
          pm(i,j)=1.0_r8/dx
          pn(i,j)=1.0_r8/(r*theta)
        enddo
      enddo
# else
      do j=JstrR,JendR
        do i=IstrR,IendR
#  ifdef BL_TEST
          dx=0.5_r8*(4000.0_r8/FLOAT(L))*FLOAT(i)+675.0_r8
#  endif
          pm(i,j)=1.0_r8/dx
          pn(i,j)=1.0_r8/dy
        enddo
      enddo
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        pm(START_2D_ARRAY))
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        pn(START_2D_ARRAY))
# endif
# if (defined CURVGRID && defined UV_ADV)
!
!---------------------------------------------------------------------
!  Compute d(1/n)/d(xi) at horizontal U-points and d(1/m)/d(eta) at
!  horizontal V-points.
!---------------------------------------------------------------------
!
      do j=Jstr,Jend
        do i=Istr,Iend
          dndx(i,j)=0.5_r8*((1.0_r8/pn(i+1,j  ))-(1.0_r8/pn(i-1,j  )))
          dmde(i,j)=0.5_r8*((1.0_r8/pm(i  ,j+1))-(1.0_r8/pm(i  ,j-1)))
        enddo
      enddo
#  if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        dndx(START_2D_ARRAY))
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        dmde(START_2D_ARRAY))
#  endif
!
!---------------------------------------------------------------------
! Angle (radians) between XI-axis and true EAST at RHO-points.
!---------------------------------------------------------------------
!
#  if defined LAB_CANYON
      do j=JstrR,JendR
        do i=IstrR,IendR
          theta=-pi+
     &          0.5_r8*dth*((cff+1.0_r8)*(FLOAT(j-1)+0.5_r8)+
     &                      (cff-1.0_r8)*(FLOAT(Mm)/twopi)*
     &                      SIN(twopi*(FLOAT(j-1)+0.5_r8)/FLOAT(Mm)))
          angler(i,j)=theta
        enddo
      enddo
#  elif defined WEDDELL
      val1=90.0_r8*deg2rad
      do j=JstrR,JendR
        do i=IstrR,IendR
          angler(i,j)=val1
        enddo
      enddo
#  endif
#  if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        angler(START_2D_ARRAY))
#  endif
# endif /* UV_ADV && CURVGRID */
!
!---------------------------------------------------------------------
!  Compute Coriolis parameter (1/s) at RHO-points.
!---------------------------------------------------------------------
!
# ifdef WEDDELL
      val1=10.4_r8/FLOAT(Lm)
      do j=JstrR,JendR
        do i=IstrR,IendR
          f(i,j)=2.0_r8*7.29_e8-5*
     &           SIN((-79.0_r8+FLOAT(i-1)*val1)*deg2rad)
        enddo
      enddo
# else
      val1=0.5_r8*Esize
      do j=JstrR,JendR
        do i=IstrR,IendR
          f(i,j)=f0+beta*(yr(i,j)-val1)
        enddo
      enddo
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        f(START_2D_ARRAY))
# endif
!
!---------------------------------------------------------------------
!  Set bathymetry (meters; positive) at RHO-points.
!---------------------------------------------------------------------
!
# if defined BL_TEST
      do j=JstrR,JendR
        do i=IstrR,IendR
          val1=(xr(i,j)+500.0_r8)/15000.0_r8
          h(i,j)=14.0_r8+
     &           25.0_r8*(1.0_r8-EXP(-pi*xr(i,j)*1.0_e8-5))-
     &           8.0_r8*exp(-val1*val1)
        enddo
      enddo
# elif defined CANYON_A || defined CANYON_B
      do j=JstrR,JendR
        do i=IstrR,IendR
          val1=32000.0_r8-16000.0_r8*(SIN(pi*xr(i,j)/Xsize))**24
          h(i,j)=20.0_r8+0.5_r8*(depth-20.0_r8)*
     &           (1.0_r8+TANH((yr(i,j)-val1)/10000.0_r8))
        enddo
      enddo
# elif defined LAB_CANYON
      do j=JstrR,JendR
        do i=IstrR,IendR
          r=0.35_r8+dx*(FLOAT(i-1)+0.5_r8)
          theta=-pi+
     &           0.5_r8*dth*((cff+1.0_r8)*(FLOAT(j-1)+0.5_r8)+
     &                       (cff-1.0_r8)*(FLOAT(Mm)/twopi)*
     &                       SIN(dth*(FLOAT(j-1)+0.5_r8)/FLOAT(Mm)))
          val1=0.55_r8-0.15_r8*(COS(pi*theta*0.55_r8/0.2_r8)**2) !r_small
          val2=0.15_r8+0.15_r8*(COS(pi*theta*0.55_r8/0.2_r8)**2) !lambda
          if (ABS(theta).ge.0.181818181818_r8) then
            if (r.le.0.55_r8) then
              h(i,j)=0.025_r8                      ! shelf
            elseif (r.ge.0.7_r8) then
              h(i,j)=0.125_r8                      ! deep
            else
              h(i,j)=0.125_r8-0.1_r8*
     &               (COS(0.5_r8*pi*(r-0.55_r8)/0.15_r8)**2)
            endif
          else
            if (r.le.val1) then
              h(i,j)=0.025_r8                      ! shelf
            elseif (r.ge.0.7_r8) then
              h(i,j)=0.125_r8                      ! deep
            else
              h(i,j)=0.125_r8-0.1_r8*
     &               (COS(0.5_r8*pi*(r-val1)/val2)**2)
            endif
          endif
        enddo
      enddo
# elif defined OVERFLOW
      val1=200.0_r8
      do j=JstrR,JendR
        do i=IstrR,IendR
          h(i,j)=val1+0.5_r8*(depth-val1)*
     &           (1.0_r8+TANH((yr(i,j)-100000.0_r8)/20000.0_r8))
        enddo
      enddo
# elif defined RIVERPLUME
      do j=JstrR,JendR
        do i=IstrR,MIN(5,IendR)
          h(i,j)=15.0_r8
        enddo
        do i=MAX(6,IstrR),IendR
          h(i,j)=depth+FLOAT(Lm-i)*(15.0_r8-depth)/FLOAT(Lm-6)
        enddo
      enddo
# elif defined SEAMOUNT
      do j=JstrR,JendR
        do i=IstrR,IendR
          val1=(xr(i,j)-xr(Lm/2,Mm/2))/40000.0_r8
          val2=(yr(i,j)-yr(Lm/2,Mm/2))/40000.0_r8
          h(i,j)=depth-4500.0_r8*EXP(-(val1*val1+val2*val2))
        enddo
      enddo
# elif defined SHELFRONT
      do j=JstrR,JendR
        do i=IstrR,IendR
          val1=yr(i,j)/1000.0_r8
          if (val1.lt.50.0_r8) then
            h(i,j)=50.0_r8+2.0_r8*val1
          elseif (val1.lt.60.0_r8) then
            h(i,j)=160.0_r8+1.5_r8*(val1-50.0_r8)**2
     &                      0.1_r8*(val1-60.0_r8)**2
          elseif (val1.lt.100.0_r8) then
            h(i,j)=310.0_r8+30.0_r8*(val1-60.0_r8)
          elseif (val1.lt.110.0_r8) then
            h(i,j)=1660.0_r8-1.5_r8*(val1-110.0_r8)**2
          else
            h(i,j)=1660.0_r8
          endif
        enddo
      enddo
# elif defined UPWELLING
      do j=JstrR,JendR
        if (j.le.Mm/2) then
          val1=FLOAT(j)
        else
          val1=FLOAT(Mm+1-j)
        endif
        val2=MIN(depth,84.5_r8+66.526_r8*TANH((val1-10.0_r8)/7.0_r8))
        do i=IstrR,IendR
          h(i,j)=val2
        enddo
      enddo
# elif defined WEDDELL
      val1=98.80_r8
      val2=0.8270_r8
      do k=-1,26
        xwrk(k)=FLOAT(k-1)*15.0_r8*1000.0_r8
        hwrk(k)=375.0_r8
      enddo
      do k=27,232
        zwrk=-2.0_r8+FLOAT(k-1)*0.020_r8
        xwrk(k)=(520.0_r8+val1+zwrk*val1+
     &           val1*val2*LOG(COSH(zwrk)))*1000.0_r8
        hwrk(k)=-75.0_r8+2198.0_r8*(1.0_r8+val2*TANH(zwrk))
      enddo
      do k=233,235
        xwrk(k)=(850.0_r8+FLOAT(k-228)*50.0_r8)*1000.0_r8
        hwrk(k)=4000.0_r8
      enddo
      do j=JstrR,JendR
        do i=IstrR,IendR
          do k=1,234
            if ((xwrk(k).le.xr(i,1)).and.(xr(i,1).lt.xwrk(k+1))) then
               cff=1.0_r8/(xwrk(k+1)-xwrk(k))
               h(i,j)=cff*(xwrk(k+1)-xr(i,j))*hwrk(k  )+
     &                cff*(xr(i,j)-xwrk(k  ))*hwrk(k+1)
            endif
          enddo
        enddo
      enddo
# else
      do j=JstrR,JendR
        do i=IstrR,IendR
          h(i,j)=depth
        enddo
      enddo
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        h(START_2D_ARRAY))
# endif
!
! Determine minimum depth: first, determine minimum values of depth
! within each subdomain (stored as private variable cff), then
! determine global minimum by comparing these  subdomain minima.
!
      cff=h(IstrR,JstrR)
      do j=JstrR,JendR
        do i=IstrR,IendR
          cff=MIN(cff,h(i,j))
        enddo
      enddo
      if (WESTERN_EDGE .and. EASTERN_EDGE .and.
     &    SOUTHERN_EDGE .and. NORTHERN_EDGE) then
        NSUB=0.5_r8
      else
        NSUB=FLOAT(NSUB_X*NSUB_E)-0.5_r8
      endif
      call my_setlock (lock)
      if (dia_count.lt.0.5_r8) then
        hmin=cff
      else
        hmin=MIN(hmin,cff)
      endif
      dia_count=dia_count+1.0_r8
      if (dia_count.gt.NSUB) then
        dia_count=0.0_r8
      endif
      call my_unsetlock (lock)
# ifdef ICESHELF
!
!---------------------------------------------------------------------
!  Set depth of ice shelf (meters; negative) at RHO-points.
!---------------------------------------------------------------------
!
#  ifdef WEDDELL
      val1=340.0_r8/16.0_r8
      do j=JstrR,JendR
        do i=IstrR,IendR
          if (i.gt.20) then
            zice(i,j)=0.0_r8
          elseif (i.gt.4) then
            zice(i,j)=-340.0_r8+FLOAT(i-1)*val1
          else
            zice(i,j)=-340.0_r8
          endif
        enddo
      enddo
#  else
      do j=JstrR,JendR
        do i=IstrR,IendR
          zice(i,j)=0.0_r8
        enddo
      enddo
#  endif
#  if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        zice(START_2D_ARRAY))
#  endif
# endif
      return
      end
#endif /* ANA_GRID */
#if defined ANA_HUMIDITY && defined BULK_FLUXES
      subroutine ana_humid (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets surface air humidity (moisture) using an        !
!  analytical expression.  There three types of humidity:            !
!                                                                    !
!     1) Absolute humidity: density of water vapor.                  !
!     2) Specific humidity: ratio of the mass of water vapor to      !
!        the mass of moist air cointaining the vapor (g/kg)          !
!     3) Relative humidity: ratio of the actual mixing ratio to      !
!        saturation mixing ratio of the air at given temperature     !
!        and pressure (percentage).                                  !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_humid_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_humid_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "forces.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set analytical surface air humidity.
!---------------------------------------------------------------------
!
# ifdef BL_TEST
      do j=JstrR,JendR
        do i=IstrR,IendR
          Hair(i,j)=0.776_r8
        enddo
      enddo
# else
      ANA_HUMIDITY: no values provided for HAIR.
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        Hair(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_HUMIDITY && BULK_FLUXES */
#ifdef ANA_INITIAL
      subroutine ana_initial (tile)
!
!=====================================================================
!                                                                    !
!  This subroutine sets initial conditions for momentum and tracer   !
!  type variables using analytical expressions.                      !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_initial_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_initial_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "ocean.h"
# include "scalars.h"
# ifdef  ANA_BRY
#  include "boundary.h"
# else
#  include "clima.h"
# endif
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, itrc, j, k
      REAL_TYPE
     &        val1, val2, val3, val4, x, x0, y, y0
     &   , cff, AmpU
  

!
# include "set_bounds.h"
!
      itrc=0
      k=0
      val1=0.0_r8
      val2=0.0_r8
      val3=0.0_r8
      val4=0.0_r8
      x=0.0_r8
      y=0.0_r8
      x0=0.0_r8
      y0=0.0_r8
!
!---------------------------------------------------------------------
!  Initial conditions for 2D momentum (m/s) components.
!---------------------------------------------------------------------
!
# if defined SOLITON
      x0=2.0_r8*xl/3.0_r8
      y0=0.5_r8*el
      val1=0.395_r8
      val2=0.771_r8*(val1*val1)
      do j=JstrR,JendR
        do i=Istr,IendR
          x=0.5_r8*(xr(i-1,j)+xr(i,j))-x0
          y=0.5_r8*(yr(i-1,j)+yr(i,j))-y0
          val3=EXP(-val1*x)
          val4=val2*((2.0_r8*val3/(1.0_r8+(val3*val3)))**2)
          ubar(i,j,1)=0.25_r8*val4*(6.0_r8*y*y-9.0_r8)*
     &                EXP(-0.5_r8*y*y)
          ubar(i,j,2)=ubar(i,j,1)
        enddo
      enddo
      do j=Jstr,JendR
        do i=IstrR,IendR
          x=0.5_r8*(xr(i,j-1)+xr(i,j))-x0
          y=0.5_r8*(yr(i,j-1)+yr(i,j))-y0
          val3=EXP(-val1*x)
          val4=val2*((2.0_r8*val3/(1.0_r8+(val3*val3)))**2)
          vbar(i,j,1)=2.0_r8*val4*y*(-2.0_r8*val1*TANH(val1*x))*
     &                EXP(-0.5_r8*y*y)
          vbar(i,j,2)=vbar(i,j,1)
        enddo
      enddo
# elif defined ISWAKE

      AmpU=0.15     ! Rich Signell non-stratified setup
      cff=AmpU/g
      x0=xl/2.
      y0=el/2.

      do j=jstrR,jendR
        do i=istrR,iendR
          zeta(i,j,1)=-cff*f(i,j)*(yr(i,j)-y0)
          ubar(i,j,1)=AmpU
          vbar(i,j,1)=0.
        enddo
      enddo
      do k=1,N
        do j=jstrR,jendR
          do i=istrR,iendR
            u(i,j,k,1)=ubar(i,j,1)
            v(i,j,k,1)=vbar(i,j,1)
            t(i,j,k,1,itemp)=0.
          enddo
        enddo
      enddo


#  ifdef ANA_BRY
      if (WESTERN_EDGE) then
#    ifdef M2_FRC_BRY
        do j=jstrR,jendR                     ! Save perimeter values
          zeta_west(j)=zeta(istr-1,j,1)      ! of the initial fields
          ubar_west(j)=ubar(istr  ,j,1)      ! to be used as external
        enddo
        do j=jstr,jendR
          vbar_west(j)=vbar(istr-1,j,1)      ! forcing data for side
        enddo                                ! boundaries.
#    endif
#    ifdef SOLVE3D
        do k=1,N
#     ifdef M3_FRC_BRY
          do j=jstrR,jendR
            u_west(j,k)=u(istr  ,j,k,1)
          enddo
          do j=jstr,jendR
            v_west(j,k)=v(istr-1,j,k,1)
          enddo
#     endif
#     ifdef T_FRC_BRY
          do j=jstrR,jendR
            t_west(j,k,itemp)=t(istr-1,j,k,1,itemp)
          enddo
#     endif
        enddo
#    endif
      endif


      if (EASTERN_EDGE) then
#    ifdef M2_FRC_BRY
        do j=jstrR,jendR
          zeta_east(j)=zeta(iend+1,j,1)
          ubar_east(j)=ubar(iend+1,j,1)
          vbar_east(j)=vbar(iend+1,j,1)
        enddo
#    endif
/* --->
#    ifdef SOLVE3D
        do k=1,N
          do j=jstrR,jendR
#     ifdef M3_FRC_BRY
            u_east(j,k)=u(iend+1,j,k,1)
            v_east(j,k)=v(iend+1,j,k,1)
#     endif
#     ifdef T_FRC_BRY
            t_east(j,k,itemp)=t(iend+1,j,k,1,itemp)
#     endif
          enddo
        enddo
#    endif
---> */
      endif


#    ifdef OBC_SOUTH
      if (SOUTHERN_EDGE) then
#     ifdef M2_FRC_BRY
        do i=istrR,iendR
          zeta_south(i)=zeta(i,jstr-1,1)
          ubar_south(i)=ubar(i,jstr-1,1)
          vbar_south(i)=vbar(i,jstr  ,1)
        enddo
#     endif
#     ifdef SOLVE3D
        do k=1,N
          do i=istrR,iendR
#      ifdef M3_FRC_BRY
            u_south(i,k)=u(i,jstr-1,k,1)
            v_south(i,k)=v(i,jstr  ,k,1)
#      endif
#      ifdef T_FRC_BRY
            t_south(i,k,itemp)=t(i,jstr-1,k,1,itemp)
#      endif
          enddo
        enddo
#     endif
      endif
#    endif

#    ifdef OBC_NORTH
      if (NORTHERN_EDGE) then
#     ifdef M2_FRC_BRY
        do i=istrR,iendR
          zeta_north(i)=zeta(i,jend+1,1)
          ubar_north(i)=ubar(i,jend+1,1)
          vbar_north(i)=vbar(i,jend+1,1)
        enddo
#     endif
#     ifdef SOLVE3D
        do k=1,N
          do i=istrR,iendR
#      ifdef M3_FRC_BRY
            u_north(i,k)=u(i,jend+1,k,1)
            v_north(i,k)=v(i,jend+1,k,1)
#      endif
#      ifdef T_FRC_BRY
            t_north(i,k,itemp)=t(i,jend+1,k,1,itemp)
#      endif
          enddo
        enddo
#     endif
      endif
#    endif

#  else
#   ifdef UCLIMATOLOGY
      do j=jstrR,jendR                   ! Save initial data into
        do i=istrR,iendR                 ! climatology for the
          ubclm(i,j)=ubar(i,j,1)         ! subsequent use as inflow
          vbclm(i,j)=vbar(i,j,1)         ! boundary conditions
        enddo
      enddo
#   endif
#   ifdef SOLVE3D
      do k=1,N
        do j=jstrR,jendR
          do i=istrR,iendR
#    ifdef UCLIMATOLOGY
            uclm(i,j,k)=u(i,j,k,1)
            vclm(i,j,k)=v(i,j,k,1)
#    endif
#    ifdef TCLIMATOLOGY
            tclm(i,j,k,itemp)=t(i,j,k,1,itemp)
#    endif
          enddo
        enddo
      enddo
#   endif /* SOLVE3D */
#  endif /* ANA_BRY */



# elif defined SED_TEST1
      val1=100.0_r8
      do j=JstrR,JendR
        do i=Istr,IendR
          ubar(i,j,1)=-10.0_r8/(10.0_r8+9_e8-6*FLOAT(i)*val1)
          ubar(i,j,2)=ubar(i,j,1)
        enddo
      enddo
      do j=Jstr,JendR
        do i=IstrR,IendR
          vbar(i,j,1)=0.0_r8
          vbar(i,j,2)=vbar(i,j,1)
        enddo
      enddo
# else
      do j=JstrR,JendR
        do i=Istr,IendR
          ubar(i,j,1)=0.0_r8
          ubar(i,j,2)=ubar(i,j,1)
        enddo
      enddo
      do j=Jstr,JendR
        do i=IstrR,IendR
          vbar(i,j,1)=0.0_r8
          vbar(i,j,2)=vbar(i,j,1)
        enddo
      enddo
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      do i=1,2
        call exchange_u2d_tile (Istr,Iend,Jstr,Jend,
     &                        ubar(START_2D_ARRAY,i))
        call exchange_v2d_tile (Istr,Iend,Jstr,Jend,
     &                        vbar(START_2D_ARRAY,i))
      enddo
# endif
!
!---------------------------------------------------------------------
!  Initial conditions for free-surface (m).
!---------------------------------------------------------------------
!
# if defined SOLITON
      x0=2.0_r8*xl/3.0_r8
      y0=0.5_r8*el
      val1=0.395_r8
      val2=0.771_r8*(val1*val1)
      do j=JstrR,JendR
        do i=IstrR,IendR
          x=xr(i,j)-x0
          y=yr(i,j)-y0
          val3=EXP(-val1*x)
          val4=val2*((2.0_r8*val3/(1.0_r8+(val3*val3)))**2)
          zeta(i,j,1)=0.25_r8*val4*(6.0_r8*y*y+3.0_r8)*
     &                EXP(-0.5_r8*y*y)
          zeta(i,j,2)=zeta(i,j,1)
        enddo
      enddo
#elif defined SED_TEST1
      val1=100.0_r8
      do j=JstrR,JendR
        do i=IstrR,IendR
          zeta(i,j,1)=9_e8-6*FLOAT(i)*val1
          zeta(i,j,2)=zeta(i,j,1)
        enddo
      enddo
# else
      do j=JstrR,JendR
        do i=IstrR,IendR
          zeta(i,j,1)=0.0_r8
          zeta(i,j,2)=zeta(i,j,1)
        enddo
      enddo
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      do i=1,2
        call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                          zeta(START_2D_ARRAY,i))
      enddo
# endif
# ifdef SOLVE3D
!
!---------------------------------------------------------------------
!  Initial conditions for 3D momentum components (m/s).
!---------------------------------------------------------------------
!
#  if defined SED_TEST1
      do k=1,N
       do j=JstrR,JendR
         do i=Istr,IendR
            u(i,j,k,1)=-1.0_r8*LOG((h(i,j)+z_r(i,j,k))/Zob)/
     &                 (LOG(h(i,j)/Zob)-1.0_r8+Zob/h(i,j))
            u(i,j,k,2)=u(i,j,k,1)
          enddo
        enddo
        do j=Jstr,JendR
          do i=IstrR,IendR
            v(i,j,k,1)=0.0_r8
            v(i,j,k,2)=v(i,j,k,1)
          enddo
        enddo
      enddo
#  elif defined ISWAKE
       ! defined above
#  else
      do k=1,N
       do j=JstrR,JendR
         do i=Istr,IendR
            u(i,j,k,1)=0.0_r8
            u(i,j,k,2)=u(i,j,k,1)
          enddo
        enddo
        do j=Jstr,JendR
          do i=IstrR,IendR
            v(i,j,k,1)=0.0_r8
            v(i,j,k,2)=v(i,j,k,1)
          enddo
        enddo
      enddo
#  endif
#  if defined EW_PERIODIC || defined NS_PERIODIC
      do i=1,2
        call exchange_u3d_tile (Istr,Iend,Jstr,Jend,
     &                          u(START_2D_ARRAY,1,i))
        call exchange_v3d_tile (Istr,Iend,Jstr,Jend,
     &                          v(START_2D_ARRAY,1,i))
      enddo
#  endif
!
!---------------------------------------------------------------------
!  Initial conditions for tracer type variables.
!---------------------------------------------------------------------
!
!  Set initial conditions for potential temperature (Celsius) and
!  salinity (PSU).
!
#  ifdef BASIN
      val1=(44.69_r8/39.382_r8)**2
      val2=val1*(rho0*800.0_r8/g)*(5.0_e8-5/((42.689_r8/44.69_r8)**2))
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=val2*EXP(z_r(i,j,k)/800.0_r8)*
     &                       (0.6_r8-0.4_r8*TANH(z_r(i,j,k)/800.0_r8))
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
          enddo
        enddo
      enddo
#  elif defined BL_TEST
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            val1=TANH(1.1_r8*z_r(i,j,k)+11.0_r8)
            t(i,j,k,1,itemp)=17.25_r8+6.25_r8*val1
            t(i,j,k,1,isalt)=31.0_r8-0.75_r8*val1
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
            t(i,j,k,2,isalt)=t(i,j,k,1,isalt)
          enddo
        enddo
      enddo
#  elif defined CANYON_A
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=T0
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
          enddo
        enddo
      enddo
#  elif defined CANYON_B
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=3.488_r8*EXP(z_r(i,j,k)/800.0_r8)*
     &                       (1.0_r8-(2.0_r8/3.0_r8)*
     &                               TANH(z_r(i,j,k)/800.0_r8))
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
          enddo
        enddo
      enddo
#  elif defined FLT_TEST
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=T0
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
          enddo
        enddo
      enddo
#  elif defined GRAV_ADJ
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,MIN(L/2,IendR)
            t(i,j,k,1,itemp)=T0+5.0_r8
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
          enddo
          do i=MAX(L/2+1,IstrR),IendR
            t(i,j,k,1,itemp)=T0
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
          enddo
!!        do i=IstrR,IendR
!!         if (i.lt.Lm/2) then
!!            t(i,j,k,1,itemp)=T0+5.0_r8
!!            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
!!         elseif (i.eq.Lm/2) then
!!            t(i,j,k,1,itemp)=T0+4.0_r8
!!            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
!!          elseif (i.eq.Lm/2+1) then
!!            t(i,j,k,1,itemp)=T0+1.0_r8
!!            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
!!          else
!!            t(i,j,k,1,itemp)=T0
!!            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
!!          endif
!!        enddo
        enddo
      enddo
#  elif defined LAB_CANYON
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=-659.34183_r8*z_r(i,j,k)
          enddo
        enddo
      enddo
#  elif defined LMD_TEST
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=MIN(13.0_r8,
     &                           7.0_r8+0.2_r8*(z_r(i,j,k)+50.0_r8))
            t(i,j,k,1,isalt)=35.0_r8
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
            t(i,j,k,2,isalt)=t(i,j,k,1,isalt)
          enddo
        enddo
      enddo
#  elif defined NJ_BIGHT
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            if (z_r(i,j,k).ge.-15.0_r8) then
              t(i,j,k,1,itemp)= 2.049264257728403_e8+01-z_r(i,j,k)*
     &                         (2.640850848793918_e8-01+z_r(i,j,k)*
     &                         (2.751125328535212_e8-01+z_r(i,j,k)*
     &                         (9.207489761648872_e8-02+z_r(i,j,k)*
     &                         (1.449075725742839_e8-02+z_r(i,j,k)*
     &                         (1.078215685912076_e8-03+z_r(i,j,k)*
     &                         (3.240318053903974_e8-05+
     &                          1.262826857690271_e8-07*z_r(i,j,k)))))))
              t(i,j,k,1,isalt)= 3.066489149193135_e8+01-z_r(i,j,k)*
     &                         (1.476725262946735_e8-01+z_r(i,j,k)*
     &                         (1.126455760313399_e8-01+z_r(i,j,k)*
     &                         (3.900923281871022_e8-02+z_r(i,j,k)*
     &                         (6.939014937447098_e8-03+z_r(i,j,k)*
     &                         (6.604436696792939_e8-04+z_r(i,j,k)*
     &                         (3.191792361954220_e8-05+
     &                          6.177352634409320_e8-07*z_r(i,j,k)))))))
            else
               t(i,j,k,1,itemp)=14.6_r8+
     &                          6.70_r8*TANH(1.1_r8*z_r(i,j,k)+15.9_r8)
               t(i,j,k,1,isalt)=31.3_r8-
     &                          0.55_r8*TANH(1.1_r8*z_r(i,j,k)+15.9_r8)
            endif
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
            t(i,j,k,2,isalt)=t(i,j,k,1,isalt)
          enddo
        enddo
      enddo
#  elif defined OVERFLOW
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=T0-0.5_r8*T0*(1.0_r8+
     &                       TANH((yr(i,j)-60000.0_r8)/2000.0_r8))
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
          enddo
        enddo
      enddo
#  elif defined RIVERPLUME
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=T0+0.01_r8*FLOAT(k)
            t(i,j,k,1,isalt)=S0
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
            t(i,j,k,2,isalt)=t(i,j,k,1,isalt)
          enddo
        enddo
      enddo
#  elif defined SEAMOUNT
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=T0+7.5_r8*EXP(z_r(i,j,k)/1000.0_r8)
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
          enddo
        enddo
      enddo
#  elif defined SHELFRONT
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=T0+2.5_r8*
     &                          TANH((yr(i,j)-50000.0_r8)/20000.0_r8)
            t(i,j,k,1,isalt)=S0+
     &                          TANH((yr(i,j)-50000.0_r8)/20000.0_r8)
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
            t(i,j,k,2,isalt)=t(i,j,k,1,isalt)
          enddo
        enddo
      enddo
#  elif defined SED_TEST1
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=20.0_r8
            t(i,j,k,1,isalt)=0.0_r8
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
            t(i,j,k,2,isalt)=t(i,j,k,1,isalt)
          enddo
        enddo
      enddo
#  elif defined UPWELLING
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=14.0_r8+8.0_r8*EXP(z_r(i,j,k)/50.0_r8)-T0
!!          t(i,j,k,1,itemp)=14.0_r8+(z_r(i,j,k)+75.0_r8)/150.0_r8+
!!   &                       4.0_r8*(1.0_r8+TANH((z_r(i,j,k)+35.0_r8)/
!!   &                                           6.5_r8))
            t(i,j,k,1,isalt)=1.0_e8-4*yr(i,j)-S0
!!          t(i,j,k,1,isalt)=S0
!!          if (j.lt.Mm/2) then
!!            t(i,j,k,1,isalt)=0.0_r8
!!          elseif (j.eq.Mm/2) then
!!            t(i,j,k,1,isalt)=0.5_r8
!!          elseif (j.gt.Mm/2) then
!!            t(i,j,k,1,isalt)=1.0_r8
!!          endif
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
            t(i,j,k,2,isalt)=t(i,j,k,1,isalt)
          enddo
        enddo
      enddo
#  elif defined ISWAKE
    ! defined above
#  else
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            t(i,j,k,1,itemp)=T0
            t(i,j,k,1,isalt)=S0
            t(i,j,k,2,itemp)=t(i,j,k,1,itemp)
            t(i,j,k,2,isalt)=t(i,j,k,1,isalt)
          enddo
        enddo
      enddo
#  endif
#  if defined EW_PERIODIC || defined NS_PERIODIC
      do itrc=1,NAT
        do i=1,2
          call exchange_r3d_tile (Istr,Iend,Jstr,Jend,
     &                            t(START_2D_ARRAY,1,i,itrc))
        enddo
      enddo
#  endif
# endif /* SOLVE3D */
      return
      end
#endif /* ANA_INITIAL */
#if defined ANA_M2CLIMA && defined M2CLIMATOLOGY
      subroutine ana_m2clima (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets analytical 2D momentum climatology fields.      !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_m2clima_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_m2clima_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "clima.h"
# include "grid.h"
# include "ocean.h"
# include "mask.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set 2D momentum climatology.
!---------------------------------------------------------------------
!
      do j=JstrR,JendR
        do i=Istr,IendR
          ubarclm(i,j)=???
        enddo
      enddo
      do j=Jstr,JendR
        do i=IstrR,IendR
          vbarclm(i,j)=???
        enddo
      enddo
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_u2d_tile (Istr,Iend,Jstr,Jend,
     &                        ubarclm(START_2D_ARRAY))
      call exchange_v2d_tile (Istr,Iend,Jstr,Jend,
     &                        vbarclm(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_M2CLIMA && M2CLIMATOLOGY */
#ifdef ANA_M2OBC
      subroutine ana_m2obc (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets 2D momentum open boundary conditions using      !
!  analytical expressions.                                           !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_m2obc_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_m2obc_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "boundary.h"
# include "grid.h"
# include "ocean.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
      REAL_TYPE
     &        angle, fac, major, minor, omega, phase, val
!
# include "set_bounds.h"
!
      i=0
      j=0
      fac=0.0_r8
      omega=0.0_r8
      angle=0.0_r8
      major=0.0_r8
      minor=0.0_r8
      phase=0.0_r8
      val=0.0_r8
!
!---------------------------------------------------------------------
!  2D momentum open boundary conditions.
!---------------------------------------------------------------------
!
# if defined NJ_BIGHT
      if (EASTERN_EDGE) then
        do j=JstrR,JendR
          ubar_east(j)=0.0_r8
        enddo
        do j=Jstr,JendR
          vbar_east(j)=0.0_r8
        enddo
      endif
      if (SOUTHERN_EDGE) then
        do i=Istr,IendR
          ubar_south(i)=0.0_r8
        enddo
        do i=IstrR,IendR
          vbar_south(i)=0.0_r8
        enddo
      endif
# elif defined SED_TEST1
      if (WESTERN_EDGE) then
        do j=JstrR,JendR
          val=0.5_r8*(zeta(0 ,j,knew)+h(0 ,j)+
     &                zeta(1 ,j,knew)+h(1 ,j))
          ubar_west(j)=-10.0_r8/val
        enddo
        do j=Jstr,JendR
          vbar_west(j)=0.0_r8
        enddo
      endif
      if (EASTERN_EDGE) then
        do j=JstrR,JendR
          val=0.5_r8*(zeta(Lm,j,knew)+h(Lm,j)+
     &                zeta(L ,j,knew)+h(L ,j))
          ubar_east(j)=-10.0_r8/val2
        enddo
        do j=Jstr,JendR
          vbar_east(j)=0.0_r8
        enddo
      endif
# elif defined WEDDELL
      if (WESTERN_EDGE) then
        fac=TANH((tdays-dstart)/1.0_r8)
        omega=2.0_r8*pi*time/(12.42_r8*3600.0_r8)    !  M2 Tide period
        minor=0.0143_r8+(0.0143_r8+0.010_r8)/FLOAT(L)
        major=0.1144_r8+(0.1144_r8-0.013_r8)/FLOAT(L)
        phase=(318.0_r8+(318.0_r8-355.0_r8)/FLOAT(L))*deg2rad
        angle=(125.0_r8+(125.0_r8- 25.0_r8)/FLOAT(L))*deg2rad
        do j=JstrR,JendR
          val=0.5_r8*(angler(0 ,j)+angler(1,j))
          ubar_west(j)=fac*(major*COS(angle-val)*COS(omega-phase)-
     &                      minor*SIN(angle-val)*SIN(omega-phase))
        enddo
        do j=Jstr,JendR
          val=0.5_r8*(angler(0,j-1)+angler(0,j))
          vbar_west(j)=fac*(major*SIN(angle-val)*COS(omega-phase)-
     &                      minor*SIN(angle-val)*COS(omega-phase))
        enddo
      endif
      if (EASTERN_EDGE) then
        fac=TANH((tdays-dstart)/1.0_r8)
        omega=2.0_r8*pi*time/(12.42_r8*3600.0_r8)    !  M2 Tide period
        minor=0.0143_r8+(0.0143_r8+0.010_r8)
        major=0.1144_r8+(0.1144_r8-0.013_r8)
        phase=(318.0_r8+(318.0_r8-355.0_r8))*deg2rad
        angle=(125.0_r8+(125.0_r8- 25.0_r8))*deg2rad
        do j=JstrR,JendR
          val=0.5_r8*(angler(Lm,j)+angler(L,j))
          ubar_east(j)=fac*(major*COS(angle-val)*COS(omega-phase)-
     &                      minor*SIN(angle-val)*SIN(omega-phase))
        enddo
        do j=Jstr,JendR
          val=0.5_r8*(angler(L,j-1)+angler(L,j))
          vbar_east(j)=fac*(major*SIN(angle-val)*COS(omega-phase)-
     &                      minor*SIN(angle-val)*COS(omega-phase))
        enddo
      endif
# elif defined ISWAKE
! nothing, because it is defined by ana_initial, and kept constant.
# else
      ANA_M2OBC: no values provided for 2D momentum.
# endif
      return
      end
#endif /* ANA_M2OBC */
#if defined ANA_M3CLIMA && defined M3CLIMATOLOGY
      subroutine ana_m3clima (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets analytical 3D momentum climatology fields.      !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_m3clima_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_m3clima_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "clima.h"
# include "grid.h"
# include "ocean.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set 3D momentum climatology.
!---------------------------------------------------------------------
!
      do k=1,N
        do j=JstrR,JendR
          do i=Istr,IendR
            uclm(i,j,k)=???
          enddo
        enddo
        do j=Jstr,JendR
          do i=IstrR,IendR
            vclm(i,j,k)=???
          enddo
        enddo
      enddo
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_u3d_tile (Istr,Iend,Jstr,Jend,
     &                        uclm(START_2D_ARRAY,1))
      call exchange_v3d_tile (Istr,Iend,Jstr,Jend,
     &                        vclm(START_2D_ARRAY,1))
# endif
      return
      end
#endif /* ANA_M3CLIMA && M3CLIMATOLOGY */
#ifdef ANA_M3OBC
      subroutine ana_m3obc (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets 3D momentum open boundary conditions using      !
!  analytical expressions.                                           !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_m3obc_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_m3obc_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "boundary.h"
# include "grid.h"
# include "ocean.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
      REAL_TYPE
     &        fac, val
!
# include "set_bounds.h"
!
      i=0
      j=0
      fac=0.0_r8
      val=0.0_r8
!
!---------------------------------------------------------------------
!  3D momentum open boundary conditions.
!---------------------------------------------------------------------
!
# if defined SED_TEST1
      if (WESTERN_EDGE) then
        fac=5.0_e8-6
        do k=1,N
          do j=JstrR,JendR
            val=0.5_r8*(zeta(0 ,j,knew)+h(0 ,j)+
     &                  zeta(1 ,j,knew)+h(1 ,j))
            u_west(j,k)=-LOG((val+0.5*(z_r(0,j,k)+z_r(1,j,k)))/fac)/
     &                  (LOG(val/fac)-1.0_r8+fac/val)
          enddo
          do j=Jstr,JendR
            v_west(j,k)=0.0_r8
          enddo
        enddo
      endif
      if (EASTERN_EDGE) then
        fac=5.0_e8-6
        do k=1,N
          do j=JstrR,JendR
            val=0.5_r8*(zeta(Lm,j,knew)+h(Lm,j)+
     &                  zeta(L ,j,knew)+h(L ,j))
            u_east(j,k)=-LOG((val+0.5*(z_r(Lm,j,k)+z_r(L,j,k)))/fac)/
     &                  (LOG(val/fac)-1.0_r8+fac/val)
          enddo
          do j=Jstr,JendR
            v_east(j,k)=0.0_r8
          enddo
        enddo
      endif
# elif defined ISWAKE
! nothing, because it is defined by ana_initial, and kept constant.
# else
      ANA_M3OBC: no values provided for 3D momentum.
# endif
      return
      end
#endif /* ANA_M3OBC */
#if defined ANA_MASK && defined MASKING
      subroutine ana_mask (tile)
!
!=====================================================================
!                                                                    !
!  This subroutine sets analytical Land/Sea masking.                 !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "scratch.h"
# include "tile.h"
!
      call ana_mask_tile (Istr,Iend,Jstr,Jend, A2d(1,1))
      return
      end
!
!*********************************************************************
      subroutine ana_mask_tile (Istr,Iend,Jstr,Jend, mask)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "mask.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
      REAL_TYPE
     &        mask(PRIVATE_2D_SCRATCH_ARRAY)
     &   , rd_inner, rd, Xsize,Esize, x0,y0
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set Land/Sea mask of RHO-points: Land=0, Sea=1.
!---------------------------------------------------------------------
!
!  Notice that private scratch array "mask" is used to allow
!  computation within a parallel loop.
!
# if defined FLT_TEST
      do j=Jstr-2,Jend+2
        do i=Istr-2,Iend+2
          mask(i,j)=1.0_r8
          if (j.eq.1 ) mask(i,j)=0.0_r8
          if (j.eq.Mm) mask(i,j)=0.0_r8
          if (i.ge.L/2.and.i.le.L/2+1.and.
     &        j.ge.M/2.and.j.le.M/2+1) mask(i,j)=0.0_r8
        enddo
      enddo
# elif defined RIVERPLUME
      do j=Jstr-2,Jend+2
        do i=Istr-2,Iend+2
          mask(i,j)=1.0_r8
        enddo
      enddo
      do i=Istr-2,MIN(5,Iend+2)
        do j=Jstr-2,MIN(M-19,Jend+2)
          mask(i,j)=0.0_r8
        enddo
        do j=MAX(Jstr-2,M-17),Jend+2
          mask(i,j)=0.0_r8
        enddo
      enddo
# elif defined ISWAKE
!  Notice that private scratch array "mask" is used to allow
!  computation within a parallel loop.
!



      Esize=80.0e+3                               ! width of channel
      Xsize=Esize*dble(Lm)/dble(Mm)               ! length of channel


      y0=Esize/2.                ! y location of island center
      x0=y0                      ! x location of island center
      rd_inner=10.e+3            ! radius of island

      DO j=Jstr-1,Jend+1
        DO i=Istr-1,Iend+1
          mask(i,j)=1.0_r8
          rd=sqrt((xr(i,j)-x0)**2+(yr(i,j)-y0)**2)
          if (rd.le.rd_inner) then
            mask(i,j)=0.0_r8
          endif
        END DO
      END DO
!
      DO j=JstrR,JendR
        DO i=IstrR,IendR
            rmask(i,j)=mask(i,j)
        END DO
      END DO
!
!--------------------------------------------------------------------
!  Compute Land/Sea mask of U- and V-points.
!--------------------------------------------------------------------
!
      DO j=JstrR,JendR
        DO i=Istr,IendR
          umask(i,j)=mask(i-1,j)*mask(i,j)
        END DO
      END DO
      DO j=Jstr,JendR
        DO i=IstrR,IendR
          vmask(i,j)=mask(i,j-1)*mask(i,j)
        END DO
      END DO
!
!--------------------------------------------------------------------
!  Compute Land/Sea mask of PSI-points.
!--------------------------------------------------------------------
!
      DO j=Jstr,JendR
        DO i=Istr,IendR
          pmask(i,j)=mask(i-1,j-1)*mask(i,j-1)
     &              *mask(i-1,j  )*mask(i,j  )
        END DO
      END DO

# else
      ANA_MASK: no values provided for RMASK.
# endif
!
      do j=JstrR,JendR
        do i=IstrR,IendR
          rmask(i,j)=mask(i,j)
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute Land/Sea mask of U- and V-points.
!---------------------------------------------------------------------
!
      do j=JstrR,JendR
        do i=Istr,IendR
          umask(i,j)=mask(i-1,j)*mask(i,j)
        enddo
      enddo
      do j=Jstr,JendR
        do i=IstrR,IendR
          vmask(i,j)=mask(i,j-1)*mask(i,j)
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Compute Land/Sea mask of PSI-points.  If applicable,  set
!  slipperiness mask (Sea=1, Land=0, boundary=1-gamma2).
!---------------------------------------------------------------------
!
      do j=Jstr,JendR
        do i=Istr,IendR
          pmask(i,j)=mask(i-1,j-1)*mask(i,j-1)*
     &               mask(i-1,j  )*mask(i,j  )
        enddo
      enddo
      if (gamma2.lt.0.0_r8) then
        do j=Jstr,JendR
          do i=Istr,IendR
            pmask(i,j)=2.0_r8-pmask(i,j)
          enddo
        enddo
      endif
      return
      end
#endif /* ANA_MASK && MASKING */
#if defined ANA_PAIR && defined BULK_FLUXES
      subroutine ana_pair (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets surface air pressure (mb) using an analytical   !
!  expression.                                                       !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_pair_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_pair_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "forces.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set analytical surface air pressure (mb).
!  (1 mb = 100 Pa = 1 hPa,  1 bar = 1.0e+5 N/m2 = 1.0e+5 dynes/cm2).
!---------------------------------------------------------------------
!
# ifdef BL_TEST
      do j=JstrR,JendR
        do i=IstrR,IendR
          Pair(i,j)=1013.48_r8
        enddo
      enddo
# else
      ANA_PAIR: no values provided for PAIR.
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        Pair(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_PAIR && BULK_FLUXES */
#if (defined TS_PSOURCE || defined UV_PSOURCE) && defined ANA_PSOURCE
      subroutine ana_psource
!
!=====================================================================
!                                                                    !
!  This subroutine sets analytical tracer and mass point Sources     !
!  and/or Sinks.  River runoff can be consider as a point source.    !
!                                                                    !
!=====================================================================
!
      implicit none
# include "param.h"
# include "grid.h"
# include "ocean.h"
# include "scalars.h"
# include "sources.h"
!
      INTEGER_TYPE
     &        is, i, j, k
      REAL_TYPE
     &        fac, my_area
!
      i=0
      j=0
      k=0
      my_area=0.0_r8
!
!---------------------------------------------------------------------
!  Set tracer and/or mass point sources and/or sink.
!---------------------------------------------------------------------
!
      if (iic.eq.ntstart) then
!
!  Set-up point Sources/Sink number (Nsrc), direction (Dsrc), I- and
!  J-grid locations (Isrc,Jsrc), and logical switch for type of tracer
!  to apply (Lsrc). Currently, the direction can be along XI-direction
!  (Dsrc = 0) or along ETA-direction (Dsrc > 0).  The mass sources are
!  located at U- or V-points so the grid locations should range from
!  1 =< Isrc =< L  and  1 =< Jsrc =< M.
!
# ifdef RIVERPLUME
        Nsrc=1
        Dsrc(Nsrc)=0.0_r8
        Isrc(Nsrc)=1
        Jsrc(Nsrc)=50
        Lsrc(Nsrc,itemp)=.true.
        Lsrc(Nsrc,isalt)=.true.
# elif defined SED_TEST1
        Nsrc=Mm*2
        do is=1,Nsrc/2
          Dsrc(is)=0.0_r8
          Isrc(is)=1
          Jsrc(is)=is
          Lsrc(is,itemp)=.false.
          Lsrc(is,isalt)=.false.
        enddo
        do is=Nsrc/2+1,Nsrc
          Dsrc(is)=0.0_r8
          Isrc(is)=L
          Jsrc(is)=is-Mm
          Lsrc(is,itemp)=.false.
          Lsrc(is,isalt)=.false.
        enddo
# else
        ANA_PSOURCE: no values provided for Dsrc, Isrc, Jsrc, Lsrc.
# endif
      endif
# ifdef SOLVE3D
!
!  If appropriate, set-up nondimensional shape function to distribute
!  mass point sources/sinks vertically.  It most add to unity!!.
!
#  if defined SED_TEST1
        do k=1,N
          do is=1,Nsrc
            i=Isrc(is)
            j=Jsrc(is)
            Qshape(is,k)=ABS(u(i,j,k,nnew)/ubar(i,j,knew))*
     &                   (z_w(i-1,Mm/2,k)-z_w(i-1,Mm/2,k-1)+
     &                    z_w(i  ,Mm/2,k)-z_w(i  ,Mm/2,k-1))/
     &                   (z_w(i-1,Mm/2,N)-z_w(i-1,Mm/2,0  )+
     &                    z_w(i  ,Mm/2,N)-z_w(i  ,Mm/2,0  ))
          enddo
        enddo
#  else
        do k=1,N
          do is=1,Nsrc
            Qshape(is,k)=1.0_r8/FLOAT(N)
          enddo
        enddo
#  endif
# endif /* SOLVE3D */
# ifdef UV_PSOURCE
!
!  Set-up vertically integrated mass transport (m3/s) of point
!  Sources/Sinks (positive in the positive U- or V-direction and
!  viceversa).
!
#  if defined RIVERPLUME
      if ((tdays-dstart).lt.0.5_r8) then
        fac=1.0_r8+TANH((time-43200.0_r8)/43200.0_r8)
      else
        fac=1.0_r8
      endif
      do is=1,Nsrc
        Qbar(is)=fac*1500.0_r8
      enddo
#  elif defined SED_TEST1
      my_area=0.0_r8         ! West end
      do is=1,Nsrc/2
        i=Isrc(is)
        j=Jsrc(is)
        my_area=my_area+0.5_r8*(zeta(i-1,j,knew)+h(i-1,j)+
     &                          zeta(i  ,j,knew)+h(i  ,j))*on_u(i,j)
      enddo
      fac=-1000.0_r8*10.0_r8*1.0_r8
      do is=1,Nsrc/2
        i=Isrc(is)
        j=Jsrc(is)
        Qbar(is)=fac*(0.5_r8*(zeta(i-1,j,knew)+h(i-1,j)+
     &                        zeta(i  ,j,knew)+h(i  ,j)))*
     &           on_u(i,j)/my_area
      enddo
      my_area=0.0_r8         ! East end
      do is=Nsrc/2+1,Nsrc
        i=Isrc(is)
        j=Jsrc(is)
        my_area=my_area+0.5_r8*(zeta(i-1,j,knew)+h(i-1,j)+
     &                          zeta(i  ,j,knew)+h(i  ,j))*on_u(i,j)
      enddo
      fac=-1000.0_r8*10.0_r8*1.0_r8
      do is=Nsrc/2+1,Nsrc
        i=Isrc(is)
        j=Jsrc(is)
        Qbar(is)=fac*(0.5_r8*(zeta(i-1,j,knew)+h(i-1,j)+
     &                        zeta(i  ,j,knew)+h(i  ,j)))*
     &           on_u(i,j)/my_area
      enddo
#  else
      ANA_PSOURCE: no values provided for Qbar.
#  endif
#  ifdef SOLVE3D
!
!  Set-up mass transport profile (m3/s) of point Sources/Sinks.
!
      do k=1,N
        do is=1,Nsrc
          Qsrc(is,k)=Qbar(is)*Qshape(is,k)
        enddo
      enddo
#  endif /* SOLVE3D */
# endif /* UV_PSOURCE */
# if defined TS_PSOURCE && defined SOLVE3D
!
!  Set-up tracer (tracer units) point Sources/Sinks.
!
#  ifdef RIVERPLUME
      do k=1,N
        do is=1,Nsrc
          Tsrc(is,k,itemp)=T0
          Tsrc(is,k,isalt)=0.0_r8
        enddo
      enddo
#  else
      ANA_PSOURCE: no values provided for Tsrc.
#  endif
# endif /* TS_PSOURCE && SOLVE3D */
      return
      end
#endif /* (TS_PSOURCE || UV_PSOURCE) && ANA_PSOURCE */
#if defined ANA_RAIN && defined BULK_FLUXES
      subroutine ana_rain (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets precipitation rate (kg/m2/s) using an           !
!  analytical expression.                                            !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_rain_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_rain_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "forces.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set analytical precipitation rate (kg/m2/s).
!---------------------------------------------------------------------
!
      do j=JstrR,JendR
        do i=IstrR,IendR
          rain(i,j)=0.0_r8
        enddo
      enddo
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        rain(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_RAIN && BULK_FLUXES */
#if defined ANA_SEDIMENT && defined SEDIMENT
      subroutine ana_sediment (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets initial conditions for  sediment tracer fields  !
!  concentrations (mg/l) using analytical expressions. It also sets  !
!  initial bed conditions in each sediment layer.                    !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_sediment_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_sediment_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "ocean.h"
# include "scalars.h"
# include "sediment.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, itrc, j, k
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Initial sediment concentrations in the water column.
!---------------------------------------------------------------------
!
      do itrc=1,NST
        do k=1,N
          do j=JstrR,JendR
            do i=IstrR,IendR
              t(i,j,k,1,idsed(itrc))=Csed(itrc)
              t(i,j,k,2,idsed(itrc))=t(i,j,k,1,idsed(itrc))
            enddo
          enddo
        enddo
      enddo
!
!---------------------------------------------------------------------
!  Initial bed layer thicknesses (m).
!---------------------------------------------------------------------
!
# ifdef SED_TEST1
      do k=1,Nbed
        do j=JstrR,JendR
          do i=IstrR,IendR
            bed(i,j,k,ithck)=1.0_r8
          enddo
        enddo
      enddo
# else
      do k=1,Nbed
        do j=JstrR,JendR
          do i=IstrR,IendR
            bed(i,j,k,ithck)=???
          enddo
        enddo
      enddo
      ANA_SEDIMENT: no values provided for bed(:,:,:,ithck)
# endif
      return
      end
#endif /* ANA_SEDIMENT && SEDIMENT */
#ifdef ANA_SMFLUX
      subroutine ana_smflux (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets kinematic surface momentum flux (wind stress)   !
!  "sustr" and "svstr" (m2/s2) using an analytical expression.       !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_smflux_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_smflux_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "ocean.h"
# include "forces.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
      REAL_TYPE
     &        Ewind, Nwind, val1, val2, windamp, winddir
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set kinematic surface momentum flux (wind stress) component in the
!  XI-direction (m2/s2) at horizontal U-points.
!---------------------------------------------------------------------
!
      Ewind=0.0_r8
      Nwind=0.0_r8
      windamp=0.0_r8
      winddir=0.0_r8
      val1=0.0_r8
      val2=0.0_r8
!
# ifdef BASIN
      val1=5.0_e8-5*(1.0_r8+TANH((time-6.0_r8*86400.0_r8)/
     &              (3.0_r8*86400.0_r8)))
      val2=2.0_r8*pi/el
      do j=JstrR,JendR
        do i=Istr,IendR
          sustr(i,j)=-val1*COS(val2*yr(i,j))
        enddo
      enddo
# elif defined BL_TEST
      Ewind=0.0_r8/rho0
      Nwind=0.3_r8/rho0
      do j=JstrR,JendR
        do i=IstrR,IendR
          sustr(i,j)=Ewind
        enddo
      enddo
# elif defined CANYON_A || defined CANYON_B
      do j=JstrR,JendR
        do i=Istr,IendR
          sustr(i,j)=5.0_e8-5*SIN(2.0_r8*pi*(tdays-dstart)/10.0_r8)*
     &               (1.0_r8-TANH((yr(i,j)-0.5_r8*el)/10000.0_r8))
        enddo
      enddo
# elif defined DOUBLE_GYRE
      if ((tdays-dstart).le.2.0_r8) then
        windamp=-0.05_r8*SIN(pi*tdays/4.0_r8)/rho0
      else
        windamp=-0.05_r8/rho0
      endif
      val1=2.0_r8*pi/el
      do j=JstrR,JendR
        do i=Istr,IendR
          sustr(i,j)=windamp*COS(val1*yr(i,j))
        enddo
      enddo
# elif defined FLT_TEST
      do j=JstrR,JendR
        do i=Istr,IendR
          sustr(i,j)=1.0_e8-3
        enddo
      enddo
# elif defined LMD_TEST
      if (time.le.57600.0_r8) then
        windamp=-0.6_r8*SIN(pi*time/57600.0_r8)*
     &                  SIN(2.0_r8*pi/57600.0_r8)/rho0
      else
        windamp=0.0_r8
      endif
      do j=JstrR,JendR
        do i=Istr,IendR
          sustr(i,j)=windamp
        enddo
      enddo
# elif defined NJ_BIGHT
!!    windamp=0.086824313_r8
!!    winddir=0.5714286_r8
!!    if ((tdays-dstart).le.0.5_r8) then
!!      Ewind=windamp*winddir*SIN(pi*(tdays-dstart))/rho0
!!      Nwind=windamp*SIN(pi*(tdays-dstart))/rho0
!!    else
!!      Ewind=windamp*winddir/rho0
!!      Nwind=windamp/rho0
!!    endif
      if ((tdays-dstart).le.3.0_r8) then
         winddir=60.0_r8
         windamp=0.1_r8
      elseif (((tdays-dstart).gt.3.0_r8).and.
     &        ((tdays-dstart).le.4.0_r8)) then
         winddir= 60.0_r8*((tdays-dstart)-2.0_r8)-
     &           120.0_r8*((tdays-dstart)-2.0_r8)
         windamp=0.0_r8
      else
         winddir=-120.0_r8
         windamp=0.0_r8
      endif
      Ewind=windamp*COS(pi*winddir/180.0_r8)/rho0
      Nwind=windamp*SIN(pi*winddir/180.0_r8)/rho0
      do j=JstrR,JendR
        do i=Istr,IendR
!!        val1=0.5_r8*(angler(i-1,j)+angler(i,j))
!!        sustr(i,j)=Ewind*COS(val1)+Nwind*SIN(val1)
          sustr(i,j)=0.0_r8
        enddo
      enddo
# elif defined UPWELLING
      if ((tdays-dstart).le.2.0_r8) then
        windamp=-0.1_r8*SIN(pi*(tdays-dstart)/4.0_r8)/rho0
      else
        windamp=-0.1_r8/rho0
      endif
      do j=JstrR,JendR
        do i=Istr,IendR
          sustr(i,j)=windamp
        enddo
      enddo
# elif defined USWEST
      do j=JstrR,JendR
        do i=Istr,IendR
!!        val1=(latr(i,j)-latr(Lm/2,Mm/2))/20.0_r8
!!        sustr(i,j)=1.0_e8-4*val1
!!        sustr(i,j)=-1.0_e8-4
          sustr(i,j)=0.0_r8
        enddo
      enddo
# else
      do j=JstrR,JendR
        do i=Istr,IendR
          sustr(i,j)=0.0_r8
        enddo
      enddo
# endif
!
!---------------------------------------------------------------------
!  Set kinematic surface momentum flux (wind stress) component in the
!  ETA-direction (m2/s2) at horizontal V-points.
!---------------------------------------------------------------------
!
# if defined BL_TEST
      do j=JstrR,JendR
        do i=IstrR,IendR
          svstr(i,j)=Nwind
        enddo
      enddo
# elif defined LMD_TEST
      if (time.le.57600.0_r8) then
        windamp=-0.6_r8*SIN(pi*time/57600.0_r8)*
     &                  COS(2.0_r8*pi/57600.0_r8)/rho0
      else
        windamp=0.0_r8
      endif
      do j=Jstr,JendR
        do i=IstrR,IendR
          svstr(i,j)=windamp
        enddo
      enddo
# elif defined NJ_BIGHT
      do j=Jstr,JendR
        do i=IstrR,IendR
!!        val1=0.5_r8*(angler(i,j)+angler(i,j-1))
!!        svstr(i,j)=-Ewind*SIN(val1)+Nwind*COS(val1)
          svstr(i,j)=0.0_r8
        enddo
      enddo
# elif defined USWEST
      do j=Jstr,JendR
        do i=IstrR,IendR
          svstr(i,j)=-1.0_e8-4
        enddo
      enddo
# else
      do j=Jstr,JendR
        do i=IstrR,IendR
          svstr(i,j)=0.0_r8
        enddo
      enddo
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_u2d_tile (Istr,Iend,Jstr,Jend,
     &                        sustr(START_2D_ARRAY))
      call exchange_v2d_tile (Istr,Iend,Jstr,Jend,
     &                        svstr(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_SMFLUX */
#ifdef ANA_SPINNING
      subroutine ana_spinning (tile)
!
!=======================================================================
!                                                                      !
!  This subroutine sets time-varying rotation force as the sum of      !
!  Coriolis and Centripetal accelerations.  This is used in polar      !
!  coordinate applications (annulus grid).                             !
!                                                                      !
!=======================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_spinning_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_spinning_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "scalars.h"

      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
# ifdef LAB_CANYON
      REAL_TYPE
     &        Bu, Omega0, Omega1, Omega1_of_t, Ro, Ro_t, Rs, Width,
     &        d_rho_dz, d_Omega1_dt, fcor, hd, little_omega, time_fac
# endif
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Compute time-varying rotation force: Coriolis plus Centripetal
!  accelerations.
!---------------------------------------------------------------------
!
# ifdef LAB_CANYON
      Omega0=2.0_r8*pi/25.0_r8
      Width=0.20_r8
      Ro=0.10_r8
      Rs=0.55_r8
      little_omega=2.0_r8*pi/24.0_r8
      Bu=10.0_r8
      hd=0.125_r8
!
      fcor=2.0_r8*Omega0
      Omega1=fcor*Width*Ro/Rs
      Ro_t=little_omega/fcor
      d_rho_dz=(1000.0_r8*Bu/g)*(fcor*Width/hd)**2
      time_fac=1.0_r8+(Omega1/Omega0)*SIN(little_omega*time)
      Omega1_of_t=Omega1*SIN(little_omega*time) 
      d_Omega1_dt=Omega1*little_omega*COS(little_omega*time)
!
      do j=JstrR,JendR
        do i=IstrR,IendR
          fomn(i,j)=(f(i,j)*time_fac+
     &               SQRT(xr(i,j)*xr(i,j)+yr(i,j)*yr(i,j))*
     &               ((2.0_r8*Omega0+Omega1_of_t)*Omega1_of_t))*
     &              omn(i,j)
        enddo
      enddo
# endif
      return
      end
#endif /* ANA_SPINNING */
#ifdef ANA_SRFLUX
      subroutine ana_srflux (tile)
!
!=====================================================================
!                                                                    !
!  This subroutine sets kinematic surface solar shortwave radiation  !
!  flux "srflx" (degC m/s) using an analytical expression.           !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_srflux_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_srflux_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "forces.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
# ifdef ALBEDO
      INTEGER_TYPE
     &        iday, month, year
      REAL_TYPE
     &        Dangle, Hangle, LatRad, albedo, cff, hour, surfSR, yday,
     &        zenith
# endif
!
# include "set_bounds.h"
# ifdef ALBEDO
!
!---------------------------------------------------------------------
!  Compute shortwave radiation (degC m/s) under cloudless skies.
!  To obtain total incoming shortwave radiation multyply by
!  (1.0-0.6*c**3), where c is the fractional cloud cover.
!---------------------------------------------------------------------
!
!  Assume time is in modified Julian day.  Get hour and year day.
!
      call caldate (r_date,tdays,year,yday,month,iday,hour)
#  ifdef NJ_BIGHT
      if (hour.ge.4.0_r8) then
        hour=hour-4.0_r8
      else
        hour=hour+20.0_r8
      endif
#  endif
!
!  Estimate solar declination angle (radians).
!
      Dangle=23.44_r8*COS((172.0_r8-yday)*pi/180.0_r8)
      Dangle=Dangle*deg2rad
!
!  Compute hour angle (radians).
!
      Hangle=(12.0_r8-hour)*pi/12.0_r8
!
!  Estimate variation in optical thickness of the atmosphere over
!  the course of a day.
!
      surfSR=MAX(0.0_r8,Csolar*COS(Hangle))
!
!  Compute shortwave radiation flux.  Notice that flux is scaled
!  from W/m2 to degC m/s by dividing by (rho0*Cp).
!
      cff=surfSR/(rho0*Cp)
      do j=JstrR,JendR
        do i=IstrR,IendR
          LatRad=latr(i,j)*deg2rad
          zenith=ACOS(SIN(LatRad)*SIN(Dangle)+
     &                COS(LatRad)*COS(Dangle)*COS(Hangle))
          zenith=MIN(90.0_r8,zenith*rad2deg)
          albedo=0.03_r8+0.97_r8*EXP(-0.12_r8*(90.0_r8-zenith))
          srflx(i,j)=(1.0_r8-albedo)*cff
     &              *(1.0_r8-0.6_r8*cloud(i,j)*cloud(i,j)*cloud(i,j))
        enddo
      enddo
# else
!
!---------------------------------------------------------------------
!  Set incoming solar shortwave radiation (W/m2).
!---------------------------------------------------------------------
!
      do j=JstrR,JendR
        do i=IstrR,IendR
          srflx(i,j)=0.0_r8
        enddo
      enddo
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        srflx(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_SRFLUX */
#if defined ANA_SSH && defined ZCLIMATOLOGY
      subroutine ana_ssh (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets analytical sea surface height climatology.      !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_ssh_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_ssh_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "clima.h"
# include "mask.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set sea surface height (meters).
!---------------------------------------------------------------------
!
      do j=JstrR,JendR
        do i=IstrR,IendR
          ssh(i,j)=???
        enddo
      enddo
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        ssh(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_SSH && ZCLIMATOLOGY */
#if defined SALINITY    && defined ANA_SSS && \
   (defined SCORRECTION || defined SRELAXATION)
      subroutine ana_sss (tile)
!
!=====================================================================
!                                                                    !
!  This subroutine sets sea surface salinity SST (PSU) which is      !
!  used for surface water flux correction.                           !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_sss_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_sss_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "forces.h"
# include "grid.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set sea surface salinity (PSU).
!---------------------------------------------------------------------
!
      do j=JstrR,JendR
        do i=IstrR,IendR
          sss(i,j)=???
        enddo
      enddo
      ANA_SST: no values provided for SST.
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        sss(START_2D_ARRAY))
# endif
      return
      end
#endif /* SALINITY && ANA_SSS && (SCORRECTION || SRELAXATION) */
#if defined ANA_SST && defined QCORRECTION
      subroutine ana_sst (tile)
!
!=====================================================================
!                                                                    !
!  This subroutine sets sea surface temperature SST  (Celsius)  and  !
!  surface net heat flux sensitivity dQdSTT to SST using analytical  !
!  expressions.  The forcing dQdSTT is usually computed in units of  !
!  (Watts/m2/degC).  It needs to be scaled to (m/s) by dividing by   !
!  rho0*Cp.  These forcing fields are used  when flux correction is  !
!  activated:                                                        !
!                                                                    !
!       Q_model ~ Q + dQdSST * (T_model - SST)                       !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_sst_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_sst_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "forces.h"
# include "grid.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set sea surface temperature (Celsius) and heat flux sensitivity to
!  SST (Watts/m2).
!---------------------------------------------------------------------
!
      do j=JstrR,JendR
        do i=IstrR,IendR
          sst(i,j)=???
          dqdt(i,j)=???
        enddo
      enddo
      ANA_SST: no values provided for SST and DQDT.
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        sst(START_2D_ARRAY))
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        dqdt(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_SST && QCORRECTION */
#if defined ANA_STFLUX || defined ANA_SSFLUX || defined ANA_SPFLUX
      subroutine ana_stflux (tile,itrc)
!
!=====================================================================
!                                                                    !
!  This routine sets kinematic surface flux of tracer type variables !
!  "stflx" (tracer units m/s) using analytical expressions.          !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        itrc, tile
# include "param.h"
# include "tile.h"
!
      call ana_stflux_tile (Istr,Iend,Jstr,Jend,itrc)
      return
      end
!
!*********************************************************************
      subroutine ana_stflux_tile (Istr,Iend,Jstr,Jend,itrc)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "forces.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, itrc, j
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set kinematic surface heat flux (degC m/s) at horizontal
!  RHO-points.
!---------------------------------------------------------------------
!
      if (itrc.eq.itemp) then
        do j=JstrR,JendR
          do i=IstrR,IendR
# ifdef BL_TEST
            stflx(i,j,itrc)=srflx(i,j)
# else
            stflx(i,j,itrc)=0.0_r8
# endif
          enddo
        enddo
!
!---------------------------------------------------------------------
!  Set kinematic surface freshwater flux (m/s) at horizontal
!  RHO-points, scaling by surface salinity is done in STEP3D.
!---------------------------------------------------------------------
!
      elseif (itrc.eq.isalt) then
        do j=JstrR,JendR
          do i=IstrR,IendR
            stflx(i,j,itrc)=0.0_r8
          enddo
        enddo
!
!---------------------------------------------------------------------
!  Set kinematic surface flux (T m/s) of passive tracers, if any.
!---------------------------------------------------------------------
!
      else
        do j=JstrR,JendR
          do i=IstrR,IendR
            stflx(i,j,itrc)=0.0_r8
          enddo
        enddo
      endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        stflx(START_2D_ARRAY,itrc))
# endif
      return
      end
#endif /* ANA_STFLUX || ANA_SSFLUX || ANA_SPFLUX */
#if defined ANA_TAIR && defined BULK_FLUXES
      subroutine ana_tair (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets surface air temperature (degC) using an         !
!  analytical expression.                                            !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_tair_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_tair_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "forces.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set analytical surface air temperature (degC).
!---------------------------------------------------------------------
!
# ifdef BL_TEST
      do j=JstrR,JendR
        do i=IstrR,IendR
          Tair(i,j)=23.567_r8
        enddo
      enddo
# else
      ANA_TAIR: no values provided for TAIR.
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        Tair(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_TAIR && BULK_FLUXES */
#if defined ANA_TCLIMA && defined TCLIMATOLOGY
      subroutine ana_tclima (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets analytical tracer climatology fields.           !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_tclima_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_tclima_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "clima.h"
# include "grid.h"
# include "ocean.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set tracer climatology.
!---------------------------------------------------------------------
!
      do k=1,N
        do j=JstrR,JendR
          do i=IstrR,IendR
            tclm(i,j,k,itemp)=???
            tclm(i,j,k,isalt)=???
          enddo
        enddo
      enddo
# if defined EW_PERIODIC || defined NS_PERIODIC
      do itrc=1,NAT
        call exchange_r3d_tile (Istr,Iend,Jstr,Jend,
     &                          tclm(START_2D_ARRAY,1,itrc))
      enddo
# endif
      return
      end
#endif /* ANA_TCLIMA && TCLIMATOLOGY */
#ifdef ANA_TOBC
      subroutine ana_tobc (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets tracer-type variables open boundary conditions  !
!  using analytical expressions.                                     !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_tobc_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_tobc_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "boundary.h"
# include "grid.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Tracers open boundary conditions.
!---------------------------------------------------------------------
!
# ifdef NJ_BIGHT
      if (EASTERN_EDGE) then
        do k=1,N
          do j=JstrR,JendR
            if (z_r(L,j,k).ge.-15.0_r8) then
              t_east(j,k,itemp)= 2.04926425772840_e8+01-z_r(L,j,k)*
     &                          (2.64085084879392_e8-01+z_r(L,j,k)*
     &                          (2.75112532853521_e8-01+z_r(L,j,k)*
     &                          (9.20748976164887_e8-02+z_r(L,j,k)*
     &                          (1.44907572574284_e8-02+z_r(L,j,k)*
     &                          (1.07821568591208_e8-03+z_r(L,j,k)*
     &                          (3.24031805390397_e8-05+
     &                           1.26282685769027_e8-7*z_r(L,j,k)))))))
              t_east(j,k,isalt)= 3.06648914919313_e8+01-z_r(L,j,k)*
     &                          (1.47672526294673_e8-01+z_r(L,j,k)*
     &                          (1.12645576031340_e8-01+z_r(L,j,k)*
     &                          (3.90092328187102_e8-02+z_r(L,j,k)*
     &                          (6.93901493744710_e8-03+z_r(L,j,k)*
     &                          (6.60443669679294_e8-04+z_r(L,j,k)*
     &                          (3.19179236195422_e8-05+
     &                           6.17735263440932_e8-7*z_r(L,j,k)))))))
            else
              t_east(j,k,itemp)=14.6_r8+
     &                          6.70_r8*TANH(1.1_r8*z_r(L,j,k)+15.9_r8)
              t_east(j,k,isalt)=31.3_r8-
     &                          0.55_r8*TANH(1.1_r8*z_r(L,j,k)+15.9_r8)
            endif
          enddo
        enddo
      endif
      if (SOUTHERN_EDGE) then
        do k=1,N
          do i=IstrR,IendR
            if (z_r(i,0,k).ge.-15.0_r8) then
              t_south(i,k,itemp)= 2.04926425772840_e8+01-z_r(i,0,k)*
     &                           (2.64085084879392_e8-01+z_r(i,0,k)*
     &                           (2.75112532853521_e8-01+z_r(i,0,k)*
     &                           (9.20748976164887_e8-02+z_r(i,0,k)*
     &                           (1.44907572574284_e8-02+z_r(i,0,k)*
     &                           (1.07821568591208_e8-03+z_r(i,0,k)*
     &                           (3.24031805390397_e8-05+
     &                            1.26282685769027_e8-7*z_r(i,0,k)))))))
              t_south(i,k,isalt)= 3.06648914919313_e8+01-z_r(i,0,k)*
     &                           (1.47672526294673_e8-01+z_r(i,0,k)*
     &                           (1.12645576031340_e8-01+z_r(i,0,k)*
     &                           (3.90092328187102_e8-02+z_r(i,0,k)*
     &                           (6.93901493744710_e8-03+z_r(i,0,k)*
     &                           (6.60443669679294_e8-04+z_r(i,0,k)*
     &                           (3.19179236195422_e8-05+
     &                            6.17735263440932_e8-7*z_r(i,0,k)))))))
            else
              t_south(i,k,itemp)=14.6_r8+
     &                           6.70_r8*TANH(1.1_r8*z_r(i,0,k)+15.9_r8)
              t_south(i,k,isalt)=31.3_r8-
     &                           0.55_r8*TANH(1.1_r8*z_r(i,0,k)+15.9_r8)
            endif
          enddo
        enddo
      endif
# elif defined ISWAKE
! nothing, because it is defined by ana_initial, and kept constant.
# else
      ANA_TOBC: no values provided for tracers.
# endif
      return
      end
#endif /* ANA_TOBC */
#ifdef ANA_VMIX
      subroutine ana_vmix (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets vertical mixing coefficients for momentum "Akv" !
!  and tracers "Akt" (m2/s) using analytical expressions.            !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_vmix_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_vmix_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "grid.h"
# include "mixing.h"
# include "ocean.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j, k
!
#include "set_bounds.h"
!
!--------------------------------------------------------------------
!  Set vertical viscosity coefficient (m2/s).
!--------------------------------------------------------------------
!
# if defined CANYON_B
      do k=1,Nm
        do j=JstrR,JendR
          do i=IstrR,IendR
            Akv(i,j,k)=1.0_e8-3+95.0_e8-4*EXP(z_w(i,j,k)/50.0_r8)+
     &                 95.0_e8-4*EXP(-(z_w(i,j,k)+h(i,j))/50.0_r8)
          enddo
        enddo
      enddo
# elif defined NJ_BIGHT
      do k=1,Nm
        do j=JstrR,JendR
          do i=IstrR,IendR
            Akv(i,j,k)=1.0_e8-3+2.0_e8-4*EXP(z_r(i,j,k)/10.0_r8)
          enddo
        enddo
      enddo
# elif defined SED_TEST1
      do k=1,Nm                         !  vonkar*ustar*z*(1-z/D)
        do j=JstrR,JendR
          do i=IstrR,IendR
            Akv(i,j,k)=0.025_r8*(h(i,j)+z_w(i,j,k))* 
     &                 (1.0_r8-(h(i,j)+z_w(i,j,k))/
     &                  (h(i,j)+zeta(i,j,knew)))
            Akt(i,j,k,itemp)=Akv(i,j,k)*0.49_r8/0.39_r8
            Akt(i,j,k,isalt)=Akt(i,j,k,itemp) 
          enddo
        enddo
      enddo
# elif defined UPWELLING
      do k=1,Nm
        do j=JstrR,JendR
          do i=IstrR,IendR
            Akv(i,j,k)=2.0_e8-3+8.0_e8-3*EXP(z_w(i,j,k)/150.0_r8)
          enddo
        enddo
      enddo
# else
      ANA_VMIX: no values provided for AKV.
# endif /* CANYON_B */
!
!---------------------------------------------------------------------
!  Set vertical diffusion coefficient (m2/s).
!---------------------------------------------------------------------
!
# if defined CANYON_B
      do k=1,Nm
        do j=JstrR,JendR
          do i=IstrR,IendR
            Akt(i,j,k,itemp)=Akt_bak(itemp)
          enddo
        enddo
      enddo
# elif defined NJ_BIGHT
      do k=1,Nm
        do j=JstrR,JendR
          do i=IstrR,IendR
            Akt(i,j,k,itemp)=1.0_e8-5+
     &                       2.0_e8-6*EXP(z_r(i,j,k)/10.0_r8)
            Akt(i,j,k,isalt)=Akt(i,j,k,itemp)
          enddo
        enddo
      enddo
# elif defined UPWELLING
      do k=1,Nm
        do j=JstrR,JendR
          do i=IstrR,IendR
            Akt(i,j,k,itemp)=Akt_bak(itemp)
            Akt(i,j,k,isalt)=Akt_bak(isalt)
          enddo
        enddo
      enddo
# else
      ANA_VMIX: no values provided for AKT.
# endif
      return
      end
#endif /* ANA_VMIX */
#if defined ANA_WINDS && defined BULK_FLUXES
      subroutine ana_winds (tile)
!
!=====================================================================
!                                                                    !
!  This routine sets surface wind components using an analytical     !
!  expression.                                                       !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_winds_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_winds_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "forces.h"
# include "grid.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i,j
      REAL_TYPE
     &        Wdir, Wmag, u_wind, v_wind
!
# include "set_bounds.h"
!
      Wdir=0.0_r8
      Wmag=0.0_r8
      u_wind=0.0_r8
      v_wind=0.0_r8
!
!---------------------------------------------------------------------
!  Set surface wind components (m/s) at RHO-points.
!---------------------------------------------------------------------
!
# ifdef BL_TEST
      if ((tdays-dstart).le.6.0_r8) then
        u_wind=0.0_r8
!!      v_wind=4.7936_r8
        v_wind=10.0_r8
      endif
      do j=JstrR,JendR
        do i=IstrR,IendR
          Uwind(i,j)=u_wind
          Vwind(i,j)=v_wind
        enddo
      enddo
# else
      ANA_WINDS: no values provided for UWIND and VWIND.
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        Uwind(START_2D_ARRAY))
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        Vwind(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_WINDS && BULK_FLUXES */
#if defined ANA_WWAVE && defined BBL
      subroutine ana_wwave (tile)
!
!=====================================================================
!                                                                    !
!  This subroutine sets wind induced wave amplitude, direction and   !
!  period to be used in the bottom boundary layer formulation.       !
!                                                                    !
!=====================================================================
!
      implicit none
      INTEGER_TYPE
     &        tile
# include "param.h"
# include "tile.h"
!
      call ana_wwave_tile (Istr,Iend,Jstr,Jend)
      return
      end
!
!*********************************************************************
      subroutine ana_wwave_tile (Istr,Iend,Jstr,Jend)
!*********************************************************************
!
      implicit none
# include "param.h"
# include "forces.h"
# include "grid.h"
# include "scalars.h"
!
      INTEGER_TYPE
     &        Iend, Istr, Jend, Jstr, i, j
      REAL_TYPE
     &        cff, wdir
!
# include "set_bounds.h"
!
!---------------------------------------------------------------------
!  Set wind induced wave amplitude (m), direction (radians) and
!  period (s) at RHO-points.
!---------------------------------------------------------------------
!
# if defined NJ_BIGHT
!!    wdir=210.0_r8*deg2rad
      wdir=150.0_r8*deg2rad
      if ((tdays-dstart).lt.1.5_r8) then
        cff=TANH(0.5_r8*(tdays-dstart))
        cff=1.0_r8
      else
        cff=1.0_r8
      endif
      do j=JstrR,JendR
        do i=IstrR,IendR
          Awave(i,j)=0.5_r8
          Dwave(i,j)=wdir-angler(i,j)
          Pwave(i,j)=10.0_r8
        enddo
      enddo
# elif defined BL_TEST
      wdir=210.0_r8*deg2rad
      do j=JstrR,JendR
        do i=IstrR,IendR
          Awave(i,j)=0.5_r8
          Dwave(i,j)=wdir
          Pwave(i,j)=8.0_r8
        enddo
      enddo
# else
      ANA_WWAVE: no values provided for AWAVE, DWAVE, and PWAVE.
# endif
# if defined EW_PERIODIC || defined NS_PERIODIC
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        Awave(START_2D_ARRAY))
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        Dwave(START_2D_ARRAY))
      call exchange_r2d_tile (Istr,Iend,Jstr,Jend,
     &                        Pwave(START_2D_ARRAY))
# endif
      return
      end
#endif /* ANA_WWAVE && BBL */
